﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nvngdSupport/FrameBufferManager.h>
#include <nvn/nvn_FuncPtrInline.h>

/**
 * @examplesource{FrameBufferManager.cpp,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines 2 classes:
 *      - A pure virtual class that multi buffered memory managers
 *        derive from.
 *      - A simple manager class that handles safely fencing off
 *        memory used by class that derive from the above.
 *
 *  After all the memory managers have finished using the memory they
 *  have access to this frame, InsertFence() should be called on the
 *  Sync Manager.  This will insert and flush an NVN sync into the
 *  queue that will be waited on in a later frame.  SwapPools()
 *  should then be called, moving each memory manager to use the next
 *  chunk of memory it will use in the next frame, while also waiting
 *  on the sync that was set in an earlier frame to ensure that the
 *  memory is safe to start overwriting.
 */

/*
 * FrameBufferedSyncManager Constructor
 * ------------------------------------
 * Initialize the sync manager with the number of syncs
 * requested (default = 2)
 */
FrameBufferedSyncManager::FrameBufferedSyncManager(NVNdevice* pDevice, NVNqueue* pQueue, int numSyncs /* = 2 */)
    :
    m_pDevice(pDevice),
    m_pQueue(pQueue),
    m_NumSyncs(numSyncs),
    m_pSyncs(NULL),
    m_CurrentChunk(0)
{
    NN_ASSERT(pDevice && pQueue, "Device and/or Queue pointer NULL\n");
    NN_ASSERT(numSyncs > 0, "Zero or negative number of syncs requested\n");

        /* Initialize the sync objects */
    m_pSyncs = new NVNsync[numSyncs];
    for (int i = 0; i < numSyncs; ++i)
    {
        nvnSyncInitialize(&m_pSyncs[i], m_pDevice);
        nvnQueueFenceSync(m_pQueue,
            &m_pSyncs[i],
            NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE,
            NVN_SYNC_FLAG_FLUSH_FOR_CPU_BIT);
    }
}

/*
 * FrameBufferedSyncManager::RegisterMemoryManager
 * -----------------------------------------------
 * Registers a memory manager to have its memory protected
 * by an NVN sync.  The number of chunks of memory that each
 * manager uses must be equivalent to the number of syncs
 * this class was initialized with.
 */
void FrameBufferedSyncManager::RegisterMemoryManager(FrameBufferedMemoryManager* pMemoryManager)
{
    NN_ASSERT(pMemoryManager != NULL);
    NN_ASSERT(m_NumSyncs == pMemoryManager->GetNumChunks());

    m_MemoryManagers.push_back(pMemoryManager);
}

/*
 * FrameBufferedSyncManager::SwapPools
 * -----------------------------------
 * Swaps each registered memory manager's memory with
 * the next chunk it has available for use in the next
 * frame.  The NVN syncs set in InsertFence are waited
 * on here.
 */
void FrameBufferedSyncManager::SwapPools()
{
    m_CurrentChunk = (m_CurrentChunk + 1) % m_NumSyncs;

        /* When sync is hit we can safely overwrite the memory protected by the sync */
    nvnQueueFlush(m_pQueue);
    NVNsyncWaitResult result = nvnSyncWait(&m_pSyncs[m_CurrentChunk], NVN_WAIT_TIMEOUT_MAXIMUM);
    NN_ASSERT(result == NVN_SYNC_WAIT_RESULT_ALREADY_SIGNALED ||
        result == NVN_SYNC_WAIT_RESULT_CONDITION_SATISFIED, "Failed to wait on sync in Sync Manager\n");

    for (uint32_t i = 0; i < m_MemoryManagers.size(); ++i)
    {
        m_MemoryManagers[i]->SwapPools();
    }
}

/*
 * FrameBufferedSyncManager::InsertFence
 * -------------------------------------
 * Inserts an NVN sync to protect the memory used
 * this frame.
 */
void FrameBufferedSyncManager::InsertFence()
{
    /* Insert new sync */
    nvnQueueFenceSync(m_pQueue,
        &m_pSyncs[m_CurrentChunk],
        NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE,
        NVN_SYNC_FLAG_FLUSH_FOR_CPU_BIT);
}

/*
 * FrameBufferedSyncManager Destructor
 * -----------------------------------
 * Destroys the syncs created by this class.
 */
FrameBufferedSyncManager::~FrameBufferedSyncManager()
{
    if (m_pSyncs)
    {
        for (int i = 0; i < m_NumSyncs; ++i)
        {
            nvnSyncFinalize(&m_pSyncs[i]);
        }

        delete[] m_pSyncs;
    }
}

/*
 * FrameBufferedMemoryManager Virtual Destructor
 * ---------------------------------------------
 * Virtual destructor for abstract class
 */
FrameBufferedMemoryManager::~FrameBufferedMemoryManager()
{
}
