﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
/**
 * @examplesource{FrameBufferManager.h,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines 2 classes:
 *      - An abstract class that multi buffered memory managers
 *        derive from.
 *      - A simple manager class that handles safely fencing off
 *        memory used by class that derive from the above.
 *
 *  After all the memory managers have finished using the memory they
 *  have access to this frame, InsertFence() should be called on the
 *  Sync Manager.  This will insert and flush an NVN sync into the
 *  queue that will be waited on in a later frame.  SwapPools()
 *  should then be called, moving each memory manager to use the next
 *  chunk of memory it will use in the next frame, while also waiting
 *  on the sync that was set in an earlier frame to ensure that the
 *  memory is safe to start overwriting.
 */

#include <vector>
#include <nvn/nvn.h>

class FrameBufferedMemoryManager
{
public:
    virtual void SwapPools() = 0;
    virtual int GetNumChunks() = 0;

    virtual ~FrameBufferedMemoryManager();
};

class FrameBufferedSyncManager
{
public:
    FrameBufferedSyncManager(NVNdevice* pDevice, NVNqueue* pQueue, int numSyncs = 2);
    ~FrameBufferedSyncManager();

    void RegisterMemoryManager(FrameBufferedMemoryManager* pMemoryManager);

    void SwapPools();
    void InsertFence();

private:
    NVNdevice* m_pDevice;
    NVNqueue* m_pQueue;

    int         m_NumSyncs;
    NVNsync*    m_pSyncs;
    int         m_CurrentChunk;
    std::vector<FrameBufferedMemoryManager*> m_MemoryManagers;
};
