﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/hid.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_Vector.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <nvn/nvn_FuncPtrImpl.h>
#ifdef _WIN32
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nn/nn_Windows.h>
#endif
#include <nvngdSupport/TutorialBaseClass.h>
#include <nvngdSupport/ShaderHeaders/SimpleTexturedModelDataHelper.h>
#include <nvngdSupport/AssetFileLoadingHelper.h>
#include <nvngdSupport/TutorialUtil.h>
#include <nvngdSupport/TextureIDManager.h>
#include <nvngdSupport/UniformBuffer.h>
#include <nvngdSupport/StatTracker.h>
#include <nvngdSupport/DebugTextRenderer.h>
#include <lopProfiler/LOP_Profiler.h>

static const size_t g_CommandMemorySize = 4096;
static const size_t g_ControlMemorySize = 1024;
static const int    g_NumColorBuffers   = 2;
static int          g_TessellationLevel = 0;

#define TEST_LOP

class TestLop_PrimitiveFlow : public TutorialBaseClass
{
    NN_DISALLOW_COPY(TestLop_PrimitiveFlow);

public:
    TestLop_PrimitiveFlow();
    virtual ~TestLop_PrimitiveFlow();
    virtual void Init(PFNNVNBOOTSTRAPLOADERPROC pLoader, NVNnativeWindow nativeWindow);
    virtual void Shutdown();

    virtual void Draw(uint64_t millisec);
    virtual bool TestDraw(uint64_t millisec);
    virtual void Resize(int width, int height);

    void SetContinueRendering(bool c)
    {
        m_ContinueRendering = c;
    }

    bool ContinueRendering()
    {
        return m_ContinueRendering;
    }

    std::vector<std::pair<std::string, double> >& GetMetricValues()
    {
        return m_MetricValues;
    }

private:
    static void NVNAPIENTRY DebugLayerCallback(
        NVNdebugCallbackSource source,
        NVNdebugCallbackType type,
        int id,
        NVNdebugCallbackSeverity severity,
        const char* message,
        void* pUser
    );

    void SetupUniformBlock();
    void UpdateUniformBlock(const nn::util::Matrix4x4fType& modelMatrix, const nn::util::Matrix4x4fType& projectionMatrix);
    int UpdateRenderTargets();
    bool UpdateTessellation();

    NVNdevice               m_Device;
    NVNqueue                m_Queue;
    void*                   m_pQueueMemory;

    MemoryPool*             m_pCommandMemoryPool;

    ptrdiff_t               m_CommandPoolOffset;
    void*                   m_pControlPool;
    NVNcommandBuffer        m_CommandBuffer;
    NVNcommandHandle        m_CommandHandle;

    NVNtextureBuilder       m_RenderTargetBuilder;
    NVNtexture*             m_pRenderTargets[g_NumColorBuffers];
    NVNtexture*             m_pDepthBuffer;
    NVNsamplerBuilder       m_SamplerBuilder;
    NVNbufferBuilder        m_BufferBuilder;

    NVNsync                 m_RenderTargetPresentSync;
    NVNsync                 m_CommandBufferSync;
    NVNsync                 m_UniformBlockUpdateSync;
    AssetFileLoadingHelper* m_pAssetLoader;
    AssetFileDataHolder*    m_pDataHolder;

    NVNblendState           m_BlendState;
    NVNchannelMaskState     m_ChannelMaskState;
    NVNcolorState           m_ColorState;
    NVNdepthStencilState    m_DepthStencilState;
    NVNmultisampleState     m_MultisampleState;
    NVNpolygonState         m_PolygonState;

    MemoryPool*             m_pUniformBufferMemoryPool;
    UniformBuffer           m_UniformBlockVertex;
    UniformBuffer           m_UniformBlockTessControl;
    TextureIDManager*       m_pTextureIDManager;

    MemoryPool*             m_pRenderTargetMemoryPool;

    ptrdiff_t               m_RenderTargetCommandPoolOffset;
    void*                   m_pRenderTargetControlPool;
    NVNcommandBuffer        m_RenderTargetCommandBuffer;
    NVNcommandHandle        m_RenderTargetCommandHandle;

    NVNwindow*              m_pWindow;
    NVNwindowBuilder        m_WindowBuilder;
    int                     m_CurrentWindowIndex;
    NVNsync                 m_WindowSync;

    int                     m_CommandMemoryUsed;
    int                     m_ControlMemoryUsed;

    int                     m_ScreenWidth;
    int                     m_ScreenHeight;
    float                   m_RotY;

    size_t                  m_ColorTargetSize;
    size_t                  m_DepthTargetSize;

    DebugTextRenderer*      m_pDebugTextRenderer;
    FrameBufferedSyncManager*   m_pFrameBufferedSyncManager;
    std::vector<std::pair<float, float> > m_TessellationLevels;

#ifdef USE_LOP
    lop::LOP_Profiler*                              m_pProfiler;
    std::vector<uint8_t>                            m_ConfigImage;
    std::vector<lop::MetricSpec>                    m_SelectedMetrics;
    std::vector<std::pair<std::string, double> >    m_MetricValues;
    bool                                            m_ContinueRendering;
#endif
};

/*
* AssetFileLoading Constructor
* ----------------------------
* Sets up default values for the members of the class.
*/
TestLop_PrimitiveFlow::TestLop_PrimitiveFlow() :
    m_pQueueMemory(NULL),
    m_pCommandMemoryPool(NULL),
    m_pControlPool(NULL),
    m_CommandHandle(0),
    m_pDepthBuffer(NULL),
    m_pAssetLoader(NULL),
    m_pDataHolder(NULL),
    m_pUniformBufferMemoryPool(NULL),
    m_pTextureIDManager(NULL),
    m_pRenderTargetMemoryPool(NULL),
    m_pRenderTargetControlPool(NULL),
    m_RenderTargetCommandHandle(NULL),
    m_pWindow(NULL),
    m_CurrentWindowIndex(-1),
    m_CommandMemoryUsed(0),
    m_ControlMemoryUsed(0),
    m_ScreenWidth(0),
    m_ScreenHeight(0),
    m_RotY(45.0f),
    m_ColorTargetSize(0),
    m_DepthTargetSize(0),
    m_pDebugTextRenderer(NULL),
    m_pFrameBufferedSyncManager(NULL)
#ifdef USE_LOP
    , m_pProfiler(NULL)
    , m_ContinueRendering(true)
#endif
{
    for (int i = 0; i < g_NumColorBuffers; ++i)
    {
        m_pRenderTargets[i] = NULL;
    }
}

/*
* AssetFileLoading Destructor
* ---------------------------
* Empty destructor.
*/
TestLop_PrimitiveFlow::~TestLop_PrimitiveFlow()
{
}

/*
* AssetFileLoading::SetupUniformBlock
* -----------------------------------
* Initializes the uniform buffers used for the
* shader program.
*/
void TestLop_PrimitiveFlow::SetupUniformBlock()
{
    m_pUniformBufferMemoryPool = new MemoryPool();

    int bufferAlignment = 0;
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_UNIFORM_BUFFER_ALIGNMENT, &bufferAlignment);
    m_pUniformBufferMemoryPool->Init(
        NULL,
        Align(sizeof(SimpleTexturedModel::BlockVSUniformBlockData), bufferAlignment) +
        Align(sizeof(SimpleTexturedModel::BlockFSUniformBlockData), bufferAlignment),
        NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT,
        &m_Device);

    nvnMemoryPoolSetDebugLabel(m_pUniformBufferMemoryPool->GetMemoryPool(), "TessellationTest_Uniform_Buffer_Memory_Pool");

    /*!
    * Initialize the uniform buffers with a pointer to the
    * device and the appropriate size for the shader.
    */
    m_UniformBlockVertex.Init(&m_Device, sizeof(SimpleTexturedModel::BlockVSUniformBlockData), m_pUniformBufferMemoryPool);
    m_UniformBlockTessControl.Init(&m_Device, 8, m_pUniformBufferMemoryPool);
}

/*
* AssetFileLoading::UpdateUniformBlock
* ------------------------------------
* Updates the uniform blocks with the given data.
*/
void TestLop_PrimitiveFlow::UpdateUniformBlock(const nn::util::Matrix4x4fType& modelMatrix, const nn::util::Matrix4x4fType& projectionMatrix)
{
    /*! Creates an instance of the vertex shader's helper class. */
    SimpleTexturedModel::BlockVSUniformBlockData vertexShaderData;

    /*!
    * Sets the data in the helper class. The helper class functions
    * automatically handle placing the data at the correct offset
    * in the uniform block.
    */
    nn::util::Float4x4 temp;
    nn::util::MatrixStore(&temp, modelMatrix);
    vertexShaderData.SetUniform_u_modelMtx(*reinterpret_cast<const float(*)[16]>(&temp));

    nn::util::Matrix4x4fType cameraMat;
    nn::util::MatrixIdentity(&cameraMat);
    nn::util::MatrixStore(&temp, cameraMat);
    vertexShaderData.SetUniform_u_viewMtx(*reinterpret_cast<const float(*)[16]>(&temp));

    nn::util::MatrixStore(&temp, projectionMatrix);
    vertexShaderData.SetUniform_u_projMtx(*reinterpret_cast<const float(*)[16]>(&temp));

    /*! Copy the set up data into the uniform buffer. */
    m_UniformBlockVertex.SetData(&vertexShaderData, sizeof(vertexShaderData));

    std::pair<float, float>& TessellationLevels = m_TessellationLevels[g_TessellationLevel];
    float TessellationControlValues[2] = { TessellationLevels.first, TessellationLevels.second };
    m_UniformBlockTessControl.SetData(TessellationControlValues, sizeof(TessellationControlValues));
}

#if defined (NN_BUILD_TARGET_PLATFORM_OS_WIN )
//#define NVN_GRAPHICS_DEBUGGER
#endif

#ifdef NVN_GRAPHICS_DEBUGGER
extern "C" PFNNVNGENERICFUNCPTRPROC nvnDeviceGetToolsProcAddress(NVNdevice *device, const char* name);
#endif

/*!
* AssetFileLoading::Init
* ----------------------
* Initialize NVN, load asset files, and create objects needed for the
* application to run.
*/
void TestLop_PrimitiveFlow::Init(PFNNVNBOOTSTRAPLOADERPROC pLoader, NVNnativeWindow nativeWindow)
{
#ifdef NVN_GRAPHICS_DEBUGGER
    NN_UNUSED(pLoader);
    /* Load procs from NVNGD instead of from the driver, this must be called once prior to nvnDeviceInitialize. */
    nvnLoadCProcs(nullptr, (PFNNVNDEVICEGETPROCADDRESSPROC)nvnDeviceGetToolsProcAddress);

    NVNdeviceBuilder deviceBuilder;
    nvnDeviceBuilderSetDefaults(&deviceBuilder);
    int deviceFlags = NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_4_BIT;
    nvnDeviceBuilderSetFlags(&deviceBuilder, deviceFlags);

    if (nvnDeviceInitialize(&m_Device, &deviceBuilder) == false)
    {
        NN_ASSERT(0, "nvnDeviceInitialize");
    }

    nvnDeviceSetDebugLabel(&m_Device, "TessellationTest_Device");

    /* Never call nvnLoadCProcs after nvnDeviceInitialize. */
#else /* NVN_GRAPHICS_DEBUGGER */
    pfnc_nvnDeviceInitialize = reinterpret_cast<PFNNVNDEVICEINITIALIZEPROC>((*pLoader)("nvnDeviceInitialize"));
    pfnc_nvnDeviceGetProcAddress = reinterpret_cast<PFNNVNDEVICEGETPROCADDRESSPROC>((*pLoader)("nvnDeviceGetProcAddress"));
    if (!pfnc_nvnDeviceInitialize)
    {
        /* This can happen if an NVN driver is not installed on a Windows PC. */
        NN_ASSERT(0, "BootstrapLoader failed to find nvnDeviceInitialize");
    }

    nvnLoadCProcs(NULL, pfnc_nvnDeviceGetProcAddress);

    int MajorVersion, MinorVersion;
    nvnDeviceGetInteger(NULL, NVN_DEVICE_INFO_API_MAJOR_VERSION, &MajorVersion);
    nvnDeviceGetInteger(NULL, NVN_DEVICE_INFO_API_MINOR_VERSION, &MinorVersion);

    if (MajorVersion != NVN_API_MAJOR_VERSION || MinorVersion < NVN_API_MINOR_VERSION)
    {
        NN_ASSERT(0, "NVN SDK not supported by current driver.");
    }

    /* If debug or develop is enabled, turn on NVN's debug layer. */
    int deviceFlags = 0;
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    deviceFlags = NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_4_BIT;
#endif

    NVNdeviceBuilder deviceBuilder;
    nvnDeviceBuilderSetDefaults(&deviceBuilder);
    nvnDeviceBuilderSetFlags(&deviceBuilder, deviceFlags);

    if (nvnDeviceInitialize(&m_Device, &deviceBuilder) == false)
    {
        /*
        * This can fail for a few reasons; the most likely on Horizon is
        * insufficent device memory.
        */
        NN_ASSERT(0, "nvnDeviceInitialize");
    }

    nvnLoadCProcs(&m_Device, pfnc_nvnDeviceGetProcAddress);
#endif /* NVN_GRAPHICS_DEBUGGER */

    /*! Setup the debug callback for the debug layer. */
    if (deviceFlags & NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_4_BIT)
    {
        nvnDeviceInstallDebugCallback(
            &m_Device,
            reinterpret_cast<PFNNVNDEBUGCALLBACKPROC>(&DebugLayerCallback),
            NULL, // For testing purposes; any pointer is OK here.
            NVN_TRUE // NVN_TRUE = Enable the callback.
        );
    }

    /* Initialize the queue. */
    NVNqueueBuilder queueBuilder;
    nvnQueueBuilderSetDevice(&queueBuilder, &m_Device);
    nvnQueueBuilderSetDefaults(&queueBuilder);
    nvnQueueBuilderSetComputeMemorySize(&queueBuilder, 0);

    int minQueueCommandMemorySize = 0;
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_QUEUE_COMMAND_MEMORY_MIN_SIZE, &minQueueCommandMemorySize);
    nvnQueueBuilderSetCommandMemorySize(&queueBuilder, minQueueCommandMemorySize);
    nvnQueueBuilderSetCommandFlushThreshold(&queueBuilder, minQueueCommandMemorySize);

    size_t neededQueueMemorySize = nvnQueueBuilderGetQueueMemorySize(&queueBuilder);

    if ((neededQueueMemorySize % NVN_MEMORY_POOL_STORAGE_GRANULARITY) != 0)
    {
        NN_ASSERT(0, "Memory size reported for queue is not the proper granularity");
    }

#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
    m_pQueueMemory = NULL;
#else
    m_pQueueMemory = AlignedAllocate(neededQueueMemorySize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
#endif

    nvnQueueBuilderSetQueueMemory(&queueBuilder, m_pQueueMemory, neededQueueMemorySize);

    if (nvnQueueInitialize(&m_Queue, &queueBuilder) == false)
    {
        NN_ASSERT(0, "nvnQueueInitialize failed");
    }

    nvnQueueSetDebugLabel(&m_Queue, "TessellationTest_Queue");

    /*! Initialize the rendering command buffer. */
    if (nvnCommandBufferInitialize(&m_CommandBuffer, &m_Device) == NULL)
    {
        NN_ASSERT(0, "nvnCommandBufferInitialize");
    }

    nvnCommandBufferSetDebugLabel(&m_CommandBuffer, "TessellationTest_CommandBuffer");

    /*!
    * Queries the device for the proper control and command
    * memory alignment for a command buffer.
    */
    int commandBufferCommandAlignment = 0;
    int commandBufferControlAlignment = 0;
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_COMMAND_BUFFER_COMMAND_ALIGNMENT, &commandBufferCommandAlignment);
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_COMMAND_BUFFER_CONTROL_ALIGNMENT, &commandBufferControlAlignment);

    /*! Setup the command memory for the command buffers. */
    m_pCommandMemoryPool = new MemoryPool();
    m_pCommandMemoryPool->Init(NULL, Align(g_CommandMemorySize, commandBufferCommandAlignment) * 2, NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, &m_Device);

    nvnMemoryPoolSetDebugLabel(m_pCommandMemoryPool->GetMemoryPool(), "TessellationTest_CommandMemoryPool");

    /*! Grab the offset from the memory pool. */
    m_CommandPoolOffset = m_pCommandMemoryPool->GetNewMemoryChunkOffset(g_CommandMemorySize, commandBufferCommandAlignment);

    /*! Allocate control memory for the command buffer. */
    m_pControlPool = AlignedAllocate(g_ControlMemorySize, commandBufferControlAlignment);

    /*! Add the memory to the command buffer. */
    nvnCommandBufferAddCommandMemory(&m_CommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_CommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_CommandBuffer, m_pControlPool, g_ControlMemorySize);

    /*! Render Target Setting Command Buffer. */
    if (!nvnCommandBufferInitialize(&m_RenderTargetCommandBuffer, &m_Device))
    {
        NN_ASSERT(0, "Failed to initialize render target command buffer");
    }

    nvnCommandBufferSetDebugLabel(&m_RenderTargetCommandBuffer, "TessellationTest_RenderTargetCommandBuffer");

    /*! Grab the offset from the memory pool. */
    m_RenderTargetCommandPoolOffset = m_pCommandMemoryPool->GetNewMemoryChunkOffset(g_CommandMemorySize, commandBufferCommandAlignment);

    /*! Allocate control memory for the command buffer. */
    m_pRenderTargetControlPool = AlignedAllocate(g_ControlMemorySize, commandBufferControlAlignment);

    /*! Add the memory to the command buffer. */
    nvnCommandBufferAddCommandMemory(&m_RenderTargetCommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_RenderTargetCommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_RenderTargetCommandBuffer, m_pRenderTargetControlPool, g_ControlMemorySize);

    /*! Initialize the render target sync. */
    if (!nvnSyncInitialize(&m_RenderTargetPresentSync, &m_Device))
    {
        NN_ASSERT(0, "Failed to initialize render target present sync");
    }

    nvnSyncSetDebugLabel(&m_RenderTargetPresentSync, "TessellationTest_RenderTargetPresentSync");

    /*! Initialize the rendering commad buffer sync. */
    if (!nvnSyncInitialize(&m_CommandBufferSync, &m_Device))
    {
        NN_ASSERT(0, "Failed to initialize command buffer sync");
    }

    nvnSyncSetDebugLabel(&m_CommandBufferSync, "TessellationTest_CommandBufferSync");

    /*! Initialize the uniform buffer update sync. */
    if (!nvnSyncInitialize(&m_UniformBlockUpdateSync, &m_Device))
    {
        NN_ASSERT(0, "Failed to initialize uniform block update sync");
    }

    nvnSyncSetDebugLabel(&m_UniformBlockUpdateSync, "TessellationTest_UniformBlockUpdateSync");

    /*! Initialize the window sync. */
    if (!nvnSyncInitialize(&m_WindowSync, &m_Device))
    {
        NN_ASSERT(0, "Failed to initialize window sync");
    }

    nvnSyncSetDebugLabel(&m_WindowSync, "TessellationTest_WindowSync");

    nvnBlendStateSetDefaults(&m_BlendState);

    nvnChannelMaskStateSetDefaults(&m_ChannelMaskState);

    nvnColorStateSetDefaults(&m_ColorState);

    nvnDepthStencilStateSetDefaults(&m_DepthStencilState);
    nvnDepthStencilStateSetDepthTestEnable(&m_DepthStencilState, NVN_TRUE);
    nvnDepthStencilStateSetDepthWriteEnable(&m_DepthStencilState, NVN_TRUE);
    nvnDepthStencilStateSetDepthFunc(&m_DepthStencilState, NVN_DEPTH_FUNC_LESS);

    nvnMultisampleStateSetDefaults(&m_MultisampleState);

    nvnPolygonStateSetDefaults(&m_PolygonState);
    nvnPolygonStateSetFrontFace(&m_PolygonState, NVNfrontFace::NVN_FRONT_FACE_CW);
    nvnPolygonStateSetCullFace(&m_PolygonState, NVNface::NVN_FACE_BACK);
    nvnPolygonStateSetPolygonMode(&m_PolygonState, NVN_POLYGON_MODE_FILL);

    nvnSamplerBuilderSetDevice(&m_SamplerBuilder, &m_Device);
    nvnSamplerBuilderSetDefaults(&m_SamplerBuilder);

    nvnBufferBuilderSetDevice(&m_BufferBuilder, &m_Device);
    nvnBufferBuilderSetDefaults(&m_BufferBuilder);

    /*! Set up the texture builder for the render target. */
    nvnTextureBuilderSetDevice(&m_RenderTargetBuilder, &m_Device);
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_DISPLAY_BIT | NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, 1920, 1080);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_RGBA8);
    m_ColorTargetSize = nvnTextureBuilderGetStorageSize(&m_RenderTargetBuilder);

    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, 1920, 1080);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_DEPTH32F);
    m_DepthTargetSize = nvnTextureBuilderGetStorageSize(&m_RenderTargetBuilder);

    /*! Set up the render target memory pool. */
    m_pRenderTargetMemoryPool = new MemoryPool();
    m_pRenderTargetMemoryPool->Init(
        NULL,
        m_ColorTargetSize * g_NumColorBuffers + m_DepthTargetSize,
        NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT | NVN_MEMORY_POOL_FLAGS_COMPRESSIBLE_BIT,
        &m_Device);

    nvnMemoryPoolSetDebugLabel(m_pRenderTargetMemoryPool->GetMemoryPool(), "TessellationTest_RenderTargetMemoryPool");

    m_pTextureIDManager = new TextureIDManager(&m_Device);

    SetupUniformBlock();

    size_t cacheSize = 0;
    nn::Result result = nn::fs::QueryMountRomCacheSize(&cacheSize);
    NN_ASSERT(result.IsSuccess());

    char* mountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ASSERT_NOT_NULL(mountRomCacheBuffer);

    result = nn::fs::MountRom("rom", mountRomCacheBuffer, cacheSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    /*! Create a new asset loader. */
    m_pAssetLoader = new AssetFileLoadingHelper(&m_Device, m_pTextureIDManager);

    /*! Load in the asset file for the tutorial. */
    m_pDataHolder = m_pAssetLoader->LoadAssetFile("tesselationQuad.out");
    m_pDataHolder->SetupAttributeStatesNVN(SimpleTexturedModel::Attributes::GetAttributeLocation);
    m_pDataHolder->SetupTextureSamplerHandle(&m_Device, m_pTextureIDManager, &m_SamplerBuilder);

    /*! Set up the window builder. */
    nvnWindowBuilderSetDefaults(&m_WindowBuilder);
    nvnWindowBuilderSetDevice(&m_WindowBuilder, &m_Device);
    nvnWindowBuilderSetNativeWindow(&m_WindowBuilder, nativeWindow);

    m_pFrameBufferedSyncManager = new FrameBufferedSyncManager(&m_Device, &m_Queue);

    /////////////////////////////
    /////////////////////////////
    // Debug Text Renderer
    /////////////////////////////
    /////////////////////////////

    m_pDebugTextRenderer = DebugTextRenderer::GetInstance();

    m_pDebugTextRenderer->Init(&m_Device, &m_Queue, m_pTextureIDManager, m_pFrameBufferedSyncManager);
    m_pDebugTextRenderer->SetDrawEnable(true);
    m_pDebugTextRenderer->SetSpacing(false);
    m_pDebugTextRenderer->SetGridSize(120.0f, 36.0f);

    m_TessellationLevels.push_back(std::pair<float, float>(1.0f, 1.0f));
    m_TessellationLevels.push_back(std::pair<float, float>(2.0f, 2.0f));
    m_TessellationLevels.push_back(std::pair<float, float>(4.0f, 4.0f));
    m_TessellationLevels.push_back(std::pair<float, float>(8.0f, 8.0f));
    m_TessellationLevels.push_back(std::pair<float, float>(16.0f, 16.0f));
    m_TessellationLevels.push_back(std::pair<float, float>(32.0f, 32.0f));
    m_TessellationLevels.push_back(std::pair<float, float>(64.0f, 64.0f));

#ifdef USE_LOP
    /////////////////////////////
    /////////////////////////////
    // Profiler
    /////////////////////////////
    /////////////////////////////

    void* place = AlignedAllocate(sizeof(lop::LOP_Profiler), NN_ALIGNOF(lop::LOP_Profiler));
    m_pProfiler = new(place) lop::LOP_Profiler();
    m_pProfiler->Initialize(&m_Device, &m_Queue);
    m_pProfiler->InitializeChipDesc();

    // Primitive Flow
    static const char* const ppMetricNames[] = {
        "pda__input_prims",
        "pdb__input_batches",
        "wwdx__input_prims",
        "vpc__input_prims",
        "raster__setup_input_prims",
        "raster__crstr_input_prims",
        "vaf__alpha_fetched_attr_scalar",
        "sm__cycles_active_3d_vtg",
        nullptr
    };

    int metricPrint[] = {
        lop::MetricPrint_Sum,
        lop::MetricPrint_Sum,
        lop::MetricPrint_Sum,
        lop::MetricPrint_Sum,
        lop::MetricPrint_Sum,
        lop::MetricPrint_Sum,
        lop::MetricPrint_Sum,
        lop::MetricPrint_Sum | lop::MetricPrint_Pct_Peak_Sustained_Elapsed,
    };

    NN_ASSERT(m_pProfiler->GenerateMetricSpecs(ppMetricNames, metricPrint, m_SelectedMetrics));
    m_pProfiler->PrepareOfflineData(m_ConfigImage, m_SelectedMetrics);

    m_pProfiler->BeginSession(m_ConfigImage);

#endif

    nn::fs::Unmount("rom");
    delete[] mountRomCacheBuffer;
}//NOLINT(impl/function_size)

 /*!
 * AssetFileLoading::Shutdown
 * --------------------------
 * This method cleans up all nvn objects and dynamically allocated memory.
 */
void TestLop_PrimitiveFlow::Shutdown()
{
#ifdef USE_LOP
    if (m_pProfiler)
    {
        m_pProfiler->EndSession();
        AlignedDeallocate(m_pProfiler);
    }
#endif
    nvnQueueFinish(&m_Queue);
    nvnSyncFinalize(&m_RenderTargetPresentSync);
    nvnSyncFinalize(&m_CommandBufferSync);
    nvnSyncFinalize(&m_UniformBlockUpdateSync);
    nvnSyncFinalize(&m_WindowSync);

    nvnCommandBufferFinalize(&m_RenderTargetCommandBuffer);

    if (m_pRenderTargetControlPool)
    {
        AlignedDeallocate(m_pRenderTargetControlPool);
        m_pRenderTargetControlPool = NULL;
    }

    nvnCommandBufferFinalize(&m_CommandBuffer);

    if (m_pCommandMemoryPool)
    {
        m_pCommandMemoryPool->Shutdown();
        delete m_pCommandMemoryPool;
        m_pCommandMemoryPool = NULL;
    }

    if (m_pControlPool)
    {
        AlignedDeallocate(m_pControlPool);
        m_pControlPool = NULL;
    }

    if (m_pWindow)
    {
        nvnWindowFinalize(m_pWindow);
        delete m_pWindow;
        m_pWindow = NULL;
    }

    for (int i = 0; i < g_NumColorBuffers; ++i)
    {
        if (m_pRenderTargets[i])
        {
            nvnTextureFinalize(m_pRenderTargets[i]);
            delete m_pRenderTargets[i];
            m_pRenderTargets[i] = NULL;
        }
    }

    if (m_pDepthBuffer)
    {
        nvnTextureFinalize(m_pDepthBuffer);
        delete m_pDepthBuffer;
        m_pDepthBuffer = NULL;
    }

    m_UniformBlockVertex.Finalize();
    m_UniformBlockTessControl.Finalize();

    if (m_pUniformBufferMemoryPool)
    {
        m_pUniformBufferMemoryPool->Shutdown();
        delete m_pUniformBufferMemoryPool;
        m_pUniformBufferMemoryPool = NULL;
    }

    if (m_pDataHolder)
    {
        delete m_pDataHolder;
        m_pDataHolder = NULL;
    }

    if (m_pAssetLoader)
    {
        delete m_pAssetLoader;
        m_pAssetLoader = NULL;
    }

    if (m_pRenderTargetMemoryPool)
    {
        m_pRenderTargetMemoryPool->Shutdown();
        delete m_pRenderTargetMemoryPool;
        m_pRenderTargetMemoryPool = NULL;
    }

    if (m_pTextureIDManager)
    {
        delete m_pTextureIDManager;
        m_pTextureIDManager = NULL;
    }

    nvnQueueFinalize(&m_Queue);
    if (m_pQueueMemory)
    {
        AlignedDeallocate(m_pQueueMemory);
        m_pQueueMemory = NULL;
    }

    nvnDeviceFinalize(&m_Device);
}

/*!
* AssetFileLoading::Draw
* ----------------------
* This method updates uniform data and submits the commands
* recorded in the command buffer to the queue and presents
*/
void TestLop_PrimitiveFlow::Draw(uint64_t /*millisec*/)
{
    UpdateTessellation();

    /*! Get the current render target and setup/submit a command buffer to set it. */
    int index = UpdateRenderTargets();
    nvnQueueSubmitCommands(&m_Queue, 1, &m_RenderTargetCommandHandle);

    /*! Wait to update uniform data safely. */
    nvnQueueFenceSync(&m_Queue, &m_UniformBlockUpdateSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);
    nvnQueueFlush(&m_Queue);
    nvnSyncWait(&m_UniformBlockUpdateSync, NVN_WAIT_TIMEOUT_MAXIMUM);

    /*! Update the uniform data. */
    nn::util::Matrix4x3fType simpleTranslateMat;
    nn::util::MatrixIdentity(&simpleTranslateMat);
    nn::util::Vector3fType simpleTranslateVector;
    nn::util::VectorSet(&simpleTranslateVector, 2.25f, 0.0f, -7.0f);
    nn::util::MatrixSetTranslate(&simpleTranslateMat, simpleTranslateVector);

    nn::util::Matrix4x3fType simpleRotateXMat;
    nn::util::MatrixIdentity(&simpleRotateXMat);
    nn::util::Vector3fType simpleRotateXVector;
    nn::util::VectorSet(&simpleRotateXVector, -15.0f * 3.1459f / 180.0f, 0.0f, 0.0f);
    nn::util::MatrixSetRotateXyz(&simpleRotateXMat, simpleRotateXVector);

    nn::util::Matrix4x3fType simpleRotateYMat;
    nn::util::MatrixIdentity(&simpleRotateYMat);
    nn::util::Vector3fType simpleRotateYVector;
    nn::util::VectorSet(&simpleRotateYVector, 0.0f, m_RotY * 3.14159f / 180.0f, 0.0f);
    nn::util::MatrixSetRotateXyz(&simpleRotateYMat, simpleRotateYVector);

    nn::util::Matrix4x3fType simpleScaleMat;
    nn::util::MatrixIdentity(&simpleScaleMat);
    nn::util::Vector3fType simpleScaleVector;
    nn::util::VectorSet(&simpleScaleVector, 3.0f, 3.0f, 3.0f);
    nn::util::MatrixSetScale(&simpleScaleMat, simpleScaleVector);

    nn::util::Matrix4x3fType simpleModelMat;
    nn::util::Matrix4x3fType tempMat0;
    nn::util::Matrix4x3fType tempMat1;
    nn::util::MatrixMultiply(&tempMat0, simpleScaleMat, simpleRotateXMat);
    nn::util::MatrixMultiply(&tempMat1, simpleRotateYMat, tempMat0);
    nn::util::MatrixMultiply(&simpleModelMat, tempMat1, simpleTranslateMat);
    nn::util::Matrix4x4fType simpleModelMat44;
    nn::util::MatrixConvert(&simpleModelMat44, simpleModelMat);

    nn::util::Matrix4x4fType projMat44;
    nn::util::MatrixPerspectiveFieldOfViewRightHanded(&projMat44, 60.0f * 3.14159f / 180.0f, static_cast<float>(m_ScreenWidth) / static_cast<float>(m_ScreenHeight), 1.0f, 1000.0f);

    UpdateUniformBlock(simpleModelMat44, projMat44);
    m_RotY += 0.15f;

    /*!
     * Wait on sync that was received in UpdateRenderTargets now that we are
     * actually ready to use the render target
     */
    nvnSyncWait(&m_WindowSync, NVN_WAIT_TIMEOUT_MAXIMUM);

    /*! Submit the command buffer to render the cube. */
#ifdef USE_LOP
    NVPA_Bool allPassesDecoded = false;
    static bool first = true;
    int runs = 0;
    do
    {
        m_pProfiler->BeginPass();
        {
                /* Submit the commands to the queue. */
            nvnQueueSubmitCommands(&m_Queue, 1, &m_CommandHandle);
        }
        m_pProfiler->EndPass();
        nvnQueueFinish(&m_Queue);
        m_pProfiler->DecodeCounters(m_ConfigImage, allPassesDecoded);

        ++runs;

        if (allPassesDecoded)
        {
#ifdef TEST_LOP
            m_pProfiler->UnpackRawMetrics_Test(m_SelectedMetrics, m_MetricValues);
            m_ContinueRendering = false;
#else
            m_pDebugTextRenderer->SetColor(1.0f, 1.0f, 1.0f, 1.0f);
            m_pDebugTextRenderer->Printf(0, 0, "Runs: %i", runs);
            m_pDebugTextRenderer->Printf(0, 1, "Inner Tessellation: %f", m_TessellationLevels[g_TessellationLevel].first);
            m_pDebugTextRenderer->Printf(0, 2, "Outer Tessellation: %f", m_TessellationLevels[g_TessellationLevel].second);

            std::string outputString = "%-38s = %10.0f\n";
            m_pProfiler->UnpackRawMetrics(m_SelectedMetrics, false, m_pDebugTextRenderer, outputString, 3, false);
#endif
            m_pProfiler->ProfilerInitializeCounterData();
            break;
        }
    } while (!allPassesDecoded && !first);

    first = false;
#ifndef TEST_LOP
    m_pDebugTextRenderer->Draw(m_pRenderTargets[index], m_ScreenWidth, m_ScreenHeight);
#endif
#else
    nvnQueueSubmitCommands(&m_Queue, 1, &m_CommandHandle);
#endif

    /*! Present the target to the screen. */
    nvnQueuePresentTexture(&m_Queue, m_pWindow, index);

    /* Swap the multi buffered memory. */
    m_pFrameBufferedSyncManager->SwapPools();
}

void DrawHelper(TestLop_PrimitiveFlow* prim)
{
    std::vector<std::pair<std::string, double> >& metricValues = prim->GetMetricValues();
    metricValues.clear();

    while (prim->ContinueRendering())
    {
        prim->Draw(0);
    }

    prim->SetContinueRendering(true);
}

TEST(PrimitiveFlowTest, test_TessellationLevel)
{
    TestLop_PrimitiveFlow* prim = reinterpret_cast<TestLop_PrimitiveFlow*>(t());
    std::vector<std::pair<std::string, double> >& metricValues = prim->GetMetricValues();

    /////////////////////////////////
    // Tessellation level 1.0, 1.0 //
    /////////////////////////////////
    g_TessellationLevel = 0;

    DrawHelper(prim);

    EXPECT_EQ(metricValues[0].second, 20.0);
    EXPECT_EQ(metricValues[1].second, 2.0);
    EXPECT_EQ(metricValues[3].second, 20.0);
    EXPECT_EQ(metricValues[4].second, metricValues[5].second);

    /////////////////////////////////
    // Tessellation level 2.0, 2.0 //
    /////////////////////////////////
    g_TessellationLevel = 1;

    DrawHelper(prim);

    EXPECT_EQ(metricValues[0].second, 20.0);
    EXPECT_EQ(metricValues[1].second, 2.0);
    EXPECT_EQ(metricValues[3].second, 120.0);
    EXPECT_EQ(metricValues[4].second, metricValues[5].second);

    /////////////////////////////////
    // Tessellation level 4.0, 4.0 //
    /////////////////////////////////
    g_TessellationLevel = 2;

    DrawHelper(prim);

    EXPECT_EQ(metricValues[0].second, 20.0);
    EXPECT_EQ(metricValues[1].second, 2.0);
    EXPECT_EQ(metricValues[3].second, 480.0);
    EXPECT_EQ(metricValues[4].second, metricValues[5].second);

    /////////////////////////////////
    // Tessellation level 8.0, 8.0 //
    /////////////////////////////////
    g_TessellationLevel = 3;

    DrawHelper(prim);

    EXPECT_EQ(metricValues[0].second, 20.0);
    EXPECT_EQ(metricValues[1].second, 2.0);
    EXPECT_EQ(metricValues[3].second, 1920.0);
    EXPECT_EQ(metricValues[4].second, metricValues[5].second);

    ///////////////////////////////////
    // Tessellation level 16.0, 16.0 //
    ///////////////////////////////////
    g_TessellationLevel = 4;

    DrawHelper(prim);

    EXPECT_EQ(metricValues[0].second, 20.0);
    EXPECT_EQ(metricValues[1].second, 2.0);
    EXPECT_EQ(metricValues[3].second, 7680.0);
    EXPECT_EQ(metricValues[4].second, metricValues[5].second);

    ///////////////////////////////////
    // Tessellation level 32.0, 32.0 //
    ///////////////////////////////////
    g_TessellationLevel = 5;

    DrawHelper(prim);

    EXPECT_EQ(metricValues[0].second, 20.0);
    EXPECT_EQ(metricValues[1].second, 2.0);
    EXPECT_EQ(metricValues[3].second, 30720.0);
    EXPECT_EQ(metricValues[4].second, metricValues[5].second);

    ///////////////////////////////////
    // Tessellation level 64.0, 64.0 //
    ///////////////////////////////////
    g_TessellationLevel = 6;

    DrawHelper(prim);

    EXPECT_EQ(metricValues[0].second, 20.0);
    EXPECT_EQ(metricValues[1].second, 2.0);
    EXPECT_EQ(metricValues[3].second, 122880.0);
    EXPECT_EQ(metricValues[4].second, metricValues[5].second);
}

bool TestLop_PrimitiveFlow::TestDraw(uint64_t /*millisec*/)
{
#ifdef TEST_LOP
    int result = RUN_ALL_TESTS();
    m_pProfiler->EndSession();
    nnt::Exit(result);
#else
    Draw(0);
#endif
    return false;
}

/*
* AssetFileLoading::UpdateRenderTargets
* -------------------------------------
* Gets the index of the current render target from the NVNwindow
* and records a command buffer that sets it up to be rendered to.
*/
int TestLop_PrimitiveFlow::UpdateRenderTargets()
{
    /*! Sync so that we can safely overwrite the command buffer. */
    nvnQueueFenceSync(&m_Queue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);
    nvnQueueFlush(&m_Queue);
    nvnSyncWait(&m_CommandBufferSync, NVN_WAIT_TIMEOUT_MAXIMUM);

    /*! Get index to next render target to be used */
    NVNwindowAcquireTextureResult result = nvnWindowAcquireTexture(m_pWindow, &m_WindowSync, &m_CurrentWindowIndex);

    NN_ASSERT(result == NVN_WINDOW_ACQUIRE_TEXTURE_RESULT_SUCCESS);

    /*! Rest the command buffer memory. */
    nvnCommandBufferAddCommandMemory(&m_RenderTargetCommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_RenderTargetCommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_RenderTargetCommandBuffer, m_pRenderTargetControlPool, g_ControlMemorySize);

    /*! Record the command buffer to set the target. */
    nvnCommandBufferBeginRecording(&m_RenderTargetCommandBuffer);
    nvnCommandBufferPushDebugGroup(&m_RenderTargetCommandBuffer, "Set Render Target");
    nvnCommandBufferSetRenderTargets(&m_RenderTargetCommandBuffer, 1, &m_pRenderTargets[m_CurrentWindowIndex], NULL, m_pDepthBuffer, NULL);
    nvnCommandBufferPopDebugGroup(&m_RenderTargetCommandBuffer);
    m_RenderTargetCommandHandle = nvnCommandBufferEndRecording(&m_RenderTargetCommandBuffer);

    /*! Return the index. */
    return m_CurrentWindowIndex;
}

bool TestLop_PrimitiveFlow::UpdateTessellation()
{
#if defined ( NN_BUILD_TARGET_PLATFORM_OS_WIN )
#ifndef TEST_LOP
    static int cooldown = 0;
    if (cooldown < 120)
    {
        ++cooldown;
    }
    else
    {
        g_TessellationLevel = (g_TessellationLevel + 1) % m_TessellationLevels.size();
        cooldown = 0;
    }

    return false;
#else
    return false;
#endif
#else
#ifndef TEST_LOP
    static int cooldown = 10;
    if (cooldown < 10)
    {
        ++cooldown;
    }
    else
    {
        nn::hid::TouchScreenState<3> touchScreenState;

        int touchCount = nn::hid::GetTouchScreenStates(&touchScreenState, 1);
        if (touchScreenState.count > 0 && touchCount)
        {
            if (cooldown < 10)
            {
                return false;
            }
            g_TessellationLevel = (g_TessellationLevel + 1) % m_TessellationLevels.size();
            cooldown = 0;
            return true;
        }
    }
    return false;
#else
    return false;
#endif
#endif
}

/*!
* AssetFileLoading::Resize
* ------------------------
* This method is called everytime the window is resized and is passed
* the new size of the window. It frees the old render target and creates a new
* render target with the new screen size. A new set of commands for the command buffer
* is recorded.
*/
void TestLop_PrimitiveFlow::Resize(int width, int height)
{
    /*! Check for the window being minimized or having no visible surface. */
    if (width == 0 || height == 0)
    {
        return;
    }

    /*! Save the current height and width of the screen. */
    m_ScreenHeight = height;
    m_ScreenWidth = width;

    /*! Sync to safely rewrite the command buffer. */
    nvnQueueFenceSync(&m_Queue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);
    nvnQueueFlush(&m_Queue);
    nvnSyncWait(&m_CommandBufferSync, NVN_WAIT_TIMEOUT_MAXIMUM);

    /*! If it's the first time Resize is called, allocate the NVNwindow. */
    if (m_pWindow == NULL)
    {
        m_pWindow = new NVNwindow;
    }
    /*
    * Otherwise finalize (free) the NVNwindow used for the previous window size.
    * The NVNWindow must be finalized before a render target it owns is finalized.
    */
    else
    {
        nvnWindowFinalize(m_pWindow);
    }

    /*! Set up the builder for the render target. */
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_DISPLAY_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, m_ScreenWidth, m_ScreenHeight);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_RGBA8);

    std::string renderTargetName = "TessellationTest_RenderTarget_";
    for (int i = 0; i < g_NumColorBuffers; ++i)
    {
        /*! If it's the first time Resize is called, allocate the render target. */
        if (m_pRenderTargets[i] == NULL)
        {
            m_pRenderTargets[i] = new NVNtexture;
        }
        /*! Otherwise finalize (free) the render target used for the previous window size. */
        else
        {
            nvnTextureFinalize(m_pRenderTargets[i]);
        }

        nvnTextureBuilderSetStorage(&m_RenderTargetBuilder, m_pRenderTargetMemoryPool->GetMemoryPool(), m_ColorTargetSize * i);

        /*! Create the texture using the current state of the texture builder. */
        if (!nvnTextureInitialize(m_pRenderTargets[i], &m_RenderTargetBuilder))
        {
            NN_ASSERT(0, "Failed to initialize render target");
        }

        char indexAsChar = static_cast<char>(i + '0');
        std::string updatedTargetName = renderTargetName + indexAsChar;
        nvnTextureSetDebugLabel(m_pRenderTargets[i], updatedTargetName.c_str());
    }

    /*! If it's the first time Resize is called, allocate the depth buffer. */
    if (m_pDepthBuffer == NULL)
    {
        m_pDepthBuffer = new NVNtexture;
    }
    /*! Otherwise finalize (free) the depth buffer used for the previous window size. */
    else
    {
        nvnTextureFinalize(m_pDepthBuffer);
    }

    /*! Initialize depth buffer for render target. */
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, m_ScreenWidth, m_ScreenHeight);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_DEPTH32F);
    nvnTextureBuilderSetStorage(&m_RenderTargetBuilder, m_pRenderTargetMemoryPool->GetMemoryPool(), m_ColorTargetSize * g_NumColorBuffers);

    if (!nvnTextureInitialize(m_pDepthBuffer, &m_RenderTargetBuilder))
    {
        NN_ASSERT(0, "Failed to initialize depth target");
    }

    nvnTextureSetDebugLabel(m_pDepthBuffer, "TessellationTest_DepthBuffer");

    /*! Pass off the render targets to the window. */
    nvnWindowBuilderSetTextures(&m_WindowBuilder, g_NumColorBuffers, m_pRenderTargets);
    nvnWindowInitialize(m_pWindow, &m_WindowBuilder);

    nvnWindowSetDebugLabel(m_pWindow, "TessellationTest_Window");

    /*! Reset the command buffer memory. */
    nvnCommandBufferAddCommandMemory(&m_CommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_CommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_CommandBuffer, m_pControlPool, g_ControlMemorySize);

    {
        /*! Starts the recording of a new set of commands for the given command buffer. */
        nvnCommandBufferBeginRecording(&m_CommandBuffer);
        {
            nvnCommandBufferPushDebugGroup(&m_CommandBuffer, "Draw");

            /*! Bind the texture and sampler descriptor pools. */
            m_pTextureIDManager->SetSamplerPool(&m_CommandBuffer);
            m_pTextureIDManager->SetTexturePool(&m_CommandBuffer);

            // Sets the scissor rectangle and viewport to the full screen
            nvnCommandBufferSetScissor(&m_CommandBuffer, 0, 0, m_ScreenWidth, m_ScreenHeight);
            nvnCommandBufferSetViewport(&m_CommandBuffer, 0, 0, m_ScreenWidth, m_ScreenHeight);

            /*! Clears the currently set render target at a given index. */
            float clear_color[4] = { 0.4f, 0.55f, 0.6f, 1.0f };
            nvnCommandBufferClearColor(&m_CommandBuffer, 0, clear_color, NVN_CLEAR_COLOR_MASK_RGBA);
            nvnCommandBufferClearDepthStencil(&m_CommandBuffer, 1.0, NVN_TRUE, 0, 0);

            std::vector<NVNModelData*>&     modelData = m_pDataHolder->GetModelData();
            std::vector<NVNProgramData*>&   programData = m_pDataHolder->GetProgramData();

            nvnProgramSetDebugLabel(&programData[0]->m_Program, "TessellationTest_Program");

            /*! Bind the render state objects. */
            nvnCommandBufferBindBlendState(&m_CommandBuffer, &m_BlendState);
            nvnCommandBufferBindChannelMaskState(&m_CommandBuffer, &m_ChannelMaskState);
            nvnCommandBufferBindColorState(&m_CommandBuffer, &m_ColorState);
            nvnCommandBufferBindDepthStencilState(&m_CommandBuffer, &m_DepthStencilState);
            nvnCommandBufferBindMultisampleState(&m_CommandBuffer, &m_MultisampleState);
            nvnCommandBufferBindPolygonState(&m_CommandBuffer, &m_PolygonState);
            nvnCommandBufferSetSampleMask(&m_CommandBuffer, static_cast<uint32_t>(~0));

            Model* pModel = &modelData[0]->m_Model;
            NVNbufferAddress vboAddr = nvnBufferGetAddress(&modelData[0]->m_VertexBuffer);
            for (size_t i = 0; i < pModel->m_VertexAttributes.size(); ++i)
            {
                VertexAttribute& attr = pModel->m_VertexAttributes[i];
                nvnCommandBufferBindVertexBuffer(&m_CommandBuffer, attr.m_Location, vboAddr + modelData[0]->m_VertexAttributeBufferOffsets[i], attr.m_DataSize);
            }

            /*! Bind the uniform buffer for the vertex shader. */
            uint32_t vertBindingLoc = SimpleTexturedModel::BlockVSUniformBlockData::GetBinding(NVN_SHADER_STAGE_VERTEX);
            nvnCommandBufferBindUniformBuffer(&m_CommandBuffer, NVN_SHADER_STAGE_VERTEX, vertBindingLoc, m_UniformBlockVertex.GetBufferAddress(), sizeof(SimpleTexturedModel::BlockVSUniformBlockData));
            nvnCommandBufferBindUniformBuffer(&m_CommandBuffer, NVN_SHADER_STAGE_TESS_CONTROL, 0, m_UniformBlockTessControl.GetBufferAddress(), 8);

            /*! Bind the vertex attribute states and vertex stream states. */
            nvnCommandBufferBindVertexAttribState(&m_CommandBuffer, static_cast<int>(modelData[0]->m_VertexAttributeStates.size()), &modelData[0]->m_VertexAttributeStates[0]);
            nvnCommandBufferBindVertexStreamState(&m_CommandBuffer, static_cast<int>(modelData[0]->m_VertexStreamStates.size()), &modelData[0]->m_VertexStreamStates[0]);

            /*! Bind the shader program. */
            nvnCommandBufferBindProgram(&m_CommandBuffer, &programData[0]->m_Program, programData[0]->m_ShaderStages);

            NVNindexType indexType = static_cast<NVNindexType>(modelData[0]->m_Model.m_IndexData.m_IndexType);
            uint32_t numIndices = pModel->m_IndexData.m_DataSize / pModel->m_IndexData.m_Stride;

            nvnCommandBufferSetPatchSize(&m_CommandBuffer, 3);
            nvnCommandBufferDrawElements(&m_CommandBuffer,
                static_cast<NVNdrawPrimitive>(pModel->m_NvnDrawPrimitiveType),
                indexType,
                numIndices,
                nvnBufferGetAddress(&modelData[0]->m_IndexBuffer));

            // Unbind program to cleanup pipeline
            nvnCommandBufferBindProgram(&m_CommandBuffer, NULL, NVN_SHADER_STAGE_ALL_GRAPHICS_BITS);

            nvnCommandBufferPopDebugGroup(&m_CommandBuffer);
        }
        /*! End the command buffer recording. */
        m_CommandHandle = nvnCommandBufferEndRecording(&m_CommandBuffer);
    }

#ifdef CHECK_COMMAND_BUFFER_MEMORY_USAGE
    m_CommandMemoryUsed = nvnCommandBufferGetCommandMemoryUsed(&m_CommandBuffer);
    m_ControlMemoryUsed = nvnCommandBufferGetControlMemoryUsed(&m_CommandBuffer);
#endif
}//NOLINT(impl/function_size)

// static
void TestLop_PrimitiveFlow::DebugLayerCallback(
    NVNdebugCallbackSource source,
    NVNdebugCallbackType type,
    int id,
    NVNdebugCallbackSeverity severity,
    const char* message,
    void* pUser
)
{
    NN_ASSERT(pUser == NULL);

    NN_LOG("NVN Debug Layer Callback:\n");
    NN_LOG("  source:       0x%08x\n", source);
    NN_LOG("  type:         0x%08x\n", type);
    NN_LOG("  id:           0x%08x\n", id);
    NN_LOG("  severity:     0x%08x\n", severity);
    NN_LOG("  message:      %s\n", message);
}

TutorialBaseClass* t()
{
    static TestLop_PrimitiveFlow s_Tut;
    return (&s_Tut);
}

extern "C" void nnMain()
{
#ifdef TEST_LOP
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
#endif

    TutorialRun(true);
}
