﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#define USE_LOP
#ifdef USE_LOP

#include <vector>
#include <set>
#include <sstream>

#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>

#include <nvngdSupport/MemoryPool.h>
#include <nvngdSupport/DebugTextRenderer.h>

#include <nvperf/nvperf_Host.h>
#include <nvperf/nvperf_Target.h>
#include <nvperf/nvperf_Nvn.h>

#include <nvperf/nvperf_RealtimeMetrics.h>
#include <nvperf/nvperf_Gm20bRealtimeMetrics.h>
#include <nvperf/nvperf_Gm204RealtimeMetrics.h>

namespace lop
{
typedef enum MetricPrint
{
    MetricPrint_Sum                         = 0x00000001,
    MetricPrint_Avg                         = 0x00000002,
    MetricPrint_Avg_Peak_Sustained          = 0x00000004,
    MetricPrint_Avg_Cycles_Elapsed          = 0x00000008,
    MetricPrint_Sum_Per_Cycle_Elapsed       = 0x00000010,
    MetricPrint_Avg_Per_Cycle_Elapsed       = 0x00000020,
    MetricPrint_Pct_Peak_Sustained_Elapsed  = 0x00000040,

    MetricPrint_ALL                         = 0x0000007F,

    MetricPrint_NULL                        = 0x00000000,
} MetricPrint;

struct MetricSpec
{
    const nv::metrics::MetricDesc* pMetricDesc;
    int printStats;
    bool isolated;
};

class LOP_Profiler
{
public:
    LOP_Profiler();
    ~LOP_Profiler();

    void Initialize(NVNdevice* pDevice_, NVNqueue* pQueue_);
    bool InitializeChipDesc();
    void SelectAllMetrics(std::vector<MetricSpec>& selectedMetrics);

    bool GenerateMetricSpecs(const char* const* ppMetricNames, int* pPrint, std::vector<MetricSpec>& metricSpecs);
    bool GenerateLoadedMetricSpecs(std::set<std::string> metricNames, std::vector<int> printType, std::vector<MetricSpec>& metricSpecs);
    bool PrepareOfflineData(std::vector<uint8_t>& configImage, std::vector<MetricSpec>& selectedMetrics);
    bool PrepareLoadedOfflineData(std::vector<uint8_t>& counterDataPrefixImage, std::vector<MetricSpec>& selectedMetrics);
    bool ResolveDependencies(NVPA_RawMetricsConfig* pRawMetricsConfig, NVPA_CounterDataBuilder* pCounterBuilder);
    bool CreateConfigImage(std::vector<uint8_t>& configImage, NVPA_RawMetricsConfig* pRawMetricsConfig);
    bool CreateCounterDataPrefix(NVPA_CounterDataBuilder* pCounterBuilder);
    bool ProfilerInitializeCounterData();

    bool BeginSession(std::vector<uint8_t>& configImage);
    bool EndSession(bool shutdown = true);
    bool ResetSession(std::vector<uint8_t>& configImage);

    bool BeginPass();
    bool EndPass();

    bool DecodeCounters(std::vector<uint8_t>& configImage, NVPA_Bool& allPassesDecoded);

    bool UnpackRawMetrics(std::vector<MetricSpec>& selectedMetrics, bool allMetricsSelected, DebugTextRenderer* pDebugTextRenderer, const std::string& outputFormat, int linesToSkip, bool outputFile);
    bool UnpackRawMetrics_Test(std::vector<MetricSpec>& selectedMetrics, std::vector<std::pair<std::string, double> >& metricValues);
    bool UnpackAllMetrics(std::vector<MetricSpec>& selectedMetrics);
    bool SaveRawMetrics(const std::vector<MetricSpec>& metricSpecs);

    nv::metrics::ChipDesc& GetChipDesc() { return m_ChipDesc; }
    const char* GetChipName() { return m_pChipName; }

private:
    bool GenerateAllMetricRequests(const NVPA_RawMetricsConfig* pRawMetricsConfig, const std::vector<MetricSpec>& selectedMetrics, std::vector<NVPA_RawMetricRequest>& metricRequests);

    size_t m_NumRequiredPasses;

    // This data can be prepared offline and simply loaded.
    std::vector<uint8_t> m_CounterDataPrefix;

    // Profiler session state.
    NVNdevice* m_pDevice;
    NVNqueue* m_pQueue;
    std::vector<uint8_t> m_CounterDataImage;
    std::vector<uint8_t> m_CounterDataScratch;
    MemoryPool* m_pTraceArena;
    MemoryPool* m_pComputeArena;
    NVPA_NVNC_SessionOptions m_SessionOptions;

#ifdef _WIN32
    NVPA_MetricsContext* m_pMetricsContext;
    const char* m_pDeviceName;
#endif
    nv::metrics::ChipDesc m_ChipDesc;

    const char* m_pChipName;

#if !defined(_WIN32)
    uint8_t m_PerfmonBuffer[0x100000] __attribute__((aligned(4096)));
#endif
    size_t m_FrameNumber;
    size_t m_TraceBufferSize;
    size_t m_ComputeBufferSize;
};

inline std::string ssprintf(const char* pFormat, ...)
{
    va_list vl;

    va_start(vl, pFormat);
    const int length = 1 + vsnprintf(NULL, 0, pFormat, vl);
    va_end(vl);

    std::string str;
    str.resize(length);
    va_start(vl, pFormat);
    vsnprintf(&str[0], length, pFormat, vl);
    va_end(vl);
    str.erase(length - 1);

    return str;
}
}

#endif
