﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/gfx/util/gfx_TransientBufferAllocatorHolder.h>

#include <nnt.h>

TEST(TransientBufferAllocator, SingleThread)
{
    nn::gfx::Buffer buffer;

    // パラメータ
    static const int HistoryLengthMax = 5;
    const ptrdiff_t baseOffset = 5 * 1024;
    const size_t allocateSizeMin = 1;
    const size_t allocateSizeMax = 10000;
    const size_t allocateAlignmentShiftMin = 0;
    const size_t allocateAlignmentShiftMax = 10;
    //const size_t allocateAlignmentMin = 1 << allocateAlignmentShiftMin;
    const size_t allocateAlignmentMax = 1 << allocateAlignmentShiftMax;

    const int allocateCountPerSection = 50;
    const size_t offsetSize = static_cast<size_t>(HistoryLengthMax * allocateCountPerSection * (allocateSizeMax + allocateAlignmentMax));
    const int sectionCount = 100;

    // 乱数
    unsigned long randSeed =  0x2391A8BC; // 適当。
    std::mt19937 reng;
    reng.seed(randSeed);
    std::uniform_int_distribution<size_t> randSize(allocateSizeMin, allocateSizeMax);
    std::uniform_int_distribution<size_t> randAlignmentShift(allocateAlignmentShiftMin, allocateAlignmentShiftMax);


    // ここからテスト
    nn::gfx::util::TransientBufferAllocatorHolder<HistoryLengthMax> traHolder;

    EXPECT_EQ(HistoryLengthMax, static_cast<int>(traHolder.HistoryLengthMax));

    EXPECT_FALSE(traHolder.Get()->IsInitialized());
    traHolder.Initialize(&buffer, baseOffset, offsetSize, allocateAlignmentMax);
    nn::gfx::util::TransientBufferAllocator* pTra = traHolder.Get();
    EXPECT_TRUE(pTra != NULL);
    EXPECT_TRUE(pTra->IsInitialized());
    EXPECT_FALSE(pTra->IsRecording());
    EXPECT_EQ(0, pTra->GetHistoryLength());

    pTra->FillHistory();
    EXPECT_EQ(HistoryLengthMax, pTra->GetHistoryLength());

    // 履歴がいっぱいの状態で Fill しても OK
    pTra->FillHistory();

    for(int section = 0; section < sectionCount; section++)
    {
        pTra->Free();
        EXPECT_EQ(HistoryLengthMax - 1, pTra->GetHistoryLength());

        pTra->Begin();
        EXPECT_TRUE(pTra->IsRecording());

        for(int c = 0; c < allocateCountPerSection; c++)
        {
            size_t size = randSize(reng);
            size_t alignment = 1ULL << randAlignmentShift(reng);
            ptrdiff_t offset = pTra->Allocate(size, alignment);
            EXPECT_GE(offset, baseOffset);
            EXPECT_LT(offset, static_cast<ptrdiff_t>(baseOffset + offsetSize));
        }

        pTra->End();
        EXPECT_EQ(HistoryLengthMax, pTra->GetHistoryLength());
        EXPECT_FALSE(pTra->IsRecording());
    }

    traHolder.Finalize();
    EXPECT_FALSE(traHolder.Get()->IsInitialized());

}

TEST(TransientBufferAllocatorDeathTest, DeathTest)
{
    nn::gfx::Buffer buffer;

    {
        nn::gfx::util::TransientBufferAllocatorHolder<3> holder;

        // 初期化前に呼ぶと死ぬことの確認
        EXPECT_DEATH_IF_SUPPORTED(holder.Finalize(), "");

        holder.Initialize(&buffer, 0, 100, 1024);

        // 初期化後に呼ぶと死ぬことの確認
        EXPECT_DEATH_IF_SUPPORTED(holder.Initialize(&buffer, 0, 10, 1), "");

        holder.Finalize();
    }

    {
        static const int HistoryLength = 3;
        nn::gfx::util::TransientBufferAllocator tra;
        nn::gfx::util::detail::OffsetRange ranges[HistoryLength + 2];

        // 初期化前に呼ぶと死ぬことの確認
        EXPECT_DEATH_IF_SUPPORTED(tra.Finalize(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.IsRecording(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.GetBuffer(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.GetBaseOffset(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.GetSize(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.GetAllocatableAlignmentMax(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.Begin(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.End(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.Free(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.Allocate(1,1), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.GetHistoryLengthMax(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.GetHistoryLength(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.FillHistory(), "");

        // 不正なパラメータで呼ぶと死ぬことの確認
        EXPECT_DEATH_IF_SUPPORTED(tra.GetRequiredMemorySize(0), "");

        tra.SetMemory(ranges, sizeof(ranges));

        // Initialize のパラメータで死ぬことの確認
        EXPECT_DEATH_IF_SUPPORTED(tra.Initialize(NULL, 0, 10, 1, HistoryLength), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.Initialize(&buffer, -1, 10, 1, HistoryLength), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.Initialize(&buffer, 0, 0, 1, HistoryLength), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.Initialize(&buffer, 0, tra.SizeMax + 1, 1, HistoryLength), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.Initialize(&buffer, 0, 100, 0, HistoryLength), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.Initialize(&buffer, 0, 100, tra.AlignmentMax << 1, HistoryLength), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.Initialize(&buffer, 0, 100, 33, HistoryLength), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.Initialize(&buffer, 1, 100, 1024, HistoryLength), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.Initialize(&buffer, 0, 10, 1, 0), "");
        tra.SetMemory(NULL, sizeof(ranges));
        EXPECT_DEATH_IF_SUPPORTED(tra.Initialize(&buffer, 0, 10, 1, HistoryLength), "");
        if(tra.GetRequiredMemoryAlignment() > 1)
        {
            tra.SetMemory(reinterpret_cast<uint8_t*>(ranges) + 1, sizeof(nn::gfx::util::detail::OffsetRange) * (HistoryLength + 1) );
            EXPECT_DEATH_IF_SUPPORTED(tra.Initialize(&buffer, 0, 10, 1, HistoryLength), "");
        }
        tra.SetMemory(ranges, tra.GetRequiredMemorySize(HistoryLength) - 1);
        EXPECT_DEATH_IF_SUPPORTED(tra.Initialize(&buffer, 0, 10, 1, HistoryLength), "");

        // --- 初期化 ---
        tra.SetMemory(ranges, sizeof(ranges));
        tra.Initialize(&buffer, 0, 10, tra.AlignmentMax, HistoryLength);
        // --------------

        // 初期化後に呼ぶと死ぬことの確認
        EXPECT_DEATH_IF_SUPPORTED(tra.Initialize(&buffer, 0, 10, 1, HistoryLength), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.SetMemory(ranges, sizeof(ranges)), "");

        // 1 つも履歴がない状態で呼ぶと死ぬことの確認
        EXPECT_DEATH_IF_SUPPORTED(tra.Free(), "");

        // Begin 前に呼ぶと死ぬことの確認
        EXPECT_DEATH_IF_SUPPORTED(tra.End(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.Allocate(1,1), "");

        // --- Begin ---
        tra.Begin();
        // -------------

        // Begin 後に呼ぶと死ぬことの確認
        EXPECT_DEATH_IF_SUPPORTED(tra.Begin(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.FillHistory(), "");

        // 1 つも履歴がない状態で呼ぶと死ぬことの確認(Begin しただけでは履歴は増えない)
        EXPECT_DEATH_IF_SUPPORTED(tra.Free(), "");

        // --- End ---
        tra.End();
        // -------------

        tra.FillHistory();
        EXPECT_EQ(HistoryLength, tra.GetHistoryLength());

        // 履歴がいっぱいの状態で呼ぶと死ぬことの確認
        EXPECT_DEATH_IF_SUPPORTED(tra.Begin(), "");

        // Finalize しなくても OK
    }

}

