﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testGfxUtil_BmpReader.h"

#include <nn/nn_Macro.h>
#include <memory.h>

namespace {

struct BmpHeader
{
    uint8_t  magic[2];    /* Magic bytes 'B' and 'M'. */
    uint32_t fileSize;   /* Size of whole file. */
    uint32_t unused;      /* Should be 0. */
    uint32_t dataOffset; /* Offset from beginning of file to bitmap data. */

};

struct BmpInfo
{
    uint32_t infoSize;   /* Size of info struct (> sizeof(bmp_info)). */
    int32_t  width;       /* Width of image. */
    int32_t  height;      /* Height (< 0 means right-side up). */
    uint16_t planes;      /* Planes (should be 1). */
    uint16_t bits;        /* Number of bits (1, 4, 8, 16, 24, or 32). */
    uint32_t compression; /* 0 = none, 1 = 8-bit RLE, 2 = 4-bit RLE, etc. */

    // skip other fields
};

int IsEof(BmpInputStream* pIs)
{
    return pIs->readOffset == pIs->bufferSize;
}

int ReadByte(BmpInputStream* pIs)
{
    int value = pIs->pBuffer[pIs->readOffset];
    pIs->readOffset = pIs->readOffset + 1;
    return value;
}


bool ReadLittleEndian(uint32_t* dest, int bytes, BmpInputStream* pIs)
{
    uint32_t shift = 0;

    *dest = 0;

    while(bytes--)
    {
        if (IsEof(pIs))
            return false;

        int byte = ReadByte(pIs);

        *dest += (uint32_t)byte << shift;
        shift += 8;
    }

    return true;
}

template<typename T>
bool ReadLittleEndian(T* dest, BmpInputStream* pIs)
{
    uint32_t tmp = 0;
    if (ReadLittleEndian(&tmp, sizeof(T), pIs))
    {
        *dest = static_cast<T>(tmp);
        return true;
    }

    return false;

}

template<typename T>
bool ReadArrayLittleEndian(T* dest, size_t count, BmpInputStream* pIs)
{
    for (size_t i = 0; i < count; ++i)
    {
        if (!ReadLittleEndian(&dest[i], pIs))
            return false;
    }

    return true;
}


} // anonymous namespace


void DecodeLine24Bit(const uint8_t* pInputBuffer, size_t pixelCount, uint8_t* pOutputBuffer)
{
    // input is BGR
    for (size_t i = 0; i < pixelCount; ++i)
    {
        *(pOutputBuffer + 0) = *(pInputBuffer + 0);
        *(pOutputBuffer + 1) = *(pInputBuffer + 1);
        *(pOutputBuffer + 2) = *(pInputBuffer + 2);

        pOutputBuffer += 3;
        pInputBuffer += 3;
    }
}

bool Decode(const BmpFileInfo* pBmpFileInfo, const uint8_t* pInputBuffer, uint8_t* pOutputBuffer, size_t outputBufferSizeInBytes)
{
    size_t outputStride = pBmpFileInfo->stride;
    size_t ouputDataSizeInBytes = pBmpFileInfo->height * outputStride;
    if (ouputDataSizeInBytes > outputBufferSizeInBytes)
        return false;

    if (pBmpFileInfo->bpp != 24)
        return false;

    // will reverse lines
    size_t          inputStride                 = (outputStride + 3) & ~3;
    uint8_t*        pCurrentOutputLineBuffer    = pOutputBuffer + ouputDataSizeInBytes - outputStride;
    const uint8_t*  pCurrentInputLineBuffer     = pInputBuffer;
    const uint8_t*  pCurrentInputLineBufferEnd  = pInputBuffer + (inputStride * pBmpFileInfo->height);

    while (pCurrentInputLineBuffer < pCurrentInputLineBufferEnd)
    {
        DecodeLine24Bit(pCurrentInputLineBuffer, pBmpFileInfo->width, pCurrentOutputLineBuffer);

        pCurrentInputLineBuffer += inputStride;
        pCurrentOutputLineBuffer -= outputStride;
    }

    return true;
}



bool ReadBmpFromStream(BmpInputStream* pIs, BmpFileInfo* pBmpFileInfo, void* pPixelOutBuffer, size_t pixelOutBufferSizeInBytes)
{
    BmpHeader header;
    BmpInfo info;

    header.magic[0] = static_cast<uint8_t>(ReadByte(pIs));
    header.magic[1] = static_cast<uint8_t>(ReadByte(pIs));
    ReadLittleEndian(&header.fileSize, pIs);
    ReadLittleEndian(&header.unused, pIs);
    ReadLittleEndian(&header.dataOffset, pIs);

    ReadLittleEndian(&info.infoSize, pIs);
    ReadLittleEndian(&info.width, pIs);
    ReadLittleEndian(&info.height, pIs);
    ReadLittleEndian(&info.planes, pIs);
    ReadLittleEndian(&info.bits, pIs);
    ReadLittleEndian(&info.compression, pIs);

    if ((info.width == 0) || (info.height == 0))
        return false;
    if (info.planes != 1)
        return false;
    if (info.bits != 24)
        return false;
    if (info.compression != 0)
        return false;


    const size_t CHAR_BIT = 8;

    pBmpFileInfo->width     = info.width;
    pBmpFileInfo->height    = info.height;
    pBmpFileInfo->bpp       = info.bits;
    pBmpFileInfo->stride    = (info.width * info.bits) / CHAR_BIT;


    size_t dataSize = (info.height * info.width * info.bits) / CHAR_BIT;
    if (pIs->bufferSize < (dataSize  + header.dataOffset))
        return false;

    Decode(
        pBmpFileInfo, pIs->pBuffer + header.dataOffset,
        static_cast<uint8_t*>(pPixelOutBuffer), pixelOutBufferSizeInBytes);

    return true;
};

////////////

enum TgaType
{
    TgaType_Mapped      = 1,
    TgaType_Color       = 2,
    TgaType_Gray        = 3,
    TgaType_MappedRle   = 9,
    TgaType_ColorRle    = 10,
    TgaType_GrayRle     = 11,
    TgaType_Abits       = 0x0f,
    TgaType_Horizontal  = 0x10,
    TgaType_Vertical    = 0x20,
};

struct TgaHeader {
    uint8_t idLength;
    uint8_t colorMapType;

    /* The image type. */
    uint8_t imageType;

    /* Color Map Specification. */
    /* We need to separately specify high and low bytes to avoid endianness
         and alignment problems. */
    uint16_t colorMapIndex;
    uint16_t colorMapLength;
    uint8_t colorMapSize;

    /* Image Specification. */
    uint16_t xOrigin;
    uint16_t yOrigin;

    uint16_t width;
    uint16_t height;

    uint8_t bpp;

    /* Image descriptor.
       3-0: attribute bpp
       4:   left-to-right ordering
       5:   top-to-bottom ordering
       7-6: zero
       */
    uint8_t descriptor;
};

#define TGA_SIGNATURE "TRUEVISION-XFILE"

struct TgaFooter {
    uint32_t extensionAreaOffset;
    uint32_t developerDirectoryOffset;
    uint8_t signature[16];
    uint8_t dot;
    uint8_t null;
};

bool ReadTgaHeader(TgaHeader* pTgaFooter, BmpInputStream* pIs)
{
    bool bRes = true;

    bRes &= ReadLittleEndian(&pTgaFooter->idLength, pIs);
    bRes &= ReadLittleEndian(&pTgaFooter->colorMapType, pIs);
    bRes &= ReadLittleEndian(&pTgaFooter->imageType, pIs);
    bRes &= ReadLittleEndian(&pTgaFooter->colorMapIndex, pIs);
    bRes &= ReadLittleEndian(&pTgaFooter->colorMapLength, pIs);
    bRes &= ReadLittleEndian(&pTgaFooter->colorMapSize, pIs);
    bRes &= ReadLittleEndian(&pTgaFooter->xOrigin, pIs);
    bRes &= ReadLittleEndian(&pTgaFooter->yOrigin, pIs);
    bRes &= ReadLittleEndian(&pTgaFooter->width, pIs);
    bRes &= ReadLittleEndian(&pTgaFooter->height, pIs);
    bRes &= ReadLittleEndian(&pTgaFooter->bpp, pIs);
    bRes &= ReadLittleEndian(&pTgaFooter->descriptor, pIs);

    return bRes;
}

bool ReadTgaFooter(TgaFooter* pTgaFooter, BmpInputStream* pIs)
{
    bool bRes = true;

    BmpInputStream is = *pIs;
    is.readOffset = is.bufferSize - sizeof(TgaFooter);

    bRes &= ReadLittleEndian(&pTgaFooter->extensionAreaOffset, &is);
    bRes &= ReadLittleEndian(&pTgaFooter->developerDirectoryOffset, &is);
    bRes &= ReadArrayLittleEndian(pTgaFooter->signature, sizeof(pTgaFooter->signature) / sizeof(pTgaFooter->signature[0]), &is);
    bRes &= ReadLittleEndian(&pTgaFooter->dot, &is);
    bRes &= ReadLittleEndian(&pTgaFooter->null, &is);

    return bRes;
}

bool ReadTgaFromStream(BmpInputStream* pIs, BmpFileInfo* pBmpFileInfo, void* pPixelOutBuffer, size_t pixelOutBufferSizeInBytes)
{
    NN_UNUSED(pixelOutBufferSizeInBytes);
    NN_UNUSED(pPixelOutBuffer);
    NN_UNUSED(pBmpFileInfo);

    TgaHeader tgaHeader;
    TgaFooter tgaFooter;

    if (!ReadTgaFooter(&tgaFooter, pIs))
        return false;

    if (memcmp(tgaFooter.signature, TGA_SIGNATURE, sizeof(tgaFooter.signature)) != 0)
        return false;

    if (!ReadTgaHeader(&tgaHeader, pIs))
        return false;

    return true;
}


