﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/util/gfx_ScalableViewport.h>
#include <nns/gfx/gfx_DebugGraphicsFramework.h>
#include <nnt.h>

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn_FuncPtrInline.h>
#elif NN_GFX_IS_TARGET_GL
#include <GL/glew.h>
#endif

namespace
{
    const size_t g_GraphicsMemorySize = 8 * 1024 * 1024;
    nns::gfx::DebugGraphicsFramework g_GfxFw;

    void MakeCommand(nns::gfx::GraphicsFramework* pGfxFw, int bufferIndex, void* pUserPtr) NN_NOEXCEPT
    {
        NN_UNUSED(pUserPtr);

        nns::gfx::DebugGraphicsFramework* pDebugGfxFw = static_cast<nns::gfx::DebugGraphicsFramework*>(pGfxFw);
        nns::gfx::PrimitiveRenderer::Renderer* pRenderer = pDebugGfxFw->GetPrimitiveRenderer();
        pRenderer->SetDefaultParameters();
        {
            nn::util::Matrix4x3f viewMatrix;
            nn::util::MatrixLookAtRightHanded(&viewMatrix, NN_UTIL_VECTOR_3F_INITIALIZER(.0f, 1.0f, 10.0f), NN_UTIL_VECTOR_3F_INITIALIZER(.0f, 1.0f, .0f), NN_UTIL_VECTOR_3F_INITIALIZER(.0f, 1.0f, .0f));
            pRenderer->SetViewMatrix(&viewMatrix);
            nn::util::Matrix4x4f projMatrix;
            nn::util::MatrixOrthographicOffCenterRightHanded(&projMatrix, -1.0f, 1.0f, -1.0f, 1.0f, 0.1f, 100.0f);
            pRenderer->SetProjectionMatrix(&projMatrix);
        }

        const nn::gfx::util::ScalableViewport* pScalableViewport = pGfxFw->GetScalableViewport();
        const int horizontalDivideCount = 16;
        const int verticalDivideCount = 9;

        pGfxFw->BeginFrame(bufferIndex);
        {
            const nn::gfx::util::ScalableViewport::WindowCoordinate* pVirtualWindowCoord = pScalableViewport->GetVirtualWindowCoordinate();

            nn::gfx::CommandBuffer* pCommandBuffer = pGfxFw->GetRootCommandBuffer(bufferIndex);
            nn::gfx::ColorTargetView* pView = pGfxFw->GetColorTargetView();
            pCommandBuffer->SetRenderTargets(1, &pView, pGfxFw->GetDepthStencilView());
            pCommandBuffer->ClearColorTarget(pView, { {.5f, .5f, .5f, 1.0f} }, nullptr);

            for (int horizontalIndex = 0; horizontalIndex < horizontalDivideCount; ++horizontalIndex)
            {
                for (int verticalIndex = 0; verticalIndex < verticalDivideCount; ++verticalIndex)
                {
                    float horizontalWeight = static_cast<float>(horizontalIndex) / static_cast<float>(horizontalDivideCount);
                    float verticalWeight = static_cast<float>(verticalIndex) / static_cast<float>(verticalDivideCount);

                    nn::gfx::util::ScalableViewport::Rect virtualRect;
                    virtualRect.originX = pVirtualWindowCoord->GetWidth() * horizontalWeight;
                    virtualRect.originY = pVirtualWindowCoord->GetHeight() * verticalWeight;
                    virtualRect.width = pVirtualWindowCoord->GetWidth() * (1.0f / horizontalDivideCount);
                    virtualRect.height = pVirtualWindowCoord->GetHeight() * (1.0f / verticalDivideCount);

                    nn::gfx::util::ScalableViewport::Rect physicalRect;
                    pScalableViewport->ConvertRectVirtualToPhysical(&physicalRect, virtualRect);

                    nn::gfx::ViewportStateInfo viewportStateInfo;
                    viewportStateInfo.SetDefault();
                    pScalableViewport->SetupViewportStateInfo(&viewportStateInfo, physicalRect);
                    pCommandBuffer->SetViewports(0, 1, &viewportStateInfo);

                    nn::gfx::ScissorStateInfo scissorStateInfo;
                    pScalableViewport->SetupScissorStateInfo(&scissorStateInfo, physicalRect);
                    pCommandBuffer->SetScissors(0, 1, &scissorStateInfo);

                    nn::gfx::ClearColorValue clearColor;
                    clearColor = { { horizontalWeight, verticalWeight, 1.0f, 1.0f } };

#if NN_GFX_IS_TARGET_NVN
                    nvnCommandBufferClearColor(pCommandBuffer->ToData()->pNvnCommandBuffer, 0, clearColor.valueFloat, NVN_CLEAR_COLOR_MASK_RGBA);
#elif NN_GFX_IS_TARGET_GL
                    pCommandBuffer->Gl4SetUserCommandDynamic(
                        [](const void* pUserData)
                    {
                        glEnable(GL_SCISSOR_TEST);
                        const nn::gfx::ClearColorValue* pClearColor = static_cast<const nn::gfx::ClearColorValue*>(pUserData);
                        glClearColor(pClearColor->valueFloat[0], pClearColor->valueFloat[1], pClearColor->valueFloat[2], pClearColor->valueFloat[3]);
                        glClear(GL_COLOR_BUFFER_BIT);
                    }, &clearColor, sizeof(clearColor));
#else
#error
                    // サポートしていない API が選択されています。
#endif // NN_GFX_IS_TARGET

                    float val = static_cast<float>(verticalIndex + verticalDivideCount * horizontalIndex) / static_cast<float>(horizontalDivideCount * verticalDivideCount);
                    nn::util::Color4u8 color;
                    color.SetR(static_cast<uint8_t>(val * 255.0f));
                    color.SetG(static_cast<uint8_t>(val * 255.0f));
                    color.SetB(static_cast<uint8_t>(val * 255.0f));
                    color.SetA(255U);
                    pRenderer->SetColor(color);
                    pRenderer->DrawCone(pCommandBuffer, nns::gfx::PrimitiveRenderer::Surface_Solid, NN_UTIL_VECTOR_3F_INITIALIZER(.0f, .0f, .0f), 1.0f, /*1.0f / nn::util::SinEst(30.0f * 3.141592f / 180.0f)*/2.0f);
                }
            }
        }
        pGfxFw->EndFrame(bufferIndex);
    }
}

TEST(ScalableViewport, WindowCoordinateAPITest)
{
    using ScalableViewport = nn::gfx::util::ScalableViewport;
    ScalableViewport::WindowCoordinate windowCoord;

    windowCoord.SetWidth(1280.0f);
    EXPECT_FLOAT_EQ(windowCoord.GetWidth(), 1280.0f);

    windowCoord.SetHeight(720.0f);
    EXPECT_FLOAT_EQ(windowCoord.GetHeight(), 720.0f);

    windowCoord.SetSize(1.0f, 1.0f);
    EXPECT_FLOAT_EQ(windowCoord.GetWidth(), 1.0f);
    EXPECT_FLOAT_EQ(windowCoord.GetHeight(), 1.0f);

    windowCoord.SetOriginMode(ScalableViewport::OriginMode_UpperLeft);
    EXPECT_EQ(windowCoord.GetOriginMode(), ScalableViewport::OriginMode_UpperLeft);
}

TEST(ScalableViewport, ScalableViewportAPITest)
{
    using ScalableViewport = nn::gfx::util::ScalableViewport;
    ScalableViewport::WindowCoordinate virtualWindowCoord;
    ScalableViewport::WindowCoordinate physicalWindowCoord;

    virtualWindowCoord.SetWidth(1280.0f);
    virtualWindowCoord.SetHeight(720.0f);
    virtualWindowCoord.SetOriginMode(ScalableViewport::OriginMode_LowerLeft);
    physicalWindowCoord.SetWidth(1920.0f);
    physicalWindowCoord.SetHeight(1080.0f);
    physicalWindowCoord.SetOriginMode(ScalableViewport::OriginMode_UpperLeft);

    ScalableViewport scalableViewport;
    EXPECT_FALSE(scalableViewport.IsInitialized());
    scalableViewport.Initialize(virtualWindowCoord, physicalWindowCoord);
    EXPECT_TRUE(scalableViewport.IsInitialized());

    const ScalableViewport::WindowCoordinate* pVirtualWindowCoord = scalableViewport.GetVirtualWindowCoordinate();
    EXPECT_FLOAT_EQ(pVirtualWindowCoord->GetWidth(), virtualWindowCoord.GetWidth());
    EXPECT_FLOAT_EQ(pVirtualWindowCoord->GetWidth(), 1280.0f);
    EXPECT_FLOAT_EQ(pVirtualWindowCoord->GetHeight(), virtualWindowCoord.GetHeight());
    EXPECT_FLOAT_EQ(pVirtualWindowCoord->GetHeight(), 720.0f);
    EXPECT_EQ(pVirtualWindowCoord->GetOriginMode(), virtualWindowCoord.GetOriginMode());
    EXPECT_EQ(pVirtualWindowCoord->GetOriginMode(), ScalableViewport::OriginMode_LowerLeft);

    const ScalableViewport::WindowCoordinate* pPhysicalWindowCoord = scalableViewport.GetPhysicalWindowCoordinate();
    EXPECT_FLOAT_EQ(pPhysicalWindowCoord->GetWidth(), physicalWindowCoord.GetWidth());
    EXPECT_FLOAT_EQ(pPhysicalWindowCoord->GetWidth(), 1920.0f);
    EXPECT_FLOAT_EQ(pPhysicalWindowCoord->GetHeight(), physicalWindowCoord.GetHeight());
    EXPECT_FLOAT_EQ(pPhysicalWindowCoord->GetHeight(), 1080.0f);
    EXPECT_EQ(pPhysicalWindowCoord->GetOriginMode(), physicalWindowCoord.GetOriginMode());
    EXPECT_EQ(pPhysicalWindowCoord->GetOriginMode(), ScalableViewport::OriginMode_UpperLeft);

    ScalableViewport::Rect rect = { .0f, .0f, 1280.0f, 720.0f };
    scalableViewport.ConvertRectVirtualToPhysical(&rect, rect);
    EXPECT_FLOAT_EQ(rect.width, 1920.0f);
    EXPECT_FLOAT_EQ(rect.height, 1080.0f);
    scalableViewport.ConvertRectPhysicalToVirtual(&rect, rect);
    EXPECT_FLOAT_EQ(rect.originX, 0.0f);
    EXPECT_FLOAT_EQ(rect.originY, 0.0f);
    EXPECT_FLOAT_EQ(rect.width, 1280.0f);
    EXPECT_FLOAT_EQ(rect.height, 720.0f);

    nn::gfx::ViewportStateInfo viewportStateInfo;
    viewportStateInfo.SetOriginX(640.0f);
    viewportStateInfo.SetOriginY(360.0f);
    viewportStateInfo.SetWidth(640.0f);
    viewportStateInfo.SetHeight(360.0f);
    scalableViewport.SetupRectFromViewportStateInfo(&rect, viewportStateInfo);
    EXPECT_FLOAT_EQ(rect.originX, 640.0f);
    EXPECT_FLOAT_EQ(rect.originY, 360.0f);
    EXPECT_FLOAT_EQ(rect.width, 640.0f);
    EXPECT_FLOAT_EQ(rect.height, 360.0f);
}

TEST(ScalableViewport, SameSize_VirtualLower_PhysicalLower)
{
    using namespace nn::gfx::util;
    const int processFrameCount = 1 * 60;

    ScalableViewport::WindowCoordinate virtualWindowCoord;
    {
        virtualWindowCoord.SetSize(1280.0f, 720.0f);
        virtualWindowCoord.SetOriginMode(ScalableViewport::OriginMode_LowerLeft);
    }
    ScalableViewport::WindowCoordinate physicalWindowCoord;
    {
        physicalWindowCoord.SetSize(1280.0f, 720.0f);
        physicalWindowCoord.SetOriginMode(ScalableViewport::OriginMode_LowerLeft);
    }

    // フレームワークを初期化
    decltype(g_GfxFw)::FrameworkInfo info;
    info.SetDefault();
    info.SetVirtualWindowCoordinate(virtualWindowCoord);
    info.SetPhysicalWindowCoordinate(physicalWindowCoord);

    g_GfxFw.Initialize(info);

    g_GfxFw.SetMakeCommandCallback(MakeCommand, nullptr);
    for (int processFrame = 0; processFrame < processFrameCount; ++processFrame)
    {
        g_GfxFw.ProcessFrame();
    }
    g_GfxFw.QueueFinish();
    g_GfxFw.Finalize();
}

TEST(ScalableViewport, SameSize_VirtualUpper_PhysicalLower)
{
    using namespace nn::gfx::util;
    const int processFrameCount = 1 * 60;

    ScalableViewport::WindowCoordinate virtualWindowCoord;
    {
        virtualWindowCoord.SetSize(1280.0f, 720.0f);
        virtualWindowCoord.SetOriginMode(ScalableViewport::OriginMode_UpperLeft);
    }
    ScalableViewport::WindowCoordinate physicalWindowCoord;
    {
        physicalWindowCoord.SetSize(1280.0f, 720.0f);
        physicalWindowCoord.SetOriginMode(ScalableViewport::OriginMode_LowerLeft);
    }

    // フレームワークを初期化
    decltype(g_GfxFw)::FrameworkInfo info;
    info.SetDefault();
    info.SetVirtualWindowCoordinate(virtualWindowCoord);
    info.SetPhysicalWindowCoordinate(physicalWindowCoord);

    g_GfxFw.Initialize(info);

    g_GfxFw.SetMakeCommandCallback(MakeCommand, nullptr);
    for (int processFrame = 0; processFrame < processFrameCount; ++processFrame)
    {
        g_GfxFw.ProcessFrame();
    }
    g_GfxFw.QueueFinish();
    g_GfxFw.Finalize();
}

TEST(ScalableViewport, DifferentSize_VirtualLower_PhysicalLower)
{
    using namespace nn::gfx::util;
    const int processFrameCount = 1 * 60;

    ScalableViewport::WindowCoordinate virtualWindowCoord;
    {
        virtualWindowCoord.SetSize(1.0f, 1.0f);
        virtualWindowCoord.SetOriginMode(ScalableViewport::OriginMode_LowerLeft);
    }
    ScalableViewport::WindowCoordinate physicalWindowCoord;
    {
        physicalWindowCoord.SetSize(1280.0f, 720.0f);
        physicalWindowCoord.SetOriginMode(ScalableViewport::OriginMode_LowerLeft);
    }

    // フレームワークを初期化
    decltype(g_GfxFw)::FrameworkInfo info;
    info.SetDefault();
    info.SetVirtualWindowCoordinate(virtualWindowCoord);
    info.SetPhysicalWindowCoordinate(physicalWindowCoord);

    g_GfxFw.Initialize(info);

    g_GfxFw.SetMakeCommandCallback(MakeCommand, nullptr);
    for (int processFrame = 0; processFrame < processFrameCount; ++processFrame)
    {
        g_GfxFw.ProcessFrame();
    }
    g_GfxFw.QueueFinish();
    g_GfxFw.Finalize();
}

TEST(ScalableViewport, DifferentSize_VirtualUpper_PhysicalLower)
{
    using namespace nn::gfx::util;
    const int processFrameCount = 1 * 60;

    ScalableViewport::WindowCoordinate virtualWindowCoord;
    {
        virtualWindowCoord.SetSize(1.0f, 1.0f);
        virtualWindowCoord.SetOriginMode(ScalableViewport::OriginMode_UpperLeft);
    }
    ScalableViewport::WindowCoordinate physicalWindowCoord;
    {
        physicalWindowCoord.SetSize(1280.0f, 720.0f);
        physicalWindowCoord.SetOriginMode(ScalableViewport::OriginMode_LowerLeft);
    }

    // フレームワークを初期化
    decltype(g_GfxFw)::FrameworkInfo info;
    info.SetDefault();
    info.SetVirtualWindowCoordinate(virtualWindowCoord);
    info.SetPhysicalWindowCoordinate(physicalWindowCoord);

    g_GfxFw.Initialize(info);

    g_GfxFw.SetMakeCommandCallback(MakeCommand, nullptr);
    for (int processFrame = 0; processFrame < processFrameCount; ++processFrame)
    {
        g_GfxFw.ProcessFrame();
    }
    g_GfxFw.QueueFinish();
    g_GfxFw.Finalize();
}

/*
* NOTE:
* 物理ウィンドウ座標系が 左上のケースは NVN でのみ UpperLeft をテストします。
* 理由は以下の通りです。
*
* OpenGL:
* - glClipControl() の GL_UPPER_LEFTは、Normalized Device Coordinate (NDC) の Y 座標の符号が反転する(ように計算される)だけで、ウィンドウ座標の座標系が変わるわけではありません。
*   なので、Viewportのオフセットはそのまま左下が原点として作用し、Viewportの内容だけが上下反転します。
*/

#if NN_GFX_IS_TARGET_NVN

TEST(ScalableViewport, SameSize_VirtualLower_PhysicalUpper)
{
    using namespace nn::gfx::util;
    const int processFrameCount = 1 * 60;

    ScalableViewport::WindowCoordinate virtualWindowCoord;
    {
        virtualWindowCoord.SetSize(1280.0f, 720.0f);
        virtualWindowCoord.SetOriginMode(ScalableViewport::OriginMode_LowerLeft);
    }
    ScalableViewport::WindowCoordinate physicalWindowCoord;
    {
        physicalWindowCoord.SetSize(1280.0f, 720.0f);
        physicalWindowCoord.SetOriginMode(ScalableViewport::OriginMode_UpperLeft);
    }

    // フレームワークを初期化
    decltype(g_GfxFw)::FrameworkInfo info;
    info.SetDefault();
    info.SetVirtualWindowCoordinate(virtualWindowCoord);
    info.SetPhysicalWindowCoordinate(physicalWindowCoord);

    g_GfxFw.Initialize(info);

    g_GfxFw.SetMakeCommandCallback(MakeCommand, nullptr);
    for (int processFrame = 0; processFrame < processFrameCount; ++processFrame)
    {
        g_GfxFw.ProcessFrame();
    }
    g_GfxFw.QueueFinish();
    g_GfxFw.Finalize();
}

TEST(ScalableViewport, SameSize_VirtualUpper_PhysicalUpper)
{
    using namespace nn::gfx::util;
    const int processFrameCount = 1 * 60;

    ScalableViewport::WindowCoordinate virtualWindowCoord;
    {
        virtualWindowCoord.SetSize(1280.0f, 720.0f);
        virtualWindowCoord.SetOriginMode(ScalableViewport::OriginMode_UpperLeft);
    }
    ScalableViewport::WindowCoordinate physicalWindowCoord;
    {
        physicalWindowCoord.SetSize(1280.0f, 720.0f);
        physicalWindowCoord.SetOriginMode(ScalableViewport::OriginMode_UpperLeft);
    }

    // フレームワークを初期化
    decltype(g_GfxFw)::FrameworkInfo info;
    info.SetDefault();
    info.SetVirtualWindowCoordinate(virtualWindowCoord);
    info.SetPhysicalWindowCoordinate(physicalWindowCoord);

    g_GfxFw.Initialize(info);

    g_GfxFw.SetMakeCommandCallback(MakeCommand, nullptr);
    for (int processFrame = 0; processFrame < processFrameCount; ++processFrame)
    {
        g_GfxFw.ProcessFrame();
    }
    g_GfxFw.QueueFinish();
    g_GfxFw.Finalize();
}

TEST(ScalableViewport,  DifferentSize_VirtualLower_PhysicalUpper)
{
    using namespace nn::gfx::util;
    const int processFrameCount = 1 * 60;

    ScalableViewport::WindowCoordinate virtualWindowCoord;
    {
        virtualWindowCoord.SetSize(1.0f, 1.0f);
        virtualWindowCoord.SetOriginMode(ScalableViewport::OriginMode_LowerLeft);
    }
    ScalableViewport::WindowCoordinate physicalWindowCoord;
    {
        physicalWindowCoord.SetSize(1280.0f, 720.0f);
        physicalWindowCoord.SetOriginMode(ScalableViewport::OriginMode_UpperLeft);
    }

    // フレームワークを初期化
    decltype(g_GfxFw)::FrameworkInfo info;
    info.SetDefault();
    info.SetVirtualWindowCoordinate(virtualWindowCoord);
    info.SetPhysicalWindowCoordinate(physicalWindowCoord);

    g_GfxFw.Initialize(info);

    g_GfxFw.SetMakeCommandCallback(MakeCommand, nullptr);
    for (int processFrame = 0; processFrame < processFrameCount; ++processFrame)
    {
        g_GfxFw.ProcessFrame();
    }
    g_GfxFw.QueueFinish();
    g_GfxFw.Finalize();
}

TEST(ScalableViewport,  DifferentSize_VirtualUpper_PhysicalUpper)
{
    using namespace nn::gfx::util;
    const int processFrameCount = 1 * 60;

    ScalableViewport::WindowCoordinate virtualWindowCoord;
    {
        virtualWindowCoord.SetSize(1.0f, 1.0f);
        virtualWindowCoord.SetOriginMode(ScalableViewport::OriginMode_UpperLeft);
    }
    ScalableViewport::WindowCoordinate physicalWindowCoord;
    {
        physicalWindowCoord.SetSize(1280.0f, 720.0f);
        physicalWindowCoord.SetOriginMode(ScalableViewport::OriginMode_UpperLeft);
    }

    // フレームワークを初期化
    decltype(g_GfxFw)::FrameworkInfo info;
    info.SetDefault();
    info.SetVirtualWindowCoordinate(virtualWindowCoord);
    info.SetPhysicalWindowCoordinate(physicalWindowCoord);

    g_GfxFw.Initialize(info);

    g_GfxFw.SetMakeCommandCallback(MakeCommand, nullptr);
    for (int processFrame = 0; processFrame < processFrameCount; ++processFrame)
    {
        g_GfxFw.ProcessFrame();
    }
    g_GfxFw.QueueFinish();
    g_GfxFw.Finalize();
}

#endif

#ifdef NN_BUILD_TARGET_PLATFORM_OS_NN

TEST(ScalableViewport, VirtualLower_PhysicalLower_ScalingScanBuffer)
{
    using namespace nn::gfx::util;
    const int processFrameCount = 1 * 60;

    ScalableViewport::WindowCoordinate virtualWindowCoord;
    {
        virtualWindowCoord.SetSize(1280.0f, 720.0f);
        virtualWindowCoord.SetOriginMode(ScalableViewport::OriginMode_LowerLeft);
    }
    ScalableViewport::WindowCoordinate physicalWindowCoord;
    {
        float scale = 0.0625f;
        physicalWindowCoord.SetSize(1280.0f * scale, 720.0f * scale);
        physicalWindowCoord.SetOriginMode(ScalableViewport::OriginMode_LowerLeft);
    }

    // フレームワークを初期化
    decltype(g_GfxFw)::FrameworkInfo info;
    info.SetDefault();
    info.SetVirtualWindowCoordinate(virtualWindowCoord);
    info.SetPhysicalWindowCoordinate(physicalWindowCoord);
    g_GfxFw.Initialize(info);

    g_GfxFw.SetMakeCommandCallback(MakeCommand, nullptr);
    for (int processFrame = 0; processFrame < processFrameCount; ++processFrame)
    {
        g_GfxFw.ProcessFrame();
    }
    g_GfxFw.QueueFinish();
    g_GfxFw.Finalize();
}

TEST(ScalableViewport, SameSize_VirtualUpper_PhysicalLower_ScalingScanBuffer)
{
    using namespace nn::gfx::util;
    const int processFrameCount = 1 * 60;

    ScalableViewport::WindowCoordinate virtualWindowCoord;
    {
        virtualWindowCoord.SetSize(1280.0f, 720.0f);
        virtualWindowCoord.SetOriginMode(ScalableViewport::OriginMode_UpperLeft);
    }
    ScalableViewport::WindowCoordinate physicalWindowCoord;
    {
        float scale = 0.0625f;
        physicalWindowCoord.SetSize(1280.0f * scale, 720.0f * scale);
        physicalWindowCoord.SetOriginMode(ScalableViewport::OriginMode_LowerLeft);
    }

    // フレームワークを初期化
    decltype(g_GfxFw)::FrameworkInfo info;
    info.SetDefault();
    info.SetVirtualWindowCoordinate(virtualWindowCoord);
    info.SetPhysicalWindowCoordinate(physicalWindowCoord);

    g_GfxFw.Initialize(info);

    g_GfxFw.SetMakeCommandCallback(MakeCommand, nullptr);
    for (int processFrame = 0; processFrame < processFrameCount; ++processFrame)
    {
        g_GfxFw.ProcessFrame();
    }
    g_GfxFw.QueueFinish();
    g_GfxFw.Finalize();
}

TEST(ScalableViewport, DifferentSize_VirtualLower_PhysicalLower_ScalingScanBuffer)
{
    using namespace nn::gfx::util;
    const int processFrameCount = 1 * 60;

    ScalableViewport::WindowCoordinate virtualWindowCoord;
    {
        virtualWindowCoord.SetSize(1.0f, 1.0f);
        virtualWindowCoord.SetOriginMode(ScalableViewport::OriginMode_LowerLeft);
    }
    ScalableViewport::WindowCoordinate physicalWindowCoord;
    {
        float scale = 0.0625f;
        physicalWindowCoord.SetSize(1280.0f * scale, 720.0f * scale);
        physicalWindowCoord.SetOriginMode(ScalableViewport::OriginMode_LowerLeft);
    }

    // フレームワークを初期化
    decltype(g_GfxFw)::FrameworkInfo info;
    info.SetDefault();
    info.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, 64 * 1024 * 1024);
    info.SetVirtualWindowCoordinate(virtualWindowCoord);
    info.SetPhysicalWindowCoordinate(physicalWindowCoord);

    g_GfxFw.Initialize(info);

    g_GfxFw.SetMakeCommandCallback(MakeCommand, nullptr);
    for (int processFrame = 0; processFrame < processFrameCount; ++processFrame)
    {
        g_GfxFw.ProcessFrame();
    }
    g_GfxFw.QueueFinish();
    g_GfxFw.Finalize();
}

TEST(ScalableViewport, DifferentSize_VirtualUpper_PhysicalLower_ScalingScanBuffer)
{
    using namespace nn::gfx::util;
    const int processFrameCount = 1 * 60;

    ScalableViewport::WindowCoordinate virtualWindowCoord;
    {
        virtualWindowCoord.SetSize(1.0f, 1.0f);
        virtualWindowCoord.SetOriginMode(ScalableViewport::OriginMode_UpperLeft);
    }
    ScalableViewport::WindowCoordinate physicalWindowCoord;
    {
        float scale = 0.0625f;
        physicalWindowCoord.SetSize(1280.0f * scale, 720.0f * scale);
        physicalWindowCoord.SetOriginMode(ScalableViewport::OriginMode_LowerLeft);
    }

    // フレームワークを初期化
    decltype(g_GfxFw)::FrameworkInfo info;
    info.SetDefault();
    info.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, 64 * 1024 * 1024);
    info.SetVirtualWindowCoordinate(virtualWindowCoord);
    info.SetPhysicalWindowCoordinate(physicalWindowCoord);

    g_GfxFw.Initialize(info);

    g_GfxFw.SetMakeCommandCallback(MakeCommand, nullptr);
    for (int processFrame = 0; processFrame < processFrameCount; ++processFrame)
    {
        g_GfxFw.ProcessFrame();
    }
    g_GfxFw.QueueFinish();
    g_GfxFw.Finalize();
}

#if NN_GFX_IS_TARGET_NVN

TEST(ScalableViewport, SameSize_VirtualLower_PhysicalUpper_ScalingScanBuffer)
{
    using namespace nn::gfx::util;
    const int processFrameCount = 1 * 60;

    ScalableViewport::WindowCoordinate virtualWindowCoord;
    {
        virtualWindowCoord.SetSize(1280.0f, 720.0f);
        virtualWindowCoord.SetOriginMode(ScalableViewport::OriginMode_LowerLeft);
    }
    ScalableViewport::WindowCoordinate physicalWindowCoord;
    {
        float scale = 0.0625f;
        physicalWindowCoord.SetSize(1280.0f * scale, 720.0f * scale);
        physicalWindowCoord.SetOriginMode(ScalableViewport::OriginMode_UpperLeft);
    }

    // フレームワークを初期化
    decltype(g_GfxFw)::FrameworkInfo info;
    info.SetDefault();
    info.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, 64 * 1024 * 1024);
    info.SetVirtualWindowCoordinate(virtualWindowCoord);
    info.SetPhysicalWindowCoordinate(physicalWindowCoord);

    g_GfxFw.Initialize(info);

    g_GfxFw.SetMakeCommandCallback(MakeCommand, nullptr);
    for (int processFrame = 0; processFrame < processFrameCount; ++processFrame)
    {
        g_GfxFw.ProcessFrame();
    }
    g_GfxFw.QueueFinish();
    g_GfxFw.Finalize();
}

TEST(ScalableViewport, SameSize_VirtualUpper_PhysicalUpper_ScalingScanBuffer)
{
    using namespace nn::gfx::util;
    const int processFrameCount = 1 * 60;

    ScalableViewport::WindowCoordinate virtualWindowCoord;
    {
        virtualWindowCoord.SetSize(1280.0f, 720.0f);
        virtualWindowCoord.SetOriginMode(ScalableViewport::OriginMode_UpperLeft);
    }
    ScalableViewport::WindowCoordinate physicalWindowCoord;
    {
        float scale = 0.0625f;
        physicalWindowCoord.SetSize(1280.0f * scale, 720.0f * scale);
        physicalWindowCoord.SetOriginMode(ScalableViewport::OriginMode_UpperLeft);
    }

    // フレームワークを初期化
    decltype(g_GfxFw)::FrameworkInfo info;
    info.SetDefault();
    info.SetDisplayWidth(1280);
    info.SetDisplayHeight(720);
    info.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, 64 * 1024 * 1024);
    info.SetVirtualWindowCoordinate(virtualWindowCoord);
    info.SetPhysicalWindowCoordinate(physicalWindowCoord);

    g_GfxFw.Initialize(info);

    g_GfxFw.SetMakeCommandCallback(MakeCommand, nullptr);
    for (int processFrame = 0; processFrame < processFrameCount; ++processFrame)
    {
        g_GfxFw.ProcessFrame();
    }
    g_GfxFw.QueueFinish();
    g_GfxFw.Finalize();
}

TEST(ScalableViewport, DifferentSize_VirtualLower_PhysicalUpper_ScalingScanBuffer)
{
    using namespace nn::gfx::util;
    const int processFrameCount = 1 * 60;

    ScalableViewport::WindowCoordinate virtualWindowCoord;
    {
        virtualWindowCoord.SetSize(1.0f, 1.0f);
        virtualWindowCoord.SetOriginMode(ScalableViewport::OriginMode_LowerLeft);
    }
    ScalableViewport::WindowCoordinate physicalWindowCoord;
    {
        float scale = 0.0625f;
        physicalWindowCoord.SetSize(1280.0f * scale, 720.0f * scale);
        physicalWindowCoord.SetOriginMode(ScalableViewport::OriginMode_UpperLeft);
    }

    // フレームワークを初期化
    decltype(g_GfxFw)::FrameworkInfo info;
    info.SetDefault();
    info.SetDisplayWidth(1280);
    info.SetDisplayHeight(720);
    info.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, 64 * 1024 * 1024);
    info.SetVirtualWindowCoordinate(virtualWindowCoord);
    info.SetPhysicalWindowCoordinate(physicalWindowCoord);

    g_GfxFw.Initialize(info);

    g_GfxFw.SetMakeCommandCallback(MakeCommand, nullptr);
    for (int processFrame = 0; processFrame < processFrameCount; ++processFrame)
    {
        g_GfxFw.ProcessFrame();
    }
    g_GfxFw.QueueFinish();
    g_GfxFw.Finalize();
}

TEST(ScalableViewport, DifferentSize_VirtualUpper_PhysicalUpper_ScalingScanBuffer)
{
    using namespace nn::gfx::util;
    const int processFrameCount = 1 * 60;

    ScalableViewport::WindowCoordinate virtualWindowCoord;
    {
        virtualWindowCoord.SetSize(1.0f, 1.0f);
        virtualWindowCoord.SetOriginMode(ScalableViewport::OriginMode_UpperLeft);
    }
    ScalableViewport::WindowCoordinate physicalWindowCoord;
    {
        float scale = 0.0625f;
        physicalWindowCoord.SetSize(1280.0f * scale, 720.0f * scale);
        physicalWindowCoord.SetOriginMode(ScalableViewport::OriginMode_UpperLeft);
    }

    // フレームワークを初期化
    decltype(g_GfxFw)::FrameworkInfo info;
    info.SetDefault();
    info.SetDisplayWidth(1280);
    info.SetDisplayHeight(720);
    info.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, 64 * 1024 * 1024);
    info.SetVirtualWindowCoordinate(virtualWindowCoord);
    info.SetPhysicalWindowCoordinate(physicalWindowCoord);

    g_GfxFw.Initialize(info);

    g_GfxFw.SetMakeCommandCallback(MakeCommand, nullptr);
    for (int processFrame = 0; processFrame < processFrameCount; ++processFrame)
    {
        g_GfxFw.ProcessFrame();
    }
    g_GfxFw.QueueFinish();
    g_GfxFw.Finalize();
}

#endif // NN_GFX_IS_TARGET_NVN

#endif // NN_BUILD_TARGET_PLATFORM_OS_NN

extern "C" void nnMain()
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(g_GraphicsMemorySize);
    int ret = RUN_ALL_TESTS();
    nnt::Exit(ret);
}
