﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// 全メモリが確保済の状態からはじめて
// 1 つのスレッドが順次 Release() しながら
// 複数スレッドが同時に Allocate() するテスト

#include <vector>

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nnt.h>
#include <cstdlib>
#include <algorithm>

#include <nn/gfx/util/detail/gfx_OffsetRingBuffer.h>
#include "testGfxUtil_WorkerThreads.h"

namespace {
    struct AllocatedOffsetInfo
    {
        ptrdiff_t offset;
        ptrdiff_t requestedSize;
        ptrdiff_t requestedAlignment;
    };

    struct ThreadParameter
    {
        nn::gfx::util::detail::OffsetRingBuffer* pRing;
        int allocateCount;
        ptrdiff_t allocateSize;
        ptrdiff_t allocateAlignment;
        std::vector<AllocatedOffsetInfo> allocatedList;

        nn::gfx::util::detail::OffsetRange releaseRange;
        ptrdiff_t releaseSize;
    };

    void ThreadFunction(int tid, ThreadParameter* pParam)
    {
        const ptrdiff_t InvalidOffset = nn::gfx::util::detail::OffsetRingBuffer::InvalidOffset;
        if(tid == 0)
        {
            nn::gfx::util::detail::OffsetRange range = pParam->releaseRange;
            int allocateCount = 0;
            while(range.size > 0 || allocateCount < pParam->allocateCount)
            {
                if(range.size > 0)
                {
                    nn::gfx::util::detail::OffsetRange r;
                    r.base = range.base % pParam->pRing->GetSize();
                    r.size = std::min<ptrdiff_t>(pParam->releaseSize, range.size);
                    pParam->pRing->ReleaseOffsetRange(&r);
                    range.base += r.size;
                    range.size -= r.size;
                }

                if(allocateCount < pParam->allocateCount)
                {
                    ptrdiff_t offset = pParam->pRing->Allocate(pParam->allocateSize, pParam->allocateAlignment);
                    if(offset != InvalidOffset)
                    {
                        AllocatedOffsetInfo info;
                        info.offset = offset;
                        info.requestedSize = pParam->allocateSize;
                        info.requestedAlignment = pParam->allocateAlignment;
                        pParam->allocatedList.push_back(info);
                        allocateCount++;
                    }
                }
            }
        }
        else
        {
            for(int i = 0; i < pParam->allocateCount; i++)
            {
                ptrdiff_t offset = InvalidOffset;
                do
                {
                    offset = pParam->pRing->Allocate(pParam->allocateSize, pParam->allocateAlignment);
                }
                while(offset == InvalidOffset);
                {
                    AllocatedOffsetInfo info;
                    info.offset = offset;
                    info.requestedSize = pParam->allocateSize;
                    info.requestedAlignment = pParam->allocateAlignment;
                    pParam->allocatedList.push_back(info);
                }
            }
        }
    }
}

TEST(OffsetRingBuffer, MultiThreadRelease)
{
    using namespace nnt::gfxUtil;
    static const int AllocateCount = 4096;
    static const ptrdiff_t AllocateSize      = 128 * 3;
    static const ptrdiff_t AllocateAlignment = 256;
    static const ptrdiff_t AllocateSizePerThread = 512 * AllocateCount;
    static const ptrdiff_t ReleaseSize       = AllocateSize * 4;

    static const int StackSize = 1024 * 1024;
    static const int MaxNumberOfThreads = 64;
    NN_STATIC_ASSERT(StackSize % nn::os::ThreadStackAlignment == 0);

    //const int InvalidOffset = nn::gfx::util::detail::OffsetRingBuffer::InvalidOffset;
    nn::gfx::util::detail::OffsetRingBuffer ring;

    // ワーカースレッドを用意
    WorkerThreads<ThreadParameter> wthreads;
    wthreads.Initialize(ThreadFunction, StackSize);
    const int numThreads = wthreads.GetNumberOfThreads();
    // パラメータを設定
    ThreadParameter params[MaxNumberOfThreads];
    ptrdiff_t firstOffset = (AllocateCount * numThreads) / 2;
    ptrdiff_t offsetSize = numThreads * AllocateSizePerThread + AllocateSize + AllocateAlignment;
    for(int t = 0; t < numThreads; t++)
    {
        ThreadParameter& param = params[t];
        param.pRing = &ring;
        param.allocateCount     = AllocateCount;
        param.allocateSize      = AllocateSize;
        param.allocateAlignment = AllocateAlignment;

        param.releaseSize = ReleaseSize;
        param.releaseRange.base = firstOffset + 1;
        param.releaseRange.size = offsetSize;
        wthreads.SetThreadParameter(t, &param);
    }
    wthreads.PrepareWorker();

    ptrdiff_t ringSize = offsetSize + 1;
    ring.Initialize(0, ringSize);
    // 強制的に n - 1 個確保した状態にする
    {
        nn::gfx::util::detail::OffsetRange range;
        ring.Begin();                                      // head = 0               , tail = ringSize - 1
        ring.Allocate(firstOffset + 1, 1);                 // head = first + 1       , tail = ringSize - 1
        ring.End(&range);
        ring.ReleaseOffsetRange(&range);                   // head = first + 1       , tail = ringSize + first
        ring.Begin();
        ring.Allocate(ringSize - firstOffset - 1, 1);      // head = ringSize        , tail = ringSize + first
        ring.Allocate(firstOffset, 1);                     // head = ringSize + first, tail = ringSize + first
        ring.End(&range);                                  // head = first           , tail = first
    }
    ring.Begin();
    {
        wthreads.StartWorker();
        wthreads.WaitWorkerComplete();
    }
    nn::gfx::util::detail::OffsetRange range;
    ring.End(&range);

    // 正しく確保されているか確認
    {
        const int totalCount = AllocateCount * numThreads;
        std::vector<AllocatedOffsetInfo> allocatedInfoList;
        allocatedInfoList.reserve(static_cast<size_t>(totalCount));

        for(int i = 0; i < wthreads.GetNumberOfThreads(); i++)
        {
            const ThreadParameter* pParam = wthreads.GetThreadParameter(i);
            allocatedInfoList.insert(allocatedInfoList.end(), pParam->allocatedList.begin(), pParam->allocatedList.end());
        }

        std::sort(
            allocatedInfoList.begin(),
            allocatedInfoList.end(),
            [](const AllocatedOffsetInfo& a, const AllocatedOffsetInfo& b) -> bool
            {
                return a.offset < b.offset;
            }
        );

        // 重複なく確保されていることの確認
        for(size_t i = 1; i < allocatedInfoList.size(); i++)
        {
            const AllocatedOffsetInfo& prev = allocatedInfoList[i - 1];
            const AllocatedOffsetInfo& info = allocatedInfoList[i];
            EXPECT_GE(info.offset, prev.offset + prev.requestedSize);
        }
    }

    wthreads.Finalize();
}
