﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// MemoryPoolAllocator の確認テスト

#include <nn/nn_Assert.h>
#include <nn/gfx/util/gfx_MemoryPoolAllocator.h>
#include <nnt.h>

bool g_bUserAllocatorInvalid;
int g_UserAllocatorAllocCount;
void* g_pExpectedAllocateUserData;
void* g_pExpectedFreeUserData;

void* UserAllocatorMalloc(size_t size, void* pUserData)
{
    // Initialize() で渡したユーザーポインタと同じか
    EXPECT_EQ(g_pExpectedAllocateUserData, pUserData);

    if(g_bUserAllocatorInvalid)
    {
        return nullptr;
    }

    void* ret = malloc(size);
    if(ret)
    {
        g_UserAllocatorAllocCount++;
    }
    return ret;
}

void UserAllocatorFree(void* ptr, void* pUserData)
{
    // Initialize() で渡したユーザーポインタと同じか
    EXPECT_EQ(g_pExpectedFreeUserData, pUserData);
    free(ptr);

    g_UserAllocatorAllocCount--;
}

TEST(MemoryPoolAllocator, SingleThread)
{
    const ptrdiff_t baseOffset = 65536;
    const size_t size = nn::gfx::util::MemoryPoolAllocator::AllocatorUnitSize * 20;
    const size_t allocatableAlignmentMax = 65536;
    const ptrdiff_t InvalidOffset = nn::gfx::util::MemoryPoolAllocator::InvalidOffset;
    nn::gfx::MemoryPool memoryPool;
    nn::gfx::util::MemoryPoolAllocator allocator;
    int allocateFunctionUserData = 0;
    int freeFunctionUserData = 0;

    g_bUserAllocatorInvalid = false;
    g_UserAllocatorAllocCount = 0;
    g_pExpectedAllocateUserData = reinterpret_cast<void*>(&allocateFunctionUserData);
    g_pExpectedFreeUserData = reinterpret_cast<void*>(&freeFunctionUserData);

    // Initialize() 前は未初期化状態か
    EXPECT_FALSE(allocator.IsInitialized());

    // 初期化する
    allocator.Initialize(
        UserAllocatorMalloc,
        g_pExpectedAllocateUserData,
        UserAllocatorFree,
        g_pExpectedFreeUserData,
        &memoryPool,
        baseOffset,
        size,
        allocatableAlignmentMax,
        false
        );

    // Initialize() 後は初期化済み状態か
    EXPECT_TRUE(allocator.IsInitialized());

    // Initialize() で渡したパラメータを取得できているか
    EXPECT_EQ(allocator.GetMemoryPool(), &memoryPool);
    EXPECT_EQ(allocator.GetBaseOffset(), baseOffset);
    EXPECT_EQ(allocator.GetSize(), size);
    EXPECT_EQ(allocator.GetAllocatableAlignmentMax(), allocatableAlignmentMax);

    // size 分の範囲を１０回アロケートする。
    // １回ごとに Free() するので、すべて成功するはず。
    for(int i=0; i<10; i++)
    {
        ptrdiff_t offset = allocator.Allocate( size, 1 );

        // 戻ってきたオフセットは範囲内か
        EXPECT_GE(offset, baseOffset);
        EXPECT_LT(offset, baseOffset + static_cast<ptrdiff_t>(size));

        allocator.Free( offset );
    }

    // アライメントを指定したアロケート。
    {
        size_t alignment = 1;
        for(int i=0; i<17; i++){
            ptrdiff_t offset = allocator.Allocate( 1, alignment );

            // 戻ってきたオフセットは範囲内か
            EXPECT_GE(offset, baseOffset);
            EXPECT_LT(offset, baseOffset + static_cast<ptrdiff_t>(size));
            // アライメント確認
            EXPECT_EQ((offset % alignment), 0);

            allocator.Free( offset );
            alignment = (alignment << 1);
        }
    }

    // size を超える値でアロケートする。
    // 失敗して InvalidOffset が返るはず。
    {
        ptrdiff_t offset = allocator.Allocate( size + 1, 1 );

        // InvalidOffset が返っているか
        EXPECT_EQ(offset, InvalidOffset);

        // Free(InvalidOffset) は有効
        allocator.Free( offset );
    }

    // １ユニット分を (size / unitSize) 回アロケートする。
    // すべて成功するはず。
    int count = static_cast<int>(size) / nn::gfx::util::MemoryPoolAllocator::AllocatorUnitSize;
    for(int i=0; i<count; i++)
    {
        ptrdiff_t offset = allocator.Allocate( nn::gfx::util::MemoryPoolAllocator::AllocatorUnitSize, 1 );

        // 戻ってきたオフセットは範囲内か
        EXPECT_GE(offset, baseOffset);
        EXPECT_LT(offset, baseOffset + static_cast<ptrdiff_t>(size));
    }

    // Free() せずにさらにもう一回アロケートする。メモリプールは尽きているので
    // 失敗して InvalidOffset が返るはず。
    {
        ptrdiff_t offset = allocator.Allocate( nn::gfx::util::MemoryPoolAllocator::AllocatorUnitSize, 1 );

        // InvalidOffset が返っているか
        EXPECT_EQ(offset, InvalidOffset);
    }

    // 終了する
    allocator.Finalize();

    // Finalize() 後は未初期化状態か
    EXPECT_FALSE(allocator.IsInitialized());

    // ユーザーアロケータの Malloc(), Free() の回数が一致しているか
    EXPECT_EQ(g_UserAllocatorAllocCount, 0);


    // ユーザーアロケータの Malloc() が nullptr を返す場合のテスト。
    // Allocate() 時に確実にユーザーアロケータが呼ばれるように、
    // 再初期化して初回の Allocate() でテストする。(ついでに２回目の Initialize, Finalize のテストも行う)
    // 失敗して InvalidOffset が返るはず。
    {
        // 初期化する
        allocator.Initialize(
            UserAllocatorMalloc,
            g_pExpectedAllocateUserData,
            UserAllocatorFree,
            g_pExpectedFreeUserData,
            &memoryPool,
            baseOffset,
            size,
            allocatableAlignmentMax,
            false
            );

        // Initialize() 後は初期化済み状態か
        EXPECT_TRUE(allocator.IsInitialized());

        g_bUserAllocatorInvalid = true;

        ptrdiff_t offset = allocator.Allocate( 1, 1 );

        // InvalidOffset が返っているか
        EXPECT_EQ(offset, InvalidOffset);

        g_bUserAllocatorInvalid = false;

        // 終了する
        allocator.Finalize();

        // Finalize() 後は未初期化状態か
        EXPECT_FALSE(allocator.IsInitialized());
    }
}

TEST(MemoryPoolAllocatorDeathTest, DeathTest)
{
    const ptrdiff_t baseOffset = 65536;
    const size_t size = nn::gfx::util::MemoryPoolAllocator::AllocatorUnitSize * 20;
    const size_t allocatableAlignmentMax = 65536;
    nn::gfx::MemoryPool memoryPool;
    nn::gfx::util::MemoryPoolAllocator allocator;

    g_bUserAllocatorInvalid = false;
    g_UserAllocatorAllocCount = 0;
    g_pExpectedAllocateUserData = nullptr;
    g_pExpectedFreeUserData = nullptr;

    // 初期化前に呼ぶと死ぬことの確認
    EXPECT_DEATH_IF_SUPPORTED(allocator.Finalize(), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.Allocate(1, 1), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.Free(baseOffset), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.GetMemoryPool(), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.GetBaseOffset(), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.GetSize(), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.GetAllocatableAlignmentMax(), "");

    // Initialize のパラメータで死ぬことの確認
    EXPECT_DEATH_IF_SUPPORTED(allocator.Initialize(nullptr, nullptr, UserAllocatorFree, nullptr, &memoryPool, baseOffset, size, allocatableAlignmentMax, false), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.Initialize(UserAllocatorMalloc, nullptr, nullptr, nullptr, &memoryPool, baseOffset, size, allocatableAlignmentMax, false), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.Initialize(UserAllocatorMalloc, nullptr, UserAllocatorFree, nullptr, nullptr, baseOffset, size, allocatableAlignmentMax, false), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.Initialize(UserAllocatorMalloc, nullptr, UserAllocatorFree, nullptr, &memoryPool, -1, size, allocatableAlignmentMax, false), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.Initialize(UserAllocatorMalloc, nullptr, UserAllocatorFree, nullptr, &memoryPool, baseOffset + 1, size, allocatableAlignmentMax, false), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.Initialize(UserAllocatorMalloc, nullptr, UserAllocatorFree, nullptr, &memoryPool, baseOffset, 0, allocatableAlignmentMax, false), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.Initialize(UserAllocatorMalloc, nullptr, UserAllocatorFree, nullptr, &memoryPool, baseOffset, size, 0, false), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.Initialize(UserAllocatorMalloc, nullptr, UserAllocatorFree, nullptr, &memoryPool, baseOffset, size, allocatableAlignmentMax - 1, false), "");

    // 初期化する
    allocator.Initialize(
        UserAllocatorMalloc,
        nullptr,
        UserAllocatorFree,
        nullptr,
        &memoryPool,
        baseOffset,
        size,
        allocatableAlignmentMax,
        false
        );

    // 初期化後に呼ぶと死ぬことの確認
    EXPECT_DEATH_IF_SUPPORTED(allocator.Initialize(UserAllocatorMalloc, nullptr, UserAllocatorFree, nullptr, &memoryPool, baseOffset, size, allocatableAlignmentMax, false), "");

    // 不正なパラメータで呼ぶと死ぬことの確認
    EXPECT_DEATH_IF_SUPPORTED(allocator.Allocate(1, allocatableAlignmentMax + 1), "");
    ptrdiff_t offset = allocator.Allocate(10,1);
    EXPECT_DEATH_IF_SUPPORTED(allocator.Free(offset - 1), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.Free(offset + 1), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.Free(offset - nn::gfx::util::MemoryPoolAllocator::AllocatorUnitSize), "");
    EXPECT_DEATH_IF_SUPPORTED(allocator.Free(offset + nn::gfx::util::MemoryPoolAllocator::AllocatorUnitSize), "");
    // 二重解放で死ぬことの確認
    allocator.Free(offset);
    EXPECT_DEATH_IF_SUPPORTED(allocator.Free(offset), "");

    // 終了する
    allocator.Finalize();
}
