﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include "testGfxUtil_AlignedAllocate.h"

namespace nnt{ namespace gfxUtil{

    template<typename ThreadParameter>
    class WorkerThreads
    {
    public:
        static const int MaxNumberOfThreads = 8 * sizeof(nn::Bit64);
        typedef void (*WorkerThreadFunctionType)(int threadIndex, ThreadParameter* param);
    private:
        struct ThreadingInfo
        {
            int numberOfThreads;
            int coreNumbers[MaxNumberOfThreads];
        };
        struct WorkerThreadParameter
        {
            ThreadParameter m_UserParameter;
            WorkerThreadFunctionType m_WorkerThreadFunction;
            nn::os::BarrierType* m_pStartExitBarrier;
            int m_ThreadIndex;
        };

    public:
        void Initialize(
            WorkerThreadFunctionType workerThreadFunction,
            size_t stackSize
            ) NN_NOEXCEPT
        {
            m_ThreadingInfo = GetThreadingInfo();
            const int numThreads = m_ThreadingInfo.numberOfThreads;

            nn::os::InitializeBarrier(&m_StartExitBarrier, numThreads + 1);
            m_ThreadStack = static_cast<char*>(AlignedAllocate(stackSize * numThreads, nn::os::ThreadStackAlignment));
            for(int t = 0; t < numThreads; t++)
            {
                m_pThreadStacks[t] = m_ThreadStack + (t * stackSize);
                m_ThreadParameters[t].m_WorkerThreadFunction = workerThreadFunction;
                m_ThreadParameters[t].m_pStartExitBarrier    = &m_StartExitBarrier;
                m_ThreadParameters[t].m_ThreadIndex          = t;
            }

            m_StackSize = stackSize;
            m_IsThreadCreated = false;
        }
        void Finalize() NN_NOEXCEPT
        {
            const int numThreads = m_ThreadingInfo.numberOfThreads;
            if(m_IsThreadStarted)
            {
                for(int t = 0; t < numThreads; t++)
                {
                    nn::os::WaitThread(&m_Threads[t]);
                }
                m_IsThreadStarted = false;
            }
            if(m_IsThreadCreated)
            {
                for(int t = 0; t < numThreads; t++)
                {
                    nn::os::DestroyThread(&m_Threads[t]);
                }
                m_IsThreadCreated = false;
            }
            AlignedFree(m_ThreadStack);
            nn::os::FinalizeBarrier(&m_StartExitBarrier);
        }

        int GetNumberOfThreads() const NN_NOEXCEPT
        {
            return m_ThreadingInfo.numberOfThreads;
        }
        void SetThreadParameter(int tid, const ThreadParameter* pParam) NN_NOEXCEPT
        {
            m_ThreadParameters[tid].m_UserParameter = *pParam;
        }
        const ThreadParameter* GetThreadParameter(int tid) const NN_NOEXCEPT
        {
            return &m_ThreadParameters[tid].m_UserParameter;
        }

        //! スレッドを作成して待機状態にします。
        void PrepareWorker() NN_NOEXCEPT
        {
            const int numThreads = m_ThreadingInfo.numberOfThreads;
            for(int t = 0; t < numThreads; t++)
            {
                nn::Result result = nn::os::CreateThread(
                    &m_Threads[t],
                    WorkerThreadFunction,
                    &m_ThreadParameters[t],
                    m_pThreadStacks[t],
                    m_StackSize,
                    nn::os::DefaultThreadPriority,
                    m_ThreadingInfo.coreNumbers[t]
                    );
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                nn::os::StartThread(&m_Threads[t]);
            }
            m_IsThreadCreated = true;
        }
        //! 待機中のスレッドを開始します。
        void StartWorker() NN_NOEXCEPT
        {
            nn::os::AwaitBarrier(&m_StartExitBarrier);
            m_IsThreadStarted = true;
        }
        //! スレッドの完了を待ちます。
        void WaitWorkerComplete() NN_NOEXCEPT
        {
            nn::os::AwaitBarrier(&m_StartExitBarrier);
            m_IsThreadStarted = false;
        }
    private:
        static ThreadingInfo GetThreadingInfo() NN_NOEXCEPT
        {
            ThreadingInfo info = {};

            nn::Bit64 availableCoreMask = nn::os::GetThreadAvailableCoreMask();
            for(int i = 0; i < MaxNumberOfThreads; i++)
            {
                if((availableCoreMask & (static_cast<nn::Bit64>(1) << static_cast<nn::Bit64>(i))) != 0)
                {
                    info.coreNumbers[info.numberOfThreads] = i;
                    info.numberOfThreads++;
                }
            }

            return info;
        }
        static void WorkerThreadFunction(void* param) NN_NOEXCEPT
        {
            WorkerThreadParameter* pWorkerParam = reinterpret_cast<WorkerThreadParameter*>(param);
            nn::os::AwaitBarrier(pWorkerParam->m_pStartExitBarrier);
            ThreadFunction(pWorkerParam->m_ThreadIndex, &pWorkerParam->m_UserParameter);
            nn::os::AwaitBarrier(pWorkerParam->m_pStartExitBarrier);
        }

    private:
        ThreadingInfo         m_ThreadingInfo;
        char*                 m_ThreadStack;
        size_t                m_StackSize;

        nn::os::ThreadType    m_Threads[MaxNumberOfThreads];
        WorkerThreadParameter m_ThreadParameters[MaxNumberOfThreads];
        void*                 m_pThreadStacks[MaxNumberOfThreads];

        nn::os::BarrierType   m_StartExitBarrier;

        bool m_IsThreadCreated;
        bool m_IsThreadStarted;
    };

}}
