﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// 全スロットが未確保の状態から初めて
// 複数スレッドから同時に AcquireIndex() するテスト

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nnt.h>
#include <cstdlib>

#include <nn/gfx/util/detail/gfx_IndexRingBuffer.h>
#include "testGfxUtil_WorkerThreads.h"

namespace {
    struct ThreadParameter
    {
        nn::gfx::util::detail::IndexRingBuffer* pRing;
        int acquireCount;
        int* acquiredIndices;
    };
    void ThreadFunction(int tid, ThreadParameter* pParam)
    {
        NN_UNUSED(tid);
        for(int i = 0; i < pParam->acquireCount; i++)
        {
            pParam->acquiredIndices[i] = pParam->pRing->AcquireIndex();
        }
    }
}

TEST(IndexRingBuffer, MultiThread)
{
    using namespace nnt::gfxUtil;
    static const int AcquireCount = 4096;
    static const int StackSize = 1024 * 1024;
    static const int MaxNumberOfThreads = 64;
    NN_STATIC_ASSERT(StackSize % nn::os::ThreadStackAlignment == 0);

    const int InvalidIndex = nn::gfx::util::detail::IndexRingBuffer::InvalidIndex;
    nn::gfx::util::detail::IndexRingBuffer ring;

    // ワーカースレッドを用意
    WorkerThreads<ThreadParameter> wthreads;
    wthreads.Initialize(ThreadFunction, StackSize);
    const int numThreads = wthreads.GetNumberOfThreads();
    // パラメータを設定
    ThreadParameter params[MaxNumberOfThreads];
    for(int t = 0; t < numThreads; t++)
    {
        ThreadParameter& param = params[t];
        param.pRing = &ring;
        param.acquireCount = AcquireCount;
        param.acquiredIndices = static_cast<int*>(AlignedAllocate(sizeof(int) * AcquireCount + 1,  NN_ALIGNOF(int)));
        for(int j = 0; j < AcquireCount + 1; j++)
        {
            param.acquiredIndices[j] = InvalidIndex;
        }
        wthreads.SetThreadParameter(t, &param);
    }
    wthreads.PrepareWorker();

    ring.Initialize(0, AcquireCount * numThreads + 1);
    ring.Begin();
    {
        wthreads.StartWorker();
        wthreads.WaitWorkerComplete();
    }
    nn::gfx::util::detail::IndexRange range;
    ring.End(&range);

    // 正しく確保されているか確認
    {
        // 確保された数が一致するか確認
        EXPECT_EQ(0, range.base);
        EXPECT_EQ(AcquireCount * numThreads, range.count);
        // 確保された内容が正しいか確認
        {
            int* nextIndices[MaxNumberOfThreads];
            for(int t = 0; t < numThreads; t++)
            {
                nextIndices[t] = params[t].acquiredIndices;
            }
            for(int i = 0; i < AcquireCount * numThreads; i++)
            {
                // どこかで確保されているはず。
                bool found = false;
                for(int t = 0; t < numThreads; t++)
                {
                    int value = *nextIndices[t];
                    if(value == i)
                    {
                        found = true;
                        nextIndices[t]++;
                        break;
                    }
                }
                if(!found)
                {
                    // どこでも確保されていなかったらエラー
                    FAIL();
                }
            }
        }
    }

    // 後始末
    for(int t = 0; t < numThreads; t++)
    {
        AlignedFree(params[t].acquiredIndices);
    }
    wthreads.Finalize();
}
