﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// シングルスレッド動作の確認テスト

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nnt.h>
#include <cstdlib>

#include <nn/gfx/util/detail/gfx_IndexRingBuffer.h>

TEST(IndexRingBuffer, AcquireIndex)
{
    const int NumberList[]    = {10, 15, 20};
    const int BaseIndexList[] = {0, 5, 255};

    for(int i = 0; i < 9; i++)
    {
        int N = NumberList[i % 3];
        int base = BaseIndexList[i / 3];
        NN_LOG("  nSlots %d, base %d\n", N, base);
        nn::gfx::util::detail::IndexRingBuffer ring;
        ring.Initialize(base, N);

        EXPECT_EQ(base, ring.GetBaseIndex());
        EXPECT_EQ(N, ring.GetIndexCount());

        nn::gfx::util::detail::IndexRange range0 = {};
        nn::gfx::util::detail::IndexRange range1 = {};
        nn::gfx::util::detail::IndexRange range2 = {};
        nn::gfx::util::detail::IndexRange range3 = {};

        ring.Begin();
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (0 % N), slot); }
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (1 % N), slot); }
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (2 % N), slot); }
        ring.End(&range1);
        EXPECT_EQ(0 % N, range1.base);
        EXPECT_EQ(3, range1.count);

        ring.Begin();
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (3 % N), slot); }
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (4 % N), slot); }
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (5 % N), slot); }
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (6 % N), slot); }
        ring.End(&range2);
        EXPECT_EQ(3 % N, range2.base);
        EXPECT_EQ(4, range2.count);

        ring.Begin();
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (7 % N), slot); }
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (8 % N), slot); }
        ring.End(&range3);
        EXPECT_EQ(7 % N, range3.base);
        EXPECT_EQ(2, range3.count);

        ring.ReleaseIndexRange(&range1);

        ring.Begin();
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (9 % N), slot); }
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (10 % N), slot); }
        ring.End(&range1);
        EXPECT_EQ(9 % N, range1.base);
        EXPECT_EQ(2, range1.count);

        ring.ReleaseIndexRange(&range0);
        ring.ReleaseIndexRange(&range2);

        ring.Begin();
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (11 % N), slot); }
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (12 % N), slot); }
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (13 % N), slot); }
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (14 % N), slot); }

        ring.ReleaseIndexRange(&range3);

        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (15 % N), slot); }
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (16 % N), slot); }
        ring.End(&range2);
        EXPECT_EQ(11 % N, range2.base);
        EXPECT_EQ(6, range2.count);

        ring.Begin();
        ring.End(&range3);
        EXPECT_EQ(17 % N, range3.base);
        EXPECT_EQ(0, range3.count);

        ring.ReleaseIndexRange(&range1);
        ring.ReleaseIndexRange(&range2);
        ring.ReleaseIndexRange(&range3);
        ring.ReleaseIndexRange(&range0);

        ring.Begin();
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (17 % N), slot); }
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (18 % N), slot); }
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (19 % N), slot); }
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (20 % N), slot); }
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (21 % N), slot); }
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (22 % N), slot); }
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (23 % N), slot); }
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (24 % N), slot); }
        { int slot = ring.AcquireIndex(); EXPECT_EQ(base + (25 % N), slot); }
        ring.End(&range1);
        EXPECT_EQ(17 % N, range1.base);
        EXPECT_EQ(9, range1.count);

        ring.ReleaseIndexRange(&range0);

        ring.Begin();
        ring.End(&range2);
        EXPECT_EQ(26 % N, range2.base);
        EXPECT_EQ(0, range2.count);

        ring.ReleaseIndexRange(&range1);
        ring.ReleaseIndexRange(&range2);

        EXPECT_EQ(base, ring.GetBaseIndex());
        EXPECT_EQ(N, ring.GetIndexCount());

        ring.Finalize();
    }
}

TEST(IndexRingBuffer, AcquireIndexRange)
{
    int N = 100;
    int base = 123;
    const int InvalidIndex = nn::gfx::util::detail::IndexRingBuffer::InvalidIndex;
    NN_LOG("  nSlots %d, base %d\n", N, base);
    nn::gfx::util::detail::IndexRingBuffer ring;
    ring.Initialize(base, N);

    EXPECT_EQ(base, ring.GetBaseIndex());
    EXPECT_EQ(N, ring.GetIndexCount());
    EXPECT_EQ(false, ring.IsRecording());

    nn::gfx::util::detail::IndexRange range1 = {};
    nn::gfx::util::detail::IndexRange range2 = {};
    nn::gfx::util::detail::IndexRange range3 = {};

    ring.Begin();
    EXPECT_EQ(true, ring.IsRecording());
    // 0 個は確保できない
    { int slot = ring.AcquireIndexRange(0); EXPECT_EQ(InvalidIndex, slot); }   // head =  0, tail = 99
    // slotCount と同じ数は確保できない
    { int slot = ring.AcquireIndexRange(100); EXPECT_EQ(InvalidIndex, slot); } // head =  0, tail = 99
    // 適当に確保（2周目には入らない）
    { int slot = ring.AcquireIndexRange(10); EXPECT_EQ(base + 0, slot); }      // head = 10, tail = 99
    { int slot = ring.AcquireIndexRange(15); EXPECT_EQ(base + 10, slot); }     // head = 25, tail = 99
    { int slot = ring.AcquireIndexRange(20); EXPECT_EQ(base + 25, slot); }     // head = 45, tail = 99
    // 0 個は確保できない
    { int slot = ring.AcquireIndexRange(0); EXPECT_EQ(InvalidIndex, slot); }   // head = 45, tail = 99
    ring.End(&range1);
    EXPECT_EQ(0, range1.base);
    EXPECT_EQ(45, range1.count);
    EXPECT_EQ(false, ring.IsRecording());

    ring.Begin();
    { int slot = ring.AcquireIndexRange(10); EXPECT_EQ(base + 45, slot); }    // head = 55, tail = 99
    // tail を超える数は確保できない
    { int slot = ring.AcquireIndexRange(45); EXPECT_EQ(InvalidIndex, slot); } // head = 55, tail = 99
    // tail を超える数は確保できない
    { int slot = ring.AcquireIndexRange(55); EXPECT_EQ(InvalidIndex, slot); } // head = 55, tail = 99
    // 適当に確保
    { int slot = ring.AcquireIndexRange(33); EXPECT_EQ(base + 55, slot); }    // head = 88, tail = 99
    { int slot = ring.AcquireIndexRange(10); EXPECT_EQ(base + 88, slot); }    // head = 98, tail = 99
    // tail を超える数は確保できない
    { int slot = ring.AcquireIndexRange(2); EXPECT_EQ(InvalidIndex, slot); }  // head = 98, tail = 99
    // tail と一致する数は確保できる
    { int slot = ring.AcquireIndexRange(1); EXPECT_EQ(base + 98, slot); }     // head = 99, tail = 99
    ring.End(&range2);                                                        // head = 99, tail = 99
    EXPECT_EQ(45, range2.base);
    EXPECT_EQ(54, range2.count);

    ring.ReleaseIndexRange(&range1);                                          // head = 99, tail =144

    ring.Begin();
    // 境界をまたいで確保。1 スロット飛ばされる。
    { int slot = ring.AcquireIndexRange(5); EXPECT_EQ(base + 0, slot); }      // head =105, tail =144
    // 2 周目から確保
    { int slot = ring.AcquireIndexRange(10); EXPECT_EQ(base + 5, slot); }     // head =115, tail =144
    { int slot = ring.AcquireIndexRange(5); EXPECT_EQ(base + 15, slot); }     // head =120, tail =144
    { int slot = ring.AcquireIndexRange(5); EXPECT_EQ(base + 20, slot); }     // head =125, tail =144
    ring.End(&range3);                                                        // head = 25, tail = 44
    EXPECT_EQ(99, range3.base);
    EXPECT_EQ(26, range3.count);

    ring.ReleaseIndexRange(&range2);                                          // head = 25, tail = 98
    ring.ReleaseIndexRange(&range3);                                          // head = 25, tail =124

    // ギリギリ境界をまたがずに確保
    ring.Begin();
    { int slot = ring.AcquireIndexRange(75); EXPECT_EQ(base + 25, slot); }    // head =100, tail =124
    { int slot = ring.AcquireIndex(); EXPECT_EQ(base + 0, slot); }            // head =101, tail =124
    // 2 周目で溢れるように確保
    { int slot = ring.AcquireIndexRange(24); EXPECT_EQ(InvalidIndex, slot); } // head =101, tail =124
    { int slot = ring.AcquireIndexRange(23); EXPECT_EQ(base + 1, slot); }     // head =124, tail =124
    ring.End(&range1);                                                        // head = 24, tail = 24
    EXPECT_EQ(25, range1.base);
    EXPECT_EQ(99, range1.count);

    ring.ReleaseIndexRange(&range1);                                          // head = 24, tail =123

    ring.Begin();
    // 境界をまたぐため確保できない
    { int slot = ring.AcquireIndexRange(77); EXPECT_EQ(InvalidIndex, slot); } // head = 24, tail =123
    { int slot = ring.AcquireIndexRange(67); EXPECT_EQ(base + 24, slot); }    // head = 91, tail =123
    ring.End(&range2);
    EXPECT_EQ(24, range2.base);
    EXPECT_EQ(67, range2.count);

    ring.Begin();
    { int slot = ring.AcquireIndexRange(13); EXPECT_EQ(base + 0, slot); }     // head =113, tail =123
    ring.End(&range3);                                                        // head = 13, tail = 23
    EXPECT_EQ(91, range3.base);
    EXPECT_EQ(22, range3.count);

    ring.Begin();
    // tail よりも大きい数は確保できない
    { int slot = ring.AcquireIndexRange(24); EXPECT_EQ(InvalidIndex, slot); } // head = 13, tail = 23
    // tail と一致しても空きが足りなければ確保できない
    { int slot = ring.AcquireIndexRange(23); EXPECT_EQ(InvalidIndex, slot); } // head = 13, tail = 23
    // tail と一致しても空きが足りなければ確保できない
    { int slot = ring.AcquireIndexRange(14); EXPECT_EQ(InvalidIndex, slot); } // head = 13, tail = 23
    ring.End(&range1);                                                        // head = 13, tail = 23
    EXPECT_EQ(13, range1.base);
    EXPECT_EQ(0, range1.count);

    EXPECT_EQ(base, ring.GetBaseIndex());
    EXPECT_EQ(N, ring.GetIndexCount());

    ring.Finalize();
}

TEST(IndexRingBufferDeathTest, DeathTest)
{
    nn::gfx::util::detail::IndexRingBuffer ring;
    nn::gfx::util::detail::IndexRange range[10] = {};

    const int InvalidIndex = nn::gfx::util::detail::IndexRingBuffer::InvalidIndex;

    // Initialize 前に呼ぶと死ぬこと
    EXPECT_DEATH_IF_SUPPORTED(ring.Finalize(), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.Begin(), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.End(&range[0]), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.ReleaseIndexRange(&range[0]), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.AcquireIndex(), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.AcquireIndexRange(1), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.GetBaseIndex(), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.GetIndexCount(), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.IsRecording(), "");

    // Initialize のパラメータチェック
    EXPECT_DEATH_IF_SUPPORTED(ring.Initialize(-1, 100), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.Initialize(256, -1), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.Initialize(256, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.Initialize(9999, INT_MAX / 2 + 1), "");

    ring.Initialize(256, 5);

    // Begin 前に呼ぶと死ぬこと
    EXPECT_DEATH_IF_SUPPORTED(ring.End(&range[0]), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.AcquireIndex(), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.AcquireIndexRange(100), "");

    ring.Begin();

    // Begin 後に呼ぶと死ぬこと
    EXPECT_DEATH_IF_SUPPORTED(ring.Begin(), "");
    // 負の数を確保しようとすると死ぬこと
    EXPECT_DEATH_IF_SUPPORTED(ring.AcquireIndexRange(-10), "");

    ring.AcquireIndex();
    ring.AcquireIndex();
    ring.AcquireIndex();
    ring.AcquireIndex();
    // count - 1 個までしか確保できないこと
    EXPECT_EQ(ring.AcquireIndex(), InvalidIndex);

    // NULL で呼び出せないこと
    EXPECT_DEATH_IF_SUPPORTED(ring.End(NULL), "");
    ring.End(&range[0]);
    // End 後に呼べないこと
    EXPECT_DEATH_IF_SUPPORTED(ring.End(&range[0]), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.AcquireIndex(), "");

    ring.Begin();
    // count - 1 個までしか確保できないこと
    EXPECT_EQ(ring.AcquireIndex(), InvalidIndex);
    ring.End(&range[1]);

    // Release を NULL で呼べないこと
    EXPECT_DEATH_IF_SUPPORTED(ring.ReleaseIndexRange(NULL), "");
    // Release する順番を守らないといけないこと
    EXPECT_DEATH_IF_SUPPORTED(ring.ReleaseIndexRange(&range[1]), "");
    ring.ReleaseIndexRange(&range[0]);
    ring.ReleaseIndexRange(&range[1]);

    ring.Begin();
    ring.AcquireIndex();
    ring.AcquireIndex();
    ring.End(&range[2]);

    ring.Begin();
    ring.AcquireIndex();
    ring.AcquireIndex();
    // count - 1 個までしか確保できないこと
    EXPECT_EQ(ring.AcquireIndex(), InvalidIndex);
    ring.End(&range[3]);

    ring.ReleaseIndexRange(&range[2]);
    ring.ReleaseIndexRange(&range[3]);

    ring.Finalize();
}
