﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Assert.h>

enum PlatformMeasuringPoint
{
    PlatformMeasuringPoint_PowerCpu,
    PlatformMeasuringPoint_PowerGpu,
    PlatformMeasuringPoint_PowerDdr,
    PlatformMeasuringPoint_PowerVsysAp,
    PlatformMeasuringPoint_ThermalSensorInternal,
    PlatformMeasuringPoint_ThermalSensorExternal,
    PlatformMeasuringPoint_Count,
};

class PlatformMeasurementTracker
{
private:
    struct MeasuringPointData
    {
        bool        hasNewMax;

        int         min;
        int         max;
        int         average;
        int         current;

        int         measureCount;

        static const int MeasureHistoryLength = 64;
        int         measureHistory[MeasureHistoryLength];
    };

    MeasuringPointData      m_MeasuringPointDataArray[PlatformMeasuringPoint_Count];
    int                     m_IgnoreUpdateCount;

public:
                        PlatformMeasurementTracker();
                        ~PlatformMeasurementTracker();

    void                Initialize();
    void                Finalize();

    void                Reset();
    void                ResetAndIgnoreNextUpdates(int ignoreCount);
    void                Update();


    int                 GetCurrent(PlatformMeasuringPoint measuringPoint);
    int                 GetMax(PlatformMeasuringPoint measuringPoint);
    int                 GetMin(PlatformMeasuringPoint measuringPoint);
    int                 GetAverage(PlatformMeasuringPoint measuringPoint);

    bool                UpdatedMaxValue(PlatformMeasuringPoint measuringPoint);

private:
    void                ReadAndUpdateMeasuringPoint(PlatformMeasuringPoint measuringPoint);
    void                InitializeData(MeasuringPointData* pData);
    void                UpdateData(MeasuringPointData* pData, int newValue);
    int                 ComputeAverageFromHistory(const MeasuringPointData* pData);

    void                InitializeHostPowerMeasuring();
    void                FinalizeHostPowerMeasuring();
    int                 ReadHostCurrentPower(PlatformMeasuringPoint measuringPoint);
};

inline int PlatformMeasurementTracker::GetCurrent(PlatformMeasuringPoint measuringPoint)
{
    NN_ASSERT(measuringPoint < PlatformMeasuringPoint_Count);
    return m_MeasuringPointDataArray[measuringPoint].current;
}

inline int PlatformMeasurementTracker::GetMax(PlatformMeasuringPoint measuringPoint)
{
    NN_ASSERT(measuringPoint < PlatformMeasuringPoint_Count);
    return m_MeasuringPointDataArray[measuringPoint].max;
}

inline int PlatformMeasurementTracker::GetMin(PlatformMeasuringPoint measuringPoint)
{
    NN_ASSERT(measuringPoint < PlatformMeasuringPoint_Count);
    return m_MeasuringPointDataArray[measuringPoint].min;
}

inline int PlatformMeasurementTracker::GetAverage(PlatformMeasuringPoint measuringPoint)
{
    NN_ASSERT(measuringPoint < PlatformMeasuringPoint_Count);
    return m_MeasuringPointDataArray[measuringPoint].average;
}

inline bool PlatformMeasurementTracker::UpdatedMaxValue(PlatformMeasuringPoint measuringPoint)
{
    NN_ASSERT(measuringPoint < PlatformMeasuringPoint_Count);
    return m_MeasuringPointDataArray[measuringPoint].hasNewMax;
}


