﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testGfxUtil_ApplicationCommandLine.h"

#include <nn/nn_Log.h>

#include <cstdlib>

namespace
{

bool SetInitialApplicationMode(ApplicationConfiguration* pApplicationConfiguration, int value)
{
    if (value < ApplicationMode_Max)
    {
        pApplicationConfiguration->initialApplicationMode = static_cast<ApplicationMode>(value);
        return true;
    }

    return false;
}

bool SetInteractiveModeInitialBenchmark(ApplicationConfiguration* pApplicationConfiguration, int value)
{
    if (value < nnt::gfx::util::BenchmarkType_Max)
    {
        pApplicationConfiguration->interactiveModeInitialBenchmark = static_cast<nnt::gfx::util::BenchmarkType>(value);
        return true;
    }

    return false;
}

bool SetExitAfterTest(ApplicationConfiguration* pApplicationConfiguration, bool value)
{
    pApplicationConfiguration->exitAfterTest = value;
    return true;
}

bool SetFindMaxPowerModeOutputFilePath(ApplicationConfiguration* pApplicationConfiguration, const char* value)
{
    pApplicationConfiguration->findMaxPowerModeOutputFilePath = value;
    return true;
}

bool SetInteractiveModeOutputFile(ApplicationConfiguration* pApplicationConfiguration, const char* value)
{
    pApplicationConfiguration->interactiveModeOutputFile = value;
    return true;
}

bool SetReplayModeInputFile(ApplicationConfiguration* pApplicationConfiguration, const char* value)
{
    pApplicationConfiguration->replayModeInputFile = value;
    return true;
}

bool SetFindMaxPowerUpdateMode(ApplicationConfiguration* pApplicationConfiguration, int value)
{
    if (value < FindMaxPowerUpdateMode_Max)
    {
        pApplicationConfiguration->findMaxPowerUpdateMode = static_cast<FindMaxPowerUpdateMode>(value);
        return true;
    }

    return false;
}

bool SetFindMaxPowerModeMask(ApplicationConfiguration* pApplicationConfiguration, int value)
{
    pApplicationConfiguration->findMaxPowerModeBenchmarkMask = value;
    return true;
}

bool SetUpdateTimingsModeInputFile(ApplicationConfiguration* pApplicationConfiguration, const char* value)
{
    pApplicationConfiguration->updateTimingsModeInputFile = value;
    return true;
}

bool SetUpdateTimingsModeOutputFile(ApplicationConfiguration* pApplicationConfiguration, const char* value)
{
    pApplicationConfiguration->updateTimingsModeOutputFile = value;
    return true;
}

struct ApplicationOptionEnumValue
{
    const char*     string;
    int             value;
};

enum ApplicationOptionType
{
    ApplicationOptionType_StringEnum,
    ApplicationOptionType_StringValue,
    ApplicationOptionType_IntegerValue,
    ApplicationOptionType_FlagOn,
};


typedef bool(*ApplicationOptionFlagSetterFunc)(ApplicationConfiguration* pApplicationConfiguration, bool value);
typedef bool(*ApplicationOptionStringSetterFunc)(ApplicationConfiguration* pApplicationConfiguration, const char* value);
typedef bool(*ApplicationOptionIntegerSetterFunc)(ApplicationConfiguration* pApplicationConfiguration, int value);


union ApplicationOptionData
{
    struct
    {
        int                                 valueArrayCount;
        const ApplicationOptionEnumValue*   valueArray;
        ApplicationOptionIntegerSetterFunc  pOptionSetterFunc;
    } enumData;

    struct
    {
        ApplicationOptionFlagSetterFunc   pOptionSetterFunc;
    } flagData;

    struct
    {
        ApplicationOptionStringSetterFunc   pOptionSetterFunc;
    } stringValueData;

    struct
    {
        ApplicationOptionIntegerSetterFunc   pOptionSetterFunc;
    } integerValueData;
};

template <unsigned long Size>
static ApplicationOptionData MakeEnum(ApplicationOptionEnumValue (&valueArray)[Size], ApplicationOptionIntegerSetterFunc pOptionSetterFunc)
{
    ApplicationOptionData result;
    result.enumData.valueArrayCount = Size;
    result.enumData.valueArray = valueArray;
    result.enumData.pOptionSetterFunc = pOptionSetterFunc;
    return result;
}

static ApplicationOptionData MakeFlag(ApplicationOptionFlagSetterFunc pOptionSetterFunc)
{
    ApplicationOptionData result;
    result.flagData.pOptionSetterFunc = pOptionSetterFunc;
    return result;
}

static ApplicationOptionData MakeStringValue(ApplicationOptionStringSetterFunc pOptionSetterFunc)
{
    ApplicationOptionData result;
    result.stringValueData.pOptionSetterFunc = pOptionSetterFunc;
    return result;
}

static ApplicationOptionData MakeIntegerValue(ApplicationOptionIntegerSetterFunc pOptionSetterFunc)
{
    ApplicationOptionData result;
    result.integerValueData.pOptionSetterFunc = pOptionSetterFunc;
    return result;
}


struct ApplicationOption
{
    const char*                     name;
    const char*                     alternativeName;
    ApplicationOptionType           type;
    const char*                     helpText;
    ApplicationOptionData           data;
};

ApplicationOptionEnumValue optionApplicationModeValueArray[] =
{
    { "Interactive", ApplicationMode_Interactive },
    { "Replay", ApplicationMode_Replay },
    { "FindMaxPower", ApplicationMode_FindMaxPower },
    { "UpdateTimings", ApplicationMode_UpdateTimings },
#if defined(NNT_GFX_UTIL_ENABLE_LOP)
    { "Profile", ApplicationMode_Profile },
#endif
};

ApplicationOptionEnumValue optionFindMaxPowerUpdateModeValueArray[] =
{
    { "CreateNew", FindMaxPowerUpdateMode_CreateNew },
    { "UpdateMaxPower", FindMaxPowerUpdateMode_UpdateMaxPower },
};

#define GPU_BENCHMARK_DEFINE_OPTION_VALUES(_n) { #_n, nnt::gfx::util::BenchmarkType_ ## _n },

ApplicationOptionEnumValue optionBenchmarkTypeValueArray[] =
{
    GPU_BENCHMARK_TYPES_GEN(GPU_BENCHMARK_DEFINE_OPTION_VALUES)
};

ApplicationOption applicationOptions[] =
{
    {
        "-m", "--initial-mode",
        ApplicationOptionType_StringEnum,
        "Set application mode when starting the application",
        MakeEnum(optionApplicationModeValueArray, &SetInitialApplicationMode)
    },
    {
        "-x", "--exit-after",
        ApplicationOptionType_FlagOn,
        "Exit after processing is done",
        MakeFlag(&SetExitAfterTest),
    },
    {
        "--interactive-initial-benchmark", nullptr,
        ApplicationOptionType_StringEnum,
        "Set default benchmark type when starting interactive mode",
        MakeEnum(optionBenchmarkTypeValueArray, SetInteractiveModeInitialBenchmark),
    },
    {
        "--interactive-output-file", nullptr,
        ApplicationOptionType_StringValue,
        "Set interactive mode output file",
        MakeStringValue(&SetInteractiveModeOutputFile),
    },
    {
        "--replay-input-file", nullptr,
        ApplicationOptionType_StringValue,
        "Set replay mode input file",
        MakeStringValue(&SetReplayModeInputFile),
    },
    {
        "--find-max-power-update-mode", nullptr,
        ApplicationOptionType_StringEnum,
        "load output file (it it exists) and update content",
        MakeEnum(optionFindMaxPowerUpdateModeValueArray, &SetFindMaxPowerUpdateMode),
    },
    {
        "--find-max-power-output-file", nullptr,
        ApplicationOptionType_StringValue,
        "Set FindMaxPower mode output file",
        MakeStringValue(&SetFindMaxPowerModeOutputFilePath),
    },
    {
        "--find-max-power-mask", nullptr,
        ApplicationOptionType_IntegerValue,
        "Set FindMaxPower benchmark type mask",
        MakeIntegerValue(&SetFindMaxPowerModeMask),
    },
    {
        "--update-timings-input-file", nullptr,
        ApplicationOptionType_StringValue,
        "Set input file path",
        MakeStringValue(&SetUpdateTimingsModeInputFile),
    },
    {
        "--update-timings-output-file", nullptr,
        ApplicationOptionType_StringValue,
        "Set output file path",
        MakeStringValue(&SetUpdateTimingsModeOutputFile),
    },
};


bool ParseEnumValue(int* pOutValue, const char* value, const ApplicationOptionData* optionData)
{
    const ApplicationOptionEnumValue* valueArray = optionData->enumData.valueArray;
    int valueArrayCount = optionData->enumData.valueArrayCount;

    for (int i = 0; i < valueArrayCount; ++i)
    {
        if (strcmp(value, valueArray[i].string) == 0)
        {
            *pOutValue = valueArray[i].value;
            return true;
        }
    }

    return false;
}

bool ParseIntegerValue(int* pOutValue, const char* value)
{
    const char* hexPrefix = "0x";
    const int hexPrefixLength = 2;

    const char* binaryPrefix = "b";
    const int binaryPrefixLength = 1;

    const char* input = value;
    int base = 10;
    if (memcmp(value, hexPrefix, hexPrefixLength) == 0)
    {
        input += hexPrefixLength;
        base = 16;
    }
    if (memcmp(value, binaryPrefix, binaryPrefixLength) == 0)
    {
        input += binaryPrefixLength;
        base = 2;
    }

    char* end = nullptr;
    long result = std::strtol(input, &end, base);
    *pOutValue = static_cast<int>(result);
    return true;
}

}

void ApplicationConfiguration::SetDefault()
{
    initialApplicationMode = ApplicationMode_SelectMode;

    interactiveModeInitialBenchmark = nnt::gfx::util::BenchmarkType_CompositeTest;
    interactiveModeOutputFile = "host:/TestCases.json";

    replayModeInputFile = "host:/TestCases.json";

    findMaxPowerUpdateMode = FindMaxPowerUpdateMode_CreateNew;
    findMaxPowerModeOutputFilePath = "host:/MaxPowerTestCases.json";
    exitAfterTest = false;
    findMaxPowerModeBenchmarkMask = 0xFFFFFF;

    updateTimingsModeInputFile = "host:/TestCases.json";
    updateTimingsModeOutputFile = "host:/TestCases.json";
    updateTimingsModeTestCaseFilter = nullptr;
}


bool ParseCommandLine(int argc, char** argv, ApplicationConfiguration* pApplicationConfiguration)
{
    int argi = 1;

    while (argi < argc)
    {
        bool foundParameter = false;

        for (int optionIndex = 0; optionIndex < NN_ARRAY_SIZE(applicationOptions); ++optionIndex)
        {
            const ApplicationOption* pOption = &applicationOptions[optionIndex];
            if ((strcmp(pOption->name, argv[argi]) == 0)
                || ((pOption->alternativeName != nullptr)
                    && (strcmp(pOption->alternativeName, argv[argi]) == 0)))
            {
                argi++;

                switch (pOption->type)
                {
                case ApplicationOptionType_StringEnum:
                    {
                        int optionValue = 0;

                        if (argi >= argc)
                            return false;

                        if (!ParseEnumValue(&optionValue, argv[argi], &pOption->data))
                        {
                            NN_LOG("Unrecognized value: %s for parameter %s\n", argv[argi], pOption->name);
                            return false;
                        }

                        argi++;

                        if (!pOption->data.enumData.pOptionSetterFunc(pApplicationConfiguration, optionValue))
                        {
                            return false;
                        }
                    }
                    break;

                case ApplicationOptionType_FlagOn:
                    {
                        if (!pOption->data.flagData.pOptionSetterFunc(pApplicationConfiguration, 1))
                        {
                            return false;
                        }
                    }
                    break;

                case ApplicationOptionType_StringValue:
                    {
                        if (argi >= argc)
                            return false;

                        if (!pOption->data.stringValueData.pOptionSetterFunc(pApplicationConfiguration, argv[argi]))
                        {
                            return false;
                        }

                        argi++;
                    }
                    break;

                case ApplicationOptionType_IntegerValue:
                    {
                        int optionValue = 0;

                        if (argi >= argc)
                            return false;

                        if (!ParseIntegerValue(&optionValue, argv[argi]))
                        {
                            NN_LOG("Unrecognized value: %s for parameter %s\n", argv[argi], pOption->name);
                            return false;
                        }

                        if (!pOption->data.integerValueData.pOptionSetterFunc(pApplicationConfiguration, optionValue))
                        {
                            return false;
                        }

                        argi++;
                    }
                    break;

                default:
                    NN_UNEXPECTED_DEFAULT;
                }


                foundParameter = true;
                break;
            }
        }

        if (!foundParameter)
        {
            NN_LOG("Unknown parameter: %s\n", argv[argi]);
            return false;
        }
    }

    return true;
}

void PrintCommandLineHelp()
{
    for (int optionIndex = 0; optionIndex < NN_ARRAY_SIZE(applicationOptions); ++optionIndex)
    {
        const ApplicationOption* pOption = &applicationOptions[optionIndex];
        NN_LOG("%s ", pOption->name);
        if (pOption->alternativeName != nullptr)
        {
            NN_LOG("%s ", pOption->alternativeName, pOption->helpText);
        }
        NN_LOG("\n");
        NN_LOG("        ");

        switch (pOption->type)
        {
        case ApplicationOptionType_StringEnum:
            {
                NN_LOG("[");
                for (int valueIndex = 0; valueIndex < pOption->data.enumData.valueArrayCount; ++valueIndex)
                {
                    if (valueIndex > 0)
                        NN_LOG(", ");
                    NN_LOG("%s", pOption->data.enumData.valueArray[valueIndex].string);
                }
                NN_LOG("]");
            }
            break;

        case ApplicationOptionType_FlagOn:
            {
            }
            break;

        case ApplicationOptionType_StringValue:
            {
                NN_LOG("<string>");
            }
            break;

        case ApplicationOptionType_IntegerValue:
            {
                NN_LOG("<number>");
            }
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }
        NN_LOG("\n");

        NN_LOG("        %s", pOption->helpText);
        NN_LOG("\n");
    }

}

