﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NN_PERF_PROFILE_ENABLED
#include <nn/perf.h>
#include <nn/util/util_Color.h>

#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/vi.h>
#include <nn/gfx.h>

#include <nns/gfx/gfx_GraphicsFramework.h>

#if defined(NN_PERF_PROFILE_ENABLED)
#include "../../Samples/Sources/Applications/FrameworkDemo/FrameworkUtility.h"
#endif

#include "../../Samples/Sources/Applications/FrameworkDemo/PadUtility.h"

#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeterDrawer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeshRes.h>
#include <nn/nn_SdkText.h>

namespace {

    const int BufferCount = 2;

    nns::gfx::PrimitiveRenderer::Renderer* g_pPrimitiveRenderer;

    enum EditMode {
        EditMode_FrameworkMode,
        EditMode_PresentInterval,
        EditMode_CalculateSleepCpu,
        EditMode_MakeCommandSleepCpu,
        EditMode_DrawCount,
        EditMode_End,
    };
    int g_CurrentMenu;
    nns::gfx::GraphicsFramework::FrameworkMode g_CurrentFrameworkMode;
    int g_CurrentPresentInterval;
    int g_CurrentCalculateSleepCpu;
    int g_CurrentMakeCommandSleepCpu;
    int g_CurrentDrawCount;
    bool g_IsDrawBoxModelEnabled;

    //---------------------------------------------------------------
    // ViewMatrix を計算
    //---------------------------------------------------------------
    void CalculateViewMatrix(nn::util::Matrix4x3fType* viewMatrix, int frame)
    {
        float radius = 20.f;
        float x = radius * sin(frame / 500.f);
        float z = radius * cos(frame / 500.f);

        nn::util::Vector3fType camPos = { x, 10.f, z };
        nn::util::Vector3fType camTarget = { 0.0f, 0.0f, 0.0f };
        nn::util::Vector3fType camUp = { 0.f, 1.f, 0.f };
        nn::util::MatrixLookAtRightHanded(viewMatrix, camPos, camTarget, camUp);
    }

    //---------------------------------------------------------------
    // ProjectionMatrix を計算
    //---------------------------------------------------------------
    void CalculateProjectionMatrix(nn::util::Matrix4x4fType* projectionMatrix, int width, int height)
    {
        const float fovy = nn::util::FloatPi / 3.0f;
        const float aspect = static_cast<float>(width) / static_cast<float>(height);
        nn::util::MatrixPerspectiveFieldOfViewRightHanded(projectionMatrix, fovy, aspect, 0.1f, 1000.f);
    }

    //---------------------------------------------------------------
    // ModelMatrix を計算
    //---------------------------------------------------------------
    void CalculateModelMatrix(nn::util::Matrix4x3f* modelMatrix)
    {
        nn::util::Vector3f vecZero;
        nn::util::VectorZero(&vecZero);

        nn::util::Vector3f rotValue;
        nn::util::VectorSet(&rotValue, 0.0f, 0.0f, 0.0f);
        nn::util::MatrixIdentity(modelMatrix);
        nn::util::MatrixSetRotateXyz(modelMatrix, rotValue);
        nn::util::MatrixSetAxisW(modelMatrix, vecZero);
    }

    //------------------------------------------------------------------------------
    //  パフォーマンス計測
    //------------------------------------------------------------------------------
    nns::gfx::PrimitiveRenderer::MeterDrawer g_MeterDrawer;
    void* g_pMeterMemory;
    nn::gfx::MemoryPool g_MeterMemoryPool;
    void* g_pMeterPoolMemory;
    ptrdiff_t g_MeterMemoryPoolOffset;

    void InitializeLoadMeter(nns::gfx::GraphicsFramework* pGraphicsFramework)
    {
#if defined(NN_PERF_PROFILE_ENABLED)
        nn::perf::LoadMeterCenterInfo meterInfo;
        meterInfo.SetDefault();
        meterInfo.SetCoreCount(1);
        meterInfo.SetUserMeterCount(0);
        meterInfo.SetCpuBufferCount(3);
        meterInfo.SetGpuBufferCount(3);
        meterInfo.SetCpuSectionCountMax(64);
        meterInfo.SetGpuSectionCountMax(64);

        size_t memorySize = NN_PERF_GET_BUFFER_SIZE(meterInfo);
        size_t alignment = NN_PERF_GET_BUFFER_ALIGNMENT();
        g_pMeterMemory = pGraphicsFramework->AllocateMemory(memorySize, alignment);

        nn::gfx::MemoryPool::InfoType info;
        info.SetDefault();
        info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuCached | nn::gfx::MemoryPoolProperty_GpuUncached);

        size_t memoryPoolSize = NN_PERF_GET_MEMORY_POOL_SIZE(pGraphicsFramework->GetDevice(), meterInfo);
        alignment = NN_PERF_GET_MEMORY_POOL_ALIGNMENT(pGraphicsFramework->GetDevice(), meterInfo);
        alignment = nn::util::align_up(alignment, nn::gfx::MemoryPool::GetPoolMemoryAlignment(pGraphicsFramework->GetDevice(), info));
        memoryPoolSize = nn::util::align_up(memoryPoolSize, nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(pGraphicsFramework->GetDevice(), info));
        g_pMeterPoolMemory = pGraphicsFramework->AllocateMemory(memoryPoolSize, alignment);
        info.SetPoolMemory(g_pMeterPoolMemory, memoryPoolSize);

        g_MeterMemoryPool.Initialize(pGraphicsFramework->GetDevice(), info);
        g_MeterMemoryPoolOffset = 0;

        NN_PERF_INITIALIZE_METER(pGraphicsFramework->GetDevice(), meterInfo,
            g_pMeterMemory, memorySize,
            &g_MeterMemoryPool, g_MeterMemoryPoolOffset, memoryPoolSize);

        //NN_PERF_SET_GET_CORE_NUMBER_FUNCTION(GetCoreNumber);
        // 現在のスレッド(メインスレッド)をコア0に割り当て
        nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
#else
        NN_UNUSED(pGraphicsFramework);
#endif
    }

    void FinalizeLoadMeter(nns::gfx::GraphicsFramework* pGraphicsFramework)
    {
#if defined(NN_PERF_PROFILE_ENABLED)
        NN_PERF_FINALIZE_METER(pGraphicsFramework->GetDevice());
        g_MeterMemoryPool.Finalize(pGraphicsFramework->GetDevice());
        pGraphicsFramework->FreeMemory(g_pMeterPoolMemory);
        pGraphicsFramework->FreeMemory(g_pMeterMemory);
#else
        NN_UNUSED(pGraphicsFramework);
#endif
    }

    //------------------------------------------------------------------------------
    //  パッド操作処理
    //------------------------------------------------------------------------------
    void ProcessPad(nns::gfx::GraphicsFramework* pGraphicsFramework)
    {
        ReadPad();

        // 選択項目の切り替え
        if (!IsPadHold(BUTTON_LEFT) && !IsPadHold(BUTTON_RIGHT))
        {
            if (IsPadTriggered(BUTTON_UP))
            {
                g_CurrentMenu = (g_CurrentMenu + EditMode_End - 1) % EditMode_End;
            }
            else if (IsPadTriggered(BUTTON_DOWN))
            {
                g_CurrentMenu = (g_CurrentMenu + 1) % EditMode_End;
            }
        }

        if (g_CurrentMenu == EditMode_FrameworkMode)
        {
            // フレームワークモードの切り替え
            if (IsPadTriggered(BUTTON_LEFT))
            {
                g_CurrentFrameworkMode = static_cast<nns::gfx::GraphicsFramework::FrameworkMode>((g_CurrentFrameworkMode + 3 - 1) % 3);
                pGraphicsFramework->SetFrameworkMode(g_CurrentFrameworkMode);
            }
            else if (IsPadTriggered(BUTTON_RIGHT))
            {
                g_CurrentFrameworkMode = static_cast<nns::gfx::GraphicsFramework::FrameworkMode>((g_CurrentFrameworkMode + 1) % 3);
                pGraphicsFramework->SetFrameworkMode(g_CurrentFrameworkMode);
            }
        }
        else if (g_CurrentMenu == EditMode_PresentInterval)
        {
            // 画面の垂直同期を同期する数の切り替え
            if (IsPadTriggered(BUTTON_LEFT))
            {
                if (g_CurrentPresentInterval > 1)
                {
                    g_CurrentPresentInterval--;
                }
                pGraphicsFramework->SetPresentInterval(g_CurrentPresentInterval);
            }
            else if (IsPadTriggered(BUTTON_RIGHT))
            {
                if (g_CurrentPresentInterval < 4)
                {
                    g_CurrentPresentInterval++;
                }
                pGraphicsFramework->SetPresentInterval(g_CurrentPresentInterval);
            }
        }
        else if (g_CurrentMenu == EditMode_CalculateSleepCpu)
        {
            // CPU 処理負荷の増減
            if (IsPadHold(BUTTON_LEFT))
            {
                g_CurrentCalculateSleepCpu -= 100 * g_CurrentPresentInterval;
                if (g_CurrentCalculateSleepCpu < 0)
                {
                    g_CurrentCalculateSleepCpu = 0;
                }
            }
            else if (IsPadHold(BUTTON_RIGHT))
            {
                g_CurrentCalculateSleepCpu += 100 * g_CurrentPresentInterval;
                if (g_CurrentCalculateSleepCpu > 100000)
                {
                    g_CurrentCalculateSleepCpu = 100000;
                }
            }
        }
        else if (g_CurrentMenu == EditMode_MakeCommandSleepCpu)
        {
            // CPU 処理負荷の増減
            if (IsPadHold(BUTTON_LEFT))
            {
                g_CurrentMakeCommandSleepCpu -= 100 * g_CurrentPresentInterval;
                if (g_CurrentMakeCommandSleepCpu < 0)
                {
                    g_CurrentMakeCommandSleepCpu = 0;
                }
            }
            else if (IsPadHold(BUTTON_RIGHT))
            {
                g_CurrentMakeCommandSleepCpu += 100 * g_CurrentPresentInterval;
                if (g_CurrentMakeCommandSleepCpu > 100000)
                {
                    g_CurrentMakeCommandSleepCpu = 100000;
                }
            }
        }
        else if (g_CurrentMenu == EditMode_DrawCount)
        {
            // GPU 処理負荷の増減
            if (IsPadHold(BUTTON_LEFT))
            {
                g_CurrentDrawCount -= 2 * g_CurrentPresentInterval;
                if (g_CurrentDrawCount < 0)
                {
                    g_CurrentDrawCount = 0;
                }
            }
            else if (IsPadHold(BUTTON_RIGHT))
            {
                g_CurrentDrawCount += 2 * g_CurrentPresentInterval;
                if (g_CurrentDrawCount > 2000)
                {
                    g_CurrentDrawCount = 2000;
                }
            }
        }

        // A ボタン押下中はボックスを描画する
        g_IsDrawBoxModelEnabled = IsPadHold(BUTTON_A);
    }// NOLINT(impl/function_size)

    void CalculateCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, void* pUserData)
    {
        NN_UNUSED(pUserData);

        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(g_CurrentCalculateSleepCpu));

        // パッド処理
        ProcessPad(pGraphicsFramework);
    }

    //------------------------------------------------------------------------------
    //  コマンド生成
    //------------------------------------------------------------------------------
    struct MakeCommandUserData
    {
        nn::gfx::util::DebugFontTextWriter* pWriter;
        int frame;
    };

    int g_SubQueueThreadRoopCount = 0;
    int g_MainQueueThreadRoopCount = 0;

    void MakeCommandCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData)
    {
        MakeCommandUserData* pData = reinterpret_cast<MakeCommandUserData*>(pUserData);

        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(g_CurrentMakeCommandSleepCpu));

        g_pPrimitiveRenderer->Update(bufferIndex);

        // コマンド生成
        pGraphicsFramework->BeginFrame(bufferIndex);
        {
            nn::gfx::CommandBuffer* rootCommandBuffer = pGraphicsFramework->GetRootCommandBuffer(bufferIndex);

            // プリミティブレンダラーで描画
            {
                nn::util::Matrix4x3fType viewMatrix;
                nn::util::Matrix4x4fType projectionMatrix;
                nn::util::Matrix4x3f modelMatrix;
                nn::util::Vector3f translate;
                nn::util::Vector3fType center = { 0.f, 0.f, 0.f };
                nn::util::Vector3fType size = { 5.0f, 5.0f, 5.0f };

                nn::gfx::ColorTargetView* target = pGraphicsFramework->GetColorTargetView();
                nn::gfx::DepthStencilView* depthStencil = pGraphicsFramework->GetDepthStencilView();
                rootCommandBuffer->ClearColor(target, 0.1f, 0.1f, 0.1f, 1.0f, NULL);
                rootCommandBuffer->SetRenderTargets(1, &target, depthStencil);
                rootCommandBuffer->SetViewportScissorState(pGraphicsFramework->GetViewportScissorState());


                CalculateViewMatrix(&viewMatrix, pData->frame);
                CalculateProjectionMatrix(&projectionMatrix, pGraphicsFramework->GetDisplayWidth(), pGraphicsFramework->GetDisplayHeight());
                CalculateModelMatrix(&modelMatrix);

                g_pPrimitiveRenderer->SetDefaultParameters();

                g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
                g_pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
                g_pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);

                // ボックスを描画
                if (g_IsDrawBoxModelEnabled)
                {
                    g_pPrimitiveRenderer->SetLineWidth(1.f);
                    g_pPrimitiveRenderer->DrawCube(rootCommandBuffer, nns::gfx::PrimitiveRenderer::Surface::Surface_Solid, center, size);
                }

                // 軸を描画
                float interval = -10.f;
                nn::util::Vector3fType begin;
                nn::util::Vector3fType end;
                g_pPrimitiveRenderer->SetLineWidth(1.f);
                for (int i = 0; i < 21; i++)
                {
                    nn::util::VectorSet(&begin, -10.f, 0.f, interval);
                    nn::util::VectorSet(&end, 10.f, 0.f, interval);
                    g_pPrimitiveRenderer->SetColor(nn::util::Color4u8::White());
                    g_pPrimitiveRenderer->DrawLine(rootCommandBuffer, begin, end);
                    nn::util::VectorSet(&begin, interval, 0.f, -10.f);
                    nn::util::VectorSet(&end, interval, 0.f, 10.f);
                    g_pPrimitiveRenderer->DrawLine(rootCommandBuffer, begin, end);
                    interval += 1.0f;
                }

                // オブジェクトを描画
                for (int i = 0; i < g_CurrentDrawCount; ++i)
                {
                    rootCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_IndirectBuffer);
                    rootCommandBuffer->FlushMemory(nn::gfx::GpuAccess_IndirectBuffer);
                    nn::util::VectorSet(&translate, 0, 0, 0);
                    nn::util::MatrixSetAxisW(&modelMatrix, translate);
                    g_pPrimitiveRenderer->SetColor(nn::util::Color4u8(0, 0, 0, 0));
                    g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
                    g_pPrimitiveRenderer->SetLineWidth(14.f);
                    g_pPrimitiveRenderer->DrawCone(rootCommandBuffer, nns::gfx::PrimitiveRenderer::Surface::Surface_Solid, center, size);
                }
            }

            {
                nn::util::Color4u8Type color0 = { { 255, 255, 255, 255 } };

                // Title
                pData->pWriter->SetScale(1.0f, 1.0f);
                pData->pWriter->SetTextColor(color0);
                pData->pWriter->SetCursor(10, 0);
                pData->pWriter->Print("Framework Demo");

                if ((pData->frame % 60) < 30) {
                    pData->pWriter->Print(NN_TEXT("■"));
                }

#if defined(NN_PERF_PROFILE_ENABLED)
                // メーターの描画
                nn::perf::CpuMeter* pFrameMeter = NN_PERF_GET_FRAME_METER();

                nn::util::Float2 pos = NN_UTIL_FLOAT_2_INITIALIZER(32.f, pGraphicsFramework->GetDisplayHeight() - g_MeterDrawer.GetHeight(pFrameMeter) - g_MeterDrawer.GetBarHeight());
                g_MeterDrawer.SetDebugFontTextWriter(pData->pWriter);
                g_MeterDrawer.SetPosition(pos);
                g_MeterDrawer.SetWidth(pGraphicsFramework->GetDisplayWidth() - 64.f);
                g_MeterDrawer.SetScale(2 * g_CurrentPresentInterval);
                pData->pWriter->SetScale(1.0f, 1.0f);
                g_MeterDrawer.Draw(rootCommandBuffer, g_pPrimitiveRenderer, pFrameMeter);

                // FPS 表示
                float fps = 1000.f / (static_cast<float>(pFrameMeter->GetLastTotalSpan().GetNanoSeconds()) / 1000000.0f);
                pData->pWriter->SetTextColor(nn::util::Color4u8::White());
                pData->pWriter->SetCursor(10, 20);
                pData->pWriter->Print("%4.1f FPS", fps);

                // 各メーターの色表示
                {
                    pData->pWriter->SetScale(0.75f, 0.75f);
                    pData->pWriter->SetCursor(40, 630);
                    pData->pWriter->SetTextColor(nn::util::Color4u8::Green());
                    pData->pWriter->Print("CPU  ");
                    pData->pWriter->SetTextColor(nn::util::Color4u8::Red());
                    pData->pWriter->Print("GPU  ");
                    pData->pWriter->SetTextColor(nn::util::Color4u8::Blue());
                    pData->pWriter->Print("AcquireTexture  ");
                    pData->pWriter->SetTextColor(nn::util::Color4u8::Yellow());
                    pData->pWriter->Print("PresentTexture  ");
                    pData->pWriter->SetTextColor(nn::util::Color4u8::White());
                    pData->pWriter->Print("WaitSync  ");
                    pData->pWriter->SetScale(1.0f, 1.0f);
                }
#endif
                // BufferCount
                {
                    pData->pWriter->SetTextColor(nn::util::Color4u8::White());
                    pData->pWriter->SetCursor(20, 60);
                    pData->pWriter->Print("BufferCount: %d", BufferCount);
                }

                // Current FrameworkMode
                {
                    char str[3][32] = { "DeferredExecution", "DeferredSubmission", "Immediate" };
                    pData->pWriter->SetTextColor(nn::util::Color4u8::White());
                    pData->pWriter->SetCursor(20, 80);
                    pData->pWriter->Print("FrameworkMode: %s", str[g_CurrentFrameworkMode]);
                }

                // Current PresentInterval
                {
                    pData->pWriter->SetTextColor(nn::util::Color4u8::White());
                    pData->pWriter->SetCursor(20, 100);
                    pData->pWriter->Print("PresentInterval: %d", g_CurrentPresentInterval);
                }

                // Current CalculateSleepCpu
                {
                    pData->pWriter->SetTextColor(nn::util::Color4u8::White());
                    pData->pWriter->SetCursor(20, 120);
                    pData->pWriter->Print(NN_TEXT("CalculateSleepCpu: %d μs"), g_CurrentCalculateSleepCpu);
                }

                // Current MakeCommandSleepCpu
                {
                    pData->pWriter->SetTextColor(nn::util::Color4u8::White());
                    pData->pWriter->SetCursor(20, 140);
                    pData->pWriter->Print(NN_TEXT("MakeCommandSleepCpu: %d μs"), g_CurrentMakeCommandSleepCpu);
                }

                // Current DrawCount
                {
                    pData->pWriter->SetTextColor(nn::util::Color4u8::White());
                    pData->pWriter->SetCursor(20, 160);
                    pData->pWriter->Print("DrawCount: %d", g_CurrentDrawCount);
                }

                // Cursor
                {
                    pData->pWriter->SetTextColor(nn::util::Color4u8::White());
                    pData->pWriter->SetCursor(4, 80 + g_CurrentMenu * 20.0f);
                    pData->pWriter->Print(">");
                }

                // RoopCount
                {
                    pData->pWriter->SetTextColor(nn::util::Color4u8::Cyan());
                    pData->pWriter->SetCursor(1000.0f, 240.0f);
                    pData->pWriter->Print("MainThreadRoopCount: %d", g_MainQueueThreadRoopCount);

                    pData->pWriter->SetCursor(1000.0f, 260.0f);
                    pData->pWriter->Print("SubThreadRoopCount:  %d", g_SubQueueThreadRoopCount);
                }
            }

            // デバッグフォント用のコマンド生成
            pData->pWriter->Draw(rootCommandBuffer);
        }
        pGraphicsFramework->EndFrame(bufferIndex);

        g_MainQueueThreadRoopCount++;
    }// NOLINT(impl/function_size)



    //------------------------------------------------------------------------------
    //  SubQueue
    //------------------------------------------------------------------------------
    nn::gfx::Queue                              g_SubQueue;
    nns::gfx::GraphicsFramework::CommandBuffer  g_SubCommandBuffer;

    void InitializeSubGraphicsObjects(nns::gfx::GraphicsFramework* pGfxFramework)
    {
        auto pDevice = pGfxFramework->GetDevice();

        //  キュー
        {
            nn::gfx::Queue::InfoType info;
            info.SetDefault();
            info.SetCapability(nn::gfx::QueueCapability_Copy | nn::gfx::QueueCapability_Graphics);
            g_SubQueue.Initialize(pDevice, info);
        }

        // コマンドバッファ
        {
            nn::gfx::CommandBuffer::InfoType info;
            info.SetDefault();

            const size_t commandMemorySize = 1 * 1024 * 1024;
            const size_t controlMemorySize = 1 * 1024 * 1024;
            pGfxFramework->InitializeCommandBuffer(&g_SubCommandBuffer, info, commandMemorySize, controlMemorySize);
        }

    }

    void FinalizeSubGraphicsObjects(nns::gfx::GraphicsFramework* pGfxFramework)
    {
        auto pDevice = pGfxFramework->GetDevice();

        // 完了待ちしてから破棄
        g_SubQueue.Sync();

        pGfxFramework->FinalizeCommandBuffer(&g_SubCommandBuffer);
        g_SubQueue.Finalize(pDevice);
    }

    const size_t ThreadStackSize = 32 * 1024;
    NN_OS_ALIGNAS_THREAD_STACK char g_SubQueueThreadStack[ThreadStackSize];
    nn::os::ThreadType              g_SubQueueThread;
    bool                            g_IsCountinuedSubQueueThread = true;
    void SubQueueThreadFunction(void* arg)
    {
        NN_LOG("Start: %s\n", __FUNCTION__);

        auto* pGfxFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(arg);
        while (g_IsCountinuedSubQueueThread)
        {
            pGfxFramework->ResetCommandBuffer(&g_SubCommandBuffer);
            auto& commandBuffer = g_SubCommandBuffer.object;

            // Sync で IPC しすぎるのを緩和する。
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(2));

            commandBuffer.Begin();
            {
                // とりあえず何もしない。
            }
            commandBuffer.End();

            g_SubQueue.ExecuteCommand(&commandBuffer, NULL);
            g_SubQueue.Flush();
            g_SubQueue.Sync();

            g_SubQueueThreadRoopCount++;
        }

        NN_LOG("Exit: %s\n", __FUNCTION__);
    }

}

//------------------------------------------------------------------------------
//  メイン 関数
//------------------------------------------------------------------------------
extern "C" void nnMain()
{
    // FS を使用する場合はフレームワークよりも前に初期化します

    // フレームワーク初期化
    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(GraphicsSystemMemorySize);

    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    fwInfo.SetDefault();
    fwInfo.SetDisplayWidth(1280);
    fwInfo.SetDisplayHeight(720);
    fwInfo.SetBufferCount(BufferCount);
    fwInfo.SetSwapChainBufferCount(BufferCount);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, (BufferCount * 8 + 8) * 1024 * 1024);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_CommandBuffer, ((BufferCount + 1 ) * 8 + 8) * 1024 * 1024);
    fwInfo.SetRootCommandBufferCommandMemorySize(8 * 1024 * 1024);
    //fwInfo.SetDebugMode(nn::gfx::DebugMode_Full);
    nns::gfx::GraphicsFramework gfw;
    gfw.Initialize(fwInfo);

#if defined(NN_PERF_PROFILE_ENABLED)
    InitializeFrameworkUtility(&gfw);
#endif

    // デバッグフォント初期化
    nns::gfx::GraphicsFramework::DebugFontTextWriter writer;
    gfw.InitializeDebugFontTextWriter(&writer, 1024 * 8, BufferCount);

    // プリミティブレンダラー初期化
    {
        nns::gfx::PrimitiveRenderer::RendererInfo rendererInfo;
        rendererInfo.SetDefault();
        rendererInfo.SetAllocator(nns::gfx::GraphicsFramework::DefaultAllocateFunction, nullptr);
        rendererInfo.SetAdditionalBufferSize(1024 * 4);
        rendererInfo.SetDrawCallCountMax(1024 * 4);
        rendererInfo.SetViewFunctionCallCountMax(1024 * 4);

        rendererInfo.SetMultiBufferQuantity(BufferCount);

        g_pPrimitiveRenderer = nns::gfx::PrimitiveRenderer::CreateRenderer(gfw.GetDevice(), rendererInfo);
        g_pPrimitiveRenderer->SetScreenWidth(gfw.GetDisplayWidth());
        g_pPrimitiveRenderer->SetScreenHeight(gfw.GetDisplayHeight());
    }

    // パフォーマンス計測初期化
    InitializeLoadMeter(&gfw);

    // パッド初期化
    InitializePadUtility(gfw.GetLayer());


    g_CurrentMenu = EditMode_FrameworkMode;
    g_CurrentFrameworkMode = nns::gfx::GraphicsFramework::FrameworkMode_DeferredExecution;
    g_CurrentPresentInterval = 1;
    g_CurrentCalculateSleepCpu = 0;
    g_CurrentMakeCommandSleepCpu = 0;
    g_CurrentDrawCount = 0;

    // GraphicsFramework によるフレーム処理用の設定
    MakeCommandUserData userData;
    userData.pWriter = &writer.object;
    userData.frame = 0;

    gfw.SetCalculateCallback(CalculateCallback, nullptr);
    gfw.SetMakeCommandCallback(MakeCommandCallback, &userData);
    gfw.SetFrameworkMode(g_CurrentFrameworkMode);
    gfw.SetPresentInterval(g_CurrentPresentInterval);


    // SubQueue の初期化
    InitializeSubGraphicsObjects(&gfw);

    // スレッドを生成する
    auto result = nn::os::CreateThread(&g_SubQueueThread, SubQueueThreadFunction, &gfw, g_SubQueueThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority, 1);
    NN_ASSERT(result.IsSuccess(), "App: Cannot create SubQueueThread.");

    // スレッドの開始
    nn::os::StartThread(&g_SubQueueThread);

    // 毎フレームのレンダリング
    for (int frame = 0; frame < 1000000; ++frame)
    {
        NN_PERF_BEGIN_FRAME();
#if defined(NN_PERF_PROFILE_ENABLED)
        ProcessFrame();
#else
        gfw.ProcessFrame();
#endif
        userData.frame++;
        NN_PERF_END_FRAME();
    }
    gfw.QueueFinish();

    g_IsCountinuedSubQueueThread = false;

    // スレッドの終了を待機、破棄
    nn::os::WaitThread(&g_SubQueueThread);
    nn::os::DestroyThread(&g_SubQueueThread);
    FinalizeSubGraphicsObjects(&gfw);

    // パッド終了
    FinalizePadUtility();

    // パフォーマンス計測終了
    FinalizeLoadMeter(&gfw);

    // プリミティブレンダラー終了
    nns::gfx::PrimitiveRenderer::DestroyRenderer(g_pPrimitiveRenderer, gfw.GetDevice(), nns::gfx::GraphicsFramework::DefaultFreeFunction, nullptr);

    // デバッグフォント終了
    gfw.FinalizeDebugFontTextWriter(&writer);

    // フレームワーク終了
#if defined(NN_PERF_PROFILE_ENABLED)
    FinalizeFrameworkUtility();
#endif
    gfw.Finalize();
}
