﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nn/gfx/gfx_GpuAddress.h>
#include <nn/gfx/gfx_Buffer.h>
#include <nn/gfx/gfx_Texture.h>
#include <nn/gfx/gfx_Sampler.h>

#include "gfxUtilGpuBenchmark_ResourceAllocator.h"

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
    #if defined( NN_BUILD_TARGET_PLATFORM_OS_NN )
        #include <nvnTool/nvnTool_GlslcInterface.h>
    #endif
#endif


namespace nnt { namespace gfx { namespace util {

namespace {

int g_MemoryPoolProperty[] = {
    /* MemoryPoolType_RenderTarget */
    nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_Compressible,
    /* MemoryPoolType_Data */
    nn::gfx::MemoryPoolProperty_CpuCached | nn::gfx::MemoryPoolProperty_GpuCached,
    /* MemoryPoolType_CommandBuffer */
    nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuUncached,
    /* MemoryPoolType_ConstantBuffer */
    nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached,
    /* MemoryPoolType_Shader */
    nn::gfx::MemoryPoolProperty_CpuCached | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_ShaderCode,
    /* MemoryPoolType_QueryBuffer */
    nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuUncached,
};
NN_STATIC_ASSERT((sizeof(g_MemoryPoolProperty) / sizeof(g_MemoryPoolProperty[0])) == MemoryPoolType_End);

const size_t g_MemoryPoolMaxUsage[] =
{
    /* MemoryPoolType_RenderTarget      */  217055232 - 15728640,
    /* MemoryPoolType_Data              */  384 * 1024 * 1024,
    /* MemoryPoolType_CommandBuffer     */  25165824 - 16777216,
    /* MemoryPoolType_ConstantBuffer    */  70144,
    /* MemoryPoolType_Shader            */  547184 + 133888,
    /* MemoryPoolType_QueryBuffer       */  4096,
};
NN_STATIC_ASSERT((sizeof(g_MemoryPoolMaxUsage) / sizeof(g_MemoryPoolMaxUsage[0])) == MemoryPoolType_End);


void* DefaultalignedAllocatorMemoryAllocate(size_t size, size_t alignment, void* pUserData)
{
    NN_UNUSED(pUserData);
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    return aligned_alloc(alignment, nn::util::align_up(size, alignment));
#else
    return _aligned_malloc(size, alignment);
#endif
}

void DefaultalignedAllocatorMemoryFree(void* pMemory, void* pUserData)
{
    NN_UNUSED(pUserData);
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    free(pMemory);
#else
    _aligned_free(pMemory);
#endif
}

}

void* ResourceAllocator::MemoryPoolAllocatorMalloc(size_t size, void* pUserData)
{
    ResourceAllocator* pResourceAllocator = reinterpret_cast<ResourceAllocator*>(pUserData);
    return pResourceAllocator->m_pAllocateMemoryCallback(size, 1, pResourceAllocator->m_pMemoryCallbackUserData);
}

void ResourceAllocator::MemoryPoolAllocatorFree(void* pMemory, void* pUserData)
{
    ResourceAllocator* pResourceAllocator = reinterpret_cast<ResourceAllocator*>(pUserData);
    return pResourceAllocator->m_pFreeMemoryCallback(pMemory, pResourceAllocator->m_pMemoryCallbackUserData);
}

void* ResourceAllocator::DescriptorPoolAllocatorMalloc(size_t size, void* pUserData)
{
    ResourceAllocator* pResourceAllocator = reinterpret_cast<ResourceAllocator*>(pUserData);
    return pResourceAllocator->m_pAllocateMemoryCallback(size, 1, pResourceAllocator->m_pMemoryCallbackUserData);
}

void ResourceAllocator::DescriptorPoolAllocatorFree(void* pMemory, void* pUserData)
{
    ResourceAllocator* pResourceAllocator = reinterpret_cast<ResourceAllocator*>(pUserData);
    return pResourceAllocator->m_pFreeMemoryCallback(pMemory, pResourceAllocator->m_pMemoryCallbackUserData);
}

void ResourceAllocatorInfo::SetDefault()
{
    m_pAllocateMemoryCallback = &DefaultalignedAllocatorMemoryAllocate;
    m_pFreeMemoryCallback = &DefaultalignedAllocatorMemoryFree;
    m_pMemoryCallbackUserData = nullptr;

    SetDescriptorPoolCount(nn::gfx::DescriptorPoolType_BufferView, 16);
    SetDescriptorPoolCount(nn::gfx::DescriptorPoolType_TextureView, 512);
    SetDescriptorPoolCount(nn::gfx::DescriptorPoolType_Sampler, 512);

    for (int memoryPoolIndex = 0; memoryPoolIndex < MemoryPoolType_End; ++memoryPoolIndex)
    {
        MemoryPoolType memoryPoolType = static_cast<MemoryPoolType>(memoryPoolIndex);
        SetMemoryPoolSize(memoryPoolType, g_MemoryPoolMaxUsage[memoryPoolType]);
    }
}

void ResourceAllocatorInfo::SetMemoryAllocator(
    AllocateMemoryCallback pAllocateMemoryCallback,
    FreeMemoryCallback pFreeMemoryCallback,
    void* pMemoryCallbackUserData)
{
    m_pAllocateMemoryCallback = pAllocateMemoryCallback;
    m_pFreeMemoryCallback = pFreeMemoryCallback;
    m_pMemoryCallbackUserData = pMemoryCallbackUserData;
}

AllocateMemoryCallback ResourceAllocatorInfo::GetAllocateMemoryCallback() const
{
    return m_pAllocateMemoryCallback;
}

FreeMemoryCallback ResourceAllocatorInfo::GetFreeMemoryCallback() const
{
    return m_pFreeMemoryCallback;
}

void* ResourceAllocatorInfo::GetMemoryCallbackUserData() const
{
    return m_pMemoryCallbackUserData;
}

void ResourceAllocatorInfo::SetMemoryPoolSize(MemoryPoolType type, size_t sizeInBytes)
{
    NN_ASSERT(type < MemoryPoolType_End);
    m_MemoryPoolSize[type] = sizeInBytes;
}

size_t ResourceAllocatorInfo::GetMemoryPoolSize(MemoryPoolType type) const
{
    NN_ASSERT(type < MemoryPoolType_End);
    return m_MemoryPoolSize[type];
}

void ResourceAllocatorInfo::SetDescriptorPoolCount(nn::gfx::DescriptorPoolType type, int count)
{
    NN_ASSERT(type < nn::gfx::DescriptorPoolType_End);;
    m_DescriptorPoolCount[type] = count;
}

int ResourceAllocatorInfo::GetDescriptorPoolCount(nn::gfx::DescriptorPoolType type) const
{
    NN_ASSERT(type < nn::gfx::DescriptorPoolType_End);;
    return m_DescriptorPoolCount[type];
}

ResourceAllocator::ResourceAllocator()
{
}

ResourceAllocator::~ResourceAllocator()
{
}

void ResourceAllocator::Initialize(nn::gfx::Device* pDevice, const InfoType& info)
{
    m_pAllocateMemoryCallback   = info.GetAllocateMemoryCallback();
    m_pFreeMemoryCallback       = info.GetFreeMemoryCallback();
    m_pMemoryCallbackUserData   = info.GetMemoryCallbackUserData();

#if defined(NN_SDK_BUILD_DEBUG)
    InitializeMemoryPoolAllocatorTrackerData();
#endif

    InitializeMemoryPools(info, pDevice);
    InitializeDescriptorPools(info, pDevice);

#if defined(NN_SDK_BUILD_DEBUG)
    PushMemoryPoolAllocatorStatus();
    UpdateMemoryPoolAllocatorMaxUsage();
#endif

}

void ResourceAllocator::Finalize(nn::gfx::Device* pDevice)
{
#if defined(NN_SDK_BUILD_DEBUG)
    PopAndCompareMemoryPoolAllocatorStatus();
#endif

    FinalizeDescriptorPools(pDevice);
    FinalizeMemoryPools(pDevice);

#if defined(NN_SDK_BUILD_DEBUG)
    NN_ASSERT(m_MemoryAllocationCount == 0);
#endif
}


void* ResourceAllocator::AllocateMemory(size_t size, size_t alignment)
{
#if defined(NN_SDK_BUILD_DEBUG)
    m_MemoryAllocationCount++;
#endif

    void* pBuffer = m_pAllocateMemoryCallback(size, alignment, m_pMemoryCallbackUserData);
    NN_ASSERT_NOT_NULL(pBuffer);
    return pBuffer;
}

void ResourceAllocator::FreeMemory(void* pMemory)
{
#if defined(NN_SDK_BUILD_DEBUG)
    NN_ASSERT(m_MemoryAllocationCount > 0);
    m_MemoryAllocationCount--;
#endif

    m_pFreeMemoryCallback(pMemory, m_pMemoryCallbackUserData);
}

nn::gfx::MemoryPool* ResourceAllocator::GetMemoryPool(MemoryPoolType memoryPoolType)
{
    NN_ASSERT(memoryPoolType < MemoryPoolType_End);
    return &m_MemoryPoolArray[memoryPoolType].memoryPool;
}

ptrdiff_t ResourceAllocator::AllocatePoolMemory(MemoryPoolType memoryPoolType, size_t size, size_t alignment)
{
    NN_ASSERT(memoryPoolType < MemoryPoolType_End);
    NN_ASSERT(m_MemoryPoolArray[memoryPoolType].memorySize > 0);
    ptrdiff_t offset = m_MemoryPoolArray[memoryPoolType].allocator.Allocate(size, alignment);
    NN_ASSERT(offset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);
    return offset;
}

void ResourceAllocator::FreePoolMemory(MemoryPoolType memoryPoolType, ptrdiff_t offset)
{
    NN_ASSERT(memoryPoolType < MemoryPoolType_End);
    NN_ASSERT(m_MemoryPoolArray[memoryPoolType].memorySize > 0);
    return m_MemoryPoolArray[memoryPoolType].allocator.Free(offset);
}

size_t ResourceAllocator::GetMemoryPoolSize(MemoryPoolType memoryPoolType) const
{
    NN_ASSERT(memoryPoolType < MemoryPoolType_End);
    return m_MemoryPoolArray[memoryPoolType].memorySize;
}

size_t ResourceAllocator::GetMemoryPoolAllocatorTotalFreeSize(MemoryPoolType memoryPoolType)
{
    NN_ASSERT(memoryPoolType < MemoryPoolType_End);
    if (m_MemoryPoolArray[memoryPoolType].memorySize > 0)
    {
        return 0;
    }
    return m_MemoryPoolArray[memoryPoolType].allocator.GetTotalFreeSize();
}


int ResourceAllocator::AllocateDescriptorSlots(nn::gfx::DescriptorPoolType descriptorPoolType, int count)
{
    NN_ASSERT(descriptorPoolType < nn::gfx::DescriptorPoolType_End);
    int slotIndex = m_DescriptorPoolArray[descriptorPoolType].allocator.Allocate(count);
    NN_ASSERT(slotIndex != nn::gfx::util::DescriptorPoolAllocator::InvalidIndex);
    return slotIndex;
}

void ResourceAllocator::FreeDescriptorSlots(nn::gfx::DescriptorPoolType descriptorPoolType, int firstSlotIndex)
{
    m_DescriptorPoolArray[descriptorPoolType].allocator.Free(firstSlotIndex);
}

nn::gfx::DescriptorPool* ResourceAllocator::GetDescriptorPool(nn::gfx::DescriptorPoolType descriptorPoolType)
{
    NN_ASSERT(descriptorPoolType < nn::gfx::DescriptorPoolType_End);
    return &m_DescriptorPoolArray[descriptorPoolType].decriptorPool;
}

int ResourceAllocator::AllocateAndSetBufferViewToDescriptorPool(
    nn::gfx::Buffer* pBuffer, size_t bufferSize,
    nn::gfx::DescriptorSlot* pDescriptorSlot)
{
    nn::gfx::DescriptorPoolType descriptorPoolType = nn::gfx::DescriptorPoolType_BufferView;
    int slotIndex = AllocateDescriptorSlots(descriptorPoolType, 1);
    SetBufferViewToDescriptorPool(slotIndex, 1, pBuffer, &bufferSize, pDescriptorSlot);
    return slotIndex;
}

int ResourceAllocator::AllocateAndSetTextureViewToDescriptorPool(
    nn::gfx::TextureView* pTextureView, nn::gfx::DescriptorSlot* pDescriptorSlot)
{
    nn::gfx::DescriptorPoolType descriptorPoolType = nn::gfx::DescriptorPoolType_TextureView;
    int slotIndex = AllocateDescriptorSlots(descriptorPoolType, 1);
    SetTextureViewToDescriptorPool(slotIndex, 1, pTextureView, pDescriptorSlot);
    return slotIndex;
}

int ResourceAllocator::AllocateAndSetSamplerToDescriptorPool(
    nn::gfx::Sampler* pSampler, nn::gfx::DescriptorSlot* pDescriptorSlot)
{
    nn::gfx::DescriptorPoolType descriptorPoolType = nn::gfx::DescriptorPoolType_Sampler;
    int slotIndex = AllocateDescriptorSlots(descriptorPoolType, 1);
    SetSamplerToDescriptorPool(slotIndex, 1, pSampler, pDescriptorSlot);
    return slotIndex;
}

void ResourceAllocator::SetBufferViewToDescriptorPool(
    int slotIndex, int count,
    nn::gfx::Buffer* pBuffer, size_t* pBufferSizeArray,
    nn::gfx::DescriptorSlot* pDescriptorSlotArray)
{
    nn::gfx::DescriptorPoolType descriptorPoolType = nn::gfx::DescriptorPoolType_BufferView;
    nn::gfx::DescriptorPool* pDescriptorPool = &m_DescriptorPoolArray[descriptorPoolType].decriptorPool;

    pDescriptorPool->BeginUpdate();
    for (int i = 0; i < count; ++i)
    {
        nn::gfx::GpuAddress gpuAddress;
        pBuffer[i].GetGpuAddress(&gpuAddress);
        pDescriptorPool->SetBufferView(slotIndex + i, gpuAddress, pBufferSizeArray[i]);
    }
    pDescriptorPool->EndUpdate();

    for (int i = 0; i < count; ++i)
    {
        pDescriptorPool->GetDescriptorSlot(&pDescriptorSlotArray[i], slotIndex + i);
    }
}

void ResourceAllocator::SetTextureViewToDescriptorPool(
    int slotIndex, int count,
    nn::gfx::TextureView* pTextureViewArray, nn::gfx::DescriptorSlot* pDescriptorSlotArray)
{
    nn::gfx::DescriptorPoolType descriptorPoolType = nn::gfx::DescriptorPoolType_TextureView;
    nn::gfx::DescriptorPool* pDescriptorPool = &m_DescriptorPoolArray[descriptorPoolType].decriptorPool;

    pDescriptorPool->BeginUpdate();
    for (int i = 0; i < count; ++i)
    {
        pDescriptorPool->SetTextureView(slotIndex + i, &pTextureViewArray[i]);
    }
    pDescriptorPool->EndUpdate();

    for (int i = 0; i < count; ++i)
    {
        pDescriptorPool->GetDescriptorSlot(&pDescriptorSlotArray[i], slotIndex + i);
    }
}

void ResourceAllocator::SetSamplerToDescriptorPool(
    int slotIndex, int count,
    nn::gfx::Sampler* pSamplerArray, nn::gfx::DescriptorSlot* pDescriptorSlotArray)
{
    nn::gfx::DescriptorPoolType descriptorPoolType = nn::gfx::DescriptorPoolType_Sampler;
    nn::gfx::DescriptorPool* pDescriptorPool = &m_DescriptorPoolArray[descriptorPoolType].decriptorPool;

    pDescriptorPool->BeginUpdate();
    for (int i = 0; i < count; ++i)
    {
        pDescriptorPool->SetSampler(slotIndex + i, &pSamplerArray[i]);
    }
    pDescriptorPool->EndUpdate();

    for (int i = 0; i < count; ++i)
    {
        pDescriptorPool->GetDescriptorSlot(&pDescriptorSlotArray[i], slotIndex + i);
    }
}

void ResourceAllocator::InitializeMemoryPools(const InfoType& info, nn::gfx::Device* pDevice)
{
    for (int i = 0; i < MemoryPoolType_End; ++i)
    {
        MemoryPoolType memoryPoolType = static_cast<MemoryPoolType>(i);
        MemoryPoolData* pMemoryPoolData = &m_MemoryPoolArray[memoryPoolType];
        size_t requestedMemoryPoolSize = info.GetMemoryPoolSize(memoryPoolType);

        if (requestedMemoryPoolSize > 0)
        {
            nn::gfx::MemoryPool::InfoType memorypoolInfo;
            memorypoolInfo.SetDefault();
            memorypoolInfo.SetMemoryPoolProperty(g_MemoryPoolProperty[i]);

            size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(pDevice, memorypoolInfo);
            size_t granularity = nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(pDevice, memorypoolInfo);
            size_t poolMemorySize = nn::util::align_up(requestedMemoryPoolSize, granularity);

            void* pPoolMemory = AllocateMemory(poolMemorySize, alignment);
            NN_ASSERT_NOT_NULL(pPoolMemory);

            memorypoolInfo.SetPoolMemory(pPoolMemory, poolMemorySize);

            pMemoryPoolData->memoryPool.Initialize(pDevice, memorypoolInfo);

            pMemoryPoolData->allocator.Initialize(
                MemoryPoolAllocatorMalloc,
                reinterpret_cast<void*>(this),
                MemoryPoolAllocatorFree,
                reinterpret_cast<void*>(this),
                &pMemoryPoolData->memoryPool,
                0,
                poolMemorySize,
                nn::gfx::util::MemoryPoolAllocator::AlignmentMax,
                true
            );
            NN_SDK_ASSERT(pMemoryPoolData->allocator.IsInitialized());

            pMemoryPoolData->memorySize = poolMemorySize;
            pMemoryPoolData->pMemory = pPoolMemory;
        }
        else
        {
            pMemoryPoolData->memorySize = 0;
            pMemoryPoolData->pMemory = nullptr;
        }
    }
}

void ResourceAllocator::FinalizeMemoryPools(nn::gfx::Device* pDevice)
{
    for (int i = 0; i < MemoryPoolType_End; ++i)
    {
        MemoryPoolType memoryPoolType = static_cast<MemoryPoolType>(i);
        MemoryPoolData* pMemoryPoolData = &m_MemoryPoolArray[memoryPoolType];

        if (pMemoryPoolData->memorySize > 0)
        {
            pMemoryPoolData->allocator.Finalize();
            pMemoryPoolData->memoryPool.Finalize(pDevice);
            FreeMemory(pMemoryPoolData->pMemory);

            pMemoryPoolData->pMemory = nullptr;
            pMemoryPoolData->memorySize = 0;
        }
    }
}

void ResourceAllocator::InitializeDescriptorPools(const InfoType& info, nn::gfx::Device* pDevice)
{
    int descriptorBaseIndex[nn::gfx::DescriptorPoolType_End];
    descriptorBaseIndex[nn::gfx::DescriptorPoolType_BufferView] = 0;
    descriptorBaseIndex[nn::gfx::DescriptorPoolType_TextureView] = 0;
    descriptorBaseIndex[nn::gfx::DescriptorPoolType_Sampler] = 0;

#if NN_GFX_IS_TARGET_NVN
    nn::gfx::Device::DataType& deviceData = nn::gfx::AccessorToData(pDevice);
    nvnDeviceGetInteger(deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS, &descriptorBaseIndex[nn::gfx::DescriptorPoolType_TextureView]);
    nvnDeviceGetInteger(deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS, &descriptorBaseIndex[nn::gfx::DescriptorPoolType_Sampler]);
#endif

    for (int i = 0; i < nn::gfx::DescriptorPoolType_End; i++)
    {
        nn::gfx::DescriptorPoolType descriptorPoolType = static_cast<nn::gfx::DescriptorPoolType>(i);
        DecriptorPoolData* pDecriptorPoolData = &m_DescriptorPoolArray[descriptorPoolType];

        int requestSlotCount = info.GetDescriptorPoolCount(descriptorPoolType);

        if (requestSlotCount > 0) {
            NN_SDK_ASSERT_GREATER(requestSlotCount, descriptorBaseIndex[i]);

            nn::gfx::DescriptorPool::InfoType descriptorpoolInfo;
            descriptorpoolInfo.SetDefault();
            descriptorpoolInfo.SetDescriptorPoolType(descriptorPoolType);
            descriptorpoolInfo.SetSlotCount(requestSlotCount);

            size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(pDevice, descriptorpoolInfo);
            size_t alignment = nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(pDevice, descriptorpoolInfo);
            ptrdiff_t memoryPoolOffset = AllocatePoolMemory(DecriptorPoolData::MemoryPoolType, size, alignment);
            NN_SDK_ASSERT(memoryPoolOffset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);

            pDecriptorPoolData->decriptorPool.Initialize(
                pDevice, descriptorpoolInfo,
                GetMemoryPool(DecriptorPoolData::MemoryPoolType), memoryPoolOffset, size);

            pDecriptorPoolData->allocator.Initialize(
                DescriptorPoolAllocatorMalloc,
                reinterpret_cast<void*>(this),
                DescriptorPoolAllocatorFree,
                reinterpret_cast<void*>(this),
                &pDecriptorPoolData->decriptorPool,
                descriptorBaseIndex[i],
                requestSlotCount - descriptorBaseIndex[i],
                true
            );
            NN_SDK_ASSERT(pDecriptorPoolData->allocator.IsInitialized());

            pDecriptorPoolData->memoryPoolOffset = memoryPoolOffset;
            pDecriptorPoolData->count = requestSlotCount;
        }
        else
        {
            pDecriptorPoolData->memoryPoolOffset = 0;
            pDecriptorPoolData->count = 0;
        }
    }
}

void ResourceAllocator::FinalizeDescriptorPools(nn::gfx::Device* pDevice)
{
    for (int i = 0; i < nn::gfx::DescriptorPoolType_End; i++)
    {
        nn::gfx::DescriptorPoolType descriptorPoolType = static_cast<nn::gfx::DescriptorPoolType>(i);
        DecriptorPoolData* pDecriptorPoolData = &m_DescriptorPoolArray[descriptorPoolType];

        if (pDecriptorPoolData->count > 0)
        {
            pDecriptorPoolData->allocator.Finalize();
            pDecriptorPoolData->decriptorPool.Finalize(pDevice);
            FreePoolMemory(DecriptorPoolData::MemoryPoolType, pDecriptorPoolData->memoryPoolOffset);
            pDecriptorPoolData->memoryPoolOffset = 0;
            pDecriptorPoolData->count = 0;
        }

    }

}

#if defined(NN_SDK_BUILD_DEBUG)

namespace {

static const char* g_memoryPoolName[] =
{
    "RenderTarget",         //MemoryPoolType_RenderTarget,
    "Data",                 //MemoryPoolType_Data,
    "CommandBuffer",        //MemoryPoolType_CommandBuffer,
    "ConstantBuffer",       //MemoryPoolType_ConstantBuffer,
    "Shader",               //MemoryPoolType_Shader,
    "QueryBuffer",          //MemoryPoolType_QueryBuffer,
};
NN_STATIC_ASSERT((sizeof(g_memoryPoolName) / sizeof(g_memoryPoolName[0])) == nnt::gfx::util::MemoryPoolType_End);

}

void ResourceAllocator::InitializeMemoryPoolAllocatorTrackerData()
{
    memset(&m_MemoryPoolAllocatorTrackerData, 0, sizeof(m_MemoryPoolAllocatorTrackerData));
    m_MemoryAllocationCount = 0;
}

void ResourceAllocator::PushMemoryPoolAllocatorStatus()
{
    NN_ASSERT(m_MemoryPoolAllocatorTrackerData.statusDumpIndex < MemoryPoolAllocatorTrackerData::DumpMaxCount);
    MemoryPoolAllocatorStatusDump* pDump = &m_MemoryPoolAllocatorTrackerData.statusDump[m_MemoryPoolAllocatorTrackerData.statusDumpIndex];

    pDump->memoryAllocationCount = m_MemoryAllocationCount;

    for (unsigned int memoryPoolIndex = 0; memoryPoolIndex < nnt::gfx::util::MemoryPoolType_End; ++memoryPoolIndex)
    {
        nnt::gfx::util::MemoryPoolType type = static_cast<nnt::gfx::util::MemoryPoolType>(memoryPoolIndex);
        pDump->poolState[memoryPoolIndex].totalFreeSize = GetMemoryPoolAllocatorTotalFreeSize(type);
    }

    m_MemoryPoolAllocatorTrackerData.statusDumpIndex++;
}

void ResourceAllocator::PopAndCompareMemoryPoolAllocatorStatus()
{
    NN_ASSERT(m_MemoryPoolAllocatorTrackerData.statusDumpIndex > 0);
    MemoryPoolAllocatorStatusDump* pDump =
        &m_MemoryPoolAllocatorTrackerData.statusDump[m_MemoryPoolAllocatorTrackerData.statusDumpIndex - 1];

    for (unsigned int memoryPoolIndex = 0; memoryPoolIndex < nnt::gfx::util::MemoryPoolType_End; ++memoryPoolIndex)
    {
        nnt::gfx::util::MemoryPoolType type = static_cast<nnt::gfx::util::MemoryPoolType>(memoryPoolIndex);
        size_t refTotalFreeSize = pDump->poolState[memoryPoolIndex].totalFreeSize;
        size_t actualTotalFreeSize = GetMemoryPoolAllocatorTotalFreeSize(type);
        if (refTotalFreeSize != actualTotalFreeSize)
        {
            NN_LOG("Pool %s has remaining allocations\n", g_memoryPoolName[memoryPoolIndex]);
            NN_ABORT();
        }
    }

    if (m_MemoryAllocationCount != pDump->memoryAllocationCount)
    {
        NN_LOG("Allocation count mismatch\n");
        NN_ABORT();
    }

    m_MemoryPoolAllocatorTrackerData.statusDumpIndex--;
}

void ResourceAllocator::UpdateMemoryPoolAllocatorMaxUsage()
{
    for (unsigned int memoryPoolIndex = 0; memoryPoolIndex < nnt::gfx::util::MemoryPoolType_End; ++memoryPoolIndex)
    {
        nnt::gfx::util::MemoryPoolType type = static_cast<nnt::gfx::util::MemoryPoolType>(memoryPoolIndex);
        size_t totalSize = GetMemoryPoolSize(type);
        size_t freeSize = GetMemoryPoolAllocatorTotalFreeSize(type);
        size_t usedSize = totalSize - freeSize;

        if (usedSize > m_MemoryPoolAllocatorTrackerData.memoryPoolMaxUsage[type])
        {
            m_MemoryPoolAllocatorTrackerData.memoryPoolMaxUsage[type] = usedSize;
        }
    }
}

void ResourceAllocator::PrintMemoryPoolAllocatorMaxUsage()
{
    for (unsigned int memoryPoolIndex = 0; memoryPoolIndex < nnt::gfx::util::MemoryPoolType_End; ++memoryPoolIndex)
    {
        size_t maxRequiredSize = m_MemoryPoolAllocatorTrackerData.memoryPoolMaxUsage[memoryPoolIndex];
        NN_LOG("%20s:%d\n", g_memoryPoolName[memoryPoolIndex], maxRequiredSize);
    }
}

#endif

} } } // namespace nnt { namespace gfx { namespace util {

