﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>

namespace nnt { namespace gfx { namespace util {

class ResourceAllocator;

class GpuBenchmarkPropertyDefinitionEnum
{
private:
    const char*         m_Name;
    const char**        m_pValueNameArray;
    int                 m_ValueNameArraySize;

public:
                        GpuBenchmarkPropertyDefinitionEnum(const char* proertyName, const char** pValueNameArray, int valueNameArraySize);
                        ~GpuBenchmarkPropertyDefinitionEnum();

    const char*         GetName() const;
    int                 GetCount() const;
    const char*         GetElementNameAt(int index) const;
    int                 FindElementIndexFromName(const char* lookupName) const;
};

class GpuBenchmarkPropertyDefinitionIntegerRange
{
private:
    const char*         m_Name;
    int                 m_Min;
    int                 m_Max;
    int                 m_Step;

public:
                        GpuBenchmarkPropertyDefinitionIntegerRange(const char* proertyName, int min, int max, int step);
                        ~GpuBenchmarkPropertyDefinitionIntegerRange();

    const char*         GetName() const;
    int                 GetMin() const;
    int                 GetMax() const;
    int                 GetStep() const;

    bool                IsValidValue(int value) const;
};


enum PropertyType
{
    PropertyType_Invalid        = 0,
    PropertyType_Enumeration,
    PropertyType_IntegerRange,

    PropertyType_Max,
};


const char* GetPropertyTypeName(PropertyType type);


template<typename HolderType, int (HolderType::*pGetterFunc)() const>
int GetterWrapper(void* pContext)
{
    HolderType* pHolder = reinterpret_cast<HolderType*>(pContext);
    return (pHolder->*pGetterFunc)();
}

template<typename HolderType, void (HolderType::*pSetterFunc)(int)>
void SetterWrapper(void* pContext, int value)
{
    HolderType* pHolder = reinterpret_cast<HolderType*>(pContext);
    NN_ASSERT(!pHolder->GetGfxObjectsInitialized());
    (pHolder->*pSetterFunc)(value);
}

template<typename HolderType, typename IndexType, int (HolderType::*pGetterFunc)(IndexType) const, IndexType index>
int IndexedGetterWrapper(void* pContext)
{
    HolderType* pHolder = reinterpret_cast<HolderType*>(pContext);
    return (pHolder->*pGetterFunc)(index);
}

template<typename HolderType, typename IndexType, int (HolderType::*pSetterFunc)(IndexType, int) const, IndexType index>
void IndexedSetterWrapper(void* pContext, int value)
{
    HolderType* pHolder = reinterpret_cast<HolderType*>(pContext);
    NN_ASSERT(!pHolder->GetGfxObjectsInitialized());
    (pHolder->*pSetterFunc)(index, value);
}

template<typename HolderType, typename EnumType>
struct SetterGetterEnumContext
{
    HolderType* pHolder;
    const EnumType* pValueArray;
    int valueArraySize;
    ResourceAllocator* pResourceAllocator;
};

void* AllocateSetterGetterEnumContext(ResourceAllocator* pAllocator, size_t size, size_t alignment);
void FreeSetterGetterEnumContext(ResourceAllocator* pAllocator, void* pMemory);

template<typename HolderType, typename EnumType>
SetterGetterEnumContext<HolderType, EnumType>* InitializeSetterGetterEnumContext(
    HolderType* pHolder, ResourceAllocator* pResourceAllocator,
    const EnumType* pValueArray, int valueArraySize)
{
    typedef SetterGetterEnumContext<HolderType, EnumType> EnumContextType;

    void* pMemory = AllocateSetterGetterEnumContext(pResourceAllocator, sizeof(EnumContextType), NN_ALIGNOF(EnumContextType));
    EnumContextType *pResult = new (pMemory) EnumContextType();
    pResult->pHolder = pHolder;
    pResult->pValueArray = pValueArray;
    pResult->valueArraySize = valueArraySize;
    pResult->pResourceAllocator = pResourceAllocator;

#if defined(NN_SDK_BUILD_DEBUG)
    for (int i = 0; i < valueArraySize; ++i)
    {
        for (int j = i + 1; j < valueArraySize; ++j)
        {
            NN_ASSERT(pValueArray[i] != pValueArray[j]);
        }
    }
#endif

    return pResult;
}

template<typename HolderType, typename EnumType>
void FinalizeSetterGetterEnumContext(SetterGetterEnumContext<HolderType, EnumType>* pSetterGetterEnumContext)
{
    typedef SetterGetterEnumContext<HolderType, EnumType> EnumContextType;
    ResourceAllocator* pResourceAllocator = pSetterGetterEnumContext->pResourceAllocator;

    pSetterGetterEnumContext->~EnumContextType();
    FreeSetterGetterEnumContext(pResourceAllocator, pSetterGetterEnumContext);
}

template<typename HolderType, typename EnumType, EnumType (HolderType::*pGetterFunc)() const>
int GetterEnumWrapper(void* pContext)
{
    SetterGetterEnumContext<HolderType, EnumType>* pGetterEnumContext = static_cast<SetterGetterEnumContext<HolderType, EnumType>*>(pContext);

    HolderType* pHolder = pGetterEnumContext->pHolder;
    EnumType enumValue = (pHolder->*pGetterFunc)();
    for (int i = 0; i < pGetterEnumContext->valueArraySize; ++i)
    {
        if (pGetterEnumContext->pValueArray[i] == enumValue)
            return static_cast<int>(i);
    }

    NN_ABORT();
    return -1;
}

template<typename HolderType, typename EnumType, void (HolderType::*pSetterFunc)(EnumType)>
void SetterEnumWrapper(void* pContext, int value)
{
    SetterGetterEnumContext<HolderType, EnumType>* pGetterEnumContext = reinterpret_cast<SetterGetterEnumContext<HolderType, EnumType>*>(pContext);
    NN_ASSERT_RANGE(value, 0, static_cast<int>(pGetterEnumContext->valueArraySize));
    HolderType* pHolder = pGetterEnumContext->pHolder;
    EnumType newEnumValue = pGetterEnumContext->pValueArray[value];
    NN_ASSERT(!pHolder->GetGfxObjectsInitialized());
    (pHolder->*pSetterFunc)(newEnumValue);
}

template<typename HolderType, typename EnumType>
void ShutdownEnumWrapper(void* pContext)
{
    SetterGetterEnumContext<HolderType, EnumType>* pGetterEnumContext = reinterpret_cast<SetterGetterEnumContext<HolderType, EnumType>*>(pContext);
    FinalizeSetterGetterEnumContext(pGetterEnumContext);
}

template<typename HolderType, typename EnumType, typename IndexType, EnumType (HolderType::*pGetterFunc)(IndexType) const, IndexType indexValue>
int IndexedGetterEnumWrapper(void* pContext)
{
    SetterGetterEnumContext<HolderType, EnumType>* pGetterEnumContext = reinterpret_cast<SetterGetterEnumContext<HolderType, EnumType>*>(pContext);

    HolderType* pHolder = pGetterEnumContext->pHolder;
    EnumType enumValue = (pHolder->*pGetterFunc)(indexValue);
    for (int i = 0; i < pGetterEnumContext->valueArraySize; ++i) {

        if (pGetterEnumContext->pValueArray[i] == enumValue)
            return static_cast<int>(i);
    }

    NN_ABORT();
    return -1;
}

template<typename HolderType, typename EnumType, typename IndexType, void (HolderType::*pSetterFunc)(IndexType, EnumType), IndexType indexValue>
void IndexedSetterEnumWrapper(void* pContext, int value)
{
    SetterGetterEnumContext<HolderType, EnumType>* pGetterEnumContext = reinterpret_cast<SetterGetterEnumContext<HolderType, EnumType>*>(pContext);
    NN_ASSERT_RANGE(value, 0, static_cast<int>(pGetterEnumContext->valueArraySize));
    HolderType* pHolder = pGetterEnumContext->pHolder;
    EnumType newEnumValue = pGetterEnumContext->pValueArray[value];
    NN_ASSERT(!pHolder->GetGfxObjectsInitialized());
    (pHolder->*pSetterFunc)(indexValue, newEnumValue);
}

class GpuBenchmarkPropertyHolder
{
public:
    typedef int (*GetterFunc)(void*);
    typedef void (*SetterFunc)(void*, int);
    typedef void(*ShutdownFunc)(void*);

private:
    void*               pContext;
    GetterFunc          pGetterFunc;
    SetterFunc          pSetterFunc;
    ShutdownFunc        pShutdownFunc;

    PropertyType        type;

    union
    {
        const GpuBenchmarkPropertyDefinitionEnum*           pAsEnum;
        const GpuBenchmarkPropertyDefinitionIntegerRange*   pAsIntegerRange;
        const void*                                         pAddress;
    } propertyDefinition;

    template<typename DefinitionType>
    void SetDefinitionReference(const DefinitionType* pDefinitionType);

public:
    GpuBenchmarkPropertyHolder();
    ~GpuBenchmarkPropertyHolder();

    template<typename DefinitionType>
    void Initialize(
        const DefinitionType* pDefinition, void* pContext,
        GetterFunc pGetter, SetterFunc pSetter, ShutdownFunc pShutdown);
    void Finalize();

    void Set(int value);
    int Get() const;

    PropertyType                                            GetType() const;
    const char*                                             GetName() const;

    void                                                    Increase();
    void                                                    Decrease();

    const GpuBenchmarkPropertyDefinitionEnum*               ToEnum() const;
    const GpuBenchmarkPropertyDefinitionIntegerRange*       ToIntegerRange() const;
};

template<>
inline void GpuBenchmarkPropertyHolder::SetDefinitionReference<GpuBenchmarkPropertyDefinitionEnum>(const GpuBenchmarkPropertyDefinitionEnum* pDefinitionType)
{
    type = PropertyType_Enumeration;
    propertyDefinition.pAsEnum = pDefinitionType;
}

template<>
inline void GpuBenchmarkPropertyHolder::SetDefinitionReference<GpuBenchmarkPropertyDefinitionIntegerRange>(const GpuBenchmarkPropertyDefinitionIntegerRange* pDefinitionType)
{
    type = PropertyType_IntegerRange;
    propertyDefinition.pAsIntegerRange = pDefinitionType;
}

template<typename DefinitionType>
inline void GpuBenchmarkPropertyHolder::SetDefinitionReference(const DefinitionType* pDefinitionType)
{
    NN_ABORT();
    type = PropertyType_Invalid;
    propertyDefinition.pAddress = nullptr;
}


template<typename DefinitionType>
void GpuBenchmarkPropertyHolder::Initialize(
    const DefinitionType* pDefinition, void* pContext,
    GetterFunc pGetter, SetterFunc pSetter, ShutdownFunc pShutdown)
{
    SetDefinitionReference(pDefinition);

    this->pContext          = pContext;
    this->pGetterFunc       = pGetter;
    this->pSetterFunc       = pSetter;
    this->pShutdownFunc     = pShutdown;
}

inline void GpuBenchmarkPropertyHolder::Finalize()
{
    if (pShutdownFunc != nullptr)
        pShutdownFunc(pContext);

    pContext = nullptr;
    pGetterFunc = nullptr;
    pSetterFunc = nullptr;
    pShutdownFunc = nullptr;

    type = PropertyType_Invalid;
    propertyDefinition.pAddress = nullptr;
}


inline void GpuBenchmarkPropertyHolder::Set(int value)
{
    pSetterFunc(pContext, value);
}

inline int GpuBenchmarkPropertyHolder::Get() const
{
    return pGetterFunc(pContext);
}

inline PropertyType GpuBenchmarkPropertyHolder::GetType() const
{
    return type;
}

inline const GpuBenchmarkPropertyDefinitionEnum* GpuBenchmarkPropertyHolder::ToEnum() const
{
    NN_ASSERT(type == PropertyType_Enumeration);
    return propertyDefinition.pAsEnum;
}

inline const GpuBenchmarkPropertyDefinitionIntegerRange* GpuBenchmarkPropertyHolder::ToIntegerRange() const
{
    NN_ASSERT(type == PropertyType_IntegerRange);
    return propertyDefinition.pAsIntegerRange;
}


} } } // namespace nnt { namespace gfx { namespace util {

