﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once


#include "gfxUtilGpuBenchmark_GpuBenchmark.h"

#include "gfxUtilGpuBenchmark_Property.h"
#include "gfxUtilGpuBenchmark_PropertyArray.h"
#include "gfxUtilGpuBenchmark_ResShader.h"

#include <nn/gfx/gfx_State.h>
#include <nn/gfx/gfx_Buffer.h>
#include <nn/gfx/gfx_DescriptorSlot.h>

//#define NN_GFXUTIL_GPUBENCHMARK_VERTEXFETCH_DEBUG

namespace nnt { namespace gfx { namespace util {

class GpuBenchmarkVertexFetch : public GpuBenchmark
{
public:
    static const char*  ClassName;

private:
    static const int                    m_RenderSize            = 256;
    static const int                    m_MaxAttributeCount     = 6;

    static const nn::gfx::ImageFormat   m_RenderFormat      = nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm;
    static const nn::gfx::TileMode      m_TileMode          = nn::gfx::TileMode_Optimal;

    int                                 m_AttributeCount;
    nn::gfx::AttributeFormat            m_AttributeFormat;
    int                                 m_VertexGridSize;

    ptrdiff_t                           m_VertexStride;


    nn::gfx::Texture                    m_RenderTexture;
    nn::gfx::ColorTargetView            m_RenderTextureTargetView;
    nn::gfx::ViewportScissorState       m_RenderTextureViewportScissorState;

    ResShader                           m_DrawQuadResShader;

    nn::gfx::Buffer                     m_VertexBuffer;
    nn::gfx::VertexState                m_VertexState;

    nn::gfx::Buffer                     m_ConstantBuffer;
    int                                 m_ConstantBufferDescriptorSlotIndex;
    nn::gfx::DescriptorSlot             m_ConstantBufferDescriptorSlot;

    nn::gfx::Buffer                     m_OutputCopyBuffer;
    size_t                              m_OutputCopyBufferSize;

#if defined(NN_GFXUTIL_GPUBENCHMARK_VERTEXFETCH_DEBUG)
    ResShader                           m_RenderQuadResShader;
    nn::gfx::TextureView                m_RenderTextureView;
    int                                 m_RenderTextureViewDescriptorSlotIndex;
    nn::gfx::DescriptorSlot             m_RenderTextureViewDescriptorSlot;
    nn::gfx::Sampler                    m_RenderTextureSampler;
    int                                 m_RenderTextureSamplerDescriptorSlotIndex;
    nn::gfx::DescriptorSlot             m_RenderTextureSamplerDescriptorSlot;
#endif


    enum Property
    {
        Property_AttributeCount,
        Property_AttributeFormat,
        Property_VertexGridSize,
        Property_Count,
    };

    PropertyArray<Property_Count>   m_PropertyArray;

public:
                                                GpuBenchmarkVertexFetch();
    virtual                                     ~GpuBenchmarkVertexFetch();

    virtual void                                Initialize(ResourceAllocator* pResourceAllocator) NN_OVERRIDE;
    virtual void                                Finalize(ResourceAllocator* pResourceAllocator) NN_OVERRIDE;

    virtual void                                InitializeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice) NN_OVERRIDE;
    virtual void                                FinalizeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice) NN_OVERRIDE;

    int                                         GetAttributeCount() const;
    void                                        SetAttributeCount(int value);

    nn::gfx::AttributeFormat                    GetAttributeFormat() const;
    void                                        SetAttributeFormat(nn::gfx::AttributeFormat format);

    int                                         GetVertexGridSize() const;
    void                                        SetVertexGridSize(int value);

    virtual void                                PreBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer) NN_OVERRIDE;
    virtual void                                DoBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer, int runCount) NN_OVERRIDE;
    virtual void                                RenderDebug(nn::gfx::CommandBuffer* pTestCommandBuffer) NN_OVERRIDE;

    virtual void                                PrintResults(nn::TimeSpan cpuTimeElapsed, nn::TimeSpan gpuTimeElapsed, int runCount, nn::gfx::util::DebugFontTextWriter* pDebugFontTextWriter) NN_OVERRIDE;

    virtual void                                CopyResultToBuffer(nn::gfx::CommandBuffer* pCommandBuffer) NN_OVERRIDE;
    virtual void                                MapResultBuffer(void** pOutBuffer, size_t* pOutBufferSize) NN_OVERRIDE;
    virtual void                                UnmapResultBuffer() NN_OVERRIDE;

    virtual const char*                         GetName() const NN_OVERRIDE;
    virtual BenchmarkType                       GetType() const NN_OVERRIDE;
    virtual int                                 GetPropertyCount() const NN_OVERRIDE;
    virtual int                                 FillPropertyList(const GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize) const NN_OVERRIDE;
    virtual int                                 FillPropertyList(GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize) NN_OVERRIDE;
    virtual GpuBenchmarkPropertyHolder*         FindPropertyByName(const char* propertyName) NN_OVERRIDE;
    virtual GpuBenchmarkPropertyHolder*         GetPropertyByIndex(int index) NN_OVERRIDE;

private:
    int                                         GetVertexCount() const;

};

inline int GpuBenchmarkVertexFetch::GetAttributeCount() const
{
    return m_AttributeCount;
}

inline void GpuBenchmarkVertexFetch::SetAttributeCount(int count)
{
    NN_ASSERT(count > 0);
    NN_ASSERT(count <= m_MaxAttributeCount);
    m_AttributeCount = count;
}

inline nn::gfx::AttributeFormat GpuBenchmarkVertexFetch::GetAttributeFormat() const
{
    return m_AttributeFormat;
}

inline void GpuBenchmarkVertexFetch::SetAttributeFormat(nn::gfx::AttributeFormat format)
{
    m_AttributeFormat = format;
}

inline int GpuBenchmarkVertexFetch::GetVertexGridSize() const
{
    return m_VertexGridSize;
}

inline void GpuBenchmarkVertexFetch::SetVertexGridSize(int value)
{
    m_VertexGridSize = value;
}


inline int GpuBenchmarkVertexFetch::GetVertexCount() const
{
    return (GetVertexGridSize() + 1) * (GetVertexGridSize() + 1) * 6;
}

} } } // namespace nnt { namespace gfx { namespace util {
