﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once


#include "gfxUtilGpuBenchmark_GpuBenchmark.h"

#include "gfxUtilGpuBenchmark_Property.h"
#include "gfxUtilGpuBenchmark_PropertyArray.h"
#include "gfxUtilGpuBenchmark_ResInputTexture.h"
#include "gfxUtilGpuBenchmark_ResShader.h"

//#define NN_GFXUTIL_GPUBENCHMARK_TEXTURE_FETCH_DEBUG

namespace nnt { namespace gfx { namespace util {

class GpuBenchmarkTextureFetch : public GpuBenchmark
{

public:
    enum TextureSize
    {
        TextureSize_256 = 256,
        TextureSize_512 = 512,
        TextureSize_1024 = 1024,
    };

    static const char* ClassName;

private:
    static const int                        m_RenderSize        = 4096;
    static const nn::gfx::ImageFormat       m_RenderFormat      = nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm;
    static const nn::gfx::TileMode          m_RenderTileMode    = nn::gfx::TileMode_Optimal;
    static const int                        m_MaxTextureCount   = 6;

    int                             m_TextureCount;
    nn::gfx::FilterMode             m_TextureFilterMode;
    nn::gfx::ImageFormat            m_TextureFormat;
    TextureSize                     m_TextureSize;


    nn::gfx::Texture                m_RenderTexture;
    nn::gfx::ColorTargetView        m_RenderTextureTargetView;
    nn::gfx::ViewportScissorState   m_RenderTextureViewportScissorState;

    nn::gfx::Sampler                m_TextureSampler;
    int                             m_TextureSamplerSlotIndex;
    nn::gfx::DescriptorSlot         m_TextureSamplerDescriptorSlot;

    struct TextureData
    {
        nn::gfx::Texture            texture;
        nn::gfx::TextureView        textureView;
        int                         slotIndex;
        nn::gfx::DescriptorSlot     descriptorSlot;
    };

    TextureData                     m_TextureDataArray[m_MaxTextureCount];
    ResShader                       m_ResShader;

    nn::gfx::Buffer                 m_VertexBuffer;

    nn::gfx::Buffer                     m_OutputCopyBuffer;
    size_t                              m_OutputCopyBufferSize;

#if defined(NN_GFXUTIL_GPUBENCHMARK_TEXTURE_FETCH_DEBUG)
    ResShader                           m_RenderQuadResShader;
    nn::gfx::TextureView                m_RenderTextureView;
    int                                 m_RenderTextureViewDescriptorSlotIndex;
    nn::gfx::DescriptorSlot             m_RenderTextureViewDescriptorSlot;
    nn::gfx::Sampler                    m_RenderTextureSampler;
    int                                 m_RenderTextureSamplerDescriptorSlotIndex;
    nn::gfx::DescriptorSlot             m_RenderTextureSamplerDescriptorSlot;
#endif


    enum Property
    {
        Property_TextureFilterMode,
        Property_TextureCount,
        Property_TextureFormat,
        Property_TextureSize,
        Property_Count,
    };


    PropertyArray<Property_Count>   m_PropertyArray;


public:
                                                GpuBenchmarkTextureFetch();
    virtual                                     ~GpuBenchmarkTextureFetch();

    virtual void                                Initialize(ResourceAllocator* pResourceAllocator) NN_OVERRIDE;
    virtual void                                Finalize(ResourceAllocator* pResourceAllocator) NN_OVERRIDE;

    virtual void                                InitializeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice) NN_OVERRIDE;
    virtual void                                FinalizeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice) NN_OVERRIDE;

    int                                         GetTextureCount() const;
    void                                        SetTextureCount(int count);

    nn::gfx::ImageFormat                        GetTextureFormat() const;
    void                                        SetTextureFormat(nn::gfx::ImageFormat format);

    TextureSize                                 GetTextureSize() const;
    void                                        SetTextureSize(TextureSize size);

    nn::gfx::FilterMode                         GetTextureFilterMode() const;
    void                                        SetTextureFilterMode(nn::gfx::FilterMode mode);

    virtual void                                PreBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer) NN_OVERRIDE;
    virtual void                                DoBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer, int runCount) NN_OVERRIDE;
    virtual void                                RenderDebug(nn::gfx::CommandBuffer* pTestCommandBuffer) NN_OVERRIDE;

    virtual void                                PrintResults(nn::TimeSpan cpuTimeElapsed, nn::TimeSpan gpuTimeElapsed, int runCount, nn::gfx::util::DebugFontTextWriter* pDebugFontTextWriter) NN_OVERRIDE;

    virtual void                                CopyResultToBuffer(nn::gfx::CommandBuffer* pCommandBuffer) NN_OVERRIDE;
    virtual void                                MapResultBuffer(void** pOutBuffer, size_t* pOutBufferSize) NN_OVERRIDE;
    virtual void                                UnmapResultBuffer() NN_OVERRIDE;

    virtual const char*                         GetName() const NN_OVERRIDE;
    virtual BenchmarkType                       GetType() const NN_OVERRIDE;
    virtual int                                 GetPropertyCount() const NN_OVERRIDE;
    virtual int                                 FillPropertyList(const GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize) const NN_OVERRIDE;
    virtual int                                 FillPropertyList(GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize) NN_OVERRIDE;
    virtual GpuBenchmarkPropertyHolder*         FindPropertyByName(const char* propertyName) NN_OVERRIDE;
    virtual GpuBenchmarkPropertyHolder*         GetPropertyByIndex(int index) NN_OVERRIDE;

private:
    static int                                  GetDimensionInPixelFromSize(TextureSize size);
    static const char*                          GetFormatString(nn::gfx::ImageFormat format);
    static void                                 GetInputTextureName(
                                                    char* pBuffer, int bufferSize,
                                                    TextureSize size, nn::gfx::ImageFormat format);

};

inline int GpuBenchmarkTextureFetch::GetTextureCount() const
{
    return m_TextureCount;
}

inline void GpuBenchmarkTextureFetch::SetTextureCount(int count)
{
    NN_ASSERT(count > 0);
    NN_ASSERT(count <= m_MaxTextureCount);
    m_TextureCount = count;
}

inline nn::gfx::ImageFormat GpuBenchmarkTextureFetch::GetTextureFormat() const
{
    return m_TextureFormat;
}

inline void GpuBenchmarkTextureFetch::SetTextureFormat(nn::gfx::ImageFormat format)
{
    m_TextureFormat = format;
}

inline GpuBenchmarkTextureFetch::TextureSize GpuBenchmarkTextureFetch::GetTextureSize() const
{
    return m_TextureSize;
}

inline void GpuBenchmarkTextureFetch::SetTextureSize(TextureSize size)
{
    m_TextureSize = size;
}

inline nn::gfx::FilterMode GpuBenchmarkTextureFetch::GetTextureFilterMode() const
{
    return m_TextureFilterMode;
}
inline void GpuBenchmarkTextureFetch::SetTextureFilterMode(nn::gfx::FilterMode mode)
{
    m_TextureFilterMode = mode;
}

inline int GpuBenchmarkTextureFetch::GetDimensionInPixelFromSize(TextureSize size)
{
    return static_cast<int>(size);
}

} } } // namespace nnt { namespace gfx { namespace util {
