﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once


#include "gfxUtilGpuBenchmark_GpuBenchmark.h"

#include "gfxUtilGpuBenchmark_Property.h"
#include "gfxUtilGpuBenchmark_PropertyArray.h"
#include "gfxUtilGpuBenchmark_ResShader.h"

#include <nn/gfx/gfx_Texture.h>
#include <nn/gfx/gfx_State.h>
#include <nn/gfx/gfx_DescriptorSlot.h>

//#define NN_GFXUTIL_GPUBENCHMARK_SHADER_DEBUG

namespace nnt { namespace gfx { namespace util {

class GpuBenchmarkShader : public GpuBenchmark
{
public:
    static const char*  ClassName;

    enum ShaderTransformCount
    {
        ShaderTransformCount_8 = 0,
        ShaderTransformCount_16 = 1,
        ShaderTransformCount_32 = 2,
        ShaderTransformCount_64 = 3,
        ShaderTransformCount_Max
    };

private:
    static const int                    m_RenderSizeMin             = 32;
    static const int                    m_RenderSizeMax             = 1024;
    static const int                    m_RenderSizeIncreaseStep    = (m_RenderSizeMax - m_RenderSizeMin) / 4;
    static const nn::gfx::ImageFormat   m_RenderFormat              = nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm;
    static const nn::gfx::TileMode      m_TileMode                  = nn::gfx::TileMode_Optimal;

    const int                           m_ConstantBufferSlotIndex   = 0;
    const int                           m_ConstantBufferMatrixCount = 64;
    const size_t                        m_ConstantBufferSize        = sizeof(float) * 16 * m_ConstantBufferMatrixCount;

    int                                 m_RenderSize;
    int                                 m_GridSize;
    ShaderTransformCount                m_VertexTransformCount;
    ShaderTransformCount                m_PixelTransformCount;


    nn::gfx::Texture                    m_RenderTexture;
    nn::gfx::ColorTargetView            m_RenderTextureColorTargetView;
    nn::gfx::ViewportScissorState       m_ViewportScissorState;

    ResShader                           m_ResShader;
    nn::gfx::Buffer                     m_VertexBuffer;

    nn::gfx::Buffer                     m_ConstantBuffer;
    int                                 m_ConstantBufferDescriptorSlotIndex;
    nn::gfx::DescriptorSlot             m_ConstantBufferDescriptorSlot;

    nn::gfx::Buffer                     m_OutputCopyBuffer;
    size_t                              m_OutputCopyBufferSize;

#if defined(NN_GFXUTIL_GPUBENCHMARK_SHADER_DEBUG)
    ResShader                           m_RenderQuadResShader;
    nn::gfx::TextureView                m_RenderTextureView;
    int                                 m_RenderTextureViewDescriptorSlotIndex;
    nn::gfx::DescriptorSlot             m_RenderTextureViewDescriptorSlot;
    nn::gfx::Sampler                    m_RenderTextureSampler;
    int                                 m_RenderTextureSamplerDescriptorSlotIndex;
    nn::gfx::DescriptorSlot             m_RenderTextureSamplerDescriptorSlot;
#endif


    enum Property
    {
        Property_GridSize,
        Property_VertexTransformCount,
        Property_RenderSize,
        Property_PixelTransformCount,
        Property_Count,
    };

    PropertyArray<Property_Count>       m_PropertyArray;

public:
                                                GpuBenchmarkShader();
    virtual                                     ~GpuBenchmarkShader();

    virtual void                                Initialize(ResourceAllocator* pResourceAllocator) NN_OVERRIDE;
    virtual void                                Finalize(ResourceAllocator* pResourceAllocator) NN_OVERRIDE;

    virtual void                                InitializeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice) NN_OVERRIDE;
    virtual void                                FinalizeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice) NN_OVERRIDE;

    int                                         GetGridSize() const;
    void                                        SetGridSize(int value);

    ShaderTransformCount                        GetVertexTransformCount() const;
    void                                        SetVertexTransformCount(ShaderTransformCount value);

    int                                         GetRenderSize() const;
    void                                        SetRenderSize(int value);

    ShaderTransformCount                        GetPixelTransformCount() const;
    void                                        SetPixelTransformCount(ShaderTransformCount value);

    virtual void                                PreBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer) NN_OVERRIDE;
    virtual void                                DoBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer, int runCount) NN_OVERRIDE;
    virtual void                                RenderDebug(nn::gfx::CommandBuffer* pTestCommandBuffer) NN_OVERRIDE;

    virtual void                                PrintResults(nn::TimeSpan cpuTimeElapsed, nn::TimeSpan gpuTimeElapsed, int runCount, nn::gfx::util::DebugFontTextWriter* pDebugFontTextWriter) NN_OVERRIDE;

    virtual void                                CopyResultToBuffer(nn::gfx::CommandBuffer* pCommandBuffer) NN_OVERRIDE;
    virtual void                                MapResultBuffer(void** pOutBuffer, size_t* pOutBufferSize) NN_OVERRIDE;
    virtual void                                UnmapResultBuffer() NN_OVERRIDE;

    virtual const char*                         GetName() const NN_OVERRIDE;
    virtual BenchmarkType                       GetType() const NN_OVERRIDE;
    virtual int                                 GetPropertyCount() const NN_OVERRIDE;
    virtual int                                 FillPropertyList(const GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize) const NN_OVERRIDE;
    virtual int                                 FillPropertyList(GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize) NN_OVERRIDE;
    virtual GpuBenchmarkPropertyHolder*         FindPropertyByName(const char* propertyName) NN_OVERRIDE;
    virtual GpuBenchmarkPropertyHolder*         GetPropertyByIndex(int index) NN_OVERRIDE;

};

inline int GpuBenchmarkShader::GetGridSize() const
{
    return m_GridSize;
}

inline void GpuBenchmarkShader::SetGridSize(int value)
{
    m_GridSize = value;
}

inline GpuBenchmarkShader::ShaderTransformCount GpuBenchmarkShader::GetVertexTransformCount() const
{
    return m_VertexTransformCount;
}
inline void GpuBenchmarkShader::SetVertexTransformCount(ShaderTransformCount value)
{
    m_VertexTransformCount = value;
}

inline int GpuBenchmarkShader::GetRenderSize() const
{
    return m_RenderSize;
}

inline void GpuBenchmarkShader::SetRenderSize(int value)
{
    m_RenderSize = value;
}

inline GpuBenchmarkShader::ShaderTransformCount GpuBenchmarkShader::GetPixelTransformCount() const
{
    return m_PixelTransformCount;
}

inline void GpuBenchmarkShader::SetPixelTransformCount(ShaderTransformCount value)
{
    m_PixelTransformCount = value;
}


} } } // namespace nnt { namespace gfx { namespace util {
