﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/////////////////////////////////////////////////////////////////////////////
//
// Draws a color-interpolated triangle.
// Changes the background color to show its liveliness.
//
//////////////////////////////////////////////////////////////////////////////

#include <cstdio>
#include <cstring>

#include <gfx/demo.h>
#include <nn/nn_Log.h>
#include <nnt.h>
#include <nnt/nnt_Argument.h>

////////////////////////////////////////////////////
//
// Assets data, types and interface for demos
//
////////////////////////////////////////////////////

// A .gsh file is a GX2 specific file that contains vertex and pixel
// shader data. In this case the vertex shader just passes through
// the position and color without modification. and the pixel shader
// outputs the interpolated vertex color.
static int g_ShaderFileIdx = 0;
static const char * const SIMPLE_SHADER_FILE[] =
{
    "shaders/triangleGS",
    "shaders/triangleGSHlslcc",
};

// Position for a single triangle.
static DEMO_F32x3 TRIANGLE_POSITION_DATA[] =
{
    { { { -0.75f, 0.75f, -0.5f } } },
    { { { 0.75f, 0.75f, -0.5f } } },
    { { { 0.00f, -0.75f, -0.5f } } }
};

// Number of vertex
static const u32 TRIANGLE_VERTEX_NUM = 3;

// Struct for gs ring buffers
typedef struct _RingBufferGS
{
    u32 inputRingSize;
    u32 outputRingSize;
    void* inBuffer;
    void* outBuffer;
} RingBufferGS;

#ifdef CAFE
// GS ring buffers for triangle
RingBufferGS g_TriangleRingData;
#endif

// Attrib buffers for triangle
static DEMOGfxBuffer g_TriangleAttribData;

// Animation Information
static DEMOGfxPipeline g_Pipeline;

static nn::gfx::ViewportScissorState g_Viewport;
static void* g_pViewportData;

////////////////////////////////////////////////////
//
// Prototypes
//
////////////////////////////////////////////////////

static void InitPipeline();
static void InitAttribData(DEMOGfxBuffer *pAttribData);
static void FreeAttribData(DEMOGfxBuffer *pAttribData);
static void DrawScene(void);

#ifdef CAFE
static void InitRingBufferGS( RingBufferGS *pRingData );
static void FreeRingBufferGS( RingBufferGS *pRingData );
#endif

////////////////////////////////////////////////////
//
// Functions
//
////////////////////////////////////////////////////

// The initialization function for the rendering portions of this sample.
// It is responsible for allocating the three types of shaders and buffers
// as well as ensuring that data is flushed from the CPU to GPU memory
// for Simple Shader
static void InitPipeline()
{
    DEMOGfxLoadShadersFromFile(&g_Pipeline.shaders, 0, SIMPLE_SHADER_FILE[g_ShaderFileIdx]);

    g_Pipeline.SetDefaults();

    DEMOGfxInitShaderAttribute(&g_Pipeline.shaders, "a_position", 0, 0, nn::gfx::AttributeFormat_32_32_32_Float );
    DEMOGfxInitShaderVertexBuffer( &g_Pipeline.shaders, 0, 3 * sizeof( float ), 0 );

    g_Pipeline.blendTargetStateCount = 1;
    g_Pipeline.colorTargetStateCount = 1;
    g_Pipeline.blendTargetStateInfoArray[ 0 ].SetDefault();
    g_Pipeline.colorTargetStateInfoArray[ 0 ].SetDefault();
    g_Pipeline.colorTargetStateInfoArray[ 0 ].SetFormat( DEMOColorBufferInfo.GetImageFormat() );
    g_Pipeline.Initialize( &DEMODevice );

    // Setup the viewport
    DEMOGfxSetViewportScissorState( &g_Viewport, &g_pViewportData,
        0.0f, 0.0f,
        static_cast< float >( DEMOColorBufferInfo.GetWidth() ),
        static_cast< float >( DEMOColorBufferInfo.GetHeight() ),
        0.0f, 1.0f,
        static_cast< float >( DEMOColorBufferInfo.GetHeight() ),
        false );
}

#ifdef CAFE
static void InitRingBufferGS( RingBufferGS *pRingData )
{
    // Set up geometry shader
    pRingData->inputRingSize = GX2CalcGeometryShaderInputRingBufferSize( reinterpret_cast< GX2VertexShader* >( g_Pipeline.shaders.vertexShaderData )->ringItemsize );
    pRingData->outputRingSize = GX2CalcGeometryShaderOutputRingBufferSize( reinterpret_cast< GX2GeometryShader* >( g_Pipeline.shaders.geomShaderData )->ringItemsize );
    pRingData->inBuffer = DEMOGfxAllocMEM1( pRingData->inputRingSize, 256 );
    pRingData->outBuffer = DEMOGfxAllocMEM1( pRingData->outputRingSize, 256 );
}

static void FreeRingBufferGS( RingBufferGS *pRingData )
{
    // Free ring buffers for GS
    DEMOGfxFreeMEM1( pRingData->inBuffer );
    DEMOGfxFreeMEM1( pRingData->outBuffer );
}
#endif

// Initialize attribute buffer and data
static void InitAttribData(DEMOGfxBuffer *pAttribData)
{
    // Position Buffer
    pAttribData->Initialize( sizeof( TRIANGLE_POSITION_DATA ),
        TRIANGLE_POSITION_DATA, nn::gfx::GpuAccess_VertexBuffer, 0 );
}

// Free attribute buffer and data
static void FreeAttribData(DEMOGfxBuffer *pAttribData)
{
    pAttribData->Finalize();
}

// The draw function for the rendering portions of this sample.
static void DrawScene()
{
    static float s_grey = 0.0f;

    s_grey += 0.01f;
    if (s_grey > 1.0)
    {
        s_grey = 0.0f;
    }

    // When using the DEMO library, it is necessary to call
    // DEMOGfxBeforeRender and DEMOGfxDoneRender before and after drawing.
    DEMOGfxBeforeRender();

    nn::gfx::ColorTargetView* pCurrentScanBuffer = DEMOGetColorBufferView();

    DEMOCommandBuffer.ClearColor(pCurrentScanBuffer, s_grey, s_grey, s_grey, 1.0f, NULL);
    DEMOCommandBuffer.ClearDepthStencil(&DEMODepthBufferView, 1.0, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_GEOMETRY_SHADER );
#endif
    const nn::gfx::ColorTargetView* colorTargets[] = { pCurrentScanBuffer };

    DEMOCommandBuffer.SetRenderTargets( 1, colorTargets, &DEMODepthBufferView );
    DEMOCommandBuffer.SetViewportScissorState( &g_Viewport );
    DEMOCommandBuffer.SetPipeline( &g_Pipeline.pipeline );

#ifdef CAFE
    // Setup Geometry Shader Input Rings
    GX2SetGeometryShaderInputRingBuffer( g_TriangleRingData.inBuffer, g_TriangleRingData.inputRingSize );
    GX2SetGeometryShaderOutputRingBuffer( g_TriangleRingData.outBuffer, g_TriangleRingData.outputRingSize );
#endif

    // Set attribute buffer
    DEMOCommandBuffer.SetVertexBuffer( 0, g_TriangleAttribData.gpuAddress, 3 * sizeof(float),
        g_TriangleAttribData.size );

    // Draw Triangle.
    // This command will actually result in a draw command being issued to
    // the GPU.
    DEMOCommandBuffer.Draw( nn::gfx::PrimitiveTopology_TriangleList, TRIANGLE_VERTEX_NUM, 0, 1, 0 );

    // This function will handle presenting the rendered buffer to the screen
    // by swapping the color buffer, setting the new context state,
    // resetting the color buffer, and flushing the pipeline.
    DEMOGfxDoneRender();
}

////////////////////////////////////////////////////
//
// Main
//
////////////////////////////////////////////////////

//extern "C" void nnMain()
TEST(GfxTriangleGs, Run)
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    // Initialize the DEMO library, DEMO test system, and create the
    // primary display.
    DEMOInit();
    DEMOTestInit(argc, argv);

    // This function will create the appropriate size color and depth
    // buffers, setup the scan buffer, and initialize the viewport and
    // scissor rectangles to be the entire screen.
    DEMOGfxInit(argc, argv);

    DEMOFontInit();
    DEMOTestIsUseHlslccGlsl() ? g_ShaderFileIdx = 1 : g_ShaderFileIdx = 0;

    // Initialize attributes and attribute buffer
    InitAttribData(&g_TriangleAttribData);

    // Initialize the pipeline
    InitPipeline();

#ifdef CAFE
    InitRingBufferGS( &g_TriangleRingData );
#endif

    while (DEMOIsRunning())
    {
        // Draw Scene
        DrawScene();
    }

#ifdef CAFE
    FreeRingBufferGS( &g_TriangleRingData );
#endif

    // Free attrib buffers
    FreeAttribData( &g_TriangleAttribData );

    g_Viewport.Finalize( &DEMODevice );
    g_Pipeline.Finalize( &DEMODevice );

    // Shutdown Demo lib
    DEMOFontShutdown();
    DEMOTestShutdown();
    DEMOGfxShutdown();
    DEMOShutdown();

    SUCCEED();
}
