﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/////////////////////////////////////////////////////////////////////////////
//
// Draws supported textures with each LOD.
//
//////////////////////////////////////////////////////////////////////////////

#include <cstdio>
#include <cstring>

#include <gfx/demo.h>
#include <nn/gfx.h>

#if NN_GFX_IS_TARGET_D3D
#include <unordered_set>
#include <string>
#endif

#include <nnt.h>
#include <nnt/nnt_Argument.h>

// Grid Informations

static const int GRID_MAX_X = 14;
static const int GRID_MAX_Y = 9;

static const float GRID_W = 86.0f;
static const float GRID_H = 72.0f;

static const float GRID_OFFSET_X = 30.0f;
static const float GRID_OFFSET_Y = 50.0f;

////////////////////////////////////////////////////
//
// Assets data, types and interface for demos
//
////////////////////////////////////////////////////
//
// Texture Data
//
// Directories which has .gtx files
static const char *TextureDir[] =
{
    "textures/textureFormats",
};

// A .gsh file is a GX2 specific file that contains vertex and pixel
// shader data.
static int g_ShaderFileIdx = 0;
static const char * const SHADER_FILE[] =
{
    "shaders/textureFormats/texture2DLOD",
    "shaders/textureFormats/texture2DLODHlslcc",
};

// Max files in the directory
static const int MAX_GTX_FILES = 255;

// Max Mip level to test
static const int MAX_MIP_LEVEL = 10;

static DEMOGfxTexture s_texture[MAX_GTX_FILES];
static f32 s_MipLevel[ MAX_GTX_FILES ];
static nn::gfx::ViewportScissorState s_SmallViewport[ MAX_GTX_FILES ];
static void* s_SmallViewportMem[ MAX_GTX_FILES ];
static u32 s_FileCount = 0;

static nn::gfx::Sampler s_Sampler;
static nn::gfx::DescriptorSlot s_SamplerSlot;

static DEMOGfxBuffer vertexBuffer;

static DEMOGfxBuffer uniformBuffer[ MAX_GTX_FILES ];

static u32 samplerLoc;

static u32 ub_inpLoc;

static DEMOGfxPipeline s_Pipeline;
static nn::gfx::ViewportScissorState s_FullViewport;
static void* s_FullViewportMem;

//
// Shader Data
//

// Quad Data
static const float UNIT = 0.96f;
#define countof(buf) (sizeof(buf)/sizeof(buf[0]))

static const int BUFFER_IDX = 0;

static DEMO_F32x3F32x2 QUAD_VERTEX_DATA[] = {
    { { { { -UNIT * UNIT, UNIT, 0.0f } } },
    { { { 0.0f, 0.0f } } } },
    { { { { UNIT * UNIT, UNIT, 0.0f } } },
    { { { 1.0f, 0.0f } } } },
    { { { { -UNIT * UNIT, -UNIT, 0.0f } } },
    { { { 0.0f, 1.0f } } } },
    { { { { UNIT * UNIT, -UNIT, 0.0f } } },
    { { { 1.0f, 1.0f } } } },
};

static const int QUAD_VTX_SIZE = ( sizeof( QUAD_VERTEX_DATA ) );
static const int QUAD_VTX_STRIDE = ( sizeof( f32 ) * 3 + sizeof( f32 ) * 2 );
static const int QUAD_POS_OFFSET = 0;
static const int QUAD_TEXCOORD_OFFSET = 12;

// Struct for attribute buffers
typedef struct _AttribBuffer
{
    f32 *pVertexBuffer;
    u32 vertexSize;
    u32 vertexStride;
    u32 vertexCount;
} AttribBuffer;

// Attrib buffers for Cube
static AttribBuffer g_QuadAttribData;

static u32 g_Frame = 0;

////////////////////////////////////////////////////
//
// Prototypes
//
////////////////////////////////////////////////////
static void InitShader();
static void InitAttribData(AttribBuffer *pAttribData, u32 vertexDataSize, u32 vertexDataStride, DEMO_F32x3F32x2 *vertexData);
static void DrawTexturedQuads( AttribBuffer *pAttribData );
static char *FormatTypeToString( nn::gfx::ImageFormat eSurfaceFormat );
static char *FormatTypeToString_Short( nn::gfx::ImageFormat eSurfaceFormat );
static void PrintInfo( nn::gfx::TextureInfo* pTextureInfo, f32 value );
static void PrintInitInfo(char *fileName);

// The initialization function for the rendering portions of this sample.
// It is responsible for allocating the three types of shaders and buffers
// as well as ensuring that data is flushed from the CPU to GPU memory
// for Shader
static void InitShader()
{
    DEMOGfxLoadShadersFromFile( &s_Pipeline.shaders, 0, SHADER_FILE[g_ShaderFileIdx] );

    // Texture Sampler lookup
    samplerLoc = s_Pipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_texture" );
    DEMOGfxInitSampler( &s_Sampler, &s_SamplerSlot, nn::gfx::TextureAddressMode_Repeat, nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint, nn::gfx::ComparisonFunction_Always );

    // Uniform location lookup
    ub_inpLoc = s_Pipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_inp" );

    DEMOGfxInitShaderAttribute( &s_Pipeline.shaders, "a_position", BUFFER_IDX, QUAD_POS_OFFSET, nn::gfx::AttributeFormat_32_32_32_Float );
    DEMOGfxInitShaderAttribute( &s_Pipeline.shaders, "a_texcoord", BUFFER_IDX, QUAD_TEXCOORD_OFFSET, nn::gfx::AttributeFormat_32_32_Float );
    DEMOGfxInitShaderVertexBuffer( &s_Pipeline.shaders, BUFFER_IDX, QUAD_VTX_STRIDE, 0 );

    s_Pipeline.colorTargetStateCount = 1;
    s_Pipeline.colorTargetStateInfoArray[ 0 ].SetDefault();
    s_Pipeline.colorTargetStateInfoArray[ 0 ].SetFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );
    s_Pipeline.blendTargetStateCount = 1;
    s_Pipeline.blendTargetStateInfoArray[ 0 ].SetDefault();

    s_Pipeline.Initialize( &DEMODevice );
}

// Initialize attribute buffer and data
static void InitAttribData(AttribBuffer *pAttribData, u32 vertexDataSize, u32 vertexDataStride, DEMO_F32x3F32x2 *vertexData)
{
    // Set values for vertex
    pAttribData->vertexCount   = vertexDataSize / sizeof(vertexData[0]);
    pAttribData->vertexSize   = vertexDataSize;
    pAttribData->vertexStride = vertexDataStride;

    // Set attributes data
    pAttribData->pVertexBuffer = (f32*)vertexData;

    vertexBuffer.Initialize( QUAD_VTX_SIZE, QUAD_VERTEX_DATA, nn::gfx::GpuAccess_VertexBuffer, 0 );

    for ( u32 i = 0; i < MAX_GTX_FILES; i++ )
    {
        uniformBuffer[ i ].Initialize( sizeof( f32 ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
    }
}

#if NN_GFX_IS_TARGET_D3D
static std::unordered_set< int > s_availableTextureIndices;

static void SetTextureIndexAvailable( int idx )
{
    s_availableTextureIndices.emplace( idx );
}

static bool IsTextureAvailable( int idx )
{
    if ( s_availableTextureIndices.find( idx ) == s_availableTextureIndices.end() )
    {
        return false;
    }

    return true;
}

static bool IsTextureFileAvailable( const char* fileName )
{
    static std::unordered_set< std::string > s_unsupportedFileNameSet;

    if ( s_unsupportedFileNameSet.empty() )
    {
        const char* unsupportedFileNames[] =
        {
            "textures/textureFormats/BC1_SRGB-SRC-A8R8G8B8",
            "textures/textureFormats/BC1_UNORM-SRC-A16B16G16R16",
            "textures/textureFormats/BC1_UNORM-SRC-A16B16G16R16F",
            "textures/textureFormats/BC1_UNORM-SRC-A2B10G10R10",
            "textures/textureFormats/BC1_UNORM-SRC-A32B32G32R32F",
            "textures/textureFormats/BC1_UNORM-SRC-A8R8G8B8",
            "textures/textureFormats/BC1_UNORM-SRC-BC1",
            "textures/textureFormats/BC1_UNORM-SRC-BC2",
            "textures/textureFormats/BC1_UNORM-SRC-BC3",
            "textures/textureFormats/BC2_SRGB-SRC-A8R8G8B8",
            "textures/textureFormats/BC2_UNORM-SRC-A16B16G16R16",
            "textures/textureFormats/BC2_UNORM-SRC-A16B16G16R16F",
            "textures/textureFormats/BC2_UNORM-SRC-A2B10G10R10",
            "textures/textureFormats/BC2_UNORM-SRC-A32B32G32R32F",
            "textures/textureFormats/BC2_UNORM-SRC-A8R8G8B8",
            "textures/textureFormats/BC2_UNORM-SRC-BC1",
            "textures/textureFormats/BC2_UNORM-SRC-BC2",
            "textures/textureFormats/BC2_UNORM-SRC-BC3",
            "textures/textureFormats/BC3_SRGB-SRC-A8R8G8B8",
            "textures/textureFormats/BC3_UNORM-SRC-A16B16G16R16",
            "textures/textureFormats/BC3_UNORM-SRC-A16B16G16R16F",
            "textures/textureFormats/BC3_UNORM-SRC-A2B10G10R10",
            "textures/textureFormats/BC3_UNORM-SRC-A32B32G32R32F",
            "textures/textureFormats/BC3_UNORM-SRC-A8R8G8B8",
            "textures/textureFormats/BC3_UNORM-SRC-BC1",
            "textures/textureFormats/BC3_UNORM-SRC-BC2",
            "textures/textureFormats/BC3_UNORM-SRC-BC3",
            "textures/textureFormats/BC4_SNORM-SRC-A16B16G16R16",
            "textures/textureFormats/BC4_SNORM-SRC-A16B16G16R16F",
            "textures/textureFormats/BC4_SNORM-SRC-A2B10G10R10",
            "textures/textureFormats/BC4_SNORM-SRC-A32B32G32R32F",
            "textures/textureFormats/BC4_SNORM-SRC-A8R8G8B8",
            "textures/textureFormats/BC4_SNORM-SRC-BC1",
            "textures/textureFormats/BC4_SNORM-SRC-BC2",
            "textures/textureFormats/BC4_SNORM-SRC-BC3",
            "textures/textureFormats/BC4_UNORM-SRC-A16B16G16R16",
            "textures/textureFormats/BC4_UNORM-SRC-A16B16G16R16F",
            "textures/textureFormats/BC4_UNORM-SRC-A2B10G10R10",
            "textures/textureFormats/BC4_UNORM-SRC-A32B32G32R32F",
            "textures/textureFormats/BC4_UNORM-SRC-A8R8G8B8",
            "textures/textureFormats/BC4_UNORM-SRC-BC1",
            "textures/textureFormats/BC4_UNORM-SRC-BC2",
            "textures/textureFormats/BC4_UNORM-SRC-BC3",
            "textures/textureFormats/BC5_SNORM-SRC-A16B16G16R16",
            "textures/textureFormats/BC5_SNORM-SRC-A16B16G16R16F",
            "textures/textureFormats/BC5_SNORM-SRC-A2B10G10R10",
            "textures/textureFormats/BC5_SNORM-SRC-A32B32G32R32F",
            "textures/textureFormats/BC5_SNORM-SRC-A8R8G8B8",
            "textures/textureFormats/BC5_SNORM-SRC-BC1",
            "textures/textureFormats/BC5_SNORM-SRC-BC2",
            "textures/textureFormats/BC5_SNORM-SRC-BC3",
            "textures/textureFormats/BC5_UNORM-SRC-A16B16G16R16",
            "textures/textureFormats/BC5_UNORM-SRC-A16B16G16R16F",
            "textures/textureFormats/BC5_UNORM-SRC-A2B10G10R10",
            "textures/textureFormats/BC5_UNORM-SRC-A32B32G32R32F",
            "textures/textureFormats/BC5_UNORM-SRC-A8R8G8B8",
            "textures/textureFormats/BC5_UNORM-SRC-BC1",
            "textures/textureFormats/BC5_UNORM-SRC-BC2",
            "textures/textureFormats/BC5_UNORM-SRC-BC3",
            "textures/textureFormats/passthrough-A1R5G5B5",
            "textures/textureFormats/passthrough-BC1",
            "textures/textureFormats/passthrough-BC2",
            "textures/textureFormats/passthrough-BC3",
            "textures/textureFormats/passthrough-BC4",
            "textures/textureFormats/passthrough-BC5",
            "textures/textureFormats/passthrough-R5G6B5",
            "textures/textureFormats/passthrough-X1R5G5B5",
            "textures/textureFormats/passthrough-X4R4G4B4",
            "textures/textureFormats/R11_G11_B10_FLOAT-SRC-A32B32G32R32F"
        };

        for ( int idx = 0; idx < sizeof( unsupportedFileNames ) / sizeof( const char* ); ++idx )
        {
            s_unsupportedFileNameSet.emplace( std::string( unsupportedFileNames[ idx ] ) );
        }
    }

    if ( s_unsupportedFileNameSet.find( std::string( fileName ) ) == s_unsupportedFileNameSet.end() )
    {
        return true;
    }

    return false;
}
#endif

static void SetupGridViewports( )
{
    u32 i = 0;

    // Draw Grid Models with each format
    for ( int y = 0; y < GRID_MAX_Y; y++ )
    {
        for ( int x = 0; x < GRID_MAX_X; x++ )
        {
            f32 viewportX = ( f32 )x * GRID_W + GRID_OFFSET_X;
#if NN_GFX_IS_TARGET_D3D
            f32 viewportY = ( f32 )DEMOColorBufferInfo.GetHeight() - ( ( f32 )( y + 1 ) * GRID_H + GRID_OFFSET_Y );
#else
            f32 viewportY = ( f32 )y * GRID_H + GRID_OFFSET_Y;
#endif
            f32 viewportW = ( f32 )GRID_W;
            f32 viewportH = ( f32 )GRID_H;

            // Set Viewport
            DEMOGfxSetViewportScissorState( &s_SmallViewport[i], &s_SmallViewportMem[i], viewportX,
                viewportY, viewportW, viewportH, 0.0, 1.0, f32( DEMOColorBufferInfo.GetHeight() ), false );
            i++;
            if ( i >= s_FileCount )
            {
                break;
            }
        }
        if ( i >= s_FileCount )
        {
            break;
        }
    }
}

// The draw function for the rendering portions of this app
static void DrawTexturedQuads( AttribBuffer *pAttribData )
{
    u32 i = 0;
    DEMOGfxBeforeRender();

    // Clear buffers
    nn::gfx::ColorTargetView* colorTargets[ ] = { DEMOGetColorBufferView() };

    DEMOCommandBuffer.ClearColor( colorTargets[ 0 ], 0.25f, 0.3f, 0.45f, 1.0f, NULL );
    DEMOCommandBuffer.ClearDepthStencil( &DEMODepthBufferView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

    DEMOCommandBuffer.SetRenderTargets( 1, colorTargets, &DEMODepthBufferView );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif

    // Draw Grid Models with each format
    for (int y = 0; y < GRID_MAX_Y; y++)
    {
        for (int x = 0; x < GRID_MAX_X; x++)
        {
#if NN_GFX_IS_TARGET_D3D
            if ( !IsTextureAvailable( i ) )
            {
                i++;
                if (i >= s_FileCount)
                {
                    break;
                }
                continue;
            }
#endif
            DEMOCommandBuffer.SetPipeline( &s_Pipeline.pipeline );

            // Bind vertex buffer
            DEMOCommandBuffer.SetVertexBuffer( BUFFER_IDX, vertexBuffer.gpuAddress, QUAD_VTX_STRIDE, QUAD_VTX_SIZE );

            // Update Mip Level
            s_MipLevel[i] += 0.01f;
            f32 level = (
                s_MipLevel[ i ] <=
                s_texture[ i ].GetTextureInfo( 0 )->GetMipCount() ) ?
                s_MipLevel[ i ] :
                s_texture[ i ].GetTextureInfo( 0 )->GetMipCount() - 1;

            // Set Texture and sampler
            DEMOCommandBuffer.SetTextureAndSampler( samplerLoc, nn::gfx::ShaderStage_Pixel, s_texture[ i ].GetDescriptorSlot( 0 ), s_SamplerSlot );

            // Set Mip level (LOD)
            float *pLevel = uniformBuffer[ i ].Map< f32 >();
#if NN_GFX_IS_TARGET_GX
            GX2EndianSwap( pLevel, sizeof( *pLevel ) );
#endif

            *pLevel = level;

#if NN_GFX_IS_TARGET_GX
            GX2EndianSwap( pLevel, sizeof( *pLevel ) );
#endif
            uniformBuffer[ i ].Unmap();
            DEMOCommandBuffer.SetConstantBuffer( ub_inpLoc, nn::gfx::ShaderStage_Pixel, uniformBuffer[ i ].gpuAddress, uniformBuffer[ i ].size );

            // Set Viewport for each Model
            DEMOCommandBuffer.SetViewportScissorState( &s_SmallViewport[ i ] );

            // Draw

            DEMOCommandBuffer.Draw( nn::gfx::PrimitiveTopology_TriangleStrip,  pAttribData->vertexCount, 0 );

            // Print Info
            PrintInfo( s_texture[ i ].GetTextureInfo( 0 ), level );

            if ( s_MipLevel[ i ] >= MAX_MIP_LEVEL )
            {
                s_MipLevel[ i ] = 0;
            }

            i++;
            if (i >= s_FileCount)
            {
                break;
            }
        }
        if (i >= s_FileCount)
        {
            break;
        }
    }

    // Set Viewport to default
    DEMOCommandBuffer.SetViewportScissorState( &s_FullViewport );

    DEMOFontSetGridSize(120,36);
    DEMOFontPrintf(5, 1, "<Texture Formats : Frame = %d>", g_Frame);

    DEMOGfxDoneRender();

    g_Frame++;
}

// Print Init Informations
static void PrintInitInfo(char *fileName)
{
    DEMOGfxBeforeRender();

    DEMOCommandBuffer.ClearColor( DEMOGetColorBufferView(), 0.0f, 0.0f, 0.0f, 1.0f, NULL );
    DEMOCommandBuffer.ClearDepthStencil( &DEMODepthBufferView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

    DEMOGfxSetDefaultRenderTarget();
    DEMOGfxSetDefaultViewportScissor();

    DEMOFontPrintf(4, 11, "Loading : %s ...", fileName);
    OSReport("Loading : %s ...\n",fileName);

    DEMOGfxDoneRender();
}

// Print Informations
static void PrintInfo( nn::gfx::TextureInfo* pTextureInfo, f32 value )
{
    // Set Font Size
    DEMOFontSetGridSize(14,5);
    DEMOFontSetSpacing(1);
    DEMOFontPrintf(0.5f, 0, "%s", FormatTypeToString_Short(pTextureInfo->GetImageFormat()));

    DEMOFontSetGridSize(15,5);
    DEMOFontSetSpacing(0);
    DEMOFontPrintf( 0.5f, 1, "alm:%d", nn::gfx::Texture::CalculateMipDataAlignment( &DEMODevice, *pTextureInfo ) );
    DEMOFontPrintf(0.5f, 2, "wxh:%dx%d", pTextureInfo->GetWidth(), pTextureInfo->GetHeight());
    DEMOFontPrintf( 0.5f, 3, "mip:%d-%d:%d", 0, pTextureInfo->GetMipCount() - 1, ( u32 )value );
}

// Convert ImageFormat strings to text
static char * FormatTypeToString( nn::gfx::ImageFormat eSurfaceFormat )
{
    return (char *) DEMOGfxGetSurfaceFormatName(eSurfaceFormat);
}

// Convert ImageFormat strings short string (ImageFormat_R8_Unorm => R8_Unorm)
static char *FormatTypeToString_Short( nn::gfx::ImageFormat eSurfaceFormat )
{
    char *pFullName = FormatTypeToString(eSurfaceFormat);
    size_t StartChar = strlen("ImageFormat_");

    return pFullName + StartChar;
}

//extern "C" void nnMain()
TEST(GfxTextureFormats, Run)
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    DEMOInit();
    DEMOTestInit(argc, argv);

    // This function will create the appropriate size color and depth
    // buffers, setup the scan buffer, and initialize the viewport and
    // scissor rectangles to be the entire screen.
    DEMOGfxInit(argc, argv);

    // Initialize Demo Font
    DEMOFontInit();
    DEMOTestIsUseHlslccGlsl() ? g_ShaderFileIdx = 1 : g_ShaderFileIdx = 0;

    // After the DEMO library is initialize the sample's initialization
    // function can be called.

    // Initialize attributes and buffers
    InitAttribData(&g_QuadAttribData,
                   sizeof(QUAD_VERTEX_DATA), sizeof(DEMO_F32x3F32x2), QUAD_VERTEX_DATA);

    // Initialize shaders
    InitShader( );

    g_Frame = 0;
    memset((void *)s_MipLevel, 0, sizeof(f32) * MAX_GTX_FILES);

    OSReport( "Loading Texture Binary\n" );

    char* pFileNames[ MAX_GTX_FILES ];

    // Get Texture names in directory
    DEMOGfxScanAssetDir(TextureDir[0], &s_FileCount, MAX_GTX_FILES, pFileNames );

    // Get GX2Texture pointer from buffer
    for (size_t i = 0; i < s_FileCount; i++)
    {
        // Remove the ending from the string
        size_t len = strlen( pFileNames[ i ] );
        while ( len-- > 0 )
        {
            if ( pFileNames[ i ][ len ] == '.' )
            {
                pFileNames[ i ][ len ] = '\0';
                break;
            }
        }

        // Print Info
        PrintInitInfo(pFileNames[i]);
#if NN_GFX_IS_TARGET_D3D
        if ( !IsTextureFileAvailable( pFileNames[ i ] ) )
        {
            continue;
        }
        SetTextureIndexAvailable( static_cast< int >( i ) );
#endif
        s_texture[ i ].Initialize( pFileNames[ i ] );
    }

    DEMOGfxSetViewportScissorState( &s_FullViewport, &s_FullViewportMem, 0.0, 0.0,
        static_cast< f32 >( DEMOColorBufferInfo.GetWidth()),
        static_cast< f32 >( DEMOColorBufferInfo.GetHeight()), 0.0, 1.0,
        static_cast< f32 >( DEMOColorBufferInfo.GetHeight() ), false );

    OSReport("Done.\n");

    SetupGridViewports();

    while ( DEMOIsRunning() )
    {
        // Draw Textured Quads
        DrawTexturedQuads( &g_QuadAttribData );
    }

    // Free shaders
    s_Pipeline.Finalize( &DEMODevice );
    s_Sampler.Finalize(&DEMODevice);

    // Free texture buffers
    for (u32 i = 0; i < s_FileCount; i++)
    {
        s_texture[ i ].Finalize();
        s_SmallViewport[i].Finalize(&DEMODevice);
    }

    s_FullViewport.Finalize(&DEMODevice);
    DEMOGfxFreeMEM2( s_FullViewportMem );

    for (u32 i = 0; i < MAX_GTX_FILES; i++)
    {
        uniformBuffer[i].Finalize(&DEMODevice);
    }
    vertexBuffer.Finalize(&DEMODevice);

    DEMOFontShutdown();
    DEMOTestShutdown();
    DEMOGfxShutdown();
    DEMOShutdown();

    SUCCEED();
}
