﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/////////////////////////////////////////////////////////////////////////////
//
// Draws a rotating cube with matrix uniform blocks.
//
//////////////////////////////////////////////////////////////////////////////

#include <cstdio>
#include <cstring>

#include <gfx/demo.h>

#if NN_GFX_IS_TARGET_GL
#include <GL\glew.h>
#endif

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#endif

#include <nnt.h>
#include <nnt/nnt_Argument.h>

////////////////////////////////////////////////////
//
// Assets data, types and interface for demos
//
////////////////////////////////////////////////////

// A .gsh file is a GX2 specific file that contains vertex and pixel
// shader data.
static int g_ShaderFileIdx = 0;
static const char * const SHADER_FILE_NAME[] =
{
    "shaders/textureFetchGS",
    "shaders/textureFetchGSHlslcc",
};

static const char * const GTXList[] =
{
    "textures/textureFetchGs/mtrange_co_8_8_8_8_512x512",
    "textures/textureFetchGs/mtrange_ht_8_8_8_8_128x128",
};

// Struct for uniform blocks
// Need to match uniform blocks in shader code
typedef struct _UniformBlockDynamic
{
    Mtx44   modelMtx44; // 0
} UniformBlockDynamic;

typedef struct _UniformBlockStatic
{
    Mtx44   viewMtx44; // 0
    Mtx44   projMtx44; // 64
} UniformBlockStatic;

typedef struct _UniformBlockDynamicGS
{
    float   threadhold; // 0
} UniformBlockDynamicGS;

// Struct for uniforms
typedef struct _Uniforms
{
    // Uniform Block for dynamic updating data
    DEMOGfxBuffer dynamicVsUBCPU[2]; // UniformBlockDynamic
    DEMOGfxBuffer dynamicVsUBGPU[2]; // UniformBlockDynamic

    DEMOGfxBuffer dynamicGsUBCPU[2]; // UniformBlockDynamicGS
    DEMOGfxBuffer dynamicGsUBGPU[2]; // UniformBlockDynamicGS

    // Uniform Block for static data
    DEMOGfxBuffer staticVsUB; // UniformBlockStatic
} Uniforms;

// Struct for attribute buffers
typedef struct _AttribBuffer
{
    DEMOGfxBuffer positionBuffer;
    DEMOGfxBuffer textureBuffer;
    DEMOGfxBuffer indexBuffer;
} AttribBuffer;

// Struct for gs ring buffers
typedef struct _RingBufferGS
{
    void *pInBuffer;
    u32 inSize;
    void *pOutBuffer;
    u32 outSize;
} RingBufferGS;

// ----- Triangle Data
// Plane Data
static const int NUM_ATTRIBS = 2;
static const int NUM_GRID = 4;
static const float FIELD_SIZE = 100.0f;
static const int FIELD_INTERVAL = static_cast< int >( FIELD_SIZE * 2 / NUM_GRID );

static const int POSITION_STRIDE = sizeof( DEMO_F32x3 );
static const int POSITION_OFFSET = 0;
static const int TEXCOORD_STRIDE = sizeof( DEMO_F32x2 );
static const int TEXCOORD_OFFSET = 0;

// ----- Texture & sampler objects
static DEMOGfxTexture g_colorTex;
static DEMOGfxTexture g_heightMap;
static nn::gfx::Sampler g_MySampler;
static nn::gfx::DescriptorSlot g_MySamplerSlot;

static nn::gfx::ViewportScissorState g_viewportScissorState;
static void* g_pViewportScissorStateData;

static f32 g_SubdivideThreshold = 0.25f;

// ----- Global Frame Counter
static u32 g_FrameCount;

// Animation information

static f32 s_yrad;

static u8 g_CurrentFocus = 0;
static bool g_WireFrame = true;

////////////////////////////////////////////////////
//
// Prototypes
//
////////////////////////////////////////////////////
static void InitPipelineGS( DEMOGfxPipeline* pPipeline, AttribBuffer* pAttribData, Uniforms* pUniforms, const char * fileNames, RingBufferGS *pRingData );
static void FreePipelineGS( DEMOGfxPipeline* pPipeline, Uniforms* pUniforms, RingBufferGS* pRingData );
static void InitAttribData( AttribBuffer* pAttribData );
static void InitTexture( DEMOGfxShader* pShader );
static void InitUniforms( Uniforms* pUniforms );
static void UpdateUniforms( Uniforms* pUniforms );
static void UpdatePipeline( DEMOGfxPipeline* pPipeline );

////////////////////////////////////////////////////
//
// Functions
//
////////////////////////////////////////////////////

static void UpdatePipeline( DEMOGfxPipeline* pPipeline )
{
    DEMOQueue.Sync();
    pPipeline->pipeline.Finalize( &DEMODevice );
    if ( g_WireFrame )
    {
        pPipeline->rasterizerStateInfo.SetFillMode( nn::gfx::FillMode_Wireframe );
    }
    else
    {
        pPipeline->rasterizerStateInfo.SetFillMode( nn::gfx::FillMode_Solid );
    }
    pPipeline->Initialize( &DEMODevice );
}

// Setting Up Pad Information
static void SetPadInfo( DEMOGfxPipeline* pPipeline )
{
    u16 buttonDown = DEMOPadGetButtonDown(0);

    if (DEMO_PAD_BUTTON_A & buttonDown)
    {
        if(g_CurrentFocus==0){
            g_SubdivideThreshold += 0.5f;
            if(g_SubdivideThreshold >= 0.75f)
                g_SubdivideThreshold = 0.75f;
        }else{
            g_WireFrame = true;
            UpdatePipeline( pPipeline );
        }
    }

    if (DEMO_PAD_BUTTON_B & buttonDown)
    {
        if(g_CurrentFocus==0){
            g_SubdivideThreshold -= 0.5f;
            if(g_SubdivideThreshold <= 0.25f)
                g_SubdivideThreshold = 0.25f;
        }else{
            g_WireFrame = false;
            UpdatePipeline( pPipeline );
        }
    }

    if ( DEMO_PAD_BUTTON_DOWN & buttonDown )
    {
        g_CurrentFocus = 1;
    }

    if ( DEMO_PAD_BUTTON_UP & buttonDown )
    {
        g_CurrentFocus = 0;
    }
}

// Print Informations
static void PrintInfo()
{
    DEMOFontSetGridSize(50, 24);

    // Update Font Color
    if(g_CurrentFocus == 0)
        DEMOFontSetColor(1.0f, 0.0f, 0.0f, 1.0f);
    else
        DEMOFontSetColor(1.0f, 1.0f, 1.0f, 1.0f);
    DEMOFontPrintf(4.5f, 1.8f, "Subdivision Threshold %f", g_SubdivideThreshold);

    if(g_CurrentFocus == 0){
        DEMOFontSetColor(1.0f, 1.0f, 1.0f, 1.0f);
    }else{
        DEMOFontSetColor(1.0f, 0.0f, 0.0f, 1.0f);
    }

    if(g_WireFrame == true){
        DEMOFontPrintf(4.5f, 2.8f, "Rendering Mode : Wire Frame");
    }else{
        DEMOFontPrintf(4.5f, 2.8f, "Rendering Mode : Normal");
    }
}

// The initialization function for the rendering portions of this sample.
// It is responsible for allocating the three types of shaders and buffers
// as well as ensuring that data is flushed from the CPU to GPU memory
// for UB Shader
static void InitPipelineGS(DEMOGfxPipeline* pPipeline, AttribBuffer* pAttribData, Uniforms* pUniforms, const char* fileName, RingBufferGS* pRingData)
{
    DEMOGfxShader* pShader = &pPipeline->shaders;

    u8 i;

    // Reset the pipeline settings
    pPipeline->SetDefaults();

    DEMOGfxLoadShadersFromFile(pShader, 0, fileName);
    // Init attribute to shader
    DEMOGfxInitShaderAttribute(pShader,
                               "a_position",
                               0,
                               POSITION_OFFSET,
                               nn::gfx::AttributeFormat_32_32_32_Float);

    // Init attribute to shader
    DEMOGfxInitShaderAttribute(pShader,
                               "a_texCoord",
                               1,
                               TEXCOORD_OFFSET,
                               nn::gfx::AttributeFormat_32_32_Float);

    // Setup Position attribute buffer
    DEMOGfxInitShaderVertexBuffer( pShader, 0, POSITION_STRIDE, 0 );

    // Setup Texture attribute buffer
    DEMOGfxInitShaderVertexBuffer( pShader, 1, TEXCOORD_STRIDE, 0 );

    // Get vs uniform block location
    DEMOGfxGetVertexShaderUniformBlockLocation(pShader, "ub_staticBlock");
    DEMOGfxGetVertexShaderUniformBlockLocation(pShader, "ub_dynamicBlock");
    DEMOGfxGetGeometryShaderUniformBlockLocation(pShader, "ub_thresholdBlock");

    // Alloc uniform block buffer
    // Need to align for UB
    pUniforms->staticVsUB.Initialize( sizeof( UniformBlockStatic ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );

    for(i=0; i<2; i++){
        pUniforms->dynamicVsUBCPU[ i ].Initialize( sizeof( UniformBlockDynamic ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
        pUniforms->dynamicVsUBGPU[ i ].Initialize( sizeof( UniformBlockDynamic ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
        pUniforms->dynamicGsUBCPU[ i ].Initialize( sizeof( UniformBlockDynamicGS ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
        pUniforms->dynamicGsUBGPU[ i ].Initialize( sizeof( UniformBlockDynamicGS ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
    }

    // Setup the pipeline
    if ( g_WireFrame )
    {
        pPipeline->rasterizerStateInfo.SetFillMode( nn::gfx::FillMode_Wireframe );
    }
    else
    {
        pPipeline->rasterizerStateInfo.SetFillMode( nn::gfx::FillMode_Solid );
    }
    pPipeline->shaders.bufferCount = NUM_ATTRIBS;

    pPipeline->blendTargetStateCount = 1;
    pPipeline->blendTargetStateInfoArray[ 0 ].SetDefault();

    pPipeline->colorTargetStateCount = 1;
    pPipeline->colorTargetStateInfoArray[ 0 ].SetDefault();
    pPipeline->colorTargetStateInfoArray[ 0 ].SetFormat( DEMOColorBufferInfo.GetImageFormat() );

    pPipeline->Initialize( &DEMODevice );

#if NN_GFX_IS_TARGET_GX
    // Set up Ring Buffer for Geometry Shader's In Out
    pRingData->inSize = GX2CalcGeometryShaderInputRingBufferSize(
        reinterpret_cast< GX2VertexShader* >( pShader->vertexShaderData )->ringItemsize);
    pRingData->pInBuffer = DEMOGfxAllocMEM2(pRingData->inSize, GX2_SHADER_ALIGNMENT);

    pRingData->outSize = GX2CalcGeometryShaderOutputRingBufferSize(
        reinterpret_cast< GX2GeometryShader* >( pShader->geomShaderData )->ringItemsize);
    pRingData->pOutBuffer = DEMOGfxAllocMEM2(pRingData->outSize, GX2_SHADER_ALIGNMENT);
#else
    NN_UNUSED( pRingData );
    NN_UNUSED( pAttribData );
#endif

    // Initialize camera position
    InitUniforms(pUniforms);

    // Setup the viewport
    DEMOGfxSetViewportScissorState( &g_viewportScissorState, &g_pViewportScissorStateData, 0.0f, 0.0f,
        ( float )DEMOColorBufferInfo.GetWidth(), ( float )DEMOColorBufferInfo.GetHeight(), 0.0f, 1.0f,
        ( float )DEMOColorBufferInfo.GetHeight(), false );
}

// Free shader
static void FreePipelineGS(DEMOGfxPipeline* pPipeline, Uniforms* pUniforms, RingBufferGS* pRingData)
{
    pPipeline->Finalize( &DEMODevice );

    for ( int i = 0; i < 2; i++ )
    {
        pUniforms->dynamicVsUBCPU[ i ].Finalize();
        pUniforms->dynamicVsUBGPU[ i ].Finalize();
        pUniforms->dynamicGsUBCPU[ i ].Finalize();
        pUniforms->dynamicGsUBGPU[ i ].Finalize();
    }

    pUniforms->staticVsUB.Finalize();

#if NN_GFX_IS_TARGET_GX
    // Free ring buffers for GS
    DEMOGfxFreeMEM2(pRingData->pInBuffer);
    DEMOGfxFreeMEM2(pRingData->pOutBuffer);
#else
    NN_UNUSED( pRingData );
#endif
}

// Initialize attribute buffer and data
static void InitAttribData(AttribBuffer* pAttribData)
{
    u32 idx;
    u32 positionDataSize = ( NUM_GRID + 1 ) * ( NUM_GRID + 1 ) * sizeof( DEMO_F32x3 );
    u32 texCoordDataSize = ( NUM_GRID + 1 ) * ( NUM_GRID + 1 ) * sizeof( DEMO_F32x2 );
    u32 indexDataSize = 3 * 2 * NUM_GRID*NUM_GRID*sizeof( u32 );
    f32 *positionData = new f32[ positionDataSize / sizeof( f32 ) ];
    f32 *texCoordData = new f32[ texCoordDataSize / sizeof( f32 ) ];
    u32 *indexData = new u32[ indexDataSize / sizeof( u32 ) ];

    idx = 0;
    for ( int i = 0; i < NUM_GRID + 1; i++ )
    {
        for ( int j = 0; j < NUM_GRID + 1; j++ )
        {
            // position
            positionData[3 * idx]   = -FIELD_SIZE + j * FIELD_INTERVAL;
            positionData[3 * idx + 1] = -5.0f;
            positionData[3 * idx + 2] = -FIELD_SIZE + i * FIELD_INTERVAL;

            // texture coordinate
            texCoordData[2 * idx] = 1.0f / (f32)(NUM_GRID + 1) * j;
            texCoordData[2 * idx + 1] = 1.0f / (f32)(NUM_GRID + 1) * i;

            idx++;
        }
    }

    idx = 0;
    for ( int i = 0; i < NUM_GRID; i++ )
    {
        for ( int j = 0; j < NUM_GRID; j++ )
        {
            indexData[ 6 * idx ] = j + i * ( NUM_GRID + 1 );
            indexData[ 6 * idx + 1 ] = ( j + 1 ) + i * ( NUM_GRID + 1 );
            indexData[ 6 * idx + 2 ] = ( j + 1 ) + ( i + 1 ) * ( NUM_GRID + 1 );

            indexData[ 6 * idx + 3 ] = ( j + 1 ) + ( i + 1 ) * ( NUM_GRID + 1 );
            indexData[ 6 * idx + 4 ] = j + ( i + 1 ) * ( NUM_GRID + 1 );
            indexData[ 6 * idx + 5 ] = j + i * ( NUM_GRID + 1 );
            idx++;
        }
    }

    // Set values for position
    pAttribData->positionBuffer.Initialize( positionDataSize, positionData, nn::gfx::GpuAccess_VertexBuffer, 0 );

    // Set values for texcoord
    pAttribData->textureBuffer.Initialize( texCoordDataSize, texCoordData, nn::gfx::GpuAccess_VertexBuffer, 0 );

    // Set values for index
    pAttribData->indexBuffer.size  = indexDataSize;
    pAttribData->indexBuffer.Initialize( indexDataSize, indexData, nn::gfx::GpuAccess_IndexBuffer, 0 );

    // Set attributes data
    pAttribData->positionBuffer.data = reinterpret_cast< void* >( positionData );
    pAttribData->textureBuffer.data = reinterpret_cast< void* >( texCoordData );
    pAttribData->indexBuffer.data = reinterpret_cast< void* >( indexData );

    // Invalidate attribute buffers
    DEMOCommandBuffer.FlushMemory( nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_IndexBuffer );

    delete[] positionData;
    delete[] texCoordData;
    delete[] indexData;
}


// Init texture and sampler
static void InitTexture(DEMOGfxShader* pShader)
{
    g_colorTex.Initialize( GTXList[ 0 ] );
    g_heightMap.Initialize( GTXList[ 1 ] );

    DEMOGfxGetGeometryShaderSamplerLocation(pShader, "s_heightmap");
    DEMOGfxGetPixelShaderSamplerLocation(pShader, "s_texture");

    // Set up the sampler object
    DEMOGfxInitSampler( &g_MySampler, &g_MySamplerSlot, nn::gfx::TextureAddressMode_ClampToEdge,
        nn::gfx::FilterMode_MinLinear_MagLinear_MipLinear, nn::gfx::ComparisonFunction_Always );

}

// Init function for uniforms
static void InitUniforms(Uniforms* pUniforms)
{
    UniformBlockStatic* pUbStatic = pUniforms->staticVsUB.Map< UniformBlockStatic>();

    Mtx   lookAtMtx34;

    Vec     up = {0.20f, 0.97f, 0.0f};
    Vec  objPt = {-110.0f, -70.0f, -190.0f};
    Vec camLoc = {90.0f, 30.0f, 13.0f};

    f32   pers = 50.0f;
    f32 aspect = 1.0f;
    f32  znear = 10.0f;
    f32   zfar = 2000.0f;

    // Compute perspective matrix
    MTXPerspective( pUbStatic->projMtx44, pers, aspect, znear, zfar );

    // Compute lookAt matrix
    MTXLookAt( lookAtMtx34, &camLoc, &up, &objPt );
    MTX34To44( lookAtMtx34, pUbStatic->viewMtx44 );

#if NN_GFX_IS_TARGET_GX
    // Uniform buffers must be in little-endian mode
    GX2EndianSwap( pUbStatic, sizeof( UniformBlockStatic ) );
#endif

    pUniforms->staticVsUB.Unmap();
}

// Update uniforms
static void UpdateUniforms(Uniforms* pUniforms)
{
    UniformBlockDynamic* pVsUbCpu = pUniforms->dynamicVsUBCPU[ ( g_FrameCount + 1 ) % 2 ].Map< UniformBlockDynamic >();
    UniformBlockDynamicGS* pGsUbCpu = pUniforms->dynamicGsUBCPU[ ( g_FrameCount + 1 ) % 2 ].Map< UniformBlockDynamicGS >();

    // row major matrix
    Mtx  rotYMtx34;

    // Compute rotation matrix
    MTXRotRad(rotYMtx34, 'y', s_yrad * 2);

    // Compute model matrix
    MTX34To44(rotYMtx34, pVsUbCpu->modelMtx44);

    s_yrad += 0.0022f;

    pGsUbCpu->threadhold = g_SubdivideThreshold;

#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pVsUbCpu, sizeof( *pVsUbCpu ) );
    GX2EndianSwap( pGsUbCpu, sizeof( *pGsUbCpu ) );
#endif

    pUniforms->dynamicVsUBCPU[ ( g_FrameCount + 1 ) % 2 ].Unmap();
    pUniforms->dynamicGsUBCPU[ ( g_FrameCount + 1 ) % 2 ].Unmap();
}

static void UpdateDynamicUniform( DEMOGfxBuffer* pDstBuffer, DEMOGfxBuffer* pSrcBuffer, int offset, int size )
{
    NN_UNUSED( offset );
    u8* pSrcData = pSrcBuffer->Map< u8 >();
    u8* pDstData = pDstBuffer->Map< u8 >();

    memcpy( pDstData, pSrcData, size );

    pSrcBuffer->Unmap();
    pDstBuffer->Unmap();
}

#if NN_GFX_IS_TARGET_GL
static void myLineWidthCallback( const void* pParam )
{
    glLineWidth( *reinterpret_cast< const float* >( pParam ) );
}
#endif

// The draw function for the rendering portions of this app
static void DrawPlane(DEMOGfxPipeline* pPipeline, AttribBuffer* pAttribData, Uniforms* pUniforms, RingBufferGS* pRingData )
{
    DEMOGfxShader* pShader = &pPipeline->shaders;

    // Set Block to Dynamic
    u32 uniformBlockStaticVS = 0;
    u32 uniformBlockVS = 1;
    u32 uniformBlockGS = 0;

    // When using the DEMO library, it is necessary to call
    // DEMOGfxBeforeRender and DEMOGfxDoneRender before and after drawing.
    DEMOGfxBeforeRender();

    // Update frame counter to double buffer dynamic Uniforms
    g_FrameCount++;

    nn::gfx::ColorTargetView* pCurrentScanBuffer = DEMOGetColorBufferView();
    DEMOCommandBuffer.ClearColor( pCurrentScanBuffer, 0.2f, 0.2f, 0.2f, 1.0f, NULL );
    DEMOCommandBuffer.ClearDepthStencil( &DEMODepthBufferView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_GEOMETRY_SHADER );
#endif
    DEMOGfxSetDefaultRenderTarget();

    DEMOCommandBuffer.SetViewportScissorState( &g_viewportScissorState );

    // Set Dynamic (per-frame) uniform blocks
    UpdateDynamicUniform( &pUniforms->dynamicVsUBGPU[ g_FrameCount % 2 ],
        &pUniforms->dynamicVsUBCPU[ g_FrameCount % 2 ], 0, sizeof( UniformBlockDynamic ) );
    UpdateDynamicUniform( &pUniforms->dynamicGsUBGPU[ g_FrameCount % 2 ],
        &pUniforms->dynamicGsUBCPU[ g_FrameCount % 2 ], 0, sizeof( UniformBlockDynamicGS ) );

    // Invalidate constant cache
    DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_ConstantBuffer );

    // Set vs dynamic uniform block
    DEMOCommandBuffer.SetConstantBuffer( pShader->uniformBlocksVS.location[ uniformBlockVS ],
        nn::gfx::ShaderStage_Vertex, pUniforms->dynamicVsUBGPU[ g_FrameCount % 2 ].gpuAddress, pUniforms->dynamicVsUBGPU[ g_FrameCount % 2 ].size );

    // Set gs dynamic uniform block
    DEMOCommandBuffer.SetConstantBuffer( pShader->uniformBlocksGS.location[ uniformBlockGS ],
        nn::gfx::ShaderStage_Geometry, pUniforms->dynamicGsUBGPU[ g_FrameCount % 2 ].gpuAddress, pUniforms->dynamicGsUBGPU[ g_FrameCount % 2 ].size );

    // Set vs static uniform block
    DEMOCommandBuffer.SetConstantBuffer( pShader->uniformBlocksVS.location[ uniformBlockStaticVS ],
        nn::gfx::ShaderStage_Vertex, pUniforms->staticVsUB.gpuAddress, pUniforms->staticVsUB.size );

    DEMOCommandBuffer.SetTextureAndSampler(pShader->samplersGS.location[0], nn::gfx::ShaderStage_Geometry,
        g_heightMap.GetDescriptorSlot(0), g_MySamplerSlot);
    DEMOCommandBuffer.SetTextureAndSampler( pShader->samplersPS.location[ 0 ], nn::gfx::ShaderStage_Pixel,
        g_colorTex.GetDescriptorSlot( 0 ), g_MySamplerSlot );

    // Set Attrib buffer
    DEMOCommandBuffer.SetVertexBuffer( 0, pAttribData->positionBuffer.gpuAddress, POSITION_STRIDE, pAttribData->positionBuffer.size );
    DEMOCommandBuffer.SetVertexBuffer( 1, pAttribData->textureBuffer.gpuAddress, TEXCOORD_STRIDE, pAttribData->textureBuffer.size );

    DEMOCommandBuffer.SetPipeline( &pPipeline->pipeline );

#if NN_GFX_IS_TARGET_GX
    GX2SetGeometryShaderInputRingBuffer(pRingData->pInBuffer, pRingData->inSize);
    GX2SetGeometryShaderOutputRingBuffer(pRingData->pOutBuffer, pRingData->outSize);

    GX2SetLineWidth(3.0f);
#elif NN_GFX_IS_TARGET_GL
    NN_UNUSED( pRingData );
    float lineWidth = 3.0;
    DEMOCommandBuffer.Gl4SetUserCommand( myLineWidthCallback, &lineWidth );
#elif NN_GFX_IS_TARGET_NVN
    NN_UNUSED( pRingData );
    nvnCommandBufferSetLineWidth( DEMOCommandBuffer.ToData()->pNvnCommandBuffer, 3.0f );
#elif NN_GFX_IS_TARGET_D3D
    NN_UNUSED( pRingData );
#endif

    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32,
        pAttribData->indexBuffer.gpuAddress, static_cast< int >( pAttribData->indexBuffer.size ) / sizeof( u32 ), 0 );

    // Print Informations
    PrintInfo();

    // This function will handle presenting the rendered buffer to the screen
    // by swapping the color buffer, setting the new context state,
    // resetting the color buffer, and flushing the pipeline.
    DEMOGfxDoneRender();
}

static void FreePlaneAttributeData( AttribBuffer* pPlaneAttribData )
{
    pPlaneAttribData->indexBuffer.Finalize();
    pPlaneAttribData->positionBuffer.Finalize();
    pPlaneAttribData->textureBuffer.Finalize();
}

//extern "C" void nnMain()
TEST(GfxTextureFetchGs, Run)
{
    // Shader data
    DEMOGfxPipeline textureFetchGsPipeline;

    // Uniforms for Transform shader with Uniform Blocks
    Uniforms textureFetchGsUB;

    // Attrib buffers for Cube
    AttribBuffer planeAttribData;

    // GS ring buffers
    RingBufferGS demoRingData = {0};

    // Initialize the DEMO library, DEMO test system, and create the
    // primary display.
    DEMOInit();
    DEMOTestInit( nnt::GetHostArgc(), nnt::GetHostArgv() );

    // This function will create the appropriate size color and depth
    // buffers, setup the scan buffer, and initialize the viewport and
    // scissor rectangles to be the entire screen.
    DEMOGfxInit( nnt::GetHostArgc(), nnt::GetHostArgv() );
    DEMOFontInit();
    DEMOTestIsUseHlslccGlsl() ? g_ShaderFileIdx = 1 : g_ShaderFileIdx = 0;

    // After the DEMO library is initialize the sample's initialization
    // function can be called.

    // In case there are one-time flushes necessary:
    DEMOCommandBuffer.Begin();

    // Initialize attributes and buffers
    InitAttribData( &planeAttribData );

    // Initialize shaders and uniform blocks
    InitPipelineGS(&textureFetchGsPipeline, &planeAttribData, &textureFetchGsUB, SHADER_FILE_NAME[g_ShaderFileIdx], &demoRingData);

    // Initialize textures and samplers
    InitTexture(&textureFetchGsPipeline.shaders);

    DEMOCommandBuffer.End();

    DEMOQueue.ExecuteCommand( &DEMOCommandBuffer, NULL );
    DEMOQueue.Sync();

    g_FrameCount = 0;
    s_yrad = 0.0f;
    g_CurrentFocus = 0;
    g_WireFrame = true;
    g_SubdivideThreshold = 0.25f;

    while (DEMOIsRunning())
    {
        DEMOPadRead();
        SetPadInfo( &textureFetchGsPipeline );

        // In this example, Double buffer for uniform buffer is necessary
        // because CPU always proceeds one frame that GPU processes.
        // CPU writes N+1 th frame while GPU reads N th frame data

        // Update Model Mtx
        // 1) CPU writes uniform buffer of N + 1th frame
        UpdateUniforms(&textureFetchGsUB);

        // Draw Transformed Cube
        // 2) CPU waits VI sync (DEMOGfxBeforeRender)
        // 3) N += 1;
        // 3) GPU starts working for N th frame
        // 4) Copys framebuffer into scanbuffer and flip (DEMOGfxDoneRender)
        DrawPlane(&textureFetchGsPipeline, &planeAttribData, &textureFetchGsUB, &demoRingData);
    }

    // Free shaders and uniform blocks
    FreePipelineGS(&textureFetchGsPipeline, &textureFetchGsUB, &demoRingData);

    // Free Plane Attribute Data
    FreePlaneAttributeData( &planeAttribData );

    g_colorTex.Finalize();
    g_heightMap.Finalize();

    g_MySampler.Finalize( &DEMODevice );
    g_viewportScissorState.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2( g_pViewportScissorStateData );

    DEMOFontShutdown();
    DEMOTestShutdown();
    DEMOGfxShutdown();
    DEMOShutdown();

    SUCCEED();
}
