﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testGfx_TextureCapture.h"

#include <cstdlib>
#include <cstring>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

namespace nnt{ namespace gfx{

    TextureCapture::TextureCapture()
        : m_IsInitialized(false)
        , m_IsCapturing(false)
        , m_pAllocateFunction(nullptr)
        , m_ScreenWidth(0)
        , m_ScreenHeight(0)
        , m_pDevice(nullptr)
        , m_pMemoryPool(nullptr)
        , m_CapturedImageBuffer(nullptr)
        , m_CapturedImageBufferSize(0)
    {
    }

    TextureCapture::~TextureCapture()
    {
        this->Finalize();
    }

    //-------------------------------------------

    void* TextureCapture::AllocateMemory(size_t size, int alignment)
    {
        NN_SDK_ASSERT_NOT_NULL(m_pAllocateFunction);
        return m_pAllocateFunction->Allocate(size, alignment);
    }
    void TextureCapture::DeallocateMemory(void* ptr)
    {
        NN_SDK_ASSERT_NOT_NULL(m_pAllocateFunction);
        m_pAllocateFunction->Free(ptr);
    }

    //-------------------------------------------

    void TextureCapture::Initialize(
        ptrdiff_t* pOutNewMemoryPoolOffset,
        nn::gfx::Device* pDevice,
        nn::gfx::MemoryPool* pMemoryPool,
        ptrdiff_t currentMemoryPoolOffset,
        nnt::graphics::AllocatorFunction* pAllocatorFunction,
        int screenWidth,
        int screenHeight)
    {
        m_ScreenWidth  = screenWidth;
        m_ScreenHeight = screenHeight;
        m_pDevice      = pDevice;
        m_pMemoryPool  = pMemoryPool;
        m_pAllocateFunction = pAllocatorFunction;

        size_t imageBufferSize = screenWidth * screenHeight * 4;
        {


            nn::gfx::Buffer::InfoType info;
            info.SetDefault();
            info.SetSize(imageBufferSize);
            info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);

            if (NN_STATIC_CONDITION(nn::gfx::Texture::IsMemoryPoolRequired))
            {
                size_t alignment = nn::gfx::Buffer::GetBufferAlignment(pDevice, info);
                ptrdiff_t offset = ((currentMemoryPoolOffset + alignment - 1) / alignment) * alignment;
                size_t size = info.GetSize();
                m_CaptureBuffer.Initialize(pDevice, info, pMemoryPool, offset, size);
                *pOutNewMemoryPoolOffset = offset + size;
            }
            else
            {
                m_CaptureBuffer.Initialize(pDevice, info, NULL, 0, 0);
                *pOutNewMemoryPoolOffset = currentMemoryPoolOffset;
            }
        }

        //size_t imageBufferSize = screenWidth * screenHeight * 4;
        m_CapturedImageBuffer = static_cast<uint8_t*>(AllocateMemory(imageBufferSize, NN_ALIGNOF(std::max_align_t)));
        m_CapturedImageBufferSize = imageBufferSize;

        m_IsCapturing = false;

        m_IsInitialized = true;
    }

    void TextureCapture::Finalize()
    {
        if(!m_IsInitialized)
        {
            return;
        }
        m_CaptureBuffer.Finalize(m_pDevice);

        DeallocateMemory(m_CapturedImageBuffer);
        m_CapturedImageBuffer = NULL;
        m_CapturedImageBufferSize = 0;

        m_IsInitialized = false;
    }

    //-------------------------------------------

    bool TextureCapture::IsCapturing() const
    {
        return m_IsCapturing;
    }

    //-------------------------------------------

    void TextureCapture::PushCaptureCommand(nn::gfx::CommandBuffer* pCommandBuffer, nn::gfx::Texture* pSrcTexture)
    {
        if(m_IsCapturing)
        {
            NN_LOG("too many PushCaputureCommand().\n");
            return;
        }
        nn::gfx::TextureCopyRegion region;
        region.SetDefault();
        region.SetWidth(m_ScreenWidth);
        region.SetHeight(m_ScreenHeight);
        pCommandBuffer->CopyImageToBuffer(&m_CaptureBuffer, 0, pSrcTexture, region);
        m_IsCapturing = true;
    }

    //-------------------------------------------

    void TextureCapture::FetchCaptureResult()
    {
        if(!m_IsCapturing)
        {
            NN_LOG("too many FetchCaptureResult()\n");
            return;
        }
        const uint8_t* pBuffer = static_cast<uint8_t*>(m_CaptureBuffer.Map());

        size_t stride = 4 * m_ScreenWidth;

        int ySrc = 0;
        int yDst = m_ScreenHeight - 1;
        for(; ySrc < m_ScreenHeight; ySrc++, yDst--)
        {
            const uint8_t* pSrc = pBuffer + stride * ySrc;
            uint8_t* pDst = m_CapturedImageBuffer + stride * yDst;
            memcpy(pDst, pSrc, stride);
        }

        m_CaptureBuffer.Unmap();
        m_IsCapturing = false;
    }

    //-------------------------------------------

    const uint8_t* TextureCapture::GetCapturedImageData() const
    {
        return this->m_CapturedImageBuffer;
    }
    size_t TextureCapture::GetCapturedImageDataSize() const
    {
        return this->m_CapturedImageBufferSize;
    }
}}

