﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testGfx_ColorTargetViewCapture.h"

#include <nn/nn_Log.h>

namespace nnt{ namespace gfx{

    ColorTargetViewCapture::ColorTargetViewCapture()
        : m_IsInitialized(false)
        , m_TextureCapture()
        , m_pDevice(nullptr)
        , m_Texture()
    {
    }

    ColorTargetViewCapture::~ColorTargetViewCapture()
    {
        Finalize();
    }

    //-------------------------------------------

    void ColorTargetViewCapture::Initialize(
        ptrdiff_t* pOutNewMemoryPoolOffset,
        ptrdiff_t* pOutNewInvisibleMemoryPoolOffset,
        nn::gfx::Device* pDevice,
        nn::gfx::MemoryPool* pMemoryPool,
        nn::gfx::MemoryPool* pInvisibleMemoryPool,
        ptrdiff_t currentMemoryPoolOffset,
        ptrdiff_t currentInvisibleMemoryPoolOffset,
        nnt::graphics::AllocatorFunction* pAllocatorFunction,
        int screenWidth,
        int screenHeight)
    {
        m_TextureCapture.Initialize(pOutNewMemoryPoolOffset, pDevice, pMemoryPool, currentMemoryPoolOffset, pAllocatorFunction, screenWidth, screenHeight);
        m_pDevice = pDevice;

        {
            nn::gfx::TextureInfo info;
            info.SetDefault();
            info.SetWidth( screenWidth );
            info.SetHeight( screenHeight );
            info.SetImageFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb );
            if( NN_STATIC_CONDITION( nn::gfx::Texture::IsMemoryPoolRequired ) )
            {
                size_t size = m_Texture.CalculateMipDataSize( pDevice, info );
                ptrdiff_t offset = nn::util::align_up(
                    currentInvisibleMemoryPoolOffset,
                    m_Texture.CalculateMipDataAlignment( pDevice, info ) );
                m_Texture.Initialize( pDevice, info, pInvisibleMemoryPool, offset, size );
                *pOutNewInvisibleMemoryPoolOffset = offset + size;
            }
            else
            {
                m_Texture.Initialize( pDevice, info, NULL, 0, 0 );
                *pOutNewInvisibleMemoryPoolOffset = currentInvisibleMemoryPoolOffset;
            }
        }

        m_IsInitialized = true;
    }

    void ColorTargetViewCapture::Finalize()
    {
        if(!m_IsInitialized)
        {
            return;
        }

        m_Texture.Finalize(m_pDevice);

        m_TextureCapture.Finalize();
        m_IsInitialized = false;
    }

    //-------------------------------------------

    bool ColorTargetViewCapture::IsCapturing() const
    {
        return m_TextureCapture.IsCapturing();
    }

    //-------------------------------------------

    void ColorTargetViewCapture::PushCaptureCommand(nn::gfx::CommandBuffer* pCommandBuffer, nn::gfx::ColorTargetView* pSrcView)
    {
        if(m_TextureCapture.IsCapturing())
        {
            NN_LOG("too many PushCaputureCommand().\n");
            return;
        }
        pCommandBuffer->Resolve(&m_Texture, 0, 0, pSrcView, nullptr);
        m_TextureCapture.PushCaptureCommand(pCommandBuffer, &m_Texture);
    }

    //-------------------------------------------

    void ColorTargetViewCapture::FetchCaptureResult()
    {
        m_TextureCapture.FetchCaptureResult();
    }

    //-------------------------------------------

    const uint8_t* ColorTargetViewCapture::GetCapturedImageData() const
    {
        return m_TextureCapture.GetCapturedImageData();
    }
    size_t ColorTargetViewCapture::GetCapturedImageDataSize() const
    {
        return m_TextureCapture.GetCapturedImageDataSize();
    }
}}

