﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cmath>
#include <cfloat>

#include <gfx/demo.h>
#include <nn/gfx.h>

#include <nnt.h>
#include <nnt/nnt_Argument.h>

#if NN_GFX_IS_TARGET_GX
#include <cafe/gx2ut.h>
#endif

#ifdef WIN32
#define snprintf _snprintf
#endif


////////////////////////////////////////////////////
//
// Assets data, types and interface for demos
//
////////////////////////////////////////////////////

// ----- Surface Information

static const int SHADOWMAP_WIDTH = 1024;
static const int SHADOWMAP_HEIGHT = 1024;

#define SURFACE_WIDTH DEMOColorBufferInfo.GetWidth()
#define SURFACE_HEIGHT DEMOColorBufferInfo.GetHeight()

// Here are the shader files that contains vertex and pixel
// shader data.
static int g_ShaderFileIdx = 0;
static const char * const SHADER_FILE_DEPTH[] =
{
    "shaders/shadow/depth",
    "shaders/shadow/depthHlslcc",
};
static const char * const SHADER_FILE_SHADOW[] =
{
    "shaders/shadow/shadow",
    "shaders/shadow/shadowHlslcc",
};
static const char * const SHADER_FILE_VIEW_SM[] =
{
    "shaders/shadow/viewSM",
    "shaders/shadow/viewSMHlslcc",
};

// Model file path

static const char * const MODEL_FILE_TEMPLE = "geometries/temple.nmod";
static const char * const MODEL_FILE_TERRAIN = "geometries/ground.nmod";

// Texture directory
static const char * const MODEL_TEXTURE_DIRECTORY = "textures/shadow";

static const int NUMBER_CASCADE = 4;

static const float SHADOW_RANGE = 6000.0f;
static const float SHADOW_CASTING_RANGE = 6000.0f;
static const float SPLIT_RATIO = 0.7f;

static DemoModelData g_modelTemple;
static DemoModelData g_modelTerrain;

static Mtx44   g_viewMtx44;
static Mtx44   g_projMtx44;

static nn::gfx::Sampler g_samplerModel;
static nn::gfx::Sampler g_samplerShadow;
static nn::gfx::Sampler g_samplerViewSM;
static nn::gfx::DescriptorSlot g_samplerModelSlot;
static nn::gfx::DescriptorSlot g_samplerShadowSlot;
static nn::gfx::DescriptorSlot g_samplerViewSMSlot;

static OSTime g_lastTime = 0;
static f32 g_dtime = 0.0f;

static Vec g_up = {0.0f, 1.0f, 0.0f};
static Vec g_objPt = {0.0f, 0.0f, 0.0f};
static Vec g_camPt = {-2000.0f, 1000.0f, 0.0f};
static const Vec g_lightUp = {0.0f, 1.0f, 0.0f};
static Vec g_lightPt = {-2000.0f, 2000.0f, 0.0f};
static const Vec g_origin = {0.0f, 0.0f, 0.0f};

static float g_offsetY = 40.0f;

static nn::gfx::Texture g_shadowmap;
static nn::gfx::TextureView g_shadowmapView;
static nn::gfx::DepthStencilView g_shadowmapDepthView[NUMBER_CASCADE];
static nn::gfx::DescriptorSlot g_shadowmapSlot;
static DEMOGfxMemPool* g_shadowmapData;

static BOOL g_viewShadowMaps = 0;

static bool s_firstFrame;

static const int DEMO_BUFFER_ALIGN = 4096;
static const int viewerPositionSize = sizeof( f32 ) * 12;
static const int viewerTexCoordSize = sizeof( f32 ) * 8;
static const int viewerIndexSize = sizeof( u32 ) * 6;

DEMOGfxPipeline depthPipeline;
DEMOGfxPipeline shadowPipeline;
DEMOGfxPipeline viewerPipeline;

nn::gfx::ViewportScissorState fullViewport;
static void* pFullViewportMem;
nn::gfx::ViewportScissorState shadowViewport;
static void* pShadowViewportMem;
nn::gfx::ViewportScissorState viewerViewport[NUMBER_CASCADE];
static void* pViewerViewportMem[NUMBER_CASCADE];

struct vertexUniforms
{
    Mtx44   modelMtx44;
    Mtx44   viewMtx44;
    Mtx44   projMtx44;
};
static int depthVertexUniformBlock;
static DEMOGfxBuffer depthTempleVertexUniformBuffer[NUMBER_CASCADE];
static DEMOGfxBuffer depthTerrainVertexUniformBuffer[NUMBER_CASCADE];

static int shadowVertexUniformBlock;
static DEMOGfxBuffer shadowTempleVertexUniformBuffer;
static DEMOGfxBuffer shadowTerrainVertexUniformBuffer;

struct shadowPixelUniforms
{
    Mtx44 lightMtx[ NUMBER_CASCADE ];
    f32 splitDistance[ NUMBER_CASCADE ];
    f32 depthBias[ NUMBER_CASCADE ];
    Vec lightDir;
    f32 dummy1;
    Vec eyePos;
    f32 dummy2;
};

static int shadowPixelUniformBlock;
static DEMOGfxBuffer shadowPixelUniformBuffer;

struct viewerPixelUniforms
{
    float   index;
};

static int viewerPixelUniformBlock;
static int viewerTextureLoc;
DEMOGfxBuffer viewerPixelUniformBuffer[NUMBER_CASCADE];

DEMOGfxBuffer viewerPositionBuffer;
DEMOGfxBuffer viewerTexCoordBuffer;
DEMOGfxBuffer viewerIndexBuffer;

static int depthPixelSamplerLoc;
static int shadowPixelSamplerLoc;
static int shadowPixelMapLoc;

void* g_pMaterialData;
nn::gfx::BlendState material;
void* g_pMaterialAlphaData;
nn::gfx::BlendState materialAlpha;

////////////////////////////////////////////////////
//
// Prototypes
//
////////////////////////////////////////////////////
static void InitShaderDepth(DEMOGfxShader *pShader,const DemoAttributeData* attributes,s32 attributeCount);
static void InitShaderShadow(DEMOGfxShader *pShader,const DemoAttributeData* attributes,s32 attributeCount);
static void SetMaterial(const DemoMaterialData* pMaterial, DEMOGfxTexture* pTextures,u32 textureLoc);
static void DrawModel(const DemoModelData* pModelData,u32 textureLoc, bool bSkipMaterial);
static void InitCamera(Mtx44 resultProjMtx44, Mtx44 resultViewMtx44);
static void InitScene();
static void InitViewer(DEMOGfxShader *pShader);
static f32 ComputeLightMatrix(Mtx44 lightViewMtx44, Mtx44 lightProjMtx44, u32 split);
static void DrawSceneFromLight();
static void DrawSceneFromCamera();
static void DrawViewer();
static void PrintInfo();
static void DrawScene();
static void ProcessPad();


// The initialization function for the rendering portions of this sample.
// It is responsible for allocating the three types of shaders and buffers
// as well as ensuring that data is flushed from the CPU to GPU memory
// for Simple Shader
static void InitShaderDepth(DEMOGfxShader *pShader,const DemoAttributeData* attributes,s32 attributeCount)
{
    DEMOGfxLoadShadersFromFile(pShader, 0, SHADER_FILE_DEPTH[g_ShaderFileIdx]);

    int attributeStride = 0;

    // Set attribute buffer to shader
    for (s32 i = 0; i < attributeCount; i++)
    {
        const DemoAttributeData* pAttribute = &attributes[i];
        nn::gfx::AttributeFormat format = nn::gfx::AttributeFormat_Undefined;

        const char* attributeName = NULL;

        switch (pAttribute->type)
        {
            case DEMO_ATTRIBUTE_POSITION:
            {
                //Local coordinate attributes
                attributeName = "a_position";
                format = nn::gfx::AttributeFormat_32_32_32_Float;
                attributeStride += sizeof( float ) * 3;
            }
            break;
            case DEMO_ATTRIBUTE_TEXCOORD:
            {
                //Texture coordinate attributes
                attributeName = "a_texCoord";
                format = nn::gfx::AttributeFormat_32_32_Float;
                attributeStride += sizeof( float ) * 2;
            }
            break;
            case DEMO_ATTRIBUTE_NORMAL:
            {
                attributeStride += sizeof(float) * 3;
            }
            continue; // NORMAL is not used in this shader
            default:
            {
                //Attributes not used in shader
            }
            continue;
        }

        DEMOGfxInitShaderAttribute(pShader,
                                   attributeName,
                                   0,
                                   pAttribute->offset,
                                   format);
    }
    DEMOGfxInitShaderVertexBuffer( pShader, 0, attributeStride, 0 );

    // Uniform Location Lookup
    depthVertexUniformBlock = depthPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_matrices");
    for (int i = 0; i < NUMBER_CASCADE; i++)
    {
        depthTempleVertexUniformBuffer[ i ].Initialize( sizeof( vertexUniforms ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
        depthTerrainVertexUniformBuffer[ i ].Initialize( sizeof( vertexUniforms ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
    }
    // Texture Sampler lookup
    depthPixelSamplerLoc = depthPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_texture" );

    depthPipeline.blendTargetStateCount = 1;
    depthPipeline.blendTargetStateInfoArray[ 0 ].SetDefault();
    depthPipeline.blendTargetStateInfoArray[ 0 ].SetChannelMask( 0 );
    depthPipeline.colorTargetStateCount = 1;
    depthPipeline.colorTargetStateInfoArray[ 0 ].SetDefault();

    depthPipeline.Initialize( &DEMODevice );
}

static void InitShaderShadow(DEMOGfxShader *pShader,const DemoAttributeData* attributes,s32 attributeCount)
{

    DEMOGfxLoadShadersFromFile(pShader, 0, SHADER_FILE_SHADOW[g_ShaderFileIdx]);

    int attributeStride = 0;
    // Set attribute buffer to shader
    for (s32 i = 0; i < attributeCount; i++)
    {
        const DemoAttributeData* pAttribute = &attributes[i];
        nn::gfx::AttributeFormat format = nn::gfx::AttributeFormat_Undefined;

        const char* attributeName = NULL;

        switch (pAttribute->type){
           case DEMO_ATTRIBUTE_POSITION:
               {
                   //Local coordinate attributes
                   attributeName = "a_position";
                   format = nn::gfx::AttributeFormat_32_32_32_Float;
                   attributeStride += sizeof( float ) * 3;
           }
               break;
           case DEMO_ATTRIBUTE_NORMAL:
               {
                   //Normal attributes
                   attributeName = "a_normal";
                   format = nn::gfx::AttributeFormat_32_32_32_Float;
                   attributeStride += sizeof( float ) * 3;
           }
               break;
           case DEMO_ATTRIBUTE_TEXCOORD:
               {
                   //Texture coordinate attributes
                   attributeName = "a_texCoord";
                   format = nn::gfx::AttributeFormat_32_32_Float;
                   attributeStride += sizeof( float ) * 2;
           }
               break;
           default:
               {
                   //Attributes not used in shader
               }
               continue;
        }

        DEMOGfxInitShaderAttribute(pShader,
                                   attributeName,
                                   0,
                                   pAttribute->offset,
                                   format);
    }
    DEMOGfxInitShaderVertexBuffer( pShader, 0, attributeStride, 0 );

    // Uniform Location Lookup
    shadowVertexUniformBlock = shadowPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_matrices" );
    shadowTempleVertexUniformBuffer.Initialize( sizeof( vertexUniforms ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
    shadowTerrainVertexUniformBuffer.Initialize( sizeof( vertexUniforms ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
    shadowPixelUniformBlock = shadowPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_uniforms" );
    shadowPixelUniformBuffer.Initialize( sizeof( shadowPixelUniforms ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );

    // Texture Sampler lookup
    shadowPixelSamplerLoc = shadowPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_texture" );
    shadowPixelMapLoc = shadowPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,nn::gfx::ShaderInterfaceType_Sampler, "s_shadowmap" );
    shadowPipeline.colorTargetStateCount = 1;
    shadowPipeline.colorTargetStateInfoArray[ 0 ].SetDefault();
    shadowPipeline.colorTargetStateInfoArray[ 0 ].SetFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );
    shadowPipeline.blendTargetStateCount = 1;
    shadowPipeline.blendTargetStateInfoArray[ 0 ].SetDefault();

    shadowPipeline.rasterizerStateInfo.SetCullMode( nn::gfx::CullMode_Back );

    shadowPipeline.Initialize( &DEMODevice );
}

// Init function for setting projection matrix
static void InitCamera(Mtx44 resultProjMtx44, Mtx44 resultViewMtx44)
{
    // Row major matrices
    Mtx lookAtMtx34;

    f32 pers = 60.0f;
    f32 aspect = (f32)SURFACE_WIDTH / (f32)SURFACE_HEIGHT;
    f32 znear = 10.0f;
    f32 zfar = 100000.0f;

    // Compute perspective matrix
    MTXPerspective(resultProjMtx44, pers, aspect, znear, zfar);

    // Compute lookAt matrix
    MTXLookAt(lookAtMtx34, &g_camPt, &g_up, &g_objPt);
    MTX34To44(lookAtMtx34, resultViewMtx44);
}

// The init function for the rendering portions of this app
static void InitScene()
{
    // Load the model data
    DEMOLoadModelData(&g_modelTemple, MODEL_FILE_TEMPLE, MODEL_TEXTURE_DIRECTORY );
    DEMOLoadModelData( &g_modelTerrain, MODEL_FILE_TERRAIN, MODEL_TEXTURE_DIRECTORY );

    // Initialize shader
    InitShaderDepth( &depthPipeline.shaders, g_modelTemple.attributes,
                    g_modelTemple.header.attributeCount);
    InitShaderShadow( &shadowPipeline.shaders, g_modelTemple.attributes,
                     g_modelTemple.header.attributeCount);

    // Camera initialization
    InitCamera(g_projMtx44, g_viewMtx44);

    // Sampler setting
    DEMOGfxInitSampler( &g_samplerModel, &g_samplerModelSlot, nn::gfx::TextureAddressMode_Repeat, nn::gfx::FilterMode_MinLinear_MagLinear_MipLinear, nn::gfx::ComparisonFunction_Always );
    DEMOGfxInitSampler( &g_samplerShadow, &g_samplerShadowSlot, nn::gfx::TextureAddressMode_ClampToEdge, nn::gfx::FilterMode_Comparison_MinLinear_MagLinear_MipPoint, nn::gfx::ComparisonFunction_Less );

    // Create a texture array in depth NUMBER_CASCADE
    DEMOGfxSetupTextureBuffer(&g_shadowmap, NULL, NULL, NULL, NULL, &g_shadowmapData,
        SHADOWMAP_WIDTH, SHADOWMAP_HEIGHT, NUMBER_CASCADE, 1, nn::gfx::ImageDimension_2dArray,
        nn::gfx::ImageFormat_D32_Float, nn::gfx::DepthStencilFetchMode_DepthComponent, 0 );

    {
        nn::gfx::TextureView::InfoType info;
        info.SetDefault();
        info.SetTexturePtr( &g_shadowmap );
        info.SetChannelMapping( nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red );
        info.SetImageFormat( nn::gfx::ImageFormat_D32_Float );
        info.SetDepthStencilTextureMode( nn::gfx::DepthStencilFetchMode_DepthComponent );
        info.SetImageDimension( nn::gfx::ImageDimension_2dArray );
        info.EditSubresourceRange().EditArrayRange().SetArrayLength( NUMBER_CASCADE );
        g_shadowmapView.Initialize( &DEMODevice, info );
        int index = DEMOGfxRegisterTextureView( &g_shadowmapView );
        DEMOTextureDescriptorPool.GetDescriptorSlot( &g_shadowmapSlot, index );
    }

    {
        // Create a depth view for each member of the texture array
        nn::gfx::DepthStencilView::InfoType info;
        info.SetDefault();
        info.SetImageDimension(nn::gfx::ImageDimension_2d);

        for (int i = 0; i < NUMBER_CASCADE; i++)
        {
            info.SetTexturePtr(&g_shadowmap);
            info.EditArrayRange().SetBaseArrayIndex(i);
            info.EditArrayRange().SetArrayLength(1);
            g_shadowmapDepthView[i].Initialize(&DEMODevice, info);
#if NN_GFX_IS_TARGET_GX
            // FIXME: Temporary disable HiZ
            GX2DepthBuffer* pDepth = g_shadowmapDepthView[i].ToData()->pGx2DepthBuffer;
            GX2InitDepthBufferHiZEnable(pDepth, GX2_FALSE);
            pDepth->hiZPtr = NULL;
            pDepth->hiZSize = 0;
#endif
        }
    }
    // Create viewports
    DEMOGfxSetViewportScissorState( &fullViewport, &pFullViewportMem, 0.0f,
        0.0f,
        static_cast< float >( SURFACE_WIDTH ),
        static_cast< float >( SURFACE_HEIGHT ),
        0.0f, 1.0f,
        static_cast< float >( SURFACE_HEIGHT ), false );
    DEMOGfxSetViewportScissorState( &shadowViewport, &pShadowViewportMem, 0.0f,
        0.0f,
        static_cast< float >( SHADOWMAP_WIDTH ),
        static_cast< float >( SHADOWMAP_HEIGHT ),
        0.0f, 1.0f,
        static_cast< float >( SHADOWMAP_HEIGHT ), false );

    {
        nn::gfx::BlendStateInfo info;
        nn::gfx::BlendTargetStateInfo blendTargets[ 1 ];
        blendTargets[ 0 ].SetDefault();
        blendTargets[ 0 ].SetBlendEnabled( false );
        blendTargets[ 0 ].SetChannelMask( nn::gfx::ChannelMask_All );

        info.SetDefault();
        info.SetAlphaToCoverageEnabled( false );
        info.SetBlendTargetStateInfoArray( &blendTargets[0], 1 );
        size_t size = nn::gfx::BlendState::GetRequiredMemorySize( info );
        g_pMaterialData = DEMOGfxAllocMEM2( size, nn::gfx::BlendState::RequiredMemoryInfo_Alignment );
        material.SetMemory( g_pMaterialData, size );
        material.Initialize( &DEMODevice, info );
        info.SetAlphaToCoverageEnabled( true );
        size = nn::gfx::BlendState::GetRequiredMemorySize( info );
        g_pMaterialAlphaData = DEMOGfxAllocMEM2( size, nn::gfx::BlendState::RequiredMemoryInfo_Alignment );
        materialAlpha.SetMemory( g_pMaterialAlphaData, size );
        materialAlpha.Initialize( &DEMODevice, info );
    }
}

static void InitViewer(DEMOGfxShader *pShader)
{

    DEMOGfxLoadShadersFromFile(pShader, 0, SHADER_FILE_VIEW_SM[g_ShaderFileIdx]);
    // Set attribute buffer to shader
    DEMOGfxInitShaderAttribute(pShader,
                               "a_position",
                               0,
                               0,
                               nn::gfx::AttributeFormat_32_32_32_Float);
    DEMOGfxInitShaderAttribute(pShader,
                               "a_texCoord",
                               1,
                               0,
                               nn::gfx::AttributeFormat_32_32_Float );
    DEMOGfxInitShaderVertexBuffer(pShader, 0, 3 * sizeof(float), 0);
    DEMOGfxInitShaderVertexBuffer(pShader, 1, 2 * sizeof(float), 0);

    // Uniform Location Lookup
    viewerPixelUniformBlock = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_index");
    viewerTextureLoc = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_texture");

    f32 *posInitData = ( f32 *)DEMOAllocEx( viewerPositionSize, DEMO_BUFFER_ALIGN );
    f32 *texInitData = ( f32 * )DEMOAllocEx( viewerTexCoordSize, DEMO_BUFFER_ALIGN );
    u32 *idxInitData = ( u32 * )DEMOAllocEx( viewerIndexSize, DEMO_BUFFER_ALIGN );

    posInitData[0] = -1.0f; posInitData[1] = -1.0f;
    posInitData[2] =  0.0f; posInitData[3] =  1.0f;
    posInitData[4] = -1.0f; posInitData[5] =  0.0f;
    posInitData[6] =  1.0f; posInitData[7] =  1.0f;
    posInitData[8] =  0.0f; posInitData[9] = -1.0f;
    posInitData[10] = 1.0f; posInitData[11] = 0.0f;

    texInitData[0] =  0.0f; texInitData[1] =  1.0f;
    texInitData[2] =  1.0f; texInitData[3] =  1.0f;
    texInitData[4] =  1.0f; texInitData[5] =  0.0f;
    texInitData[6] =  0.0f; texInitData[7] =  0.0f;

    idxInitData[0] = 0; idxInitData[1] = 1;
    idxInitData[2] = 2; idxInitData[3] = 3;
    idxInitData[4] = 0; idxInitData[5] = 2;

    viewerPositionBuffer.Initialize( viewerPositionSize, posInitData, nn::gfx::GpuAccess_VertexBuffer, 0 );
    viewerTexCoordBuffer.Initialize( viewerTexCoordSize, texInitData, nn::gfx::GpuAccess_VertexBuffer, 0);
    viewerIndexBuffer.Initialize( viewerIndexSize, idxInitData, nn::gfx::GpuAccess_IndexBuffer, 0);

    DEMOFree( posInitData );
    DEMOFree( texInitData );
    DEMOFree( idxInitData );

    DEMOGfxInitSampler( &g_samplerViewSM, &g_samplerViewSMSlot, nn::gfx::TextureAddressMode_ClampToEdge, nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint, nn::gfx::ComparisonFunction_Always );

    for (int i = 0; i < NUMBER_CASCADE; i++)
    {
        viewerPixelUniformBuffer[ i ].Initialize( sizeof( viewerPixelUniforms ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
        float *idx = viewerPixelUniformBuffer[i].Map< float >();
        *idx = (float)i;
        viewerPixelUniformBuffer[i].Unmap();
    }
    float xo, yo;
    for ( u32 i = 0; i < NUMBER_CASCADE; ++i )
    {
        switch ( i )
        {
        default: /* fall through */
        case 0: xo = ( float )SURFACE_WIDTH / 8.0f * 4.0f; yo = ( float )SURFACE_HEIGHT / 5.0f * 0.0f; break;
        case 1: xo = ( float )SURFACE_WIDTH / 8.0f * 5.0f; yo = ( float )SURFACE_HEIGHT / 5.0f * 0.0f; break;
        case 2: xo = ( float )SURFACE_WIDTH / 8.0f * 6.0f; yo = ( float )SURFACE_HEIGHT / 5.0f * 0.0f; break;
        case 3: xo = ( float )SURFACE_WIDTH / 8.0f * 7.0f; yo = ( float )SURFACE_HEIGHT / 5.0f * 0.0f; break;
        }

#if NN_GFX_IS_TARGET_D3D
        // ViewportをFlipします。
        yo = ( float )SURFACE_HEIGHT - ( float )SURFACE_HEIGHT / 5.0f;
#endif

        DEMOGfxSetViewportScissorState( &viewerViewport[i], &pViewerViewportMem[i], xo, yo,
            SURFACE_WIDTH / 8.0f, SURFACE_HEIGHT / 5.0f, 0.0f, 1.0f, (float)SURFACE_HEIGHT, false );
    }

    viewerPipeline.colorTargetStateCount = 1;
    viewerPipeline.colorTargetStateInfoArray[0].SetDefault();
    viewerPipeline.colorTargetStateInfoArray[0].SetFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
    viewerPipeline.blendTargetStateCount = 1;
    viewerPipeline.blendTargetStateInfoArray[0].SetDefault();
    viewerPipeline.rasterizerStateInfo.SetCullMode(nn::gfx::CullMode_None);
    viewerPipeline.depthStencilStateInfo.SetDepthWriteEnabled(false);
    viewerPipeline.depthStencilStateInfo.SetDepthTestEnabled(false);
    viewerPipeline.Initialize(&DEMODevice);
}

static void VecBounds(const Vec& pt, Vec& minPt, Vec& maxPt)
{
    if (minPt.x > pt.x)
    {
        minPt.x = pt.x;
    }
    if (minPt.y > pt.y)
    {
        minPt.y = pt.y;
    }
    if (minPt.z > pt.z)
    {
        minPt.z = pt.z;
    }
    if (maxPt.x < pt.x)
    {
        maxPt.x = pt.x;
    }
    if (maxPt.y < pt.y)
    {
        maxPt.y = pt.y;
    }
    if (maxPt.z < pt.z)
    {
        maxPt.z = pt.z;
    }
}
static f32 ComputeLightMatrix(Mtx44 lightViewMtx44, Mtx44 lightProjMtx44, u32 split)
{
    Mtx44 invProjMtx44;
    MTX44Inverse(g_projMtx44, invProjMtx44);
    Vec cube[8] =
    {
        {-1.0f, -1.0f, -1.0f,},
        {+1.0f, -1.0f, -1.0f,},
        {-1.0f, +1.0f, -1.0f,},
        {+1.0f, +1.0f, -1.0f,},
        {-1.0f, -1.0f, +1.0f,},
        {+1.0f, -1.0f, +1.0f,},
        {-1.0f, +1.0f, +1.0f,},
        {+1.0f, +1.0f, +1.0f,},
    };
    Vec frustum[8];
    for (u32 i = 0; i < 8; ++i)
    {
        MTX44MultVec(invProjMtx44, &cube[i], &frustum[i]);
    }

    // Calculate the splitting locations for the cascade
    f32 nearDist = -frustum[0].z;
    f32 farDist = -frustum[4].z;
    f32 num = (f32)split / (f32)NUMBER_CASCADE;
    f32 nearSplitDist = SPLIT_RATIO * nearDist * powf(SHADOW_RANGE / nearDist, num)
                        + (1.0f - SPLIT_RATIO) * (nearDist + num * (SHADOW_RANGE - nearDist));
    num = (f32)(split + 1) / (f32)NUMBER_CASCADE;
    f32 farSplitDist = SPLIT_RATIO * nearDist * powf(SHADOW_RANGE / nearDist, num)
                       + (1.0f - SPLIT_RATIO) * (nearDist + num * (SHADOW_RANGE - nearDist));

    // Calculate the cascade's center
    f32 r = SHADOW_RANGE / farDist;
    f32 n = (nearSplitDist - nearDist) / (SHADOW_RANGE - nearDist);
    n = (n < 0.0f) ? 0.0f : (n > 1.0f ? 1.0f : n);

    f32 f = (farSplitDist - nearDist) / (SHADOW_RANGE - nearDist);
    f = (f < 0.0f) ? 0.0f : (f > 1.0f ? 1.0f : f);

    n *= r;
    f *= r;
    Vec minPt = {FLT_MAX, FLT_MAX, FLT_MAX};
    Vec maxPt = {-FLT_MAX, -FLT_MAX, -FLT_MAX};
    for (u32 i = 0; i < 4; ++i)
    {
        Vec t0, t1, pt;
        VECScale(&frustum[i], &t0, (1.0f - n));
        VECScale(&frustum[i + 4], &t1, n);
        VECAdd(&t0, &t1, &pt);
        VecBounds(pt, minPt, maxPt);

        VECScale(&frustum[i], &t0, (1.0f - f));
        VECScale(&frustum[i + 4], &t1, f);
        VECAdd(&t0, &t1, &pt);

        VecBounds(pt, minPt, maxPt);
    }
    Vec dir;
    VECSubtract(&maxPt, &minPt, &dir);
    VECScale(&dir, &dir, 0.5f);
    Vec centerPt;
    VECAdd(&minPt, &dir, &centerPt);
    f32 radius = 0.5f * VECDistance(&maxPt, &minPt);
    Mtx44 invViewMtx44;
    MTX44Inverse(g_viewMtx44, invViewMtx44);
    Vec center;
    MTX44MultVec(invViewMtx44, &centerPt, &center);

    Vec lightDir;
    VECNormalize(&g_lightPt, &lightDir);

    // Round cascade's center coordinates to nearest texel
    Vec ax, ay, temp;
    VECCrossProduct(&g_lightPt, &g_lightUp, &temp);
    VECNormalize(&temp, &ax);
    VECCrossProduct(&ax, &g_lightPt, &temp);
    VECNormalize(&temp, &ay);
    f32 sx, sy, sz;
    f32 ux = SHADOWMAP_WIDTH / (2.0f * radius);
    f32 uy = SHADOWMAP_HEIGHT / (2.0f * radius);
    sx = ceilf(VECDotProduct(&center, &ax) * ux) / ux;
    sy = ceilf(VECDotProduct(&center, &ay) * uy) / uy;
    sz = VECDotProduct(&center, &lightDir);
    Vec target;
    VECScale(&ax, &target, sx);
    VECScale(&ay, &temp, sy);
    VECAdd(&target, &temp, &target);
    VECScale(&lightDir, &temp, sz);
    VECAdd(&target, &temp, &target);

    // Compute lookAt matrix
    Vec lightDist;
    VECScale(&lightDir, &lightDist, radius + SHADOW_CASTING_RANGE);
    Vec lightPt;
    VECAdd(&target, &lightDist, &lightPt);
    Mtx lightViewMtx34;
    MTXLookAt(lightViewMtx34, &lightPt, &g_lightUp, &target);
    MTX34To44(lightViewMtx34, lightViewMtx44);

    // Compute projection matrix
    MTXOrtho(lightProjMtx44, radius, -radius, -radius, radius,
             0.0f, 2.0f * radius + SHADOW_CASTING_RANGE);

    return farSplitDist - nearDist;
}

// Set material
static void SetMaterial( const DemoMaterialData* pMaterial, DEMOGfxTexture* pTextures, u32 textureLoc )
{
    // Set Texture
    DEMOCommandBuffer.SetTextureAndSampler( textureLoc, nn::gfx::ShaderStage_Pixel, pTextures[ pMaterial->textureIndex ].GetDescriptorSlot( 0 ), g_samplerModelSlot );

    // Rendering state settings
    switch ( pMaterial->type )
    {
    case DEMO_MATERIAL_OPAQUE:
        //Translucent material
        DEMOCommandBuffer.SetBlendState( &material );
        break;
    case DEMO_MATERIAL_OVERLAP:
        //Translucent material (no Z-write)
        DEMOCommandBuffer.SetBlendState( &materialAlpha );
        break;
    case DEMO_MATERIAL_TRANSPARENT:
        //Translucent material
        DEMOCommandBuffer.SetBlendState( &materialAlpha );
        break;
    case DEMO_MATERIAL_ALPHA_TO_COVERAGE:
        //Coverage mask material
        DEMOCommandBuffer.SetBlendState( &materialAlpha );
        break;
    default:
        ASSERT(false && "Illegal material type");
        break;
    }
}

// Render the model data
static void DrawModel( const DemoModelData* pModelData, u32 textureLoc, bool bSkipMaterial )
{
    s32 currentMaterialIndex = -1;

    for ( s32 i = 0; i < pModelData->header.meshCount; i++ )
    {
        const DemoMeshData* pMesh = &pModelData->meshes[ i ];

        // Set material
        if ( bSkipMaterial && currentMaterialIndex != pMesh->materialIndex )
        {
            currentMaterialIndex = pMesh->materialIndex;

            SetMaterial( &pModelData->materials[ pMesh->materialIndex ],
                pModelData->pTextures, textureLoc );
        }

        // Set attribute buffers
        nn::gfx::GpuAddress gpuAddress = pModelData->attributeBuffer.buffer.gpuAddress;
        gpuAddress.Offset( pMesh->attributeOffset );
        DEMOCommandBuffer.SetVertexBuffer( 0, gpuAddress, pMesh->attributeStride, pModelData->attributeBuffer.size - pMesh->attributeOffset );

        // Draw Triangle. pMesh->indexOffset is the index offset, but we already have that in the index buffer indexView[]
        gpuAddress = pModelData->indexBuffer.buffer.gpuAddress;
        gpuAddress.Offset( pMesh->indexOffset );
        DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList,
            pModelData->indexBuffer.stride == 2 ? nn::gfx::IndexFormat_Uint16 : nn::gfx::IndexFormat_Uint32,
            gpuAddress, pMesh->indexCount, 0 );
    }
}

static void DrawSceneFromLight()
{
    DEMOCommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_ColorBuffer | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_DepthStencil | nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_VertexBuffer);
    for ( u32 i = 0; i < NUMBER_CASCADE; ++i )
    {
        DEMOCommandBuffer.ClearDepthStencil( &g_shadowmapDepthView[ i ], 1.0f, 0, nn::gfx::DepthStencilClearMode_Depth, NULL );

#if NN_GFX_IS_TARGET_GX
        GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
        DEMOCommandBuffer.SetRenderTargets( 0, NULL, &g_shadowmapDepthView[ i ] );
        DEMOCommandBuffer.SetViewportScissorState( &shadowViewport );
        DEMOCommandBuffer.SetPipeline( &depthPipeline.pipeline );

        struct vertexUniforms * pMatrices;

        pMatrices = depthTempleVertexUniformBuffer[ i ].Map< struct vertexUniforms >();
#if NN_GFX_IS_TARGET_GX
        GX2EndianSwap( pMatrices, sizeof( *pMatrices ) );
#endif
        ComputeLightMatrix( pMatrices->viewMtx44, pMatrices->projMtx44, i );

        MTX44Identity(pMatrices->modelMtx44);
        pMatrices->modelMtx44[ 1 ][ 3 ] += g_offsetY;
#if NN_GFX_IS_TARGET_GX
        GX2EndianSwap( pMatrices, sizeof( *pMatrices ) );
#endif
        depthTempleVertexUniformBuffer[i].Unmap();
        DEMOCommandBuffer.SetConstantBuffer( depthVertexUniformBlock, nn::gfx::ShaderStage_Vertex, depthTempleVertexUniformBuffer[i].gpuAddress, depthTempleVertexUniformBuffer[i].size );

        DEMOCommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_ConstantBuffer );

        DrawModel(&g_modelTemple, depthPixelSamplerLoc, false);

        pMatrices = depthTerrainVertexUniformBuffer[ i ].Map< struct vertexUniforms >();
#if NN_GFX_IS_TARGET_GX
        GX2EndianSwap( pMatrices, sizeof( *pMatrices ) );
#endif
        ComputeLightMatrix( pMatrices->viewMtx44, pMatrices->projMtx44, i );

        MTX44Identity( pMatrices->modelMtx44 );
        pMatrices->modelMtx44[ 0 ][ 0 ] *= 2.0f;
        pMatrices->modelMtx44[ 2 ][ 2 ] *= 2.0f;
        pMatrices->modelMtx44[ 1 ][ 3 ] += -1.0f;
#if NN_GFX_IS_TARGET_GX
        GX2EndianSwap( pMatrices, sizeof( *pMatrices ) );
#endif
        depthTerrainVertexUniformBuffer[i].Unmap();
        DEMOCommandBuffer.SetConstantBuffer( depthVertexUniformBlock, nn::gfx::ShaderStage_Vertex, depthTerrainVertexUniformBuffer[i].gpuAddress, depthTerrainVertexUniformBuffer[i].size );

        DEMOCommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_ConstantBuffer );

        DrawModel( &g_modelTerrain, depthPixelSamplerLoc, false );

#if 0 // FIXME: NN_GFX_IS_TARGET_GX
        GX2UTSetExpandDepthState( GX2_TRUE );
        GX2UTExpandDepthBufferOp( reinterpret_cast< GX2DepthBuffer* >( &g_shadowmapDepthView[ i ].ToData()->gx2DepthBuffer ) );
        GX2UTSetExpandDepthState( GX2_FALSE );
#endif
    }
    DEMOCommandBuffer.FlushMemory( nn::gfx::GpuAccess_DepthStencil );
    DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_ConstantBuffer );
}

static void DrawSceneFromCamera()
{
    nn::gfx::ColorTargetView* pCurrentScanBuffer = DEMOGetColorBufferView();
    DEMOCommandBuffer.ClearColor( pCurrentScanBuffer, 0.2f, 0.65f, 0.9f, 1.0f, NULL );
    DEMOCommandBuffer.ClearDepthStencil( &DEMODepthBufferView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    DEMOGfxSetDefaultRenderTarget();
    DEMOCommandBuffer.SetViewportScissorState( &fullViewport );
    DEMOCommandBuffer.SetPipeline( &shadowPipeline.pipeline );
    DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_ColorBuffer | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_VertexBuffer);

    struct shadowPixelUniforms * pUniforms = shadowPixelUniformBuffer.Map< struct shadowPixelUniforms >();
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pUniforms, sizeof( *pUniforms ) );
#endif
    Mtx44 tex;
    MTX44Identity( tex );
    tex[ 0 ][ 0 ] = 0.5f;
    tex[ 1 ][ 1 ] = -0.5f;
    tex[ 2 ][ 2 ] = 0.5f;
    tex[ 0 ][ 3 ] = 0.5f;
    tex[ 1 ][ 3 ] = 0.5f;
    tex[ 2 ][ 3 ] = 0.5f;
    for ( u32 i = 0; i < NUMBER_CASCADE; ++i )
    {
        Mtx44 lightViewMtx44, lightProjMtx44, temp;
        pUniforms->splitDistance[ i ] = ComputeLightMatrix( lightViewMtx44, lightProjMtx44, i );
        MTX44Concat( tex, lightProjMtx44, temp );
        MTX44Concat( temp, lightViewMtx44, pUniforms->lightMtx[ i ] );
    }

    pUniforms->depthBias[ 0 ] = -0.0015f;
    pUniforms->depthBias[ 1 ] = -0.002f;
    pUniforms->depthBias[ 2 ] = -0.0025f;
    pUniforms->depthBias[ 3 ] = -0.003f;

    VECSubtract( &g_lightPt, &g_origin, &pUniforms->lightDir );
    memcpy( &pUniforms->eyePos, &g_camPt, sizeof( Vec ) );

#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pUniforms, sizeof( *pUniforms ) );
#endif
    shadowPixelUniformBuffer.Unmap( );
    DEMOCommandBuffer.SetConstantBuffer( shadowPixelUniformBlock, nn::gfx::ShaderStage_Pixel, shadowPixelUniformBuffer.gpuAddress, shadowPixelUniformBuffer.size );

    // Set Shadow Map Texture
    DEMOCommandBuffer.SetTextureAndSampler(shadowPixelMapLoc, nn::gfx::ShaderStage_Pixel, g_shadowmapSlot, g_samplerShadowSlot);

    struct vertexUniforms * pMatrices = shadowTempleVertexUniformBuffer.Map< struct vertexUniforms >();
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pMatrices, sizeof( *pMatrices ) );
#endif
    memcpy( &pMatrices->viewMtx44, &g_viewMtx44, sizeof( Mtx44 ) );
    memcpy( &pMatrices->projMtx44, &g_projMtx44, sizeof( Mtx44 ) );

    MTX44Identity(pMatrices->modelMtx44);
    pMatrices->modelMtx44[ 1 ][ 3 ] += g_offsetY;
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pMatrices, sizeof( *pMatrices ) );
#endif
    shadowTempleVertexUniformBuffer.Unmap();
    DEMOCommandBuffer.SetConstantBuffer( shadowVertexUniformBlock, nn::gfx::ShaderStage_Vertex, shadowTempleVertexUniformBuffer.gpuAddress, shadowTempleVertexUniformBuffer.size );

    DEMOCommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_ConstantBuffer );

    DrawModel(&g_modelTemple, shadowPixelSamplerLoc, true);

    pMatrices = shadowTerrainVertexUniformBuffer.Map< struct vertexUniforms >();
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pMatrices, sizeof( *pMatrices ) );
#endif
    memcpy( &pMatrices->viewMtx44, &g_viewMtx44, sizeof( Mtx44 ) );
    memcpy( &pMatrices->projMtx44, &g_projMtx44, sizeof( Mtx44 ) );

    MTX44Identity( pMatrices->modelMtx44 );
    pMatrices->modelMtx44[ 0 ][ 0 ] *= 2.0f;
    pMatrices->modelMtx44[ 2 ][ 2 ] *= 2.0f;
    pMatrices->modelMtx44[ 1 ][ 3 ] += -1.0f;
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pMatrices, sizeof( *pMatrices ) );
#endif
    shadowTerrainVertexUniformBuffer.Unmap();
    DEMOCommandBuffer.SetConstantBuffer( shadowVertexUniformBlock, nn::gfx::ShaderStage_Vertex, shadowTerrainVertexUniformBuffer.gpuAddress, shadowTerrainVertexUniformBuffer.size );

    DEMOCommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_ConstantBuffer );

    DrawModel( &g_modelTerrain, shadowPixelSamplerLoc, true );
}

static void DrawViewer()
{
    DEMOCommandBuffer.SetPipeline(&viewerPipeline.pipeline);
    DEMOCommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_ColorBuffer | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_DepthStencil | nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_VertexBuffer);
    DEMOCommandBuffer.SetVertexBuffer(0, viewerPositionBuffer.gpuAddress, sizeof(float) * 3, viewerPositionBuffer.size );
    DEMOCommandBuffer.SetVertexBuffer(1, viewerTexCoordBuffer.gpuAddress, sizeof(float) * 2, viewerTexCoordBuffer.size );

    for (u32 i = 0; i < 4; ++i)
    {
        DEMOCommandBuffer.SetViewportScissorState( &viewerViewport[ i ] );
        DEMOCommandBuffer.SetTextureAndSampler(viewerTextureLoc, nn::gfx::ShaderStage_Pixel, g_shadowmapSlot, g_samplerViewSMSlot);
        DEMOCommandBuffer.SetConstantBuffer(viewerPixelUniformBlock, nn::gfx::ShaderStage_Pixel, viewerPixelUniformBuffer[i].gpuAddress, viewerPixelUniformBuffer[i].size);
        DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32,
            viewerIndexBuffer.gpuAddress, 6, 0);
    }
}

static void PrintInfo()
{
    DEMOCommandBuffer.SetViewportScissorState( &fullViewport );

    DEMOFontPrintf(5,2, "Shadow : %.1f fps", 1.0f / g_dtime > 1.0e-7 ? (1.0f / g_dtime) : 1.0e-7);
    DEMOFontPrintf(5,3, "Camera : %g,%g,%g", g_camPt.x, g_camPt.y, g_camPt.z);
    DEMOFontPrintf(5,4, "Target : %g,%g,%g", g_objPt.x, g_objPt.y, g_objPt.z);
    DEMOFontPrintf(5,5, "Move Camera (L Stick/R Stick)");
    DEMOFontPrintf(5,6, "Rotate Light Direction (up/down/left/right)");
    DEMOFontPrintf(5,7, "Show All Shadow Maps (b)");
}

// The draw function for the rendering portions of this sample.
static void DrawScene()
{
    DEMOGfxBeforeRender();

    // Draw the scene from the light view
    DrawSceneFromLight();

    DrawSceneFromCamera();
    if (g_viewShadowMaps)
    {
        DrawViewer();
    }

    if (s_firstFrame == false)
    {
        PrintInfo();
    }
    else
    {
        s_firstFrame = false;
    }

    DEMOGfxDoneRender();
}

static void ProcessPad()
{
    DEMOPadRead();
    u16 button = DEMOPadGetButton(0);
    u16 buttonDown = DEMOPadGetButtonDown(0);

    if (DEMO_PAD_BUTTON_B & buttonDown)
    {
        g_viewShadowMaps = ~g_viewShadowMaps;
    }

    {
        // L Pad rotates light direction
        static f32 v = 1.0f;
        f32 sx = 0.0f;
        f32 sy = 0.0f;
        sx += (DEMO_PAD_BUTTON_LEFT & button) ? v : 0.0f;
        sx -= (DEMO_PAD_BUTTON_RIGHT & button) ? v : 0.0f;
        sy += (DEMO_PAD_BUTTON_UP & button) ? v : 0.0f;
        sy -= (DEMO_PAD_BUTTON_DOWN & button) ? v : 0.0f;

        Mtx44 rot;
        MTX44RotAxisRad(rot, &g_lightUp, MTXDegToRad(sx));
        MTX44MultVecSR(rot, &g_lightPt, &g_lightPt);

        Vec ld;
        VECNormalize(&g_lightPt, &ld);
        f32 c = VECDotProduct(&g_lightUp, &ld);
        if ((0.25f < c && sy > 0.0f) || (c < 0.9f && sy < 0.0f))
        {
            Vec ax;
            VECCrossProduct(&g_lightUp, &g_lightPt, &ax);
            MTX44RotAxisRad(rot, &ax, MTXDegToRad(sy));
            MTX44MultVecSR(rot, &g_lightPt, &g_lightPt);
        }
    }

    {
        // L Stick translates camera
        f32 sx = (f32)DEMOPadGetStickX(0) / 255.0f;
        f32 sy = (f32)DEMOPadGetStickY(0) / 255.0f;

        Vec vec, tempVec;
        tempVec.x =  sx * 100.0f;
        tempVec.y = 0.0f;
        tempVec.z = -sy * 100.0f;

        Mtx44 inv;
        MTX44Inverse(g_viewMtx44, inv);
        MTX44MultVecSR(inv, &tempVec, &vec);
        Vec tempCamLoc = g_camPt;
        Vec tempObjPt = g_objPt;
        VECAdd(&vec, &tempCamLoc, &g_camPt);
        VECAdd(&vec, &tempObjPt, &g_objPt);

        InitCamera(g_projMtx44, g_viewMtx44);
    }

    {
        // R Stick rotates camera
        f32 sx = (f32)DEMOPadGetSubStickX(0) / 255.0f;
        f32 sy = (f32)DEMOPadGetSubStickY(0) / 255.0f;

        Vec eyev, tempEyev;
        VECSubtract(&g_objPt, &g_camPt, &eyev);
        Vec wupv =  {0.0f, 1.0f, 0.0f};
        VECCrossProduct(&eyev, &g_up, &tempEyev);
        VECNormalize(&tempEyev, &eyev);

        Mtx44 rot,rot0,rot1;
        MTX44RotAxisRad(rot0, &eyev, MTXDegToRad( sy * 5.0f));
        MTX44RotAxisRad(rot1, &wupv, MTXDegToRad(-sx * 5.0f));

        MTX44Concat(rot0, rot1, rot);

        Vec camv, tempCamv;
        VECSubtract(&g_objPt, &g_camPt, &tempCamv);
        MTX44MultVecSR(rot, &tempCamv, &camv);

        VECAdd(&camv, &g_camPt, &g_objPt);
        Vec tempUp = g_up;
        MTX44MultVecSR(rot, &tempUp, &g_up);

        InitCamera(g_projMtx44, g_viewMtx44);
    }
}

//extern "C" void nnMain()
TEST(GfxShadow, Run)
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    DEMOInit();
    DEMOTestInit(argc, argv);
    DEMOGfxInit(argc, argv);
    DEMOFontInit();
    DEMOTestIsUseHlslccGlsl() ? g_ShaderFileIdx = 1 : g_ShaderFileIdx = 0;

    InitScene();
    s_firstFrame = true;
    InitViewer(&viewerPipeline.shaders);

    while (DEMOIsRunning())
    {
        g_dtime = OSTicksToMilliseconds(OSGetTime() - g_lastTime) / 1000.0f;
        g_lastTime = OSGetTime();

        ProcessPad();
        DrawScene();
    }

    DEMOFreeModelData(&g_modelTemple);
    DEMOFreeModelData(&g_modelTerrain);

    for (int i = 0; i < NUMBER_CASCADE; i++)
    {
        depthTempleVertexUniformBuffer[i].Finalize();

        depthTerrainVertexUniformBuffer[i].Finalize();
    }
    shadowTempleVertexUniformBuffer.Finalize();
    shadowTerrainVertexUniformBuffer.Finalize();
    shadowPixelUniformBuffer.Finalize();

    depthPipeline.Finalize( &DEMODevice );
    shadowPipeline.Finalize( &DEMODevice );

    material.Finalize( &DEMODevice );
    materialAlpha.Finalize( &DEMODevice );

    viewerTexCoordBuffer.Finalize();
    viewerPositionBuffer.Finalize();
    viewerIndexBuffer.Finalize();

    for (int i = 0; i < NUMBER_CASCADE; i++)
    {
        viewerViewport[i].Finalize(&DEMODevice);
        viewerPixelUniformBuffer[i].Finalize();
        g_shadowmapDepthView[ i ].Finalize( &DEMODevice );
    }
    g_shadowmapView.Finalize(&DEMODevice);
    g_shadowmap.Finalize(&DEMODevice);
    g_shadowmapData->Finalize();
    delete g_shadowmapData;
    shadowViewport.Finalize(&DEMODevice);
    fullViewport.Finalize(&DEMODevice);

    g_samplerViewSM.Finalize(&DEMODevice);
    g_samplerModel.Finalize(&DEMODevice);
    g_samplerShadow.Finalize(&DEMODevice);

    viewerPipeline.Finalize(&DEMODevice);

    DEMOFontShutdown();
    DEMOTestShutdown();
    DEMOGfxShutdown();
    DEMOShutdown();

    SUCCEED();
}
