﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>
#include <cmath>

#include <gfx/demo.h>

#include "runtimeBC1gen.h"

#include <nnt.h>
#include <nnt/nnt_Argument.h>

u32 RENDERWidth  = 1024;
u32 RENDERHeight = 1024;

// Position for a single triangle.
static DEMO_F32x2 FULLSCREEN_POSITIONS[] =
{
    { { { -1.0f, -1.0f } } },
    { { { 1.0f, -1.0f } } },
    { { { 1.0f,  1.0f } } },
    { { { -1.0f, 1.0f } } }
};

static u32 FULLSCREEN_INDEX[] = { 0, 1, 3, 2 };

//static const int NUM_ATTRIB1 (1)
static const int NUM_VERTEX = 4;

static const int POSITION_BUFFER_INDEX = 0;
static const int POSITION_ATTRIBUTE_LOCATION = 0;

static const int BC_BLOCK_SIZE = 4;

static DEMOGfxPipeline g_RuntimeBC1GenPipeline;
static DEMOGfxPipeline g_VisualizationPipeline;

// Compressed texture (computed by GPU at runtime)
DEMOGfxMemPool*             g_pBC1TexturePool;
nn::gfx::Texture g_BC1Texture;
nn::gfx::TextureView        g_BC1TextureView;
nn::gfx::DescriptorSlot     g_BC1TextureSlot;
nn::gfx::ColorTargetView    g_AliasBC1ColorBuffer;

nn::gfx::ViewportScissorState g_BC1Viewport;
void *g_pBC1ViewportData;

static u32              g_BC1SamplerLoc;
nn::gfx::Sampler        g_BC1Sampler;
nn::gfx::DescriptorSlot g_BC1SamplerSlot;
static u32              g_VisualizationSamplerLoc;
nn::gfx::Sampler        g_VisualizationSampler;
nn::gfx::DescriptorSlot g_VisualizationSamplerSlot;

DEMOGfxBuffer g_VertexBuffer;
DEMOGfxBuffer g_IndexBuffer;

typedef struct _VSUniforms
{
    Vec2   u_scale;
    Vec2   u_shift;
} VSUniforms;
static u32 g_BC1UniformLoc;
static u32 g_VisualizationUniformLoc;
static DEMOGfxBuffer g_BC1Uniforms;
static DEMOGfxBuffer g_UniformsOne;
static DEMOGfxBuffer g_UniformsTwo;

static DEMOGfxGpuTimestamp s_startRender;
static DEMOGfxGpuTimestamp s_endRender;
static uint64_t s_startRenderTicks;
static uint64_t s_endRenderTicks;

void BC1CompressInit()
{
    const char *const SHADER_FILE = "shaders/runtimeBC1gen/runtimeBC1gen";

    // Load shader binary to memory
    DEMOGfxLoadShadersFromFile( &g_RuntimeBC1GenPipeline.shaders, 0, SHADER_FILE );

    // Uniform Location Lookup
    g_BC1UniformLoc = g_RuntimeBC1GenPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "VSUniforms" );
    g_BC1SamplerLoc = g_RuntimeBC1GenPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_texture" );

    // position setup
    DEMOGfxInitShaderAttribute( &g_RuntimeBC1GenPipeline.shaders, "a_position", POSITION_BUFFER_INDEX, POSITION_ATTRIBUTE_LOCATION, nn::gfx::AttributeFormat_32_32_Float );
    DEMOGfxInitShaderVertexBuffer( &g_RuntimeBC1GenPipeline.shaders, POSITION_BUFFER_INDEX, sizeof( DEMO_F32x2 ), 0 );

    // these can be shared w/ Visualization Pass
    // Create Vertex Buffer
    g_VertexBuffer.Initialize( sizeof( FULLSCREEN_POSITIONS ), FULLSCREEN_POSITIONS, nn::gfx::GpuAccess_VertexBuffer, 0 );

    // Create Index Buffer
    g_IndexBuffer.Initialize( sizeof ( FULLSCREEN_INDEX ), FULLSCREEN_INDEX, nn::gfx::GpuAccess_IndexBuffer, 0 );

    // Create BC1 Texture
    DEMOGfxSetupTextureBuffer( &g_BC1Texture, &g_BC1TextureView, &g_BC1TextureSlot, NULL, NULL,
        &g_pBC1TexturePool, RENDERWidth, RENDERHeight, 1, 1, nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_Bc1_Unorm, nn::gfx::DepthStencilFetchMode_DepthComponent, 0 );

    // Create Color Target Alias of BC1 texture
    DEMOGfxSetupColorView( &g_BC1Texture, &g_AliasBC1ColorBuffer, nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R16_G16_B16_A16_Uint );

#ifdef FIXME_These_would_be_nice
    ASSERT(BC1Texture.surface.imageSize == BC1ColorBuffer.surface.imageSize);
    ASSERT(BC1Texture.surface.tileMode == BC1ColorBuffer.surface.tileMode);

    OSReport("BC1Texture.surface.imageSize %d\n", BC1Texture.surface.imageSize);
    OSReport("BC1ColorBuffer.surface.imageSize %d\n", BC1ColorBuffer.surface.imageSize);
#endif

    // texture init
    DEMOGfxInitSampler( &g_BC1Sampler, &g_BC1SamplerSlot, nn::gfx::TextureAddressMode_ClampToEdge, nn::gfx::FilterMode_Comparison_MinLinear_MagLinear_MipPoint, nn::gfx::ComparisonFunction_Always );

    f32 scaleUniform[ ] = { 1.0f, 1.0f };
    f32 shiftUniform[ ] = { 0.0f, 0.0f };

    // Setup the uniform buffer
    {
        g_BC1Uniforms.Initialize( sizeof( VSUniforms ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
        VSUniforms* pUniforms = g_BC1Uniforms.Map< VSUniforms >();
        memcpy( &pUniforms->u_scale, scaleUniform, sizeof( Vec2 ) );
        memcpy( &pUniforms->u_shift, shiftUniform, sizeof( Vec2 ) );
#ifdef CAFE
        GX2EndianSwap( pUniforms, sizeof( VSUniforms ) );
#endif
        g_BC1Uniforms.Unmap();
    }

    DEMOGfxSetViewportScissorState( &g_BC1Viewport, &g_pBC1ViewportData, 0.0f, 0.0f,
        static_cast< float >( RENDERWidth / BC_BLOCK_SIZE ), static_cast< float >( RENDERHeight / BC_BLOCK_SIZE ), 0.0f, 1.0f,
        static_cast< float >( RENDERHeight / BC_BLOCK_SIZE ), true );

    g_RuntimeBC1GenPipeline.SetDefaults();

    g_RuntimeBC1GenPipeline.colorTargetStateCount = 1;
    g_RuntimeBC1GenPipeline.blendTargetStateCount = 1;
    g_RuntimeBC1GenPipeline.blendTargetStateInfoArray[ 0 ].SetDefault();
    g_RuntimeBC1GenPipeline.colorTargetStateInfoArray[ 0 ].SetDefault();
    g_RuntimeBC1GenPipeline.colorTargetStateInfoArray[ 0 ].SetFormat( nn::gfx::ImageFormat_R16_G16_B16_A16_Uint );

    // Setup the pipeline
    g_RuntimeBC1GenPipeline.depthStencilStateInfo.SetDepthComparisonFunction( nn::gfx::ComparisonFunction_Always );
    g_RuntimeBC1GenPipeline.depthStencilStateInfo.SetDepthTestEnabled( false );
    g_RuntimeBC1GenPipeline.depthStencilStateInfo.SetDepthWriteEnabled( false );

    g_RuntimeBC1GenPipeline.Initialize( &DEMODevice );
}

void BC1CompressShutdown()
{
    g_RuntimeBC1GenPipeline.Finalize( &DEMODevice );

    g_BC1Sampler.Finalize(&DEMODevice);
    g_BC1Viewport.Finalize(&DEMODevice);

    g_AliasBC1ColorBuffer.Finalize( &DEMODevice );
    g_BC1TextureView.Finalize(&DEMODevice);
    g_BC1Texture.Finalize(&DEMODevice);
    g_BC1Uniforms.Finalize();
    g_pBC1TexturePool->Finalize();
    delete g_pBC1TexturePool;
}

void UpdateTimestamps()
{
    DEMOCommandBuffer.Begin(); // Must begin the command buffer since GetTimestampResult calls End()
    s_startRenderTicks = s_startRender.GetTimestampResult();

    s_endRenderTicks = s_endRender.GetTimestampResult();
    DEMOCommandBuffer.End(); // Must End the command buffer since GetTimestampResult calls Begin() on exit
}

void BC1CompressRun()
{
    nn::gfx::ColorTargetView* renderTargets[ ] = { &g_AliasBC1ColorBuffer };
    DEMOCommandBuffer.SetRenderTargets( 1, renderTargets, NULL );

    // Setup Viewport/Stencil.
    DEMOCommandBuffer.SetViewportScissorState( &g_BC1Viewport );

    DEMOCommandBuffer.SetPipeline( &g_RuntimeBC1GenPipeline.pipeline );

    DEMOCommandBuffer.SetVertexBuffer( POSITION_BUFFER_INDEX, g_VertexBuffer.gpuAddress, sizeof( DEMO_F32x2 ), g_VertexBuffer.size );

    DEMOCommandBuffer.SetConstantBuffer( g_BC1UniformLoc, nn::gfx::ShaderStage_Vertex, g_BC1Uniforms.gpuAddress, g_BC1Uniforms.size );

    DEMOCommandBuffer.SetTextureAndSampler( g_BC1SamplerLoc, nn::gfx::ShaderStage_Pixel,
        g_R2TTextureSlot, g_R2TSamplerSlot );

    s_startRender.QueryTimestamp();
    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleStrip, nn::gfx::IndexFormat_Uint32, g_IndexBuffer.gpuAddress, NUM_VERTEX, 0 );
    s_endRender.QueryTimestamp();


    // Make sure the input texture has been flushed
    DEMOCommandBuffer.SetTextureStateTransition( &g_BC1Texture, NULL, nn::gfx::TextureState_ColorTarget, nn::gfx::ShaderStageBit_All,
        nn::gfx::TextureState_ShaderRead, nn::gfx::ShaderStageBit_All );
}

void VisualizeResultInit()
{
    const char *const SHADER_FILE = "shaders/runtimeBC1gen/fullscreentex2d";

    // Load shader binary to memory
    DEMOGfxLoadShadersFromFile( &g_VisualizationPipeline.shaders, 0, SHADER_FILE );

    // Uniform Location Lookup
    g_VisualizationUniformLoc = g_VisualizationPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "VSUniforms" );
    g_VisualizationSamplerLoc = g_VisualizationPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_texture" );

    // position setup
    DEMOGfxInitShaderAttribute( &g_VisualizationPipeline.shaders, "a_position", POSITION_BUFFER_INDEX, POSITION_ATTRIBUTE_LOCATION, nn::gfx::AttributeFormat_32_32_Float );
    DEMOGfxInitShaderVertexBuffer( &g_VisualizationPipeline.shaders, POSITION_BUFFER_INDEX, sizeof( DEMO_F32x2 ), 0 );

    f32 scaleUniform[ ] = { 0.3f, 0.5f };
    f32 shiftUniform[ ] = { -0.4f, 0.0f };
    // setup UniformsOne
    // Set up for first draw
    // Setup the uniform buffer
    {
        g_UniformsOne.Initialize( sizeof( VSUniforms ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
        VSUniforms* pUniforms = g_UniformsOne.Map< VSUniforms >();
        memcpy( &pUniforms->u_scale, scaleUniform, sizeof( Vec2 ) );
        memcpy( &pUniforms->u_shift, shiftUniform, sizeof( Vec2 ) );
#ifdef CAFE
        GX2EndianSwap( pUniforms, sizeof( VSUniforms ) );
#endif
        g_UniformsOne.Unmap();
    }
    // Set up for second draw
    shiftUniform[ 0 ] = 0.4f;
    // Setup the uniform buffer
    {
        g_UniformsTwo.Initialize( sizeof( VSUniforms ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
        VSUniforms* pUniforms = g_UniformsTwo.Map< VSUniforms >();
        memcpy( &pUniforms->u_scale, scaleUniform, sizeof( Vec2 ) );
        memcpy( &pUniforms->u_shift, shiftUniform, sizeof( Vec2 ) );
#ifdef CAFE
        GX2EndianSwap( pUniforms, sizeof( VSUniforms ) );
#endif
        g_UniformsTwo.Unmap();
    }

    g_VisualizationPipeline.SetDefaults();

    g_VisualizationPipeline.colorTargetStateCount = 1;
    g_VisualizationPipeline.blendTargetStateCount = 1;
    g_VisualizationPipeline.blendTargetStateInfoArray[ 0 ].SetDefault();
    g_VisualizationPipeline.colorTargetStateInfoArray[ 0 ].SetDefault();
    g_VisualizationPipeline.colorTargetStateInfoArray[ 0 ].SetFormat( DEMOColorBufferInfo.GetImageFormat() );

    // Setup the pipeline
    g_VisualizationPipeline.depthStencilStateInfo.SetDepthComparisonFunction( nn::gfx::ComparisonFunction_Less );
    g_VisualizationPipeline.Initialize( &DEMODevice );
}

void VisualizeResultShutdown()
{
    // Free shaders/pipeline
    g_VisualizationPipeline.Finalize( &DEMODevice );

    // Free the vertex/index buffer
    g_VertexBuffer.Finalize();
    g_IndexBuffer.Finalize();

    // Free the 2 uniform blocks
    g_UniformsOne.Finalize();
    g_UniformsTwo.Finalize();
}

void VisualizeResultRun()
{
    // Clear buffers
    nn::gfx::ColorTargetView* pScanBufferView = DEMOGetColorBufferView();

    // Clear buffers
    DEMOCommandBuffer.ClearColor( pScanBufferView, 0.3f, 0.6f, 0.9f, 1.0f, NULL );
    DEMOCommandBuffer.ClearDepthStencil( &DEMODepthBufferView, 1.0, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

    DEMOGfxSetDefaultRenderTarget();

    // Setup Viewport/Stencil
    DEMOGfxSetDefaultViewportScissor();

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif

    DEMOCommandBuffer.SetPipeline( &g_VisualizationPipeline.pipeline );

    DEMOCommandBuffer.SetVertexBuffer( POSITION_BUFFER_INDEX, g_VertexBuffer.gpuAddress, sizeof( DEMO_F32x2 ), g_VertexBuffer.size );

    DEMOCommandBuffer.SetConstantBuffer( g_VisualizationUniformLoc, nn::gfx::ShaderStage_Vertex, g_UniformsOne.gpuAddress, g_UniformsOne.size );

    DEMOCommandBuffer.SetTextureAndSampler( g_VisualizationSamplerLoc, nn::gfx::ShaderStage_Pixel,
        g_R2TTextureSlot, g_R2TSamplerSlot );

    // Make sure the input texture has been flushed
    DEMOCommandBuffer.SetTextureStateTransition( &g_R2TTexture, NULL, nn::gfx::TextureState_ColorTarget, nn::gfx::ShaderStageBit_All,
        nn::gfx::TextureState_ShaderRead, nn::gfx::ShaderStageBit_All );

    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleStrip, nn::gfx::IndexFormat_Uint32, g_IndexBuffer.gpuAddress, NUM_VERTEX, 0 );

    DEMOCommandBuffer.SetConstantBuffer( g_VisualizationUniformLoc, nn::gfx::ShaderStage_Vertex, g_UniformsTwo.gpuAddress, g_UniformsTwo.size );

    DEMOCommandBuffer.SetTextureAndSampler( g_VisualizationSamplerLoc, nn::gfx::ShaderStage_Pixel,
        g_BC1TextureSlot, g_BC1SamplerSlot );

    // Make sure the input texture has been flushed
    DEMOCommandBuffer.SetTextureStateTransition( &g_BC1Texture, NULL, nn::gfx::TextureState_ColorTarget, nn::gfx::ShaderStageBit_All,
        nn::gfx::TextureState_ShaderRead, nn::gfx::ShaderStageBit_All );

    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleStrip, nn::gfx::IndexFormat_Uint32, g_IndexBuffer.gpuAddress, NUM_VERTEX, 0 );

    //DEMOFontSetContextState();

    //DEMOFontSetColor(1.0f, 0.0f, 0.0f, 1.0f);
}

// Setting Up Pad Information
static void SetPadInfo()
{
    u16 buttonDown = DEMOPadGetButtonDown(0);

    if (DEMO_PAD_BUTTON_LEFT & buttonDown) { }

    if (DEMO_PAD_BUTTON_RIGHT & buttonDown) { }

    if (DEMO_PAD_BUTTON_A & buttonDown) {
        if(RENDERScene == 0)
            RENDERScene = 1;
        else
            RENDERScene = 0;
    }
}

//extern "C" void nnMain()
TEST(GfxRuntimeBC1Gen, Run)
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    DEMOInit();
    DEMOTestInit(argc, argv);
    DEMOGfxInit(argc, argv);
    DEMOFontInit();

    SceneInit();

    s_startRender.Initialize();
    s_endRender.Initialize();

    BC1CompressInit();
    VisualizeResultInit();

    while (DEMOIsRunning())
    {
        DEMOGfxBeforeRender();

        DEMOPadRead();
        SetPadInfo();

        SceneDraw();

        BC1CompressRun();

        VisualizeResultRun();

        DEMOFontSetColor(0.0f, 0.0f, 0.0f, 1.0f);
        DEMOFontPrintf(3, 1, "Render To Texture X :%d, Y : %d", RENDERWidth, RENDERHeight);
        DEMOFontPrintf(3, 2, "Rendering %d pixels at %d micro sec", RENDERWidth * RENDERHeight, (u32)DEMOGfxGpuTimestamp::TicksToMicroseconds(s_endRenderTicks - s_startRenderTicks));
        DEMOFontPrintf(10, 5, "Left : Rendered Image");
        DEMOFontPrintf(33, 5, "Right : Compressed BC1 Image");

        DEMOFontPrintf(3, 19, "Press A to change rendered scene");

        // Draw Infomation
        DEMOGfxDoneRender();

        // Since updating timestamps require a sync allow the time to be behind a frame.
        UpdateTimestamps();
    }

    s_startRender.Finalize();
    s_endRender.Finalize();

    BC1CompressShutdown();
    VisualizeResultShutdown();
    SceneShutdown();

    // Free various resourc
    DEMOFontShutdown();
    DEMOTestShutdown();
    DEMOGfxShutdown();
    DEMOShutdown();

    SUCCEED();
}
