﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>
#include <cmath>

#include <gfx/demo.h>

#include "runtimeBC1gen.h"

nn::gfx::ColorTargetView  g_R2TColorBuffer;
nn::gfx::Texture          g_R2TTexture;
nn::gfx::TextureView      g_R2TTextureView;
nn::gfx::DescriptorSlot   g_R2TTextureSlot;
DEMOGfxMemPool* g_pR2TTexturePool;

nn::gfx::DepthStencilView  g_R2TDepthBuffer;
nn::gfx::Texture          g_R2TDepthTexture;
nn::gfx::TextureView      g_R2TDepthTextureView;
nn::gfx::DescriptorSlot   g_R2TDepthTextureSlot;
DEMOGfxMemPool* g_pR2TDepthTexturePool;

nn::gfx::ViewportScissorState g_R2TViewport;
void *g_pR2TViewportData;

nn::gfx::Sampler g_R2TSampler;
nn::gfx::DescriptorSlot g_R2TSamplerSlot;

u8 RENDERScene = 0;

// ----- Lattice Scene

// ----- GX2 Shader information
static const char * const SHADER_FILE = "shaders/runtimeBC1gen/transform";

DEMOGfxPipeline g_LatticePipeline;

static DEMOGfxBuffer g_VertexBuffer;
static DEMOGfxBuffer g_IndexBuffer;

// Constant Buffers
static int staticMatrixLoc;
static int dynamicMatrixLoc;


//---------------------------------------------------------------------------*
//  Model Data
//---------------------------------------------------------------------------*/
static const int STRUT_LN = 130;     /// long side of strut
static const int STRUT_SD = 4;       /// short side of strut
static const int JOINT_SD = 10;      /// joint is a cube

/// Vertices
static f32 modelVtxs[] =
{
//      x             y             z          No:Id
    -STRUT_SD,     STRUT_SD,    -STRUT_SD,   // 0:0
//   r          g          b          a
     0.5f,      0.5f,      0.42f,     1.0f,
     STRUT_SD,     STRUT_SD,    -STRUT_SD,   // 1:1
     0.5f,      0.5f,      0.42f,     1.0f,
     STRUT_SD,     STRUT_SD,    -STRUT_LN,   // 2:2
     0.5f,      0.5f,      0.42f,     1.0f,
    -STRUT_SD,     STRUT_SD,    -STRUT_SD,   // 0:3
     0.5f,      0.5f,      0.42f,     1.0f,
     STRUT_SD,     STRUT_SD,    -STRUT_LN,   // 2:4
     0.5f,      0.5f,      0.42f,     1.0f,
    -STRUT_SD,     STRUT_SD,    -STRUT_LN,   // 3:5
     0.5f,      0.5f,      0.42f,     1.0f,

     STRUT_SD,     STRUT_SD,    -STRUT_SD,   // 1:6
     0.3f,      0.3f,      0.3f,     1.0f,
     STRUT_SD,    -STRUT_SD,    -STRUT_SD,   // 4:7
     0.3f,      0.3f,      0.3f,     1.0f,
     STRUT_SD,    -STRUT_SD,    -STRUT_LN,   // 5:8
     0.3f,      0.3f,      0.3f,     1.0f,
     STRUT_SD,     STRUT_SD,    -STRUT_SD,   // 1:9
     0.3f,      0.3f,      0.3f,     1.0f,
     STRUT_SD,    -STRUT_SD,    -STRUT_LN,   // 5:10
     0.3f,      0.3f,      0.3f,     1.0f,
     STRUT_SD,     STRUT_SD,    -STRUT_LN,   // 4:11
     0.3f,      0.3f,      0.3f,     1.0f,

     STRUT_SD,     STRUT_SD,    -STRUT_SD,   // 1:12
     0.5f,      0.5f,      0.42f,     1.0f,
     STRUT_SD,     STRUT_SD,     STRUT_SD,   // 6:13
     0.5f,      0.5f,      0.42f,     1.0f,
     STRUT_LN,     STRUT_SD,     STRUT_SD,   // 7:14
     0.5f,      0.5f,      0.42f,     1.0f,
     STRUT_SD,     STRUT_SD,    -STRUT_SD,   // 1:15
     0.5f,      0.5f,      0.42f,     1.0f,
     STRUT_LN,     STRUT_SD,     STRUT_SD,   // 7:16
     0.5f,      0.5f,      0.42f,     1.0f,
     STRUT_LN,     STRUT_SD,    -STRUT_SD,   // 8:17
     0.5f,      0.5f,      0.42f,     1.0f,

     STRUT_SD,     STRUT_SD,     STRUT_SD,   // 6:18
     0.2f,      0.2f,      0.2f,     1.0f,
     STRUT_SD,    -STRUT_SD,     STRUT_SD,   //10:19
     0.2f,      0.2f,      0.2f,     1.0f,
     STRUT_LN,     STRUT_SD,     STRUT_SD,   // 7:20
     0.2f,      0.2f,      0.2f,     1.0f,
     STRUT_SD,    -STRUT_SD,     STRUT_SD,   //10:21
     0.2f,      0.2f,      0.2f,     1.0f,
     STRUT_LN,    -STRUT_SD,     STRUT_SD,   // 9:22
     0.2f,      0.2f,      0.2f,     1.0f,
     STRUT_LN,     STRUT_SD,     STRUT_SD,   // 7:23
     0.2f,      0.2f,      0.2f,     1.0f,


     STRUT_SD,     STRUT_SD,    -STRUT_SD,   //  1:24
     0.3f,      0.3f,      0.3f,     1.0f,
     STRUT_SD,     STRUT_LN,     STRUT_SD,   // 11:25
     0.3f,      0.3f,      0.3f,     1.0f,
     STRUT_SD,     STRUT_SD,     STRUT_SD,   //  6:26
     0.3f,      0.3f,      0.3f,     1.0f,
     STRUT_SD,     STRUT_SD,    -STRUT_SD,   //  1:27
     0.3f,      0.3f,      0.3f,     1.0f,
     STRUT_SD,     STRUT_LN,    -STRUT_SD,   // 12:29
     0.3f,      0.3f,      0.3f,     1.0f,
     STRUT_SD,     STRUT_LN,     STRUT_SD,   // 11:28
     0.3f,      0.3f,      0.3f,     1.0f,

     STRUT_SD,     STRUT_SD,     STRUT_SD,   //  6:30
     0.2f,      0.2f,      0.2f,     1.0f,
     STRUT_SD,     STRUT_LN,     STRUT_SD,   // 11:31
     0.2f,      0.2f,      0.2f,     1.0f,
    -STRUT_SD,     STRUT_SD,     STRUT_SD,   // 14:32
     0.2f,      0.2f,      0.2f,     1.0f,
    -STRUT_SD,     STRUT_SD,     STRUT_SD,   // 14:33
     0.2f,      0.2f,      0.2f,     1.0f,
     STRUT_SD,     STRUT_LN,     STRUT_SD,   // 11:34
     0.2f,      0.2f,      0.2f,     1.0f,
    -STRUT_SD,     STRUT_LN,     STRUT_SD,   // 13:35
     0.2f,      0.2f,      0.2f,     1.0f,

    -JOINT_SD,     JOINT_SD,    -JOINT_SD,   // 20:36
     0.5f,      0.5f,      0.42f,     1.0f,
    -JOINT_SD,     JOINT_SD,     JOINT_SD,   // 21:38
     0.5f,      0.5f,      0.42f,     1.0f,
     JOINT_SD,     JOINT_SD,    -JOINT_SD,   // 22:37
     0.5f,      0.5f,      0.42f,     1.0f,
     JOINT_SD,     JOINT_SD,    -JOINT_SD,   // 20:39
     0.5f,      0.5f,      0.42f,     1.0f,
    -JOINT_SD,     JOINT_SD,     JOINT_SD,   // 22:40
     0.5f,      0.5f,      0.42f,     1.0f,
     JOINT_SD,     JOINT_SD,     JOINT_SD,   // 23:41
     0.5f,      0.5f,      0.42f,     1.0f,

     JOINT_SD,     JOINT_SD,    -JOINT_SD,   // 21:42
     0.3f,      0.3f,      0.3f,     1.0f,
     JOINT_SD,     JOINT_SD,     JOINT_SD,   // 23:43
     0.3f,      0.3f,      0.3f,     1.0f,
     JOINT_SD,    -JOINT_SD,    -JOINT_SD,   // 24:44
     0.3f,      0.3f,      0.3f,     1.0f,
     JOINT_SD,     JOINT_SD,     JOINT_SD,   // 23:45
     0.3f,      0.3f,      0.3f,     1.0f,
     JOINT_SD,    -JOINT_SD,     JOINT_SD,   // 25:46
     0.3f,      0.3f,      0.3f,     1.0f,
     JOINT_SD,    -JOINT_SD,    -JOINT_SD,   // 24:47
     0.3f,      0.3f,      0.3f,     1.0f,

    -JOINT_SD,     JOINT_SD,     JOINT_SD,   // 22:48
     0.2f,      0.2f,      0.2f,     1.0f,
    -JOINT_SD,    -JOINT_SD,     JOINT_SD,   // 26:49
     0.2f,      0.2f,      0.2f,     1.0f,
     JOINT_SD,     JOINT_SD,     JOINT_SD,   // 23:50
     0.2f,      0.2f,      0.2f,     1.0f,
     JOINT_SD,     JOINT_SD,     JOINT_SD,   // 23:51
     0.2f,      0.2f,      0.2f,     1.0f,
    -JOINT_SD,    -JOINT_SD,     JOINT_SD,   // 26:52
     0.2f,      0.2f,      0.2f,     1.0f,
     JOINT_SD,    -JOINT_SD,     JOINT_SD,   // 25:53
     0.2f,      0.2f,      0.2f,     1.0f,

};

//static const int MODEL_VTX_COUNT 54 * 3
static const int MODEL_VTX_SIZE = ( sizeof( modelVtxs ) );
static const int MODEL_VTX_STRIDE = ( sizeof( f32 ) * 3 + sizeof( f32 ) * 4 );
static const int MODEL_POS_OFFSET = 0;
static const int MODEL_CLR_OFFSET = 3;

// ----------------------------------------------------
//
//            13----------- 11
//             |          /|
//             |         / |
//             |        /  |
//             |        12 |
//             |        |  |
//             |        |  |
//             |        |  |
//             |        |  |
//            14--------|-- 6--------------- 7
//                      | /|                /|
//                      |/ |               / |
//                      |  |              /  |
//         0 ----------- 1----------------8  |
//         /           /|  | 10              |
//        /           / |  ------------------ 9
//       /           /  |
//      /           /   |
//     /           /   /4
//   3 ------------ 2 /
//                |  /
//                | /
//                |/
//                5
//
//       22 ------------ 23
//        /|             /|
//       / |            / |
//      /  |           /  |
//     /   |          /   |
//  20 --------------- 21 |
//         |          |   |
//       26 ----------|--- 25
//                    |  /
//                    | /
//                    |/
//                    24
//  y
//   ^  ^ z
//   | /
//   |/
//   ------> x
//
//---------------------------------------------------------

/// Indices
static const int MODEL_IDX_COUNT = 54;
static const int MODEL_IDX_SIZE = ( sizeof( u32 ) * MODEL_IDX_COUNT );

static u32 modelIdxs[] = {
    0,   1,  2,  3,  4,  5, // STRUT1 #0
    6,   7,  8,  9, 10, 11, // STRUT1 #1
    12, 13, 14, 15, 16, 17, // STRUT2 #2
    18, 19, 20, 21, 22, 23, // STRUT2 #3
    24, 25, 26, 27, 28, 29, // STRUT3 #4
    30, 31, 32, 33, 34, 35, // STRUT3 #5

    36, 37, 38, 39, 40, 41, // JOINT #6
    42, 43, 44, 45, 46, 47, // JOINT #7
    48, 49, 50, 51, 52, 53, // JOINT #8
};

//static const int NUM_ATTRIB2 = 2;
static const int BUFFER_IDX = 0;

// -----

// Matricies
typedef struct _StaticMatrix
{
    Mtx44   viewMtx44;
    Mtx44   projMtx44;
} StaticMatrix;

static DEMOGfxBuffer g_StaticUniforms;

typedef struct _DynamicMatrix
{
    Mtx44   modelMtx44;
} DynamicMatrix;
static size_t alignedDynamicSize = 0;

static int s_FrameCount = 0;
static f32 s_animTranZ;
static u32 s_ticks2;

typedef struct _UniformView
{
    nn::gfx::GpuAddress gpuAddress[ 2 ];
    int size;
} UniformView;

// Have one uniform block per frame to hold all of the data
static DEMOGfxBuffer dynamicUniforms[ 2 ];

// Init function for setting projection matrix
static void CameraInit(Mtx44 resultProjMtx44, Mtx44 resultViewMtx44)
{
    // row major matricies
    Mtx   lookAtMtx34;

    Vec     up = {0.20f, 0.97f, 0.0f};
    Vec  objPt = {-110.0f, -70.0f, -190.0f};
    Vec camLoc = {90.0f, 110.0f, 13.0f};

    f32   pers = 50.0f;
    f32 aspect = (f32)RENDERWidth / (f32)RENDERHeight;
    f32  znear = 50.0f;
    f32   zfar = 2000.0f;

    // Compute perspective matrix
    MTXPerspective(resultProjMtx44, pers, aspect, znear, zfar);

    // Compute lookAt matrix
    MTXLookAt(lookAtMtx34, &camLoc, &up, &objPt);
    MTX34To44(lookAtMtx34, resultViewMtx44);
}

// Update Animation
static void AnimTick()
{
    u32    animSteps    = STRUT_LN;
    f32    animLoopBack = (f32)STRUT_LN;
    f32    animStepFwd  = animLoopBack / (f32)animSteps;

    s_animTranZ += animStepFwd;

    if((s_ticks2 % animSteps) == 0)
        s_animTranZ = animLoopBack;

    s_ticks2++;
}

// Update Model function for setting model matrix
static void ModelTick(Mtx44 resultMtx44, s32 offsetX, s32 offsetY, s32 offsetZ)
{
    MTX44Identity(resultMtx44);
    resultMtx44[0][3] = offsetX + 2.0f * STRUT_LN;
    resultMtx44[1][3] = (f32)offsetY;
    resultMtx44[2][3] = offsetZ + s_animTranZ;
}

// The init function for the rendering portions of this app
static int LatticeInit()
{
    s_animTranZ = 0;
    s_ticks2 = 0;

    // Load shader binary to memory
    DEMOGfxLoadShadersFromFile( &g_LatticePipeline.shaders, 0, SHADER_FILE );

    // Uniform Location Lookup
    staticMatrixLoc = g_LatticePipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_static" );
    dynamicMatrixLoc = g_LatticePipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_dynamic" );

    // position setup
    DEMOGfxInitShaderAttribute( &g_LatticePipeline.shaders, "a_position", BUFFER_IDX, MODEL_POS_OFFSET * sizeof( f32 ), nn::gfx::AttributeFormat_32_32_32_Float );
    DEMOGfxInitShaderAttribute( &g_LatticePipeline.shaders, "a_color", BUFFER_IDX, MODEL_CLR_OFFSET * sizeof( f32 ), nn::gfx::AttributeFormat_32_32_32_32_Float );
    DEMOGfxInitShaderVertexBuffer( &g_LatticePipeline.shaders, BUFFER_IDX, MODEL_VTX_STRIDE, 0 );

    // Create Vertex Buffer
    g_VertexBuffer.Initialize( MODEL_VTX_SIZE, &modelVtxs[ 0 ], nn::gfx::GpuAccess_VertexBuffer, 0 );

    // Create Index Buffer
    g_IndexBuffer.Initialize( MODEL_IDX_SIZE, &modelIdxs[ 0 ], nn::gfx::GpuAccess_IndexBuffer, 0 );

    // Setup the static matrix buffers
    {
        g_StaticUniforms.Initialize( sizeof( StaticMatrix ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
        StaticMatrix* pMatrix = g_StaticUniforms.Map< StaticMatrix >();
        CameraInit( pMatrix->projMtx44, pMatrix->viewMtx44 );
#ifdef CAFE
        GX2EndianSwap( pMatrix, sizeof( StaticMatrix ) );
#endif
        g_StaticUniforms.Unmap();
    }

    // determine the alignment
    {
        size_t alignment = 1;
        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetGpuAccessFlags( nn::gfx::GpuAccess_ConstantBuffer );
        info.SetSize( sizeof( DynamicMatrix ) );

        alignment = nn::gfx::Buffer::GetBufferAlignment( &DEMODevice, info );
        alignedDynamicSize = ( sizeof( DynamicMatrix ) + alignment - 1 ) & ( ~( alignment - 1 ) );
    }

    dynamicUniforms[ 0 ].Initialize( 12 * 11 * 16 * alignedDynamicSize, NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
    dynamicUniforms[ 1 ].Initialize( 12 * 11 * 16 * alignedDynamicSize, NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );

    g_LatticePipeline.SetDefaults();

    g_LatticePipeline.colorTargetStateCount = 1;
    g_LatticePipeline.blendTargetStateCount = 1;
    g_LatticePipeline.blendTargetStateInfoArray[ 0 ].SetDefault();
    g_LatticePipeline.colorTargetStateInfoArray[ 0 ].SetDefault();
    g_LatticePipeline.colorTargetStateInfoArray[ 0 ].SetFormat( DEMOColorBufferInfo.GetImageFormat() );

    // Setup the pipeline
    g_LatticePipeline.depthStencilStateInfo.SetDepthComparisonFunction( nn::gfx::ComparisonFunction_Less );
    g_LatticePipeline.Initialize( &DEMODevice );

    return 1;
}

static void LatticeShutdown()
{
    // Free shaders/pipeline
    g_LatticePipeline.Finalize( &DEMODevice );

    // Free the vertex/index buffer
    g_VertexBuffer.Finalize( );
    g_IndexBuffer.Finalize( );

    // Free the static uniform
    g_StaticUniforms.Finalize( );

    dynamicUniforms[ 0 ].Finalize();
    dynamicUniforms[ 1 ].Finalize();
}


// The draw function for the rendering portions of this app
static int LatticeDraw()
{
    u32 dynamicUnformIdx = 0;
    s32 x = 0;
    s32 y = 0;
    s32 z = 0;

    // Set the Pipeline
    DEMOCommandBuffer.SetPipeline( &g_LatticePipeline.pipeline );

    // Update View Projection Matrix Uniforms
    DEMOCommandBuffer.SetConstantBuffer( staticMatrixLoc, nn::gfx::ShaderStage_Vertex, g_StaticUniforms.gpuAddress, g_StaticUniforms.size );

    // Bind vertex & color buffer
    DEMOCommandBuffer.SetVertexBuffer( BUFFER_IDX, g_VertexBuffer.gpuAddress, MODEL_VTX_STRIDE, g_VertexBuffer.size );

    // Draw Model
    DynamicMatrix* pMatrix = dynamicUniforms[ s_FrameCount % 2 ].Map< DynamicMatrix >();

    for ( x = -10 * STRUT_LN; x < 2 * STRUT_LN; x += STRUT_LN )
    {
        for ( y = -10 * STRUT_LN; y < STRUT_LN; y += STRUT_LN )
        {
            for ( z = STRUT_LN; z > -15 * STRUT_LN; z -= STRUT_LN )
            {
                // Update Model Matrix Uniform
                ModelTick( pMatrix->modelMtx44, x, y, z );
#if NN_GFX_IS_TARGET_GX
                GX2EndianSwap( pMatrix, sizeof( DynamicMatrix ) );
#endif
                // Set the uniform block
                nn::gfx::GpuAddress gpuAddress = dynamicUniforms[ s_FrameCount % 2 ].gpuAddress;
                gpuAddress.Offset( dynamicUnformIdx * alignedDynamicSize );
                DEMOCommandBuffer.SetConstantBuffer( dynamicMatrixLoc, nn::gfx::ShaderStage_Vertex,
                    gpuAddress, alignedDynamicSize );

                // Draw 1 Structure
                DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32, g_IndexBuffer.gpuAddress, MODEL_IDX_COUNT, 0 );

                dynamicUnformIdx++;
                pMatrix = reinterpret_cast< DynamicMatrix* >( reinterpret_cast< ptrdiff_t >( pMatrix ) + static_cast< ptrdiff_t >( alignedDynamicSize ) );
            }
        }
    }
    dynamicUniforms[ s_FrameCount % 2 ].Unmap();

    // Update Animation
    AnimTick();

    return 1; // 0 makes it exit
}

static const char * const SIMPLE_SHADER_FILE = "shaders/runtimeBC1gen/simple";

// Position for a single triangle.
static DEMO_F32x3 QUAD_POSITION_DATA[] =
{
    { { { -0.95f, -0.95f, -0.5f } } },
    { { { 0.95f, -0.95f, -0.5f } } },
    { { { 0.95f, 0.95f, -0.5f } } },
    { { { -0.95f, 0.95f, -0.5f } } }
};

// Color for a single triangle.
static DEMO_F32x3 QUAD_COLOR_DATA[] =
{
    { { { 1.0f, 0.0f, 0.0f } } },
    { { { 0.0f, 1.0f, 0.0f } } },
    { { { 0.0f, 0.0f, 1.0f } } },
    { { { 1.0f, 1.0f, 0.0f } } }
};

// Number of vertex
static const u32 QUAD_VERTEX_NUM = 4;

DEMOGfxPipeline g_SimplePipeline;

DEMOGfxBuffer g_PositionBuffer;
DEMOGfxBuffer g_ColorBuffer;

// Initialize attribute buffer and data
static void InitAttribData()
{
    // position setup
    DEMOGfxInitShaderAttribute( &g_SimplePipeline.shaders, "a_position", BUFFER_IDX, 0, nn::gfx::AttributeFormat_32_32_32_Float );
    DEMOGfxInitShaderAttribute( &g_SimplePipeline.shaders, "a_color", BUFFER_IDX + 1, 0, nn::gfx::AttributeFormat_32_32_32_Float );
    DEMOGfxInitShaderVertexBuffer( &g_SimplePipeline.shaders, BUFFER_IDX, sizeof(DEMO_F32x3), 0 );
    DEMOGfxInitShaderVertexBuffer( &g_SimplePipeline.shaders, BUFFER_IDX + 1, sizeof( DEMO_F32x3 ), 0 );

    // Create Vertex Buffer
    g_PositionBuffer.Initialize( sizeof( QUAD_POSITION_DATA ), QUAD_POSITION_DATA, nn::gfx::GpuAccess_VertexBuffer, 0 );
    g_ColorBuffer.Initialize( sizeof( QUAD_COLOR_DATA ), QUAD_COLOR_DATA, nn::gfx::GpuAccess_VertexBuffer, 0 );
}

// Free attribute buffer and data
static void FreeAttribData()
{
    g_PositionBuffer.Finalize();
    g_ColorBuffer.Finalize();
}

// ---------------------------------------- //

// The init function for the rendering portions of this app
void SceneInit()
{
    DEMOGfxSetupTextureBuffer( &g_R2TTexture, &g_R2TTextureView, &g_R2TTextureSlot, &g_R2TColorBuffer, NULL,
        &g_pR2TTexturePool, RENDERWidth, RENDERHeight, 1, 1, nn::gfx::ImageDimension_2d,
        nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm, nn::gfx::DepthStencilFetchMode_DepthComponent, 0 );

    // Setup Depth Buffer
    DEMOGfxSetupTextureBuffer( &g_R2TDepthTexture, &g_R2TDepthTextureView, &g_R2TDepthTextureSlot, NULL, &g_R2TDepthBuffer,
        &g_pR2TDepthTexturePool, RENDERWidth, RENDERHeight, 1, 1, nn::gfx::ImageDimension_2d,
        nn::gfx::ImageFormat_D32_Float, nn::gfx::DepthStencilFetchMode_DepthComponent, 0 );

#ifdef FIXME_HIZ
    // Setup Hi-Z buffer
    if (1) { // Allow testing with & without Hi-Z
        u32 size, align;
        GX2CalcDepthBufferHiZInfo(&R2TDepthBuffer, &size, &align);
        ptr = DEMOGfxAllocMEM1(size, align);
        GX2Invalidate(GX2_INVALIDATE_CPU, ptr, size);
        GX2InitDepthBufferHiZPtr(&R2TDepthBuffer, ptr);
    }
#endif

    DEMOGfxInitSampler( &g_R2TSampler, &g_R2TSamplerSlot, nn::gfx::TextureAddressMode_ClampToEdge, nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint, nn::gfx::ComparisonFunction_Always );

    // ------------- //
    // Scene 1 init
    LatticeInit();

    // ------------- //
    // Scene 2 init
    // Load shader binary to memory
    DEMOGfxLoadShadersFromFile( &g_SimplePipeline.shaders, 0, SIMPLE_SHADER_FILE );
    g_SimplePipeline.SetDefaults();

    // Initialize attributes and attribute buffer
    InitAttribData();

    g_SimplePipeline.colorTargetStateCount = 1;
    g_SimplePipeline.blendTargetStateCount = 1;
    g_SimplePipeline.blendTargetStateInfoArray[ 0 ].SetDefault();
    g_SimplePipeline.colorTargetStateInfoArray[ 0 ].SetDefault();
    g_SimplePipeline.colorTargetStateInfoArray[ 0 ].SetFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );

    // Setup the pipeline
    g_SimplePipeline.depthStencilStateInfo.SetDepthComparisonFunction( nn::gfx::ComparisonFunction_Less );
    g_SimplePipeline.Initialize( &DEMODevice );

    DEMOGfxSetViewportScissorState( &g_R2TViewport, &g_pR2TViewportData, 0.0f, 0.0f,
        static_cast< float >( RENDERWidth ), static_cast< float >( RENDERHeight ), 0.0f, 1.0f,
        static_cast< float >( RENDERHeight ), true );
}

void SimpleQuadShutdown()
{
    g_SimplePipeline.Finalize( &DEMODevice );
    FreeAttribData();

#ifdef FIXME_HIZ
    // Free allocated buffers
    if (R2TDepthBuffer.hiZPtr) {
        DEMOGfxFreeMEM1(R2TDepthBuffer.hiZPtr);
        R2TDepthBuffer.hiZPtr = NULL;
    }
#endif
    g_R2TTextureView.Finalize( &DEMODevice );
    g_R2TColorBuffer.Finalize( &DEMODevice );
    g_R2TTexture.Finalize( &DEMODevice );
    g_pR2TTexturePool->Finalize();
    delete g_pR2TTexturePool;
    g_R2TDepthTextureView.Finalize( &DEMODevice );
    g_R2TDepthBuffer.Finalize( &DEMODevice );
    g_R2TDepthTexture.Finalize( &DEMODevice );
    g_pR2TDepthTexturePool->Finalize();
    delete g_pR2TDepthTexturePool;

    g_R2TViewport.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2( g_pR2TViewportData );

}

void SceneDraw()
{
    // Clear buffers
    DEMOCommandBuffer.ClearColor( &g_R2TColorBuffer, 0.1f, 0.1f, 0.1f, 1.0f, NULL );
    DEMOCommandBuffer.ClearDepthStencil( &g_R2TDepthBuffer, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

    nn::gfx::ColorTargetView* renderTargets[ ] = { &g_R2TColorBuffer };
    DEMOCommandBuffer.SetRenderTargets( 1, renderTargets, &g_R2TDepthBuffer );

    // Setup Viewport/Stencil.
    DEMOCommandBuffer.SetViewportScissorState( &g_R2TViewport );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif

    if (RENDERScene == 0)
    {
        LatticeDraw();
    } else {
        // Set the Pipeline
        DEMOCommandBuffer.SetPipeline( &g_SimplePipeline.pipeline );

        // Bind vertex & color buffer
        DEMOCommandBuffer.SetVertexBuffer( BUFFER_IDX, g_PositionBuffer.gpuAddress, sizeof( DEMO_F32x3 ), g_PositionBuffer.size );
        DEMOCommandBuffer.SetVertexBuffer( BUFFER_IDX + 1, g_ColorBuffer.gpuAddress, sizeof( DEMO_F32x3 ), g_ColorBuffer.size );

        // FIXME - Need to rewrite quads attribs to triangles
        DEMOCommandBuffer.Draw( nn::gfx::PrimitiveTopology_TriangleList, QUAD_VERTEX_NUM, 0 );
    }

    DEMOCommandBuffer.FlushMemory( nn::gfx::GpuAccess_ColorBuffer );
    DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_Texture );
    // FIXME - DEMOCommandBuffer.SetTextureStateTransition?
}

void SceneShutdown()
{
    // ------------- //
    // Scene 1 shutdown
    LatticeShutdown();

    // ------------- //
    // Scene 2 shutdown
    SimpleQuadShutdown();

    g_R2TSampler.Finalize(&DEMODevice);
}
