﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/////////////////////////////////////////////////////////////////////////////
//
// geoprims.cpp
//
// Draws all primitive types
//
//////////////////////////////////////////////////////////////////////////////

#include <cstdio>
#include <cstring>
#include <cmath>
#include <algorithm>

#include <gfx/demo.h>

#if NN_GFX_IS_TARGET_GL
#include <GL/glew.h>
#endif

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#endif

#include <nnt.h>
#include <nnt/nnt_Argument.h>

// ----- Surface Information

#define SURFACE_WIDTH  (DEMOColorBufferInfo.GetWidth())
#define SURFACE_HEIGHT  (DEMOColorBufferInfo.GetHeight())

// ----- Viewport Information

static const int VIEWPORT_MAX_W =  16;
static const int VIEWPORT_MAX_H =  9;

// ----- GX2 Shader information
static const char * const GSH_SHADER_FILE = "shaders/primitiveShader";

static DEMOGfxShader theShader;

static u32 posLoc;
static u32 clrLoc;

static u32 mtxLoc;

struct MtxUniform
{
    Mtx44 modelMtx;
    Mtx44 viewMtx;
    Mtx44 projMtx;
};

static nn::gfx::Pipeline quadPointsPipeline;
static nn::gfx::Pipeline quadLinesPipeline;
static nn::gfx::Pipeline quadLineStripPipeline;
static nn::gfx::Pipeline quadTrianglesPipeline[2];
static nn::gfx::Pipeline quadTriangleStripPipeline[2];
void * quadPointsPipelineData;
void * quadLinesPipelineData;
void * quadLineStripPipelineData;
void * quadTrianglesPipelineData[2];
void * quadTriangleStripPipelineData[2];

static nn::gfx::ViewportScissorState viewportScissorState;
void * viewportScissorData;

// ----- Quad Data
static float quadVtxsInit[] =
{
    -0.7f,  0.7f,  0.0f,
    1.0f,   0.0f,  0.0f, 1.0f,
    0.7f,   0.7f,  0.0f,
    0.0f,   1.0f,  0.0f, 1.0f,
    0.7f,  -0.7f,  0.0f,
    0.0f,   0.0f,  1.0f, 1.0f,
    -0.7f, -0.7f,  0.0f,
    1.0f,   0.0f,  1.0f, 1.0f,
    -0.7f,  0.0f,  0.0f,
    1.0f,   1.0f,  0.0f, 1.0f,
    0.7f,   0.0f,  0.0f,
    0.0f,   1.0f,  1.0f, 1.0f,
};
static DEMOGfxBuffer quadVertexBuffer;

// ----- Quad Points

static const int QUAD_POINTS_IDX_COUNT = 4;
static const int QUAD_POINTS_IDX_SIZE = (sizeof(u32) * QUAD_POINTS_IDX_COUNT);
static const int NUM_ATTRIB =  2;
//static const int QUAD_VTX_COUNT = 6;
static const int QUAD_VTX_SIZE = (sizeof(quadVtxsInit));
static const int QUAD_VTX_STRIDE = (sizeof(float) * 3 + sizeof(float) * 4);
static const int QUAD_POS_OFFSET = 0;
static const int QUAD_CLR_OFFSET = 3;

static const int BUFFER_IDX = 0;

static u32 quadPointsInitIdxs[] = {0, 1, 2, 3};
static DEMOGfxBuffer quadPointsIndexBuffer;

// ----- Quad Lines

static const int QUAD_LINES_IDX_COUNT = 8;
static const int QUAD_LINES_IDX_SIZE = (sizeof(u32) * QUAD_LINES_IDX_COUNT);

static u32 quadLinesInitIdxs[] =
{
    0, 1,
    1, 2,
    2, 3,
    3, 0,
};
static DEMOGfxBuffer quadLinesIndexBuffer;

// ----- Quad Line Strip

static const int QUAD_LINE_STRIP_IDX_COUNT = 5;
static const int QUAD_LINE_STRIP_IDX_SIZE = (sizeof(u32) * QUAD_LINE_STRIP_IDX_COUNT);

static u32 quadLineStripInitIdxs[] = {0, 1, 2, 3, 0};
static DEMOGfxBuffer quadLineStripIndexBuffer;

// ----- Quad Triangles

static const int QUAD_TRIANGLES_IDX_COUNT = 6;
static const int QUAD_TRIANGLES_IDX_SIZE = (sizeof(u32) * QUAD_TRIANGLES_IDX_COUNT);

static u32 quadTrianglesInitIdxs[] =
{
    0,  1,  3,
    1,  2,  3,
};
static DEMOGfxBuffer quadTrianglesIndexBuffer;

// ----- Quad Triangle Strip

static const int QUAD_TRIANGLE_STRIP_IDX_COUNT = 4;
static const int QUAD_TRIANGLE_STRIP_IDX_SIZE = (sizeof(u32) * QUAD_TRIANGLE_STRIP_IDX_COUNT);

static u32 quadTriangleStripInitIdxs[] = {0, 1, 3, 2};
static DEMOGfxBuffer quadTriangleStripIndexBuffer;

// Matrices
//static const int MTX44_COMPONENTS_COUNTS = 16;
static DEMOGfxBuffer pointsMtx;
static DEMOGfxBuffer linesMtx;
static DEMOGfxBuffer lineStripMtx;
static DEMOGfxBuffer trianglesMtx;
static DEMOGfxBuffer triangleStripMtx;

static Mtx44   viewMtx44;
static Mtx44   projMtx44;

// Positions
static Vec    pointsPos = {-6.0f, 3.0f, 0.0f };
static Vec    linesPos = { -3.0f, 3.0f, 0.0f };
static Vec    lineStripPos = {  0.0f, 3.0f, 0.0f };
static Vec    trianglesPos = {  3.0f, 3.0f, 0.0f };
static Vec    triangleStripPos = {  6.0f, 3.0f, 0.0f };

static f32 s_yrad = 0.0f;
static f32 s_xrad = 0.0f;

// Prototype
static void CameraInit(Mtx44*, Mtx44*);
static void ModelTick(DEMOGfxBuffer&, Vec);
static void InitCameraMatrix( DEMOGfxBuffer& buffer, Mtx44* pResultProjMtx44, Mtx44* pResultViewMtx44 );
static int  SceneInit(void);
static void PrintInfo(void);
static int  SceneDraw(void);
static void InitBuffers(void);

// Init function for setting projection matrix
static void CameraInit(Mtx44* pResultProjMtx44, Mtx44* pResultViewMtx44)
{
    // row major matrices
    Mtx   lookAtMtx34;
    Vec     up = { 0.0f, 1.0f, 0.0f };
    Vec  objPt = { 0.0f, 0.0f, 0.0f };
    Vec camLoc = { 0.0f, 0.0f, 10.0f };

    f32    top  =  VIEWPORT_MAX_H / 2.0f;
    f32 bottom  = -VIEWPORT_MAX_H / 2.0f;
    f32   left  = -VIEWPORT_MAX_W / 2.0f;
    f32  right  =  VIEWPORT_MAX_W / 2.0f;
    f32  znear  = 0.0f;
    f32   zfar  = 20.0f;

    // Compute ortho matrix
    MTXOrtho( *pResultProjMtx44, top, bottom, left, right, znear, zfar );

    // Compute lookAt matrix
    MTXLookAt( lookAtMtx34, &camLoc, &up, &objPt );
    MTX34To44( lookAtMtx34, *pResultViewMtx44 );
}


static void InitCameraMatrix(DEMOGfxBuffer& buffer, Mtx44* pResultProjMtx44, Mtx44* pResultViewMtx44)
{

    buffer.Initialize(sizeof(MtxUniform), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0);

    struct MtxUniform* pMtx = buffer.Map< struct MtxUniform >( );
    MTX44Identity( pMtx->modelMtx );
    memcpy( &pMtx->viewMtx, pResultViewMtx44, sizeof(Mtx44) );
    memcpy( &pMtx->projMtx, pResultProjMtx44, sizeof(Mtx44) );
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pMtx, sizeof(*pMtx) );
#endif
    buffer.Unmap();
}

// Update Model function for setting model matrix
static void ModelTick(DEMOGfxBuffer& buf, Vec pos)
{
    // row major matrix
    Mtx  rotXMtx34;
    Mtx  rotYMtx34;
    Mtx  rotMtx34;

    // row major matrix
    struct MtxUniform* pMtx = buf.Map< struct MtxUniform >( );

#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pMtx, sizeof(*pMtx) );
#endif

    // Compute rotation matrix
    MTXRotRad( rotYMtx34, 'y', s_yrad * 2 );
    MTXRotRad( rotXMtx34, 'x', s_xrad * 2 );

    // Compute model matrix
    MTXConcat( rotXMtx34, rotYMtx34, rotMtx34 );
    MTX34To44( rotMtx34, pMtx->modelMtx );

    // Translate
    pMtx->modelMtx[ 0 ][ 3 ] = pos.x;
    pMtx->modelMtx[ 1 ][ 3 ] = pos.y;
    pMtx->modelMtx[ 2 ][ 3 ] = pos.z;

#if NN_GFX_IS_TARGET_GX
    if (!(DEMO_PAD_BUTTON_A & DEMOPadGetButton(0)))
#endif
    {
    s_yrad += 0.0015f;
    s_xrad += 0.004f;
    }

#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pMtx, sizeof(*pMtx) );
#endif

    buf.Unmap();
}

static void SetupBufferView( DEMOGfxBuffer& buffer,
            int gpuAccessFlags,
            void* initData,
            int size,
            int offset )
{

    buffer.Initialize(size, initData, gpuAccessFlags, offset);
}

static void InitBuffers()
{
    // Vertex buffer
    SetupBufferView( quadVertexBuffer, nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Read, quadVtxsInit, QUAD_VTX_SIZE, 0 );

    // Index buffer copies
    SetupBufferView( quadPointsIndexBuffer, nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_Read, quadPointsInitIdxs, QUAD_POINTS_IDX_SIZE, 0 );
    SetupBufferView( quadLinesIndexBuffer, nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_Read, quadLinesInitIdxs, QUAD_LINES_IDX_SIZE, 0 );
    SetupBufferView( quadLineStripIndexBuffer, nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_Read, quadLineStripInitIdxs, QUAD_LINE_STRIP_IDX_SIZE, 0 );
    SetupBufferView( quadTrianglesIndexBuffer, nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_Read, quadTrianglesInitIdxs, QUAD_TRIANGLES_IDX_SIZE, 0 );
    SetupBufferView( quadTriangleStripIndexBuffer, nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_Read, quadTriangleStripInitIdxs, QUAD_TRIANGLE_STRIP_IDX_SIZE, 0 );
}

static void InitMatrices()
{
    // Initialize Matrix Buffers
    CameraInit(&projMtx44, &viewMtx44);
    InitCameraMatrix(pointsMtx, &projMtx44, &viewMtx44);
    InitCameraMatrix(linesMtx, &projMtx44, &viewMtx44);
    InitCameraMatrix(lineStripMtx, &projMtx44, &viewMtx44);
    InitCameraMatrix(trianglesMtx, &projMtx44, &viewMtx44);
    InitCameraMatrix(triangleStripMtx, &projMtx44, &viewMtx44);
}

static void InitViewport()
{
    // Setup the viewport
    nn::gfx::ViewportScissorState::InfoType viewportInfo;
    nn::gfx::ScissorStateInfo scissorInfoArray[ 1 ];
    nn::gfx::ViewportStateInfo viewportInfoArray[ 1 ];
    scissorInfoArray[ 0 ].SetDefault();
    scissorInfoArray[ 0 ].SetOriginX( 0 );
    scissorInfoArray[ 0 ].SetOriginY( 0 );
    scissorInfoArray[ 0 ].SetWidth( DEMOColorBufferInfo.GetWidth() );
    scissorInfoArray[ 0 ].SetHeight( DEMOColorBufferInfo.GetHeight() );
    viewportInfoArray[ 0 ].SetDefault();
    viewportInfoArray[ 0 ].SetOriginX( 0.0f );
    viewportInfoArray[ 0 ].SetOriginY( 0.0f );
    viewportInfoArray[ 0 ].SetWidth( static_cast< float >( DEMOColorBufferInfo.GetWidth() ) );
    viewportInfoArray[ 0 ].SetHeight( static_cast< float >( DEMOColorBufferInfo.GetHeight() ) );
    viewportInfoArray[ 0 ].SetMinDepth( 0.0f );
    viewportInfoArray[ 0 ].SetMaxDepth( 1.0f );
    viewportInfo.SetDefault();
    viewportInfo.SetScissorEnabled( true );
    viewportInfo.SetScissorStateInfoArray( scissorInfoArray, 1 );
    viewportInfo.SetViewportStateInfoArray( viewportInfoArray, 1 );
    size_t viewportSize = viewportScissorState.GetRequiredMemorySize( viewportInfo );
    if ( viewportSize )
    {
        viewportScissorData = DEMOGfxAllocMEM2( viewportSize, nn::gfx::ViewportScissorState::RequiredMemoryInfo_Alignment );
    }
    else
    {
        viewportScissorData = NULL;
    }
    viewportScissorState.SetMemory( viewportScissorData, viewportSize );
    viewportScissorState.Initialize( &DEMODevice, viewportInfo );
}

static void InitPipeline( nn::gfx::Pipeline& pipeline, nn::gfx::Pipeline::InfoType& pipelineInfo, nn::gfx::PrimitiveTopology topology, void **data)
{
    size_t size = 0;

    NN_UNUSED(topology);
    size = nn::gfx::Pipeline::GetRequiredMemorySize( pipelineInfo );
    *data = DEMOGfxAllocMEM2( size, nn::gfx::Pipeline::RequiredMemoryInfo_Alignment );
    pipeline.SetMemory( *data, size );
    pipeline.Initialize( &DEMODevice, pipelineInfo );
}

// The init function for the rendering portions of this app
static int SceneInit()
{
    nn::gfx::Pipeline::InfoType pipelineInfo;

    // Start setting up the Pipeline
    pipelineInfo.SetDefault();

    s_yrad = 0.0f;
    s_xrad = 0.0f;

    DEMOGfxLoadShadersFromFile(&theShader, 0, GSH_SHADER_FILE);
    pipelineInfo.SetShaderPtr(theShader.GetShader());

    // Uniform Location Lookup
    mtxLoc = theShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "Matrices");

    // Attribute Location Lookup
    posLoc = theShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "position");
    clrLoc = theShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "color");

    nn::gfx::VertexStateInfo vertexStateInfo;

    // Setup vertex buffers and index buffers
    vertexStateInfo.SetDefault();
    nn::gfx::VertexAttributeStateInfo attribs[ NUM_ATTRIB ];
    nn::gfx::VertexBufferStateInfo buffer[ 1 ];
    attribs[ 0 ].SetDefault();
    attribs[ 0 ].SetBufferIndex( BUFFER_IDX );
    attribs[ 0 ].SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
    attribs[ 0 ].SetNamePtr( "position" );
    attribs[ 0 ].SetOffset( QUAD_POS_OFFSET * sizeof(float) );
    attribs[ 0 ].SetShaderSlot( posLoc );
    attribs[ 1 ].SetDefault();
    attribs[ 1 ].SetBufferIndex( BUFFER_IDX );
    attribs[ 1 ].SetFormat( nn::gfx::AttributeFormat_32_32_32_32_Float );
    attribs[ 1 ].SetNamePtr( "color" );
    attribs[ 1 ].SetOffset( QUAD_CLR_OFFSET * sizeof(float) );
    attribs[ 1 ].SetShaderSlot( clrLoc );
    buffer[ 0 ].SetDefault();
    buffer[ 0 ].SetStride( QUAD_VTX_STRIDE );
    vertexStateInfo.SetVertexBufferStateInfoArray( buffer, 1 );
    vertexStateInfo.SetVertexAttributeStateInfoArray( attribs, NUM_ATTRIB );
    pipelineInfo.SetVertexStateInfo(&vertexStateInfo);

    InitBuffers();

    // Set Color Mask
    nn::gfx::BlendStateInfo blendStateInfo;
    nn::gfx::BlendTargetStateInfo blendTargets[ 1 ];
    blendStateInfo.SetDefault();
    blendTargets[ 0 ].SetDefault();
    blendTargets[ 0 ].SetBlendEnabled( false );
    blendTargets[ 0 ].SetChannelMask( nn::gfx::ChannelMask_All );
    blendStateInfo.SetBlendTargetStateInfoArray( blendTargets, 1 );
    pipelineInfo.SetBlendStateInfo(&blendStateInfo);

    // Color targets
    nn::gfx::ImageFormat colorFormat = DEMOColorBufferInfo.GetImageFormat();
    nn::gfx::ImageFormat depthFormat = DEMODepthBufferInfo.GetImageFormat();

    nn::gfx::RenderTargetStateInfo renderTargetStateInfo;
    nn::gfx::ColorTargetStateInfo colorTargets[ 1 ];
    colorTargets[ 0 ].SetDefault();
    colorTargets[0].SetFormat(colorFormat);
    renderTargetStateInfo.SetDefault();
    renderTargetStateInfo.SetColorTargetStateInfoArray( colorTargets, 1 );
    renderTargetStateInfo.SetDepthStencilFormat( depthFormat );
    pipelineInfo.SetRenderTargetStateInfo(&renderTargetStateInfo);

    nn::gfx::RasterizerStateInfo rasterizerStateInfo;
    rasterizerStateInfo.SetDefault();
#if NN_GFX_IS_TARGET_GX
    // Set Point Size
    // nn::gfx doesn't support this
    GX2SetPointSize(8.0f, 2.0f); // if gl_PointSize is not used in the shader
    GX2SetPointLimits(1.0f, 10.0f); // limits gl_PointSize if used

    // Set Line Width
    // nn::gfx doesn't support this
    GX2SetLineWidth(4.0f);
#else
    // FIXME: These features are missing from nn::gfx
#endif

    nn::gfx::DepthStencilStateInfo depthStencilStateInfo;
    depthStencilStateInfo.SetDefault();
    depthStencilStateInfo.SetDepthWriteEnabled( true );
    depthStencilStateInfo.SetDepthTestEnabled( true );
    depthStencilStateInfo.SetDepthComparisonFunction( nn::gfx::ComparisonFunction_Less );
    depthStencilStateInfo.SetStencilTestEnabled( false );
    pipelineInfo.SetDepthStencilStateInfo(&depthStencilStateInfo);

    rasterizerStateInfo.SetFillMode( nn::gfx::FillMode_Solid );
    rasterizerStateInfo.SetCullMode( nn::gfx::CullMode_None );
    rasterizerStateInfo.SetFrontFace( nn::gfx::FrontFace_Ccw );
    rasterizerStateInfo.SetScissorEnabled( true );

    pipelineInfo.SetRasterizerStateInfo(&rasterizerStateInfo);

    // primitive testing requires many pipelines
    InitPipeline( quadPointsPipeline, pipelineInfo, nn::gfx::PrimitiveTopology_PointList, &quadPointsPipelineData );
    InitPipeline( quadLinesPipeline, pipelineInfo, nn::gfx::PrimitiveTopology_LineList, &quadLinesPipelineData );
    InitPipeline( quadLineStripPipeline, pipelineInfo, nn::gfx::PrimitiveTopology_LineStrip, &quadLineStripPipelineData );

    // We're going to draw twice to achieve two-sided polygon modes

    // Front face is solid
    rasterizerStateInfo.SetCullMode( nn::gfx::CullMode_Back );
    rasterizerStateInfo.SetFillMode( nn::gfx::FillMode_Solid );
    pipelineInfo.SetRasterizerStateInfo(&rasterizerStateInfo);
    InitPipeline( quadTrianglesPipeline[0], pipelineInfo, nn::gfx::PrimitiveTopology_TriangleList, &quadTrianglesPipelineData[0] );
    InitPipeline( quadTriangleStripPipeline[0], pipelineInfo, nn::gfx::PrimitiveTopology_TriangleStrip, &quadTriangleStripPipelineData[0] );

    // Back face is wireframe
    rasterizerStateInfo.SetCullMode( nn::gfx::CullMode_Front );
    rasterizerStateInfo.SetFillMode( nn::gfx::FillMode_Wireframe );
    pipelineInfo.SetRasterizerStateInfo(&rasterizerStateInfo);
    InitPipeline( quadTrianglesPipeline[1], pipelineInfo, nn::gfx::PrimitiveTopology_TriangleList, &quadTrianglesPipelineData[1] );
    InitPipeline( quadTriangleStripPipeline[1], pipelineInfo, nn::gfx::PrimitiveTopology_TriangleStrip, &quadTriangleStripPipelineData[1] );

    InitMatrices();
    InitViewport();

    return 1;
}

// Print Informations
static void PrintInfo()
{
    f32 origX      = 9.5f;
    f32 offsetX    = 16.0f;
    u32 i;

    const char *primType[] =
    {
        "POINTS",
        "LINES",
        "LINE_STRIP",
        "TRIANGLES",
        "TRIANGLE_STRIP",
    };

    DEMOFontSetGridSize(90,36);

    for(i = 0; i < 5; i++)
    {
        DEMOFontPrintf(origX + offsetX * i, 12, "%s", primType[i]);
    }
}

#if NN_GFX_IS_TARGET_GL
static void SpecialGlPipelineSetup( const void* pVoid )
{
    NN_UNUSED( pVoid );

    // This is necessary until this feature is exposed in nn::gfx
    glPointSize( 4.0f );
    glLineWidth( 4.0f );
}
#endif

static void SetupPipeline(nn::gfx::Pipeline* pPipeline)
{
    DEMOCommandBuffer.SetPipeline( pPipeline );

#if NN_GFX_IS_TARGET_GX
    //  nn::gfx doesn't support this
    GX2SetPointSize(8.0f, 2.0f); // if gl_PointSize is not used in the shader
    GX2SetPointLimits(1.0f, 10.0f); // limits gl_PointSize if used

    // Set Line Width
    // nn::gfx doesn't support this
    GX2SetLineWidth(4.0f);
#elif NN_GFX_IS_TARGET_GL
    DEMOCommandBuffer.Gl4SetUserCommand( SpecialGlPipelineSetup, NULL );
#elif NN_GFX_IS_TARGET_NVN
    nvnCommandBufferSetLineWidth( DEMOCommandBuffer.ToData()->pNvnCommandBuffer, 4.0f );
    nvnCommandBufferSetPointSize( DEMOCommandBuffer.ToData()->pNvnCommandBuffer, 4.0f );
#endif
}

static void SceneUpdate()
{
    ModelTick(pointsMtx, pointsPos);
    ModelTick(linesMtx, linesPos);
    ModelTick(lineStripMtx, lineStripPos);
    ModelTick(trianglesMtx, trianglesPos);
    ModelTick(triangleStripMtx, triangleStripPos);
}

// The draw function for the rendering portions of this app
static int SceneDraw()
{
    DEMOGfxBeforeRender();
    nn::gfx::ColorTargetView* pScanBufferView = DEMOGetColorBufferView();

    SceneUpdate();

    DEMOCommandBuffer.ClearColor( pScanBufferView, 0.2f, 0.2f, 0.2f, 1.0f, NULL);
    DEMOCommandBuffer.ClearDepthStencil( &DEMODepthBufferView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL);

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    // Setup the color/depth buffers
    const nn::gfx::ColorTargetView* renderTargets[1];
    renderTargets[ 0 ] = pScanBufferView;
    DEMOCommandBuffer.SetRenderTargets( 1, renderTargets, &DEMODepthBufferView );

    // Setup viewport scissor
    DEMOCommandBuffer.SetViewportScissorState( &viewportScissorState );

    // Bind vertex buffer
    DEMOCommandBuffer.SetVertexBuffer( BUFFER_IDX, quadVertexBuffer.gpuAddress, QUAD_VTX_STRIDE, quadVertexBuffer.size );

    // Draw Points
    DEMOCommandBuffer.SetConstantBuffer( mtxLoc, nn::gfx::ShaderStage_Vertex, pointsMtx.gpuAddress, pointsMtx.size );

    SetupPipeline( &quadPointsPipeline );
    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_PointList, nn::gfx::IndexFormat_Uint32, quadPointsIndexBuffer.gpuAddress, QUAD_POINTS_IDX_COUNT, 0, 1, 0 );

    // Draw Lines
    DEMOCommandBuffer.SetConstantBuffer( mtxLoc, nn::gfx::ShaderStage_Vertex, linesMtx.gpuAddress, linesMtx.size );

    SetupPipeline( &quadLinesPipeline );
    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_LineList, nn::gfx::IndexFormat_Uint32, quadLinesIndexBuffer.gpuAddress, QUAD_LINES_IDX_COUNT, 0, 1, 0 );

    // Draw Line Strips
    DEMOCommandBuffer.SetConstantBuffer( mtxLoc, nn::gfx::ShaderStage_Vertex, lineStripMtx.gpuAddress, lineStripMtx.size );

    SetupPipeline( &quadLineStripPipeline );
    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_LineStrip, nn::gfx::IndexFormat_Uint32, quadLineStripIndexBuffer.gpuAddress,
        QUAD_LINE_STRIP_IDX_COUNT, 0, 1, 0 );

    // Draw Triangles
    DEMOCommandBuffer.SetConstantBuffer( mtxLoc, nn::gfx::ShaderStage_Vertex, trianglesMtx.gpuAddress, trianglesMtx.size );

    SetupPipeline( &quadTrianglesPipeline[0] );
    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32, quadTrianglesIndexBuffer.gpuAddress, QUAD_TRIANGLES_IDX_COUNT, 0, 1, 0 );
    SetupPipeline( &quadTrianglesPipeline[1] );
    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32, quadTrianglesIndexBuffer.gpuAddress, QUAD_TRIANGLES_IDX_COUNT, 0, 1, 0 );

    // Draw Triangle Strips
    DEMOCommandBuffer.SetConstantBuffer( mtxLoc, nn::gfx::ShaderStage_Vertex, triangleStripMtx.gpuAddress, triangleStripMtx.size );

    SetupPipeline( &quadTriangleStripPipeline[0] );
    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleStrip, nn::gfx::IndexFormat_Uint32, quadTriangleStripIndexBuffer.gpuAddress, QUAD_TRIANGLE_STRIP_IDX_COUNT, 0, 1, 0 );
    SetupPipeline( &quadTriangleStripPipeline[1] );
    DEMOCommandBuffer.DrawIndexed(nn::gfx::PrimitiveTopology_TriangleStrip, nn::gfx::IndexFormat_Uint32, quadTriangleStripIndexBuffer.gpuAddress, QUAD_TRIANGLE_STRIP_IDX_COUNT, 0, 1, 0);

    PrintInfo();

    DEMOGfxDoneRender();

    return 1;
}

void FreeResources()
{
    // Free shaders
    DEMOGfxFreeShaders(&theShader);

    // Free pipelines
    quadPointsPipeline.Finalize( &DEMODevice );
    DEMOFree( quadPointsPipelineData );
    quadLinesPipeline.Finalize( &DEMODevice );
    DEMOFree( quadLinesPipelineData );
    quadLineStripPipeline.Finalize( &DEMODevice );
    DEMOFree( quadLineStripPipelineData );
    quadTrianglesPipeline[0].Finalize( &DEMODevice );
    quadTrianglesPipeline[1].Finalize( &DEMODevice );
    DEMOFree( quadTrianglesPipelineData[0] );
    DEMOFree( quadTrianglesPipelineData[1] );
    quadTriangleStripPipeline[0].Finalize( &DEMODevice );
    quadTriangleStripPipeline[1].Finalize( &DEMODevice );
    DEMOFree( quadTriangleStripPipelineData[0] );
    DEMOFree( quadTriangleStripPipelineData[1] );

    // Free scissor state
    void* freeData = viewportScissorState.GetMemory();
    viewportScissorState.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2( freeData );

    // Free buffers
    quadTriangleStripIndexBuffer.Finalize(  );
    quadTrianglesIndexBuffer.Finalize(  );
    quadLineStripIndexBuffer.Finalize( );
    quadLinesIndexBuffer.Finalize( );
    quadPointsIndexBuffer.Finalize( );
    quadVertexBuffer.Finalize( );

    pointsMtx.Finalize( );
    linesMtx.Finalize( );
    lineStripMtx.Finalize( );
    trianglesMtx.Finalize( );
    triangleStripMtx.Finalize( );
}

//extern "C" void nnMain()
TEST(GfxPrimitives, Run)
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    DEMOInit();
    DEMOTestInit(argc, argv);
    DEMOGfxInit(argc, argv);
    DEMOFontInit();

    SceneInit();
    while (DEMOIsRunning())
    {
#if NN_GFX_IS_TARGET_GX
        DEMOPadRead();
#endif
        SceneDraw();
    }

    FreeResources();
    DEMOFontShutdown();
    DEMOTestShutdown();
    DEMOGfxShutdown();
    DEMOShutdown();

    SUCCEED();
}
