﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>
#include <cmath>

#include <gfx/demo.h>
#include <nnt.h>
#include <nnt/nnt_Argument.h>

// ----- Surface Information

static const int RENDERTARGET_WIDTH = 120;
static const int RENDERTARGET_HEIGHT = 80;

#define SURFACE_WIDTH  ( DEMOColorBufferInfo.GetWidth() )
#define SURFACE_HEIGHT  ( DEMOColorBufferInfo.GetHeight() )

// ----- Shader information
static int g_ShaderFileIdx = 0;
static const char * const SHADER_FILE_1ST[] =
{
    "shaders/multipleRenderTargets/multipleRenderTarget",
    "shaders/multipleRenderTargets/multipleRenderTargetHlslcc",
};
static const char * const SHADER_FILE_2ND[] =
{
    "shaders/multipleRenderTargets/texture2D",
    "shaders/multipleRenderTargets/texture2DHlslcc",
};

static const int POS_IDX = 0;
static const int CLR_IDX = 1;
static const int TEX_IDX = 1;

typedef struct _FirstPassUniform
{
    Mat44 viewMtx;
    Mat44 projMtx;
} FirstPassUniform;

typedef struct _FirstPassDynamicUniform
{
    Mat44 modelMtx;
} FirstPassDynamicUniform;

typedef struct _SecondPassUniform
{
    float offsets[ 4 ];
} SecondPassUniform;

static struct _FirstPass
{
    DEMOGfxPipeline pipeline;

    // Constant Uniform Buffers
    DEMOGfxBuffer constants;
    int constantLocation;

    // Dynamic Uniform Buffers
    DEMOGfxBuffer dynamicUniforms[ 2 ][ 2 ];
    int dynamicUniformLoc;

    // -- Initial Data -- //
    int numIndices;

    // -- Attribute Buffers -- //
    f32* posBuf;
    f32* clrBuf;
    u32* idxBuf;
    DEMOGfxBuffer position;
    DEMOGfxBuffer color;
    DEMOGfxBuffer index;

    // -- Texture Setting -- //
    nn::gfx::Texture myColorBuffer[ 2 ];
    nn::gfx::ColorTargetView myColorBufferView[ 2 ];
    DEMOGfxMemPool* myColorBufferData[ 2 ];

    nn::gfx::Texture myDepthBuffer;
    nn::gfx::DepthStencilView myDepthBufferView;
    DEMOGfxMemPool* myDepthBufferData;
} FirstPass;

static struct _SecondPass
{
    DEMOGfxPipeline pipeline;

    // -- Shader Location -- //
    u32 textureLoc;

    DEMOGfxBuffer constants[ 2 ];
    u32 offsetLoc;

    // -- Initial Data -- //
    int numIndices;

    // -- Attribute Buffers -- //
    f32* posBuf;
    f32* texcoordBuf;
    u32* idxBuf;
    DEMOGfxBuffer position;
    DEMOGfxBuffer texcoord;
    DEMOGfxBuffer index;

    // -- Texture Setting -- //
    nn::gfx::TextureView myTextureBufferView[ 2 ];
    nn::gfx::DescriptorSlot myTextureBufferSlot[ 2 ];
    nn::gfx::Sampler mySampler;
    nn::gfx::DescriptorSlot mySamplerSlot;
} SecondPass;

static nn::gfx::ViewportScissorState g_ViewportScissorState;
static void* g_ViewportScissorStateData;

static float rotateY;

static int s_FrameCount = 0;

// Prototype
static int SceneInit();
static int SceneDraw();

static void FirstPassInit()
{
    rotateY = 0.0f;

    // ----------------- //
    // First pass setup
    // ----------------- //

    // geometry setup
    float posInitData[] = {
        -0.4f, -0.4f, 0.0f,
         0.4f, -0.4f, 0.0f,
         0.0f,  0.4f, 0.0f,
    };

    float clrInitData[] = {
        1.0f, 0.0f, 0.0f,
        0.0f, 1.0f, 0.0f,
        0.0f, 0.0f, 1.0f,
    };

    uint32_t idxInitData[] = {
        0, 1, 2
    };

    FirstPass.numIndices = 3;

    DEMOGfxLoadShadersFromFile(&FirstPass.pipeline.shaders, 0, SHADER_FILE_1ST[g_ShaderFileIdx]);
    // Setup the constant uniform data
    {
        int size = sizeof( FirstPassUniform );
        float projmat[ 4 ][ 4 ] = { { 1.0f, 0.0f, 0.0f, 0.0f },
            { 0.0f, 1.0f, 0.0f, 0.0f },
            { 0.0f, 0.0f, 1.0f, 0.0f },
            { 0.0f, 0.0f, 0.0f, 1.0f } };
        float viewmat[ 4 ][ 4 ] = { { 1.0f, 0.0f, 0.0f, 0.0f },
            { 0.0f, 1.0f, 0.0f, 0.0f },
            { 0.0f, 0.0f, 1.0f, 0.0f },
            { 0.0f, 0.0f, 0.0f, 1.0f } };
        FirstPass.constants.Initialize( size, NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );

        FirstPassUniform* pUniform = FirstPass.constants.Map< FirstPassUniform >();
        memcpy( &pUniform->viewMtx, viewmat, sizeof( Mtx44 ) );
        memcpy( &pUniform->projMtx, projmat, sizeof( Mtx44 ) );
#if NN_GFX_IS_TARGET_GX
        GX2EndianSwap( pUniform, sizeof( *pUniform ) );
#endif
        FirstPass.constantLocation = FirstPass.pipeline.shaders.GetInterfaceSlot(
            nn::gfx::ShaderStage_Vertex,
            nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_constants" );

        FirstPass.constants.Unmap();
    }

    // Setup the dynamic uniform data
    {
        int size = sizeof( FirstPassDynamicUniform );
        FirstPass.dynamicUniformLoc = FirstPass.pipeline.shaders.GetInterfaceSlot(
            nn::gfx::ShaderStage_Vertex,
            nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_dynamics" );
        for ( int frame = 0; frame < 2; frame++ )
        {
            for ( int drawCall = 0; drawCall < 2; drawCall++ )
            {
                FirstPass.dynamicUniforms[ frame ][ drawCall ].Initialize( size, NULL,
                    nn::gfx::GpuAccess_ConstantBuffer, 0 );
            }
        }
    }

    // Attribute Format Setup
    DEMOGfxInitShaderAttribute( &FirstPass.pipeline.shaders, "a_position", POS_IDX, 0, nn::gfx::AttributeFormat_32_32_32_Float );
    DEMOGfxInitShaderAttribute( &FirstPass.pipeline.shaders, "a_color", CLR_IDX, 0, nn::gfx::AttributeFormat_32_32_32_Float );

    // Vertex Buffer Setup
    DEMOGfxInitShaderVertexBuffer( &FirstPass.pipeline.shaders, POS_IDX, sizeof( float ) * 3, 0 );
    DEMOGfxInitShaderVertexBuffer( &FirstPass.pipeline.shaders, CLR_IDX, sizeof( float ) * 3, 0 );

    // Vertex position buffer
    FirstPass.position.Initialize( sizeof( posInitData ), posInitData, nn::gfx::GpuAccess_VertexBuffer, 0 );

    // Vertex color buffer
    FirstPass.color.Initialize( sizeof( clrInitData ), clrInitData, nn::gfx::GpuAccess_VertexBuffer, 0 );

    // Index buffer
    FirstPass.index.Initialize( sizeof( idxInitData ), idxInitData, nn::gfx::GpuAccess_IndexBuffer, 0 );

    // Initialize render depth buffer
    DEMOGfxSetupTextureBuffer( &FirstPass.myDepthBuffer, NULL, NULL, NULL, &FirstPass.myDepthBufferView,
        &FirstPass.myDepthBufferData, RENDERTARGET_WIDTH, RENDERTARGET_HEIGHT, 1, 1, nn::gfx::ImageDimension_2d,
        nn::gfx::ImageFormat_D32_Float, nn::gfx::DepthStencilFetchMode_DepthComponent, 0 );

    // Initialize render color buffer
    DEMOGfxSetupTextureBuffer( &FirstPass.myColorBuffer[ 0 ], NULL, NULL,
        &FirstPass.myColorBufferView[ 0 ], NULL,
        &FirstPass.myColorBufferData[ 0 ], RENDERTARGET_WIDTH, RENDERTARGET_HEIGHT, 1, 1,
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm,
        nn::gfx::DepthStencilFetchMode_DepthComponent, 0 );

    // Initialize render color buffer
    DEMOGfxSetupTextureBuffer( &FirstPass.myColorBuffer[ 1 ], NULL, NULL,
        &FirstPass.myColorBufferView[ 1 ], NULL,
        &FirstPass.myColorBufferData[ 1 ], RENDERTARGET_WIDTH, RENDERTARGET_HEIGHT, 1, 1,
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm,
        nn::gfx::DepthStencilFetchMode_DepthComponent, 0 );

    // Set up the sampler object
    DEMOGfxInitSampler( &SecondPass.mySampler, &SecondPass.mySamplerSlot, nn::gfx::TextureAddressMode_ClampToEdge,
        nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint, nn::gfx::ComparisonFunction_Always );

    // Setup pipeline
    FirstPass.pipeline.colorTargetStateCount = 2;
    FirstPass.pipeline.colorTargetStateInfoArray[ 0 ].SetDefault();
    FirstPass.pipeline.colorTargetStateInfoArray[ 0 ].SetFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );
    FirstPass.pipeline.colorTargetStateInfoArray[ 1 ].SetDefault();
    FirstPass.pipeline.colorTargetStateInfoArray[ 1 ].SetFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );
    FirstPass.pipeline.blendTargetStateCount = 2;
    FirstPass.pipeline.blendTargetStateInfoArray[ 0 ].SetDefault();
    FirstPass.pipeline.blendTargetStateInfoArray[ 1 ].SetDefault();
    FirstPass.pipeline.Initialize( &DEMODevice );
}

static void SecondPassInit()
{
    // ------------------ //
    // Second pass setup
    // ------------------ //
    float posInitData[] = {
        -0.4f, -0.6f, 0.0f,
         0.4f, -0.6f, 0.0f,
        -0.4f,  0.6f, 0.0f,
         0.4f,  0.6f, 0.0f,
    };

    float texInitData[] = {
        0.0f, 1.0f,
        1.0f, 1.0f,
        0.0f, 0.0f,
        1.0f, 0.0f,
    };

    uint32_t idxInitData[] = {
        0, 1, 2, 3,
    };

    SecondPass.numIndices = 4;

    DEMOGfxLoadShadersFromFile(&SecondPass.pipeline.shaders, 0, SHADER_FILE_2ND[g_ShaderFileIdx]);

    SecondPass.textureLoc = SecondPass.pipeline.shaders.GetInterfaceSlot(
        nn::gfx::ShaderStage_Pixel,
        nn::gfx::ShaderInterfaceType_Sampler, "s_texture");

    // Uniform Location lookup
    for ( int idx = 0; idx < 2; idx++ )
    {
        f32 mOffset[4] = { 0.0f, 0.0f, 0.0f, 0.0f };
        int size = sizeof( mOffset );
        mOffset[0] = (idx == 0) ? -0.5f : 0.5f;

#if NN_GFX_IS_TARGET_GX
        GX2EndianSwap( mOffset, sizeof( mOffset ) );
#endif
        SecondPass.offsetLoc = SecondPass.pipeline.shaders.GetInterfaceSlot(
            nn::gfx::ShaderStage_Vertex,
            nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_constants" );
        SecondPass.constants[ idx ].Initialize( size, mOffset,
            nn::gfx::GpuAccess_ConstantBuffer, 0 );
    }

    // Attribute Format Setup
    DEMOGfxInitShaderAttribute( &SecondPass.pipeline.shaders, "a_position", POS_IDX, 0, nn::gfx::AttributeFormat_32_32_32_Float );
    DEMOGfxInitShaderAttribute( &SecondPass.pipeline.shaders, "a_texCoord", TEX_IDX, 0, nn::gfx::AttributeFormat_32_32_Float );

    // Vertex Buffer Setup
    DEMOGfxInitShaderVertexBuffer( &SecondPass.pipeline.shaders, POS_IDX, sizeof( float ) * 3, 0 );
    DEMOGfxInitShaderVertexBuffer( &SecondPass.pipeline.shaders, TEX_IDX, sizeof( float ) * 2, 0 );

    // Vertex position buffer
    SecondPass.position.Initialize( sizeof( posInitData ), posInitData, nn::gfx::GpuAccess_VertexBuffer, 0 );

    // Vertex texcoord buffer
    SecondPass.texcoord.Initialize( sizeof( texInitData ), texInitData, nn::gfx::GpuAccess_VertexBuffer, 0 );

    // Index buffer
    SecondPass.index.Initialize( sizeof( idxInitData ), idxInitData, nn::gfx::GpuAccess_IndexBuffer, 0 );

    // Setup the texture views into the MRT
    DEMOGfxSetupTextureView( &FirstPass.myColorBuffer[ 0 ], &SecondPass.myTextureBufferView[ 0 ], &SecondPass.myTextureBufferSlot[ 0 ], nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm,
        nn::gfx::DepthStencilFetchMode_DepthComponent );
    DEMOGfxSetupTextureView( &FirstPass.myColorBuffer[ 1 ], &SecondPass.myTextureBufferView[ 1 ], &SecondPass.myTextureBufferSlot[ 1 ], nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm,
        nn::gfx::DepthStencilFetchMode_DepthComponent );

    // Setup pipeline
    SecondPass.pipeline.colorTargetStateCount = 1;
    SecondPass.pipeline.colorTargetStateInfoArray[ 0 ].SetDefault();
    SecondPass.pipeline.colorTargetStateInfoArray[ 0 ].SetFormat( DEMOColorBufferInfo.GetImageFormat() );
    SecondPass.pipeline.blendTargetStateCount = 1;
    SecondPass.pipeline.blendTargetStateInfoArray[ 0 ].SetDefault();
    SecondPass.pipeline.Initialize( &DEMODevice );
}

// The init function for the rendering portions of this app
static int SceneInit()
{
    FirstPassInit();

    SecondPassInit();

    DEMOGfxSetViewportScissorState( &g_ViewportScissorState, &g_ViewportScissorStateData, 0.0f, 0.0f,
        ( float )RENDERTARGET_WIDTH, ( float )RENDERTARGET_HEIGHT, 0.0f, 1.0f, ( float )RENDERTARGET_HEIGHT, false );

    return 1;
}

static void FirstPassDraw()
{
    // ------------------------------- //
    //   1ST PASS (RENDERING TEXTURE)
    // ------------------------------- //
    const nn::gfx::ColorTargetView* const colorTargets[] = {
        &FirstPass.myColorBufferView[ 0 ],
        &FirstPass.myColorBufferView[ 1 ]
    };

    DEMOCommandBuffer.ClearColor( &FirstPass.myColorBufferView[ 0 ], 0.4f, 0.4f, 0.4f, 0.0f, NULL );
    DEMOCommandBuffer.ClearColor( &FirstPass.myColorBufferView[ 1 ], 0.2f, 0.75f, 0.2f, 0.0f, NULL );
    DEMOCommandBuffer.ClearDepthStencil( &FirstPass.myDepthBufferView, 1.0f, 0, nn::gfx::DepthStencilClearMode_Depth, NULL );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    DEMOCommandBuffer.SetRenderTargets( 2, colorTargets, &FirstPass.myDepthBufferView );
    DEMOCommandBuffer.SetViewportScissorState( &g_ViewportScissorState );

    DEMOCommandBuffer.SetPipeline( &FirstPass.pipeline.pipeline );

    // Bind position buffer
    DEMOCommandBuffer.SetVertexBuffer( POS_IDX, FirstPass.position.gpuAddress, sizeof( float ) * 3, FirstPass.position.size );
    DEMOCommandBuffer.SetVertexBuffer( CLR_IDX, FirstPass.color.gpuAddress, sizeof( float ) * 3, FirstPass.color.size );

    // Set the constant uniform
    DEMOCommandBuffer.SetConstantBuffer( FirstPass.constantLocation, nn::gfx::ShaderStage_Vertex,
        FirstPass.constants.gpuAddress, FirstPass.constants.size );

    // Update the dynamic uniform
    float modelmat[ 4 ][ 4 ] = { { 1.5f, 0.0f, 0.0f, -0.5f },
                                 { 0.0f, 1.5f, 0.0f, 0.0f },
                                 { 0.0f, 0.0f, 1.0f, 0.0f },
                                 { 0.0f, 0.0f, 0.0f, 1.0f } };
    modelmat[0][0] = cos(rotateY);  modelmat[2][0] = -sin(rotateY);
    modelmat[0][2] = sin(rotateY);  modelmat[2][2] = cos(rotateY);


    FirstPassDynamicUniform* pDynamic = FirstPass.dynamicUniforms[ s_FrameCount & 1 ][ 0 ].Map< FirstPassDynamicUniform >();
    memcpy( &pDynamic->modelMtx, modelmat, sizeof( Mtx44 ) );
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pDynamic, sizeof( *pDynamic ) );
#endif
    FirstPass.dynamicUniforms[ s_FrameCount & 1 ][ 0 ].Unmap();

    // Set the Dynamci Uniform
    DEMOCommandBuffer.SetConstantBuffer( FirstPass.dynamicUniformLoc, nn::gfx::ShaderStage_Vertex,
        FirstPass.dynamicUniforms[ s_FrameCount & 1 ][ 0 ].gpuAddress, FirstPass.dynamicUniforms[ s_FrameCount & 1 ][ 0 ].size );

    // -----------------------
    // draw first triangle
    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32,
        FirstPass.index.gpuAddress, FirstPass.numIndices, 0 );

    // -----------------------
    // draw second triangle
    modelmat[0][0] = cos(-rotateY / 2.0f);  modelmat[2][0] = -sin(-rotateY / 2.0f);
    modelmat[0][2] = sin(-rotateY / 2.0f);  modelmat[2][2] =  cos(-rotateY / 2.0f);
    modelmat[0][3] += 1.0f;

    // Uniform Location Lookup
    pDynamic = FirstPass.dynamicUniforms[ s_FrameCount & 1 ][ 1 ].Map< FirstPassDynamicUniform >();
    memcpy( &pDynamic->modelMtx, modelmat, sizeof( Mtx44 ) );
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pDynamic, sizeof( *pDynamic ) );
#endif
    FirstPass.dynamicUniforms[ s_FrameCount & 1 ][ 1 ].Unmap();
    DEMOCommandBuffer.SetConstantBuffer( FirstPass.dynamicUniformLoc, nn::gfx::ShaderStage_Vertex,
        FirstPass.dynamicUniforms[ s_FrameCount & 1 ][ 1 ].gpuAddress, sizeof( FirstPassDynamicUniform ) );

    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32,
        FirstPass.index.gpuAddress, FirstPass.numIndices, 0 );

    DEMOCommandBuffer.FlushMemory( nn::gfx::GpuAccess_ColorBuffer | nn::gfx::GpuAccess_DepthStencil );
    DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_Texture );
}

static void SecondPassDraw()
{
    // ------------------------------- //
    //   2ND PASS (VISALIZE TEXTURE)
    // ------------------------------- //

    nn::gfx::ColorTargetView* pCurrentScanBuffer = DEMOGetColorBufferView();
    DEMOCommandBuffer.ClearColor( pCurrentScanBuffer, 0.3f, 0.6f, 0.9f, 0.0f, NULL );
    DEMOCommandBuffer.ClearDepthStencil( &DEMODepthBufferView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    DEMOGfxSetDefaultRenderTarget();
    DEMOGfxSetDefaultViewportScissor();

    DEMOCommandBuffer.SetPipeline( &SecondPass.pipeline.pipeline );

    // Bind position buffer
    DEMOCommandBuffer.SetVertexBuffer( POS_IDX, SecondPass.position.gpuAddress, sizeof( float ) * 3, SecondPass.position.size );

    // Bind texcoord buffer
    DEMOCommandBuffer.SetVertexBuffer( TEX_IDX, SecondPass.texcoord.gpuAddress, sizeof( float ) * 2, SecondPass.texcoord.size );

    // ------------------------------------------------------------

    DEMOCommandBuffer.SetConstantBuffer( SecondPass.offsetLoc, nn::gfx::ShaderStage_Vertex,
        SecondPass.constants[ 0 ].gpuAddress, SecondPass.constants[ 0 ].size );

    DEMOCommandBuffer.SetTextureAndSampler( SecondPass.textureLoc, nn::gfx::ShaderStage_Pixel,
        SecondPass.myTextureBufferSlot[ 0 ], SecondPass.mySamplerSlot );

    // -----------------------
    // draw first quad
    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleStrip, nn::gfx::IndexFormat_Uint32,
        SecondPass.index.gpuAddress, SecondPass.numIndices, 0 );

    // ------------------------------------------------------------

    DEMOCommandBuffer.SetConstantBuffer( SecondPass.offsetLoc, nn::gfx::ShaderStage_Vertex,
        SecondPass.constants[ 1 ].gpuAddress, SecondPass.constants[ 1 ].size );

    DEMOCommandBuffer.SetTextureAndSampler( SecondPass.textureLoc, nn::gfx::ShaderStage_Pixel,
        SecondPass.myTextureBufferSlot[ 1 ], SecondPass.mySamplerSlot );

    // -----------------------
    // draw first quad
    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleStrip, nn::gfx::IndexFormat_Uint32,
        SecondPass.index.gpuAddress, SecondPass.numIndices, 0 );
}

// The draw function for the rendering portions of this app
static int SceneDraw()
{
    DEMOGfxBeforeRender();

    FirstPassDraw();
    SecondPassDraw();

    // -----------------------
    // DEMODoneRender
    DEMOGfxDoneRender();

    rotateY += 0.06f;
    s_FrameCount++;

    return 1; // 0 makes it exit
}

//extern "C" void nnMain()
TEST(GfxMultipleRenderTarget, Run)
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    DEMOInit();
    DEMOTestInit(argc, argv);
    DEMOGfxInit(argc, argv);
    DEMOFontInit();
    DEMOTestIsUseHlslccGlsl() ? g_ShaderFileIdx = 1 : g_ShaderFileIdx = 0;

    SceneInit();
    while (DEMOIsRunning())
    {
        SceneDraw();
    }

    // Free pipelines/shaders
    FirstPass.pipeline.Finalize( &DEMODevice );
    SecondPass.pipeline.Finalize( &DEMODevice );

    // Free vertex/index buffers
    FirstPass.position.Finalize();
    FirstPass.color.Finalize();
    FirstPass.index.Finalize();

    SecondPass.position.Finalize();
    SecondPass.texcoord.Finalize();
    SecondPass.index.Finalize();

    // Free the textures
    FirstPass.myColorBufferView[ 0 ].Finalize( &DEMODevice );
    FirstPass.myColorBufferView[ 1 ].Finalize( &DEMODevice );
    FirstPass.myColorBuffer[ 0 ].Finalize( &DEMODevice );
    FirstPass.myColorBuffer[ 1 ].Finalize( &DEMODevice );
    FirstPass.myDepthBufferView.Finalize( &DEMODevice );
    FirstPass.myDepthBuffer.Finalize( &DEMODevice );
    SecondPass.myTextureBufferView[ 0 ].Finalize( &DEMODevice );
    SecondPass.myTextureBufferView[ 1 ].Finalize( &DEMODevice );
    SecondPass.mySampler.Finalize( &DEMODevice );
    FirstPass.myColorBufferData[ 0 ]->Finalize();
    FirstPass.myColorBufferData[ 1 ]->Finalize();
    FirstPass.myDepthBufferData->Finalize();

    // Free the viewport
    {
        g_ViewportScissorState.Finalize( &DEMODevice );
        DEMOGfxFreeMEM2( g_ViewportScissorStateData );
    }

    // Free Uniforms
    FirstPass.constants.Finalize();
    FirstPass.dynamicUniforms[0][0].Finalize();
    FirstPass.dynamicUniforms[0][1].Finalize();
    FirstPass.dynamicUniforms[1][0].Finalize();
    FirstPass.dynamicUniforms[1][1].Finalize();
    SecondPass.constants[ 0 ].Finalize();
    SecondPass.constants[ 1 ].Finalize();

    DEMOFontShutdown();
    DEMOTestShutdown();
    DEMOGfxShutdown();
    DEMOShutdown();

    SUCCEED();
}
