﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/gfx.h>

#if defined( NN_SDK_BUILD_DEBUG ) || defined( NN_SDK_BUILD_DEVELOP )
    #define EXPECT_ASSERT( statement, regex ) \
        EXPECT_DEATH_IF_SUPPORTED( statement, regex )
#else
    #define EXPECT_ASSERT( statement, regex )
#endif

namespace {

template< typename TInfo,
    typename TValue,
    void ( TInfo::*Set )( TValue ),
    TValue ( TInfo::*Get )() const >
struct InfoProperty
{
    typedef TInfo InfoType;
    typedef TValue ValueType;
};

template< typename TInfo,
    typename TValue0,
    typename TValue1,
    void ( TInfo::*Set )( TValue0, TValue1 ),
    TValue0 ( TInfo::*Get0 )() const,
    TValue1 ( TInfo::*Get1 )() const >
struct InfoPairedProperty
{
    typedef TInfo InfoType;
    typedef TValue0 Value0Type;
    typedef TValue1 Value1Type;
};

template< typename TInfo >
void InverseBit( TInfo* pInfo )
{
    NN_STATIC_ASSERT( ( sizeof( TInfo ) & 0x03 ) == 0 );
    for( int* pMemory = reinterpret_cast< int* >( pInfo ),
        *pEnd = pMemory + sizeof( TInfo ) / 4; pMemory < pEnd; ++pMemory )
    {
        *pMemory = *pMemory ^ -1;
    }
}

template< typename T >
struct InfoPropertyTester;

template< typename TInfo,
    typename TValue,
    void ( TInfo::*Set )( TValue ),
    TValue ( TInfo::*Get )() const >
struct InfoPropertyTester< InfoProperty< TInfo, TValue, Set, Get > >
{
    static void Test( TValue value ) NN_NOEXCEPT
    {
        TInfo info;
        ( info.*Set )( value );
        EXPECT_EQ( value, ( info.*Get )( ) );
    }

    static void TestAssert( TValue value ) NN_NOEXCEPT
    {
        TInfo info;
        EXPECT_ASSERT( ( info.*Set )( value ), "" );
    }

    static void TestDefault( TValue value ) NN_NOEXCEPT
    {
        TInfo info;
        info.SetDefault();
        InverseBit( &info );
        info.SetDefault();
        EXPECT_EQ( value, ( info.*Get )() );
    }
};

template< typename TInfo,
    typename TValue0,
    typename TValue1,
    void ( TInfo::*Set )( TValue0, TValue1 ),
    TValue0 ( TInfo::*Get0 )() const,
    TValue1 ( TInfo::*Get1 )() const >
struct InfoPropertyTester< InfoPairedProperty< TInfo, TValue0, TValue1, Set, Get0, Get1 > >
{
    static void Test( TValue0 value0, TValue1 value1 ) NN_NOEXCEPT
    {
        TInfo info;
        ( info.*Set )( value0, value1 );
        EXPECT_EQ( value0, ( info.*Get0 )() );
        EXPECT_EQ( value1, ( info.*Get1 )() );
    }

    static void TestAssert( TValue0 value0, TValue1 value1 ) NN_NOEXCEPT
    {
        TInfo info;
        EXPECT_ASSERT( ( info.*Set )( value0, value1 ), "" );
    }

    static void TestDefault( TValue0 value0, TValue1 value1 ) NN_NOEXCEPT
    {
        TInfo info;
        info.SetDefault();
        InverseBit( &info );
        info.SetDefault();
        EXPECT_EQ( value0, ( info.*Get0 )() );
        EXPECT_EQ( value1, ( info.*Get1 )() );
    }
};

#if 0 // VS2012 が切れたら移行
template< typename TInfo, typename Head, class... Args >
void InfoPropertyTest( Head head, Args... args ) NN_NOEXCEPT
{
    InfoPropertyTester< TInfo >::Test( head );
    InfoPropertyTest< TInfo >( args... );
}
template< typename TInfo >
void InfoPropertyTest() NN_NOEXCEPT
{
}

template< typename TInfo, typename Head, class... Args >
void InfoPropertyTestAssert( Head head, Args... args ) NN_NOEXCEPT
{
    InfoPropertyTester< TInfo >::TestAssert( head );
    InfoPropertyTestAssert< TInfo >( args... );
}
template< typename TInfo >
void InfoPropertyTestAssert() NN_NOEXCEPT
{
}
#else
template< typename TInfo >
void InfoPropertyTest( typename TInfo::ValueType value ) NN_NOEXCEPT
{
    InfoPropertyTester< TInfo >::Test( value );
}
template< typename TInfo >
void InfoPropertyTest( typename TInfo::ValueType value0,
    typename TInfo::ValueType value1 ) NN_NOEXCEPT
{
    InfoPropertyTester< TInfo >::Test( value0 );
    InfoPropertyTester< TInfo >::Test( value1 );
}
template< typename TInfo >
void InfoPropertyTest( typename TInfo::ValueType value0, typename TInfo::ValueType value1,
    typename TInfo::ValueType value2 ) NN_NOEXCEPT
{
    InfoPropertyTester< TInfo >::Test( value0 );
    InfoPropertyTester< TInfo >::Test( value1 );
    InfoPropertyTester< TInfo >::Test( value2 );
}
template< typename TInfo >
void InfoPropertyTest( typename TInfo::ValueType value0, typename TInfo::ValueType value1,
    typename TInfo::ValueType value2, typename TInfo::ValueType value3 ) NN_NOEXCEPT
{
    InfoPropertyTester< TInfo >::Test( value0 );
    InfoPropertyTester< TInfo >::Test( value1 );
    InfoPropertyTester< TInfo >::Test( value2 );
    InfoPropertyTester< TInfo >::Test( value3 );
}

template< typename TInfo >
void InfoPropertyTestAssert( typename TInfo::ValueType value ) NN_NOEXCEPT
{
    InfoPropertyTester< TInfo >::TestAssert( value );
}
template< typename TInfo >
void InfoPropertyTestAssert( typename TInfo::ValueType value0,
    typename TInfo::ValueType value1 ) NN_NOEXCEPT
{
    InfoPropertyTester< TInfo >::TestAssert( value0 );
    InfoPropertyTester< TInfo >::TestAssert( value1 );
}
#endif

template< typename TInfo >
void InfoPropertyTestDefault( typename TInfo::ValueType value ) NN_NOEXCEPT
{
    InfoPropertyTester< TInfo >::TestDefault( value );
}

}

TEST( BufferInfoTest, Size )
{
    typedef InfoProperty< nn::gfx::BufferInfo, size_t,
        &nn::gfx::BufferInfo::SetSize, &nn::gfx::BufferInfo::GetSize > SizeProperty;

    InfoPropertyTest< SizeProperty >( 0, 1024 * 1024 );
    InfoPropertyTestDefault< SizeProperty >( 0 );
}

TEST( BufferInfoTest, GpuAccessFlags )
{
    typedef InfoProperty< nn::gfx::BufferInfo, int, &nn::gfx::BufferInfo::SetGpuAccessFlags,
        &nn::gfx::BufferInfo::GetGpuAccessFlags > GpuAccessFlagsProperty;

    InfoPropertyTest< GpuAccessFlagsProperty >( nn::gfx::GpuAccess_IndexBuffer,
        nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_ConstantBuffer );
    InfoPropertyTestDefault< GpuAccessFlagsProperty >( 0 );
}

TEST( CommandBufferInfoTest, CommandBufferType )
{
    typedef InfoProperty< nn::gfx::CommandBufferInfo, nn::gfx::CommandBufferType,
        &nn::gfx::CommandBufferInfo::SetCommandBufferType,
        &nn::gfx::CommandBufferInfo::GetCommandBufferType > CommandBufferTypeProperty;

    InfoPropertyTest< CommandBufferTypeProperty >( nn::gfx::CommandBufferType_Nested );
    InfoPropertyTestDefault< CommandBufferTypeProperty >( nn::gfx::CommandBufferType_Direct );
}

TEST( CommandBufferInfoTest, QueueCapability )
{
    typedef InfoProperty< nn::gfx::CommandBufferInfo, int,
        &nn::gfx::CommandBufferInfo::SetQueueCapability,
        &nn::gfx::CommandBufferInfo::GetQueueCapability > QueueCapabilityProperty;

    InfoPropertyTest< QueueCapabilityProperty >( nn::gfx::QueueCapability_Graphics,
        nn::gfx::QueueCapability_Compute | nn::gfx::QueueCapability_Graphics );
    InfoPropertyTestDefault< QueueCapabilityProperty >( nn::gfx::QueueCapability_Graphics |
        nn::gfx::QueueCapability_Compute | nn::gfx::QueueCapability_Copy );
}

TEST( DescriptorPoolInfoTest, DescriptorPoolType )
{
    typedef InfoProperty< nn::gfx::DescriptorPoolInfo, nn::gfx::DescriptorPoolType,
        &nn::gfx::DescriptorPoolInfo::SetDescriptorPoolType,
        &nn::gfx::DescriptorPoolInfo::GetDescriptorPoolType > DescriptorPoolTypeProperty;

    InfoPropertyTest< DescriptorPoolTypeProperty >( nn::gfx::DescriptorPoolType_Sampler,
        nn::gfx::DescriptorPoolType_TextureView );
    InfoPropertyTestDefault< DescriptorPoolTypeProperty >( nn::gfx::DescriptorPoolType_BufferView );
}

TEST( DescriptorPoolInfoTest, SlotCount )
{
    typedef InfoProperty< nn::gfx::DescriptorPoolInfo, int, &nn::gfx::DescriptorPoolInfo::SetSlotCount,
        &nn::gfx::DescriptorPoolInfo::GetSlotCount > SlotCountProperty;

    InfoPropertyTest< SlotCountProperty >( 0, 5 );
    InfoPropertyTestAssert< SlotCountProperty >( -4 );
    InfoPropertyTestDefault< SlotCountProperty >( 0 );
}

TEST( DeviceInfoTest, ApiVersion )
{
    typedef InfoPairedProperty< nn::gfx::DeviceInfo, int, int, &nn::gfx::DeviceInfo::SetApiVersion,
        &nn::gfx::DeviceInfo::GetApiMajorVersion, &nn::gfx::DeviceInfo::GetApiMinorVersion > ApiVersionProperty;

    InfoPropertyTester< ApiVersionProperty >::Test( 4, 8 );
    InfoPropertyTester< ApiVersionProperty >::TestDefault( 0, 0 );
}

TEST( DeviceInfoTest, DebugMode )
{
    typedef InfoProperty< nn::gfx::DeviceInfo, nn::gfx::DebugMode,
        &nn::gfx::DeviceInfo::SetDebugMode, &nn::gfx::DeviceInfo::GetDebugMode > DebugModeProperty;

    InfoPropertyTest< DebugModeProperty >( nn::gfx::DebugMode_Disable, nn::gfx::DebugMode_Full );
    InfoPropertyTestDefault< DebugModeProperty >( nn::gfx::DebugMode_Disable );
}

TEST( MemoryPoolInfoTest, MemoryPoolProperty )
{
    typedef InfoProperty< nn::gfx::MemoryPoolInfo, int, &nn::gfx::MemoryPoolInfo::SetMemoryPoolProperty,
        &nn::gfx::MemoryPoolInfo::GetMemoryPoolProperty > MemoryPoolPropertyProperty;

    InfoPropertyTest< MemoryPoolPropertyProperty >( nn::gfx::MemoryPoolProperty_CpuCached
        | nn::gfx::MemoryPoolProperty_GpuInvisible, nn::gfx::MemoryPoolProperty_CpuInvisible
        | nn::gfx::MemoryPoolProperty_GpuUncached | nn::gfx::MemoryPoolProperty_Compressible,
        nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached
        | nn::gfx::MemoryPoolProperty_ShaderCode );
    InfoPropertyTestDefault< MemoryPoolPropertyProperty >(
        nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached );
}

TEST( MemoryPoolInfoTest, PoolMemory )
{
    typedef InfoPairedProperty< nn::gfx::MemoryPoolInfo, void*, size_t,
        &nn::gfx::MemoryPoolInfo::SetPoolMemory, &nn::gfx::MemoryPoolInfo::GetPoolMemory,
        &nn::gfx::MemoryPoolInfo::GetPoolMemorySize > PoolMemoryProperty;

    InfoPropertyTester< PoolMemoryProperty >::Test(
        reinterpret_cast< void* >( 100 ), 256 );
    InfoPropertyTester< PoolMemoryProperty >::TestDefault( NULL, 0 );
}

TEST( QueueInfoTest, Capability )
{
    typedef InfoProperty< nn::gfx::QueueInfo, int,
        &nn::gfx::QueueInfo::SetCapability, &nn::gfx::QueueInfo::GetCapability > CapabilityProperty;

    InfoPropertyTest< CapabilityProperty >( nn::gfx::QueueCapability_Compute,
        nn::gfx::QueueCapability_Graphics | nn::gfx::QueueCapability_Copy );
    InfoPropertyTestDefault< CapabilityProperty >( nn::gfx::QueueCapability_Graphics |
        nn::gfx::QueueCapability_Compute | nn::gfx::QueueCapability_Copy );
}

TEST( SamplerInfoTest, FilterMode )
{
    typedef InfoProperty< nn::gfx::SamplerInfo, nn::gfx::FilterMode,
        &nn::gfx::SamplerInfo::SetFilterMode, &nn::gfx::SamplerInfo::GetFilterMode > FilterModeProperty;

    InfoPropertyTest< FilterModeProperty >(
        nn::gfx::FilterMode_Comparison_MinPoint_MagPoint_MipPoint,
        nn::gfx::FilterMode_Maximum_MinPoint_MagLinear_MipPoint );
    InfoPropertyTestDefault< FilterModeProperty >( nn::gfx::FilterMode_MinLinear_MagLinear_MipLinear );
}

TEST( SamplerInfoTest, AddressU )
{
    typedef InfoProperty< nn::gfx::SamplerInfo, nn::gfx::TextureAddressMode,
        &nn::gfx::SamplerInfo::SetAddressU, &nn::gfx::SamplerInfo::GetAddressU > AddressUProperty;

    InfoPropertyTest< AddressUProperty >( nn::gfx::TextureAddressMode_Mirror,
        nn::gfx::TextureAddressMode_ClampToEdge );
    InfoPropertyTestDefault< AddressUProperty >( nn::gfx::TextureAddressMode_ClampToEdge );
}

TEST( SamplerInfoTest, AddressV )
{
    typedef InfoProperty< nn::gfx::SamplerInfo, nn::gfx::TextureAddressMode,
        &nn::gfx::SamplerInfo::SetAddressV, &nn::gfx::SamplerInfo::GetAddressV > AddressVProperty;

    InfoPropertyTest< AddressVProperty >( nn::gfx::TextureAddressMode_Mirror,
        nn::gfx::TextureAddressMode_ClampToEdge );
    InfoPropertyTestDefault< AddressVProperty >( nn::gfx::TextureAddressMode_ClampToEdge );
}

TEST( SamplerInfoTest, AddressW )
{
    typedef InfoProperty< nn::gfx::SamplerInfo, nn::gfx::TextureAddressMode,
        &nn::gfx::SamplerInfo::SetAddressW, &nn::gfx::SamplerInfo::GetAddressW > AddressWProperty;

    InfoPropertyTest< AddressWProperty >( nn::gfx::TextureAddressMode_Mirror,
        nn::gfx::TextureAddressMode_ClampToEdge );
    InfoPropertyTestDefault< AddressWProperty >( nn::gfx::TextureAddressMode_ClampToEdge );
}

TEST( SamplerInfoTest, ComparisonFunction )
{
    typedef InfoProperty< nn::gfx::SamplerInfo, nn::gfx::ComparisonFunction,
        &nn::gfx::SamplerInfo::SetComparisonFunction, &nn::gfx::SamplerInfo::GetComparisonFunction >
        ComparisonFunctionProperty;

    InfoPropertyTest< ComparisonFunctionProperty >(
        nn::gfx::ComparisonFunction_Greater,
        nn::gfx::ComparisonFunction_Never );
    InfoPropertyTestDefault< ComparisonFunctionProperty >( nn::gfx::ComparisonFunction_Never );
}

TEST( SamplerInfoTest, BorderColorType )
{
    typedef InfoProperty< nn::gfx::SamplerInfo, nn::gfx::TextureBorderColorType,
        &nn::gfx::SamplerInfo::SetBorderColorType, &nn::gfx::SamplerInfo::GetBorderColorType >
        BorderColorTypeProperty;

    InfoPropertyTest< BorderColorTypeProperty >( nn::gfx::TextureBorderColorType_OpaqueBlack,
        nn::gfx::TextureBorderColorType_White );
    InfoPropertyTestDefault< BorderColorTypeProperty >( nn::gfx::TextureBorderColorType_White );
}

TEST( SamplerInfoTest, MaxAnisotropy )
{
    typedef InfoProperty< nn::gfx::SamplerInfo, int, &nn::gfx::SamplerInfo::SetMaxAnisotropy,
        &nn::gfx::SamplerInfo::GetMaxAnisotropy > MaxAnisotropyProperty;

    InfoPropertyTest< MaxAnisotropyProperty >( 0, 8 );
    InfoPropertyTestAssert< MaxAnisotropyProperty >( -4 );
    InfoPropertyTestDefault< MaxAnisotropyProperty >( 1 );
}

TEST( SamplerInfoTest, MinLod )
{
    typedef InfoProperty< nn::gfx::SamplerInfo, float,
        &nn::gfx::SamplerInfo::SetMinLod, &nn::gfx::SamplerInfo::GetMinLod > MinLodProperty;

    InfoPropertyTest< MinLodProperty >( 0.0f, 10.0f );
    InfoPropertyTestDefault< MinLodProperty >( -1000.0f );
}

TEST( SamplerInfoTest, MaxLod )
{
    typedef InfoProperty< nn::gfx::SamplerInfo, float,
        &nn::gfx::SamplerInfo::SetMaxLod, &nn::gfx::SamplerInfo::GetMaxLod > MaxLodProperty;

    InfoPropertyTest< MaxLodProperty >( 0.0f, 10.0f );
    InfoPropertyTestDefault< MaxLodProperty >( 1000.0f );
}

TEST( SamplerInfoTest, LodBias )
{
    typedef InfoProperty< nn::gfx::SamplerInfo, float,
        &nn::gfx::SamplerInfo::SetLodBias, &nn::gfx::SamplerInfo::GetLodBias > LodBiasProperty;

    InfoPropertyTest< LodBiasProperty >( 0.0f, 2.0f );
    InfoPropertyTestDefault< LodBiasProperty >( 0.0f );
}

TEST( ShaderInfoTest, SeparationEnabled )
{
    typedef InfoProperty< nn::gfx::ShaderInfo, bool, &nn::gfx::ShaderInfo::SetSeparationEnabled,
        &nn::gfx::ShaderInfo::IsSeparationEnabled > SeparationEnabledProperty;

    InfoPropertyTest< SeparationEnabledProperty >( false, true );
    InfoPropertyTestDefault< SeparationEnabledProperty >( false );
}

TEST( ShaderInfoTest, CodeType )
{
    typedef InfoProperty< nn::gfx::ShaderInfo, nn::gfx::ShaderCodeType,
        &nn::gfx::ShaderInfo::SetCodeType, &nn::gfx::ShaderInfo::GetCodeType > CodeTypeProperty;

    InfoPropertyTest< CodeTypeProperty >(
        nn::gfx::ShaderCodeType_Binary, nn::gfx::ShaderCodeType_SourceArray );
    InfoPropertyTestDefault< CodeTypeProperty >( nn::gfx::ShaderCodeType_Source );
}

TEST( ShaderInfoTest, SourceFormat )
{
    typedef InfoProperty< nn::gfx::ShaderInfo, nn::gfx::ShaderSourceFormat,
        &nn::gfx::ShaderInfo::SetSourceFormat, &nn::gfx::ShaderInfo::GetSourceFormat > SourceFormatProperty;

    InfoPropertyTest< SourceFormatProperty >( nn::gfx::ShaderSourceFormat_Glsl );
    InfoPropertyTestDefault< SourceFormatProperty >( nn::gfx::ShaderSourceFormat_Glsl );
}

TEST( ShaderInfoTest, BinaryFormat )
{
    typedef InfoProperty< nn::gfx::ShaderInfo, int, &nn::gfx::ShaderInfo::SetBinaryFormat,
        &nn::gfx::ShaderInfo::GetBinaryFormat > BinaryFormatProperty;

    InfoPropertyTest< BinaryFormatProperty >( 4 );
    InfoPropertyTestDefault< BinaryFormatProperty >( 0 );
}

TEST( ShaderInfoTest, ShaderCodePtrProperty )
{
    nn::gfx::ShaderInfo info;

    const void* pVertexCode = reinterpret_cast< const void* >( 100 );
    const void* pHullCode = reinterpret_cast< const void* >( 200 );
    const void* pDomainCode = reinterpret_cast< const void* >( 300 );
    const void* pGeometryCode = reinterpret_cast< const void* >( 400 );
    const void* pPixelCode = reinterpret_cast< const void* >( 500 );
    const void* pComputeCode = reinterpret_cast< const void* >( 600 );
    info.SetShaderCodePtr( nn::gfx::ShaderStage_Vertex, pVertexCode );
    info.SetShaderCodePtr( nn::gfx::ShaderStage_Hull, pHullCode );
    info.SetShaderCodePtr( nn::gfx::ShaderStage_Domain, pDomainCode );
    info.SetShaderCodePtr( nn::gfx::ShaderStage_Geometry, pGeometryCode );
    info.SetShaderCodePtr( nn::gfx::ShaderStage_Pixel, pPixelCode );
    info.SetShaderCodePtr( nn::gfx::ShaderStage_Compute, pComputeCode );
    EXPECT_EQ( pVertexCode, info.GetShaderCodePtr( nn::gfx::ShaderStage_Vertex ) );
    EXPECT_EQ( pHullCode, info.GetShaderCodePtr( nn::gfx::ShaderStage_Hull ) );
    EXPECT_EQ( pDomainCode, info.GetShaderCodePtr( nn::gfx::ShaderStage_Domain ) );
    EXPECT_EQ( pGeometryCode, info.GetShaderCodePtr( nn::gfx::ShaderStage_Geometry ) );
    EXPECT_EQ( pPixelCode, info.GetShaderCodePtr( nn::gfx::ShaderStage_Pixel ) );
    EXPECT_EQ( pComputeCode, info.GetShaderCodePtr( nn::gfx::ShaderStage_Compute ) );

    info.SetDefault();
    InverseBit( &info );
    info.SetDefault();
    EXPECT_EQ( NULL, info.GetShaderCodePtr( nn::gfx::ShaderStage_Vertex ) );
    EXPECT_EQ( NULL, info.GetShaderCodePtr( nn::gfx::ShaderStage_Hull ) );
    EXPECT_EQ( NULL, info.GetShaderCodePtr( nn::gfx::ShaderStage_Domain ) );
    EXPECT_EQ( NULL, info.GetShaderCodePtr( nn::gfx::ShaderStage_Geometry ) );
    EXPECT_EQ( NULL, info.GetShaderCodePtr( nn::gfx::ShaderStage_Pixel ) );
    EXPECT_EQ( NULL, info.GetShaderCodePtr( nn::gfx::ShaderStage_Compute ) );
}

TEST( MultisampleStateInfoTest, SampleCount )
{
    typedef InfoProperty< nn::gfx::MultisampleStateInfo, int,
        &nn::gfx::MultisampleStateInfo::SetSampleCount,
        &nn::gfx::MultisampleStateInfo::GetSampleCount > SampleCountProperty;

    InfoPropertyTest< SampleCountProperty >( 0, 1, 4 );
    InfoPropertyTestAssert< SampleCountProperty >( -1 );
    InfoPropertyTestDefault< SampleCountProperty >( 1 );
}

TEST( MultisampleStateInfoTest, SampleMask )
{
    typedef InfoProperty< nn::gfx::MultisampleStateInfo, int,
        &nn::gfx::MultisampleStateInfo::SetSampleMask,
        &nn::gfx::MultisampleStateInfo::GetSampleMask > SampleMaskProperty;

    InfoPropertyTest< SampleMaskProperty >( 0, 0xFF );
    InfoPropertyTestDefault< SampleMaskProperty >( 0xFFFFFFFF );
}

TEST( MultisampleStateInfoTest, AlphaToCoverageEnabled )
{
    typedef InfoProperty< nn::gfx::MultisampleStateInfo, bool,
        &nn::gfx::MultisampleStateInfo::SetAlphaToCoverageEnabled,
        &nn::gfx::MultisampleStateInfo::IsAlphaToCoverageEnabled > AlphaToCoverageEnabled;

    InfoPropertyTest< AlphaToCoverageEnabled >( false, true );
    InfoPropertyTestDefault< AlphaToCoverageEnabled >( false );
}

TEST( RasterizerStateInfoTest, FillMode )
{
    typedef InfoProperty< nn::gfx::RasterizerStateInfo, nn::gfx::FillMode,
        &nn::gfx::RasterizerStateInfo::SetFillMode,
        &nn::gfx::RasterizerStateInfo::GetFillMode > FillModeProperty;

    InfoPropertyTest< FillModeProperty >( nn::gfx::FillMode_Solid, nn::gfx::FillMode_Wireframe );
    InfoPropertyTestDefault< FillModeProperty >( nn::gfx::FillMode_Solid );
}

TEST( RasterizerStateInfoTest, FrontFace )
{
    typedef InfoProperty< nn::gfx::RasterizerStateInfo, nn::gfx::FrontFace,
        &nn::gfx::RasterizerStateInfo::SetFrontFace,
        &nn::gfx::RasterizerStateInfo::GetFrontFace > FrontFaceProperty;

    InfoPropertyTest< FrontFaceProperty >( nn::gfx::FrontFace_Cw, nn::gfx::FrontFace_Ccw );
    InfoPropertyTestDefault< FrontFaceProperty >( nn::gfx::FrontFace_Ccw );
}

TEST( RasterizerStateInfoTest, CullMode )
{
    typedef InfoProperty< nn::gfx::RasterizerStateInfo, nn::gfx::CullMode,
        &nn::gfx::RasterizerStateInfo::SetCullMode,
        &nn::gfx::RasterizerStateInfo::GetCullMode > CullModeProperty;

    InfoPropertyTest< CullModeProperty >( nn::gfx::CullMode_Back, nn::gfx::CullMode_None );
    InfoPropertyTestDefault< CullModeProperty >( nn::gfx::CullMode_Back );
}

TEST( RasterizerStateInfoTest, PrimitiveTopologyType )
{
    typedef InfoProperty< nn::gfx::RasterizerStateInfo, nn::gfx::PrimitiveTopologyType,
        &nn::gfx::RasterizerStateInfo::SetPrimitiveTopologyType,
        &nn::gfx::RasterizerStateInfo::GetPrimitiveTopologyType > PrimitiveTopologyTypeProperty;

    InfoPropertyTest< PrimitiveTopologyTypeProperty >(
        nn::gfx::PrimitiveTopologyType_Point, nn::gfx::PrimitiveTopologyType_Triangle );
    InfoPropertyTestDefault< PrimitiveTopologyTypeProperty >(
        nn::gfx::PrimitiveTopologyType_Triangle );
}

TEST( RasterizerStateInfoTest, RasterEnabled )
{
    typedef InfoProperty< nn::gfx::RasterizerStateInfo, bool,
        &nn::gfx::RasterizerStateInfo::SetRasterEnabled,
        &nn::gfx::RasterizerStateInfo::IsRasterEnabled > RasterEnabledProperty;

    InfoPropertyTest< RasterEnabledProperty >( false, true );
    InfoPropertyTestDefault< RasterEnabledProperty >( true );
}

TEST( RasterizerStateInfoTest, MultisampleEnabled )
{
    typedef InfoProperty< nn::gfx::RasterizerStateInfo, bool,
        &nn::gfx::RasterizerStateInfo::SetMultisampleEnabled,
        &nn::gfx::RasterizerStateInfo::IsMultisampleEnabled > MultisampleEnabledProperty;

    InfoPropertyTest< MultisampleEnabledProperty >( false, true );
    InfoPropertyTestDefault< MultisampleEnabledProperty >( false );
}

TEST( RasterizerStateInfoTest, DepthClipEnabled )
{
    typedef InfoProperty< nn::gfx::RasterizerStateInfo, bool,
        &nn::gfx::RasterizerStateInfo::SetDepthClipEnabled,
        &nn::gfx::RasterizerStateInfo::IsDepthClipEnabled > DepthClipEnabledProperty;

    InfoPropertyTest< DepthClipEnabledProperty >( false, true );
    InfoPropertyTestDefault< DepthClipEnabledProperty >( false );
}

TEST( RasterizerStateInfoTest, ScissorEnabled )
{
    typedef InfoProperty< nn::gfx::RasterizerStateInfo, bool,
        &nn::gfx::RasterizerStateInfo::SetScissorEnabled,
        &nn::gfx::RasterizerStateInfo::IsScissorEnabled > ScissorEnabledProperty;

    InfoPropertyTest< ScissorEnabledProperty >( false, true );
    InfoPropertyTestDefault< ScissorEnabledProperty >( false );
}

TEST( RasterizerStateInfoTest, SlopeScaledDepthBias )
{
    typedef InfoProperty< nn::gfx::RasterizerStateInfo, float,
        &nn::gfx::RasterizerStateInfo::SetSlopeScaledDepthBias,
        &nn::gfx::RasterizerStateInfo::GetSlopeScaledDepthBias > SlopeScaledDepthBiasProperty;

    InfoPropertyTest< SlopeScaledDepthBiasProperty >( 0.0f, 4.0f );
    InfoPropertyTestDefault< SlopeScaledDepthBiasProperty >( 0.0f );
}

TEST( RasterizerStateInfoTest, DepthBias )
{
    typedef InfoProperty< nn::gfx::RasterizerStateInfo, int,
        &nn::gfx::RasterizerStateInfo::SetDepthBias,
        &nn::gfx::RasterizerStateInfo::GetDepthBias > DepthBiasProperty;

    InfoPropertyTest< DepthBiasProperty >( 0, 4 );
    InfoPropertyTestDefault< DepthBiasProperty >( 0 );
}

TEST( RasterizerStateInfoTest, DepthBiasClamp )
{
    typedef InfoProperty< nn::gfx::RasterizerStateInfo, float,
        &nn::gfx::RasterizerStateInfo::SetDepthBiasClamp,
        &nn::gfx::RasterizerStateInfo::GetDepthBiasClamp > DepthBiasClampProperty;

    InfoPropertyTest< DepthBiasClampProperty >( 0.0f, 1.0f );
    InfoPropertyTestDefault< DepthBiasClampProperty >( 0.0f );
}

TEST( RasterizerStateInfoTest, ConservativeRasterizationMode )
{
    typedef InfoProperty< nn::gfx::RasterizerStateInfo, nn::gfx::ConservativeRasterizationMode,
        &nn::gfx::RasterizerStateInfo::SetConservativeRasterizationMode,
        &nn::gfx::RasterizerStateInfo::GetConservativeRasterizationMode >
        ConservativeRasterizationModeProperty;

    InfoPropertyTest< ConservativeRasterizationModeProperty >(
        nn::gfx::ConservativeRasterizationMode_Enable );
    InfoPropertyTestDefault< ConservativeRasterizationModeProperty >(
        nn::gfx::ConservativeRasterizationMode_Disable );
}

TEST( BlendTargetStateInfoTest, BlendEnabled )
{
    typedef InfoProperty< nn::gfx::BlendTargetStateInfo, bool,
        &nn::gfx::BlendTargetStateInfo::SetBlendEnabled,
        &nn::gfx::BlendTargetStateInfo::IsBlendEnabled > BlendEnabledProperty;

    InfoPropertyTest< BlendEnabledProperty >( false, true );
    InfoPropertyTestDefault< BlendEnabledProperty >( false );
}

TEST( BlendTargetStateInfoTest, SourceColorBlendFactor )
{
    typedef InfoProperty< nn::gfx::BlendTargetStateInfo, nn::gfx::BlendFactor,
        &nn::gfx::BlendTargetStateInfo::SetSourceColorBlendFactor,
        &nn::gfx::BlendTargetStateInfo::GetSourceColorBlendFactor > SourceColorBlendFactorProperty;

    InfoPropertyTest< SourceColorBlendFactorProperty >(
        nn::gfx::BlendFactor_OneMinusConstantAlpha );
    InfoPropertyTestDefault< SourceColorBlendFactorProperty >( nn::gfx::BlendFactor_One );
}

TEST( BlendTargetStateInfoTest, DestinationColorBlendFactor )
{
    typedef InfoProperty< nn::gfx::BlendTargetStateInfo, nn::gfx::BlendFactor,
        &nn::gfx::BlendTargetStateInfo::SetDestinationColorBlendFactor,
        &nn::gfx::BlendTargetStateInfo::GetDestinationColorBlendFactor > DestinationColorBlendFactorProperty;

    InfoPropertyTest< DestinationColorBlendFactorProperty >(
        nn::gfx::BlendFactor_SourceAlphaSaturate );
    InfoPropertyTestDefault< DestinationColorBlendFactorProperty >( nn::gfx::BlendFactor_Zero );
}

TEST( BlendTargetStateInfoTest, ColorBlendFunction )
{
    typedef InfoProperty< nn::gfx::BlendTargetStateInfo, nn::gfx::BlendFunction,
        &nn::gfx::BlendTargetStateInfo::SetColorBlendFunction,
        &nn::gfx::BlendTargetStateInfo::GetColorBlendFunction > ColorBlendFunctionProperty;

    InfoPropertyTest< ColorBlendFunctionProperty >( nn::gfx::BlendFunction_Min,
        nn::gfx::BlendFunction_ReverseSubtract );
    InfoPropertyTestDefault< ColorBlendFunctionProperty >( nn::gfx::BlendFunction_Add );
}

TEST( BlendTargetStateInfoTest, SourceAlphaBlendFactor )
{
    typedef InfoProperty< nn::gfx::BlendTargetStateInfo, nn::gfx::BlendFactor,
        &nn::gfx::BlendTargetStateInfo::SetSourceAlphaBlendFactor,
        &nn::gfx::BlendTargetStateInfo::GetSourceAlphaBlendFactor > SourceAlphaBlendFactorProperty;

    InfoPropertyTest< SourceAlphaBlendFactorProperty >(
        nn::gfx::BlendFactor_ConstantAlpha );
    InfoPropertyTestDefault< SourceAlphaBlendFactorProperty >( nn::gfx::BlendFactor_One );
}

TEST( BlendTargetStateInfoTest, DestinationAlphaBlendFactor )
{
    typedef InfoProperty< nn::gfx::BlendTargetStateInfo, nn::gfx::BlendFactor,
        &nn::gfx::BlendTargetStateInfo::SetDestinationAlphaBlendFactor,
        &nn::gfx::BlendTargetStateInfo::GetDestinationAlphaBlendFactor > DestinationAlphaBlendFactorProperty;

    InfoPropertyTest< DestinationAlphaBlendFactorProperty >(
        nn::gfx::BlendFactor_DestinationAlpha );
    InfoPropertyTestDefault< DestinationAlphaBlendFactorProperty >( nn::gfx::BlendFactor_Zero );
}

TEST( BlendTargetStateInfoTest, AlphaBlendFunction )
{
    typedef InfoProperty< nn::gfx::BlendTargetStateInfo, nn::gfx::BlendFunction,
        &nn::gfx::BlendTargetStateInfo::SetAlphaBlendFunction,
        &nn::gfx::BlendTargetStateInfo::GetAlphaBlendFunction > AlphaBlendFunctionProperty;

    InfoPropertyTest< AlphaBlendFunctionProperty >(
        nn::gfx::BlendFunction_Max, nn::gfx::BlendFunction_Add );
    InfoPropertyTestDefault< AlphaBlendFunctionProperty >( nn::gfx::BlendFunction_Add );
}

TEST( BlendTargetStateInfoTest, ChannelMask )
{
    typedef InfoProperty< nn::gfx::BlendTargetStateInfo, int,
        &nn::gfx::BlendTargetStateInfo::SetChannelMask,
        &nn::gfx::BlendTargetStateInfo::GetChannelMask > ChannelMaskProperty;

    InfoPropertyTest< ChannelMaskProperty >( 0, nn::gfx::ChannelMask_All,
        nn::gfx::ChannelMask_Red | nn::gfx::ChannelMask_Green );
    InfoPropertyTestDefault< ChannelMaskProperty >( nn::gfx::ChannelMask_All );
}

TEST( BlendStateInfoTest, AlphaToCoverageEnabled )
{
    typedef InfoProperty< nn::gfx::BlendStateInfo, bool,
        &nn::gfx::BlendStateInfo::SetAlphaToCoverageEnabled,
        &nn::gfx::BlendStateInfo::IsAlphaToCoverageEnabled > AlphaToCoverageEnabledProperty;

    InfoPropertyTest< AlphaToCoverageEnabledProperty >( false, true );
    InfoPropertyTestDefault< AlphaToCoverageEnabledProperty >( false );
}

TEST( BlendStateInfoTest, DualSourceBlendEnabled )
{
    typedef InfoProperty< nn::gfx::BlendStateInfo, bool,
        &nn::gfx::BlendStateInfo::SetDualSourceBlendEnabled,
        &nn::gfx::BlendStateInfo::IsDualSourceBlendEnabled > DualSourceBlendEnabledProperty;

    InfoPropertyTest< DualSourceBlendEnabledProperty >( false, true );
    InfoPropertyTestDefault< DualSourceBlendEnabledProperty >( false );
}

TEST( BlendStateInfoTest, IndependentBlendEnabled )
{
    typedef InfoProperty< nn::gfx::BlendStateInfo, bool,
        &nn::gfx::BlendStateInfo::SetIndependentBlendEnabled,
        &nn::gfx::BlendStateInfo::IsIndependentBlendEnabled > IndependentBlendEnabledProperty;

    InfoPropertyTest< IndependentBlendEnabledProperty >( false, true );
    InfoPropertyTestDefault< IndependentBlendEnabledProperty >( false );
}

TEST( BlendStateInfoTest, LogicOperationEnabled )
{
    typedef InfoProperty< nn::gfx::BlendStateInfo, bool,
        &nn::gfx::BlendStateInfo::SetLogicOperationEnabled,
        &nn::gfx::BlendStateInfo::IsLogicOperationEnabled > LogicOperationEnabledProperty;

    InfoPropertyTest< LogicOperationEnabledProperty >( false, true );
    InfoPropertyTestDefault< LogicOperationEnabledProperty >( false );
}

TEST( BlendStateInfoTest, BlendConstant )
{
    nn::gfx::BlendStateInfo info;
    float red = 1.0f;
    float green = 0.3f;
    float blue = 0.24f;
    float alpha = 0.01f;
    info.SetBlendConstant( red, green, blue, alpha );
    EXPECT_EQ( red, info.GetBlendConstant( nn::gfx::ColorChannel_Red ) );
    EXPECT_EQ( green, info.GetBlendConstant( nn::gfx::ColorChannel_Green ) );
    EXPECT_EQ( blue, info.GetBlendConstant( nn::gfx::ColorChannel_Blue ) );
    EXPECT_EQ( alpha, info.GetBlendConstant( nn::gfx::ColorChannel_Alpha ) );

    info.SetDefault();
    InverseBit( &info );
    info.SetDefault();
    EXPECT_EQ( 0.0f, info.GetBlendConstant( nn::gfx::ColorChannel_Red ) );
    EXPECT_EQ( 0.0f, info.GetBlendConstant( nn::gfx::ColorChannel_Green ) );
    EXPECT_EQ( 0.0f, info.GetBlendConstant( nn::gfx::ColorChannel_Blue ) );
    EXPECT_EQ( 1.0f, info.GetBlendConstant( nn::gfx::ColorChannel_Alpha ) );
}

TEST( BlendStateInfoTest, BlendTargetStateInfoArray )
{
    typedef InfoPairedProperty< nn::gfx::BlendStateInfo, const nn::gfx::BlendTargetStateInfo*, int,
        &nn::gfx::BlendStateInfo::SetBlendTargetStateInfoArray,
        &nn::gfx::BlendStateInfo::GetBlendTargetStateInfoArray,
        &nn::gfx::BlendStateInfo::GetBlendTargetCount > BlendTargetStateInfoArrayProperty;

    InfoPropertyTester< BlendTargetStateInfoArrayProperty >::Test(
        reinterpret_cast< const nn::gfx::BlendTargetStateInfo* >( 20 ), 3 );
    InfoPropertyTester< BlendTargetStateInfoArrayProperty >::TestAssert( NULL, -3 );
    InfoPropertyTester< BlendTargetStateInfoArrayProperty >::TestDefault( NULL, 0 );
}

TEST( StencilStateInfoTest, StencilFailOperation )
{
    typedef InfoProperty< nn::gfx::StencilStateInfo, nn::gfx::StencilOperation,
        &nn::gfx::StencilStateInfo::SetStencilFailOperation,
        &nn::gfx::StencilStateInfo::GetStencilFailOperation > StencilFailOperationProperty;

    InfoPropertyTest< StencilFailOperationProperty >(
        nn::gfx::StencilOperation_Invert, nn::gfx::StencilOperation_DecrementClamp );
    InfoPropertyTestDefault< StencilFailOperationProperty >( nn::gfx::StencilOperation_Keep );
}

TEST( StencilStateInfoTest, DepthFailOperation )
{
    typedef InfoProperty< nn::gfx::StencilStateInfo, nn::gfx::StencilOperation,
        &nn::gfx::StencilStateInfo::SetDepthFailOperation,
        &nn::gfx::StencilStateInfo::GetDepthFailOperation > DepthFailOperationProperty;

    InfoPropertyTest< DepthFailOperationProperty >(
        nn::gfx::StencilOperation_IncrementClamp, nn::gfx::StencilOperation_Replace );
    InfoPropertyTestDefault< DepthFailOperationProperty >( nn::gfx::StencilOperation_Keep );
}

TEST( StencilStateInfoTest, DepthPassOperation )
{
    typedef InfoProperty< nn::gfx::StencilStateInfo, nn::gfx::StencilOperation,
        &nn::gfx::StencilStateInfo::SetDepthPassOperation,
        &nn::gfx::StencilStateInfo::GetDepthPassOperation > DepthPassOperationProperty;

    InfoPropertyTest< DepthPassOperationProperty >(
        nn::gfx::StencilOperation_Zero, nn::gfx::StencilOperation_DecrementWrap );
    InfoPropertyTestDefault< DepthPassOperationProperty >( nn::gfx::StencilOperation_Keep );
}

TEST( StencilStateInfoTest, ComparisonFunction )
{
    typedef InfoProperty< nn::gfx::StencilStateInfo, nn::gfx::ComparisonFunction,
        &nn::gfx::StencilStateInfo::SetComparisonFunction,
        &nn::gfx::StencilStateInfo::GetComparisonFunction > ComparisonFunctionProperty;

    InfoPropertyTest< ComparisonFunctionProperty >( nn::gfx::ComparisonFunction_Never,
        nn::gfx::ComparisonFunction_GreaterEqual );
    InfoPropertyTestDefault< ComparisonFunctionProperty >( nn::gfx::ComparisonFunction_Always );
}

TEST( StencilStateInfoTest, StencilRef )
{
    typedef InfoProperty< nn::gfx::StencilStateInfo, int, &nn::gfx::StencilStateInfo::SetStencilRef,
        &nn::gfx::StencilStateInfo::GetStencilRef > StencilRefProperty;

    InfoPropertyTest< StencilRefProperty >( 0, 4 );
    InfoPropertyTestDefault< StencilRefProperty >( 0 );
}

TEST( DepthStencilStateInfoTest, DepthComparisonFunction )
{
    typedef InfoProperty< nn::gfx::DepthStencilStateInfo, nn::gfx::ComparisonFunction,
        &nn::gfx::DepthStencilStateInfo::SetDepthComparisonFunction,
        &nn::gfx::DepthStencilStateInfo::GetDepthComparisonFunction > DepthComparisonFunctionProperty;

    InfoPropertyTest< DepthComparisonFunctionProperty >(
        nn::gfx::ComparisonFunction_Equal, nn::gfx::ComparisonFunction_Less );
    InfoPropertyTestDefault< DepthComparisonFunctionProperty >( nn::gfx::ComparisonFunction_Less );
}

TEST( DepthStencilStateInfoTest, DepthTestEnabled )
{
    typedef InfoProperty< nn::gfx::DepthStencilStateInfo, bool,
        &nn::gfx::DepthStencilStateInfo::SetDepthTestEnabled,
        &nn::gfx::DepthStencilStateInfo::IsDepthTestEnabled > DepthTestEnabledProperty;

    InfoPropertyTest< DepthTestEnabledProperty >( false, true );
    InfoPropertyTestDefault< DepthTestEnabledProperty >( false );
}

TEST( DepthStencilStateInfoTest, DepthWriteEnabled )
{
    typedef InfoProperty< nn::gfx::DepthStencilStateInfo, bool,
        &nn::gfx::DepthStencilStateInfo::SetDepthWriteEnabled,
        &nn::gfx::DepthStencilStateInfo::IsDepthWriteEnabled > DepthWriteEnabledProperty;

    InfoPropertyTest< DepthWriteEnabledProperty >( false, true );
    InfoPropertyTestDefault< DepthWriteEnabledProperty >( false );
}

TEST( DepthStencilStateInfoTest, StencilTestEnabled )
{
    typedef InfoProperty< nn::gfx::DepthStencilStateInfo, bool,
        &nn::gfx::DepthStencilStateInfo::SetStencilTestEnabled,
        &nn::gfx::DepthStencilStateInfo::IsStencilTestEnabled > StencilTestEnabledProperty;

    InfoPropertyTest< StencilTestEnabledProperty >( false, true );
    InfoPropertyTestDefault< StencilTestEnabledProperty >( false );
}

TEST( DepthStencilStateInfoTest, DepthBoundsTestEnabled )
{
    typedef InfoProperty< nn::gfx::DepthStencilStateInfo, bool,
        &nn::gfx::DepthStencilStateInfo::SetDepthBoundsTestEnabled,
        &nn::gfx::DepthStencilStateInfo::IsDepthBoundsTestEnabled > DepthBoundsTestEnabledProperty;

    InfoPropertyTest< DepthBoundsTestEnabledProperty >( false, true );
    InfoPropertyTestDefault< DepthBoundsTestEnabledProperty >( false );
}

TEST( DepthStencilStateInfoTest, StencilReadMask )
{
    typedef InfoProperty< nn::gfx::DepthStencilStateInfo, int,
        &nn::gfx::DepthStencilStateInfo::SetStencilReadMask,
        &nn::gfx::DepthStencilStateInfo::GetStencilReadMask > StencilReadMaskProperty;

    InfoPropertyTest< StencilReadMaskProperty >( 0, 0xFF );
    InfoPropertyTestAssert< StencilReadMaskProperty >( -1, 300 );
    InfoPropertyTestDefault< StencilReadMaskProperty >( 0xFF );
}

TEST( DepthStencilStateInfoTest, StencilWriteMask )
{
    typedef InfoProperty< nn::gfx::DepthStencilStateInfo, int,
        &nn::gfx::DepthStencilStateInfo::SetStencilWriteMask,
        &nn::gfx::DepthStencilStateInfo::GetStencilWriteMask > StencilWriteMaskProperty;

    InfoPropertyTest< StencilWriteMaskProperty >( 0, 0xFF );
    InfoPropertyTestAssert< StencilWriteMaskProperty >( -100, 1000 );
    InfoPropertyTestDefault< StencilWriteMaskProperty >( 0xFF );
}

TEST( ColorTargetStateInfoTest, Format )
{
    typedef InfoProperty< nn::gfx::ColorTargetStateInfo, nn::gfx::ImageFormat,
        &nn::gfx::ColorTargetStateInfo::SetFormat,
        &nn::gfx::ColorTargetStateInfo::GetFormat > FormatProperty;

    InfoPropertyTest< FormatProperty >( nn::gfx::ImageFormat_A4_B4_G4_R4_Unorm,
        nn::gfx::ImageFormat_R32_Uint, nn::gfx::ImageFormat_Astc_6x5_UnormSrgb );
    InfoPropertyTestDefault< FormatProperty >( nn::gfx::ImageFormat_Undefined );
}

TEST( RenderTargetStateInfoTest, DepthStencilFormat )
{
    typedef InfoProperty< nn::gfx::RenderTargetStateInfo, nn::gfx::ImageFormat,
        &nn::gfx::RenderTargetStateInfo::SetDepthStencilFormat,
        &nn::gfx::RenderTargetStateInfo::GetDepthStencilFormat > DepthStencilFormatProperty;

    InfoPropertyTest< DepthStencilFormatProperty >( nn::gfx::ImageFormat_D16_Unorm,
        nn::gfx::ImageFormat_D24_Unorm_S8_Uint, nn::gfx::ImageFormat_D32_Float );
    InfoPropertyTestDefault< DepthStencilFormatProperty >( nn::gfx::ImageFormat_Undefined );
}

TEST( RenderTargetStateInfoTest, ColorTargetStateInfoArray )
{
    typedef InfoPairedProperty< nn::gfx::RenderTargetStateInfo,
        const nn::gfx::ColorTargetStateInfo*, int,
        &nn::gfx::RenderTargetStateInfo::SetColorTargetStateInfoArray,
        &nn::gfx::RenderTargetStateInfo::GetColorTargetStateInfoArray,
        &nn::gfx::RenderTargetStateInfo::GetColorTargetCount > ColorTargetStateInfoArrayProperty;

    InfoPropertyTester< ColorTargetStateInfoArrayProperty >::Test(
        reinterpret_cast< const nn::gfx::ColorTargetStateInfo* >( 2 ), 4 );
    InfoPropertyTester< ColorTargetStateInfoArrayProperty >::TestAssert( NULL, -2 );
    InfoPropertyTester< ColorTargetStateInfoArrayProperty >::TestDefault( NULL, 0 );
}

TEST( VertexAttributeStateInfoTest, SemanticIndex )
{
    typedef InfoProperty< nn::gfx::VertexAttributeStateInfo, int,
        &nn::gfx::VertexAttributeStateInfo::SetSemanticIndex,
        &nn::gfx::VertexAttributeStateInfo::GetSemanticIndex > SemanticIndexProperty;

    InfoPropertyTest< SemanticIndexProperty >( 0, 4 );
    InfoPropertyTestAssert< SemanticIndexProperty >( -1, 400 );
    InfoPropertyTestDefault< SemanticIndexProperty >( 0 );
}

TEST( VertexAttributeStateInfoTest, ShaderSlot )
{
    typedef InfoProperty< nn::gfx::VertexAttributeStateInfo, int,
        &nn::gfx::VertexAttributeStateInfo::SetShaderSlot,
        &nn::gfx::VertexAttributeStateInfo::GetShaderSlot > ShaderSlotProperty;

    InfoPropertyTest< ShaderSlotProperty >( -1, 0, 4 );
    InfoPropertyTestAssert< ShaderSlotProperty >( -2, 50000 );
    InfoPropertyTestDefault< ShaderSlotProperty >( -1 );
}

TEST( VertexAttributeStateInfoTest, BufferIndex )
{
    typedef InfoProperty< nn::gfx::VertexAttributeStateInfo, int,
        &nn::gfx::VertexAttributeStateInfo::SetBufferIndex,
        &nn::gfx::VertexAttributeStateInfo::GetBufferIndex > BufferIndexProperty;

    InfoPropertyTest< BufferIndexProperty >( 0, 4 );
    InfoPropertyTestAssert< BufferIndexProperty >( -1, 90000 );
    InfoPropertyTestDefault< BufferIndexProperty >( 0 );
}

TEST( VertexAttributeStateInfoTest, Offset )
{
    typedef InfoProperty< nn::gfx::VertexAttributeStateInfo, ptrdiff_t,
        &nn::gfx::VertexAttributeStateInfo::SetOffset,
        &nn::gfx::VertexAttributeStateInfo::GetOffset > OffsetProperty;

    InfoPropertyTest< OffsetProperty >( -1, 0, 4 );
    InfoPropertyTestDefault< OffsetProperty >( 0 );
}

TEST( VertexAttributeStateInfoTest, Format )
{
    typedef InfoProperty< nn::gfx::VertexAttributeStateInfo, nn::gfx::AttributeFormat,
        &nn::gfx::VertexAttributeStateInfo::SetFormat,
        &nn::gfx::VertexAttributeStateInfo::GetFormat > FormatProperty;

    InfoPropertyTest< FormatProperty >(  nn::gfx::AttributeFormat_16_16_16_16_Float,
        nn::gfx::AttributeFormat_32_Uint, nn::gfx::AttributeFormat_32_32_Uint );
    InfoPropertyTestDefault< FormatProperty >( nn::gfx::AttributeFormat_Undefined );
}

TEST( VertexAttributeStateInfoTest, NamePtr )
{
    typedef InfoProperty< nn::gfx::VertexAttributeStateInfo, const char*,
        &nn::gfx::VertexAttributeStateInfo::SetNamePtr,
        &nn::gfx::VertexAttributeStateInfo::GetNamePtr > NamePtrProperty;

    InfoPropertyTest< NamePtrProperty >( NULL, "NamePtrProperty" );
    InfoPropertyTestDefault< NamePtrProperty >( NULL );
}

TEST( VertexBufferStateInfoTest, Stride )
{
    typedef InfoProperty< nn::gfx::VertexBufferStateInfo, ptrdiff_t,
        &nn::gfx::VertexBufferStateInfo::SetStride,
        &nn::gfx::VertexBufferStateInfo::GetStride > StrideProperty;

    InfoPropertyTest< StrideProperty >( 0, 100 );
    InfoPropertyTestDefault< StrideProperty >( 0 );
}

TEST( VertexBufferStateInfoTest, Divisor )
{
    typedef InfoProperty< nn::gfx::VertexBufferStateInfo, int,
        &nn::gfx::VertexBufferStateInfo::SetDivisor,
        &nn::gfx::VertexBufferStateInfo::GetDivisor > DivisorProperty;

    InfoPropertyTest< DivisorProperty >( 0, 2 );
    InfoPropertyTestAssert< DivisorProperty >( -2 );
    InfoPropertyTestDefault< DivisorProperty >( 0 );
}

TEST( VertexStateInfoTest, VertexAttributeStateInfoArray )
{
    typedef InfoPairedProperty< nn::gfx::VertexStateInfo, const nn::gfx::VertexAttributeStateInfo*, int,
        &nn::gfx::VertexStateInfo::SetVertexAttributeStateInfoArray,
        &nn::gfx::VertexStateInfo::GetVertexAttributeStateInfoArray,
        &nn::gfx::VertexStateInfo::GetVertexAttributeCount > VertexAttributeStateInfoArrayProperty;

    InfoPropertyTester< VertexAttributeStateInfoArrayProperty >::Test(
        reinterpret_cast< const nn::gfx::VertexAttributeStateInfo* >( 4 ), 8 );
    InfoPropertyTester< VertexAttributeStateInfoArrayProperty >::TestAssert( NULL, -3 );
    InfoPropertyTester< VertexAttributeStateInfoArrayProperty >::TestDefault( NULL, 0 );
}

TEST( VertexStateInfoTest, VertexBufferStateInfoArray )
{
    typedef InfoPairedProperty< nn::gfx::VertexStateInfo, const nn::gfx::VertexBufferStateInfo*, int,
        &nn::gfx::VertexStateInfo::SetVertexBufferStateInfoArray,
        &nn::gfx::VertexStateInfo::GetVertexBufferStateInfoArray,
        &nn::gfx::VertexStateInfo::GetVertexBufferCount > VertexBufferStateInfoArrayProperty;

    InfoPropertyTester< VertexBufferStateInfoArrayProperty >::Test(
        reinterpret_cast< const nn::gfx::VertexBufferStateInfo* >( 10 ), 8 );
    InfoPropertyTester< VertexBufferStateInfoArrayProperty >::TestAssert( NULL, -1 );
    InfoPropertyTester< VertexBufferStateInfoArrayProperty >::TestDefault( NULL, 0 );
}

TEST( TessellationStateInfoTest, PatchControlPointCount )
{
    typedef InfoProperty< nn::gfx::TessellationStateInfo, int,
        &nn::gfx::TessellationStateInfo::SetPatchControlPointCount,
        &nn::gfx::TessellationStateInfo::GetPatchControlPointCount > PatchControlPointCountProperty;

    InfoPropertyTest< PatchControlPointCountProperty >( 0, 3 );
    InfoPropertyTestAssert< PatchControlPointCountProperty >( -1 );
    // TODO: SIGLO-28321
    //InfoPropertyTestDefault< PatchControlPointCountProperty >( 0 );
}

TEST( ViewportStateInfoTest, OriginX )
{
    typedef InfoProperty< nn::gfx::ViewportStateInfo, float,
        &nn::gfx::ViewportStateInfo::SetOriginX,
        &nn::gfx::ViewportStateInfo::GetOriginX > OriginXProperty;

    InfoPropertyTest< OriginXProperty >( 0.0f, 4.0f );
    InfoPropertyTestDefault< OriginXProperty >( 0.0f );
}

TEST( ViewportStateInfoTest, OriginY )
{
    typedef InfoProperty< nn::gfx::ViewportStateInfo, float,
        &nn::gfx::ViewportStateInfo::SetOriginY,
        &nn::gfx::ViewportStateInfo::GetOriginY > OriginYProperty;

    InfoPropertyTest< OriginYProperty >( 0.0f, 10.0f );
    InfoPropertyTestDefault< OriginYProperty >( 0.0f );
}

TEST( ViewportStateInfoTest, Width )
{
    typedef InfoProperty< nn::gfx::ViewportStateInfo, float,
        &nn::gfx::ViewportStateInfo::SetWidth,
        &nn::gfx::ViewportStateInfo::GetWidth > WidthProperty;

    InfoPropertyTest< WidthProperty >( 0.0f, 400.0f );
    InfoPropertyTestDefault< WidthProperty >( 0.0f );
}

TEST( ViewportStateInfoTest, Height )
{
    typedef InfoProperty< nn::gfx::ViewportStateInfo, float,
        &nn::gfx::ViewportStateInfo::SetHeight,
        &nn::gfx::ViewportStateInfo::GetHeight > HeightProperty;

    InfoPropertyTest< HeightProperty >( 0.0f, 800.0f );
    InfoPropertyTestDefault< HeightProperty >( 0.0f );
}

TEST( ViewportStateInfoTest, MinDepth )
{
    typedef InfoProperty< nn::gfx::ViewportStateInfo, float,
        &nn::gfx::ViewportStateInfo::SetMinDepth,
        &nn::gfx::ViewportStateInfo::GetMinDepth > MinDepthProperty;

    InfoPropertyTest< MinDepthProperty >( 0.0f, 0.3f );
    InfoPropertyTestDefault< MinDepthProperty >( 0.0f );
}

TEST( ViewportStateInfoTest, MaxDepth )
{
    typedef InfoProperty< nn::gfx::ViewportStateInfo, float,
        &nn::gfx::ViewportStateInfo::SetMaxDepth,
        &nn::gfx::ViewportStateInfo::GetMaxDepth > MaxDepthProperty;

    InfoPropertyTest< MaxDepthProperty >( 0.0f, 2.0f );
    InfoPropertyTestDefault< MaxDepthProperty >( 1.0f );
}

TEST( ScissorStateInfoTest, OriginX )
{
    typedef InfoProperty< nn::gfx::ScissorStateInfo, int,
        &nn::gfx::ScissorStateInfo::SetOriginX, &nn::gfx::ScissorStateInfo::GetOriginX > OriginXProperty;

    InfoPropertyTest< OriginXProperty >( 0, 3 );
    InfoPropertyTestAssert< OriginXProperty >( -4 );
    InfoPropertyTestDefault< OriginXProperty >( 0 );
}

TEST( ScissorStateInfoTest, OriginY )
{
    typedef InfoProperty< nn::gfx::ScissorStateInfo, int,
        &nn::gfx::ScissorStateInfo::SetOriginY, &nn::gfx::ScissorStateInfo::GetOriginY > OriginYProperty;

    InfoPropertyTest< OriginYProperty >( 0, 10 );
    InfoPropertyTestAssert< OriginYProperty >( -22 );
    InfoPropertyTestDefault< OriginYProperty >( 0 );
}

TEST( ScissorStateInfoTest, Width )
{
    typedef InfoProperty< nn::gfx::ScissorStateInfo, int,
        &nn::gfx::ScissorStateInfo::SetWidth, &nn::gfx::ScissorStateInfo::GetWidth > WidthProperty;

    InfoPropertyTest< WidthProperty >( 0, 2000 );
    InfoPropertyTestAssert< WidthProperty >( -400 );
    InfoPropertyTestDefault< WidthProperty >( 0 );
}

TEST( ScissorStateInfoTest, Height )
{
    typedef InfoProperty< nn::gfx::ScissorStateInfo, int,
        &nn::gfx::ScissorStateInfo::SetHeight, &nn::gfx::ScissorStateInfo::GetHeight > HeightProperty;

    InfoPropertyTest< HeightProperty >( 0, 300 );
    InfoPropertyTestAssert< HeightProperty >( -67 );
    InfoPropertyTestDefault< HeightProperty >( 0 );
}

TEST( ViewportScissorStateInfoTest, ScissorEnabled )
{
    typedef InfoProperty< nn::gfx::ViewportScissorStateInfo, bool,
        &nn::gfx::ViewportScissorStateInfo::SetScissorEnabled,
        &nn::gfx::ViewportScissorStateInfo::IsScissorEnabled > ScissorEnabledProperty;

    InfoPropertyTest< ScissorEnabledProperty >( false, true );
    InfoPropertyTestDefault< ScissorEnabledProperty >( false );
}

TEST( ViewportScissorStateInfoTest, ViewportStateInfoArray )
{
    typedef InfoPairedProperty< nn::gfx::ViewportScissorStateInfo,
        const nn::gfx::ViewportStateInfo*, int,
        &nn::gfx::ViewportScissorStateInfo::SetViewportStateInfoArray,
        &nn::gfx::ViewportScissorStateInfo::GetViewportStateInfoArray,
        &nn::gfx::ViewportScissorStateInfo::GetViewportCount > ViewportStateInfoArrayProperty;

    InfoPropertyTester< ViewportStateInfoArrayProperty >::Test(
        reinterpret_cast< const nn::gfx::ViewportStateInfo* >( 400 ), 1 );
    InfoPropertyTester< ViewportStateInfoArrayProperty >::TestAssert( NULL, -3 );
    InfoPropertyTester< ViewportStateInfoArrayProperty >::TestDefault( NULL, 0 );
}

TEST( ViewportScissorStateInfoTest, ScissorStateInfoArray )
{
    typedef InfoPairedProperty< nn::gfx::ViewportScissorStateInfo,
        const nn::gfx::ScissorStateInfo*, int,
        &nn::gfx::ViewportScissorStateInfo::SetScissorStateInfoArray,
        &nn::gfx::ViewportScissorStateInfo::GetScissorStateInfoArray,
        &nn::gfx::ViewportScissorStateInfo::GetScissorCount > ScissorStateInfoArrayProperty;

    InfoPropertyTester< ScissorStateInfoArrayProperty >::Test(
        reinterpret_cast< const nn::gfx::ScissorStateInfo* >( 400 ), 1 );
    InfoPropertyTester< ScissorStateInfoArrayProperty >::TestAssert( NULL, -3 );
    InfoPropertyTester< ScissorStateInfoArrayProperty >::TestDefault( NULL, 0 );
}

TEST( SwapChainInfoTest, Layer )
{
    typedef InfoProperty< nn::gfx::SwapChainInfo, nn::vi::Layer*,
        &nn::gfx::SwapChainInfo::SetLayer,
        &nn::gfx::SwapChainInfo::GetLayer > LayerProperty;

    InfoPropertyTest< LayerProperty >( reinterpret_cast< nn::vi::Layer* >( 44 ) );
    InfoPropertyTestDefault< LayerProperty >( NULL );
}

TEST( SwapChainInfoTest, Format )
{
    typedef InfoProperty< nn::gfx::SwapChainInfo, nn::gfx::ImageFormat,
        &nn::gfx::SwapChainInfo::SetFormat,
        &nn::gfx::SwapChainInfo::GetFormat > FormatProperty;

    InfoPropertyTest< FormatProperty >( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm,
        nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb );
    InfoPropertyTestDefault< FormatProperty >( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );
}

TEST( SwapChainInfoTest, BufferCount )
{
    typedef InfoProperty< nn::gfx::SwapChainInfo, int,
        &nn::gfx::SwapChainInfo::SetBufferCount,
        &nn::gfx::SwapChainInfo::GetBufferCount > BufferCountProperty;

    InfoPropertyTest< BufferCountProperty >( 1, 2, 3 );
    InfoPropertyTestDefault< BufferCountProperty >( 2 );
}

TEST( SwapChainInfoTest, Width )
{
    typedef InfoProperty< nn::gfx::SwapChainInfo, int,
        &nn::gfx::SwapChainInfo::SetWidth,
        &nn::gfx::SwapChainInfo::GetWidth > WidthProperty;

    InfoPropertyTest< WidthProperty >( 600 );
    InfoPropertyTestDefault< WidthProperty >( 1280 );
}

TEST( SwapChainInfoTest, Height )
{
    typedef InfoProperty< nn::gfx::SwapChainInfo, int,
        &nn::gfx::SwapChainInfo::SetHeight,
        &nn::gfx::SwapChainInfo::GetHeight > HeightProperty;

    InfoPropertyTest< HeightProperty >( 400 );
    InfoPropertyTestDefault< HeightProperty >( 720 );
}

TEST( TextureInfoTest, ImageStorageDimension )
{
    typedef InfoProperty< nn::gfx::TextureInfo, nn::gfx::ImageStorageDimension,
        &nn::gfx::TextureInfo::SetImageStorageDimension,
        &nn::gfx::TextureInfo::GetImageStorageDimension > ImageStorageDimensionProperty;

    InfoPropertyTest< ImageStorageDimensionProperty >(
        nn::gfx::ImageStorageDimension_3d, nn::gfx::ImageStorageDimension_1d );
    InfoPropertyTestDefault< ImageStorageDimensionProperty >( nn::gfx::ImageStorageDimension_2d );
}

TEST( TextureInfoTest, ImageFormat )
{
    typedef InfoProperty< nn::gfx::TextureInfo, nn::gfx::ImageFormat,
        &nn::gfx::TextureInfo::SetImageFormat,
        &nn::gfx::TextureInfo::GetImageFormat > ImageFormatProperty;

    InfoPropertyTest< ImageFormatProperty >( nn::gfx::ImageFormat_A1_B5_G5_R5_Unorm,
        nn::gfx::ImageFormat_Astc_10x8_UnormSrgb, nn::gfx::ImageFormat_R16_Sint );
    InfoPropertyTestDefault< ImageFormatProperty >( nn::gfx::ImageFormat_Undefined );
}

TEST( TextureInfoTest, GpuAccessFlags )
{
    typedef InfoProperty< nn::gfx::TextureInfo, int,
        &nn::gfx::TextureInfo::SetGpuAccessFlags,
        &nn::gfx::TextureInfo::GetGpuAccessFlags > GpuAccessFlagsProperty;

    InfoPropertyTest< GpuAccessFlagsProperty >( nn::gfx::GpuAccess_Texture,
        nn::gfx::GpuAccess_ColorBuffer | nn::gfx::GpuAccess_Read );
    InfoPropertyTestDefault< GpuAccessFlagsProperty >( 0 );
}

TEST( TextureInfoTest, TileMode )
{
    typedef InfoProperty< nn::gfx::TextureInfo, nn::gfx::TileMode,
        &nn::gfx::TextureInfo::SetTileMode,
        &nn::gfx::TextureInfo::GetTileMode > TileModeProperty;

    InfoPropertyTest< TileModeProperty >( nn::gfx::TileMode_Linear, nn::gfx::TileMode_Optimal );
    InfoPropertyTestDefault< TileModeProperty >( nn::gfx::TileMode_Optimal );
}

TEST( TextureInfoTest, Width )
{
    typedef InfoProperty< nn::gfx::TextureInfo, int,
        &nn::gfx::TextureInfo::SetWidth,
        &nn::gfx::TextureInfo::GetWidth > WidthProperty;

    InfoPropertyTest< WidthProperty >( 0, 1, 100 );
    InfoPropertyTestAssert< WidthProperty >( -1 );
    InfoPropertyTestDefault< WidthProperty >( 1 );
}

TEST( TextureInfoTest, Height )
{
    typedef InfoProperty< nn::gfx::TextureInfo, int,
        &nn::gfx::TextureInfo::SetHeight,
        &nn::gfx::TextureInfo::GetHeight > HeightProperty;

    InfoPropertyTest< HeightProperty >( 0, 50 );
    InfoPropertyTestAssert< HeightProperty >( -90 );
    InfoPropertyTestDefault< HeightProperty >( 1 );
}

TEST( TextureInfoTest, Depth )
{
    typedef InfoProperty< nn::gfx::TextureInfo, int,
        &nn::gfx::TextureInfo::SetDepth,
        &nn::gfx::TextureInfo::GetDepth > DepthProperty;

    InfoPropertyTest< DepthProperty >( 0, 1 );
    InfoPropertyTestAssert< DepthProperty >( -1 );
    InfoPropertyTestDefault< DepthProperty >( 1 );
}

TEST( TextureInfoTest, ArrayLength )
{
    typedef InfoProperty< nn::gfx::TextureInfo, int,
        &nn::gfx::TextureInfo::SetArrayLength,
        &nn::gfx::TextureInfo::GetArrayLength > ArrayLengthProperty;

    InfoPropertyTest< ArrayLengthProperty >( 0, 1, 20 );
    InfoPropertyTestAssert< ArrayLengthProperty >( -1 );
    InfoPropertyTestDefault< ArrayLengthProperty >( 0 );
}

TEST( TextureInfoTest, Swizzle )
{
    typedef InfoProperty< nn::gfx::TextureInfo, int,
        &nn::gfx::TextureInfo::SetSwizzle,
        &nn::gfx::TextureInfo::GetSwizzle > SwizzleProperty;

    InfoPropertyTest< SwizzleProperty >( 0, 4 );
    InfoPropertyTestAssert< SwizzleProperty >( -3 );
    InfoPropertyTestDefault< SwizzleProperty >( 0 );
}

TEST( TextureInfoTest, MultisampleCount )
{
    typedef InfoProperty< nn::gfx::TextureInfo, int,
        &nn::gfx::TextureInfo::SetMultiSampleCount,
        &nn::gfx::TextureInfo::GetMultisampleCount > MultisampleCountProperty;

    InfoPropertyTest< MultisampleCountProperty >( 0, 1, 4 );
    InfoPropertyTestAssert< MultisampleCountProperty >( -5 );
    InfoPropertyTestDefault< MultisampleCountProperty >( 1 );
}

TEST( TextureInfoTest, MipCount )
{
    typedef InfoProperty< nn::gfx::TextureInfo, int,
        &nn::gfx::TextureInfo::SetMipCount,
        &nn::gfx::TextureInfo::GetMipCount > MipCountProperty;

    InfoPropertyTest< MipCountProperty >( 0, 1, 4 );
    InfoPropertyTestAssert< MipCountProperty >( -4 );
    InfoPropertyTestDefault< MipCountProperty >( 1 );
}

TEST( TextureViewInfoTest, ImageDimension )
{
    typedef InfoProperty< nn::gfx::TextureViewInfo, nn::gfx::ImageDimension,
        &nn::gfx::TextureViewInfo::SetImageDimension,
        &nn::gfx::TextureViewInfo::GetImageDimension > ImageDimensionProperty;

    InfoPropertyTest< ImageDimensionProperty >( nn::gfx::ImageDimension_2d,
        nn::gfx::ImageDimension_CubeMapArray, nn::gfx::ImageDimension_2dMultisample );
    InfoPropertyTestDefault< ImageDimensionProperty >( nn::gfx::ImageDimension_2d );
}

TEST( TextureViewInfoTest, DepthStencilTextureMode )
{
    typedef InfoProperty< nn::gfx::TextureViewInfo, nn::gfx::DepthStencilFetchMode,
        &nn::gfx::TextureViewInfo::SetDepthStencilTextureMode,
        &nn::gfx::TextureViewInfo::GetDepthStencilTextureMode > DepthStencilTextureModeProperty;

    InfoPropertyTest< DepthStencilTextureModeProperty >(
        nn::gfx::DepthStencilFetchMode_DepthComponent, nn::gfx::DepthStencilFetchMode_StencilIndex );
    InfoPropertyTestDefault< DepthStencilTextureModeProperty >(
        nn::gfx::DepthStencilFetchMode_DepthComponent );
}

TEST( TextureViewInfoTest, ImageFormat )
{
    typedef InfoProperty< nn::gfx::TextureViewInfo, nn::gfx::ImageFormat,
        &nn::gfx::TextureViewInfo::SetImageFormat,
        &nn::gfx::TextureViewInfo::GetImageFormat > ImageFormatProperty;

    InfoPropertyTest< ImageFormatProperty >( nn::gfx::ImageFormat_R16_Float,
        nn::gfx::ImageFormat_R32_G32_B32_Float, nn::gfx::ImageFormat_R8_G8_B8_A8_Snorm );
    InfoPropertyTestDefault< ImageFormatProperty >( nn::gfx::ImageFormat_Undefined );
}

TEST( TextureViewInfoTest, TexturePtr )
{
    nn::gfx::TextureViewInfo info;

    const nn::gfx::Texture* pTexture = reinterpret_cast< const nn::gfx::Texture* >( &info );
    info.SetTexturePtr( pTexture );
    EXPECT_EQ( pTexture, info.GetTexturePtr() );

    pTexture = NULL;
    info.SetTexturePtr( pTexture );
    EXPECT_EQ( pTexture, info.GetTexturePtr() );

    info.SetDefault();
    InverseBit( &info );
    info.SetDefault();
    EXPECT_EQ( NULL, info.GetTexturePtr() );
}

TEST( TextureViewInfoTest, ChannelMapping )
{
    nn::gfx::TextureViewInfo info;

    nn::gfx::ChannelMapping red = nn::gfx::ChannelMapping_One;
    nn::gfx::ChannelMapping green = nn::gfx::ChannelMapping_Zero;
    nn::gfx::ChannelMapping blue = nn::gfx::ChannelMapping_Alpha;
    nn::gfx::ChannelMapping alpha = nn::gfx::ChannelMapping_Green;
    info.SetChannelMapping( red, green, blue, alpha );
    EXPECT_EQ( red, info.GetChannelMapping( nn::gfx::ColorChannel_Red ) );
    EXPECT_EQ( green, info.GetChannelMapping( nn::gfx::ColorChannel_Green ) );
    EXPECT_EQ( blue, info.GetChannelMapping( nn::gfx::ColorChannel_Blue ) );
    EXPECT_EQ( alpha, info.GetChannelMapping( nn::gfx::ColorChannel_Alpha ) );

    info.SetDefault();
    InverseBit( &info );
    info.SetDefault();
    EXPECT_EQ( nn::gfx::ChannelMapping_Red, info.GetChannelMapping( nn::gfx::ColorChannel_Red ) );
    EXPECT_EQ( nn::gfx::ChannelMapping_Green, info.GetChannelMapping( nn::gfx::ColorChannel_Green ) );
    EXPECT_EQ( nn::gfx::ChannelMapping_Blue, info.GetChannelMapping( nn::gfx::ColorChannel_Blue ) );
    EXPECT_EQ( nn::gfx::ChannelMapping_Alpha, info.GetChannelMapping( nn::gfx::ColorChannel_Alpha ) );
}

TEST( ColorTargetViewInfoTest, ImageDimension )
{
    typedef InfoProperty< nn::gfx::ColorTargetViewInfo, nn::gfx::ImageDimension,
        &nn::gfx::ColorTargetViewInfo::SetImageDimension,
        &nn::gfx::ColorTargetViewInfo::GetImageDimension > ImageDimensionProperty;

    InfoPropertyTest< ImageDimensionProperty >(
        nn::gfx::ImageDimension_1d, nn::gfx::ImageDimension_3d );
    // TODO: SIGLO-28321
    //InfoPropertyTestDefault< ImageDimensionProperty >( nn::gfx::ImageDimension_2d );
}

TEST( ColorTargetViewInfoTest, ImageFormat )
{
    typedef InfoProperty< nn::gfx::ColorTargetViewInfo, nn::gfx::ImageFormat,
        &nn::gfx::ColorTargetViewInfo::SetImageFormat,
        &nn::gfx::ColorTargetViewInfo::GetImageFormat > ImageFormatProperty;

    InfoPropertyTest< ImageFormatProperty >( nn::gfx::ImageFormat_R16_G16_B16_A16_Snorm );
    InfoPropertyTestDefault< ImageFormatProperty >( nn::gfx::ImageFormat_Undefined );
}

TEST( ColorTargetViewInfoTest, MipLevel )
{
    typedef InfoProperty< nn::gfx::ColorTargetViewInfo, int,
        &nn::gfx::ColorTargetViewInfo::SetMipLevel,
        &nn::gfx::ColorTargetViewInfo::GetMipLevel > MipLevelProperty;

    InfoPropertyTest< MipLevelProperty >( 0, 1, 100 );
    InfoPropertyTestAssert< MipLevelProperty >( -1, 90000 );
    InfoPropertyTestDefault< MipLevelProperty >( 0 );
}

TEST( ColorTargetViewInfoTest, TexturePtr )
{
    nn::gfx::ColorTargetViewInfo info;

    const nn::gfx::Texture* pTexture = reinterpret_cast< const nn::gfx::Texture* >( &info );
    info.SetTexturePtr( pTexture );
    EXPECT_EQ( pTexture, info.GetTexturePtr() );

    pTexture = NULL;
    info.SetTexturePtr( pTexture );
    EXPECT_EQ( pTexture, info.GetTexturePtr() );

    info.SetDefault();
    InverseBit( &info );
    info.SetDefault();
    EXPECT_EQ( NULL, info.GetTexturePtr() );
}

TEST( DepthStencilViewInfoTest, ImageDimension )
{
    typedef InfoProperty< nn::gfx::DepthStencilViewInfo, nn::gfx::ImageDimension,
        &nn::gfx::DepthStencilViewInfo::SetImageDimension,
        &nn::gfx::DepthStencilViewInfo::GetImageDimension > ImageDimensionProperty;

    InfoPropertyTest< ImageDimensionProperty >(
        nn::gfx::ImageDimension_2d, nn::gfx::ImageDimension_2dArray );
    // TODO: SIGLO-28321
    //InfoPropertyTestDefault< ImageDimensionProperty >( nn::gfx::ImageDimension_2d );
}

TEST( DepthStencilViewInfoTest, MipLevel )
{
    typedef InfoProperty< nn::gfx::DepthStencilViewInfo, int,
        &nn::gfx::DepthStencilViewInfo::SetMipLevel,
        &nn::gfx::DepthStencilViewInfo::GetMipLevel > MipLevelProperty;

    InfoPropertyTest< MipLevelProperty >( 0, 1, 8 );
    InfoPropertyTestAssert< MipLevelProperty >( -1 );
    InfoPropertyTestDefault< MipLevelProperty >( 0 );
}

TEST( DepthStencilViewInfoTest, TexturePtr )
{
    nn::gfx::DepthStencilViewInfo info;

    const nn::gfx::Texture* pTexture = reinterpret_cast< const nn::gfx::Texture* >( &info );
    info.SetTexturePtr( pTexture );
    EXPECT_EQ( pTexture, info.GetTexturePtr() );

    pTexture = NULL;
    info.SetTexturePtr( pTexture );
    EXPECT_EQ( pTexture, info.GetTexturePtr() );

    info.SetDefault();
    InverseBit( &info );
    info.SetDefault();
    EXPECT_EQ( NULL, info.GetTexturePtr() );
}

TEST( TextureMipRangeTest, MinMipLevel )
{
    typedef InfoProperty< nn::gfx::TextureMipRange, int,
        &nn::gfx::TextureMipRange::SetMinMipLevel,
        &nn::gfx::TextureMipRange::GetMinMipLevel > MinMipLevelProperty;

    InfoPropertyTest< MinMipLevelProperty >( 0, 1, 2 );
    InfoPropertyTestAssert< MinMipLevelProperty >( -1, 80000 );
    InfoPropertyTestDefault< MinMipLevelProperty >( 0 );
}

TEST( TextureMipRangeTest, MipCount )
{
    typedef InfoProperty< nn::gfx::TextureMipRange, int,
        &nn::gfx::TextureMipRange::SetMipCount,
        &nn::gfx::TextureMipRange::GetMipCount > MipCountProperty;

    InfoPropertyTest< MipCountProperty >( 0, 1, 2 );
    InfoPropertyTestAssert< MipCountProperty >( -2, 783921 );
    InfoPropertyTestDefault< MipCountProperty >( 1 );
}

TEST( TextureArrayRange, BaseArrayIndex )
{
    typedef InfoProperty< nn::gfx::TextureArrayRange, int,
        &nn::gfx::TextureArrayRange::SetBaseArrayIndex,
        &nn::gfx::TextureArrayRange::GetBaseArrayIndex > BaseArrayIndexProperty;

    InfoPropertyTest< BaseArrayIndexProperty >( 0, 1, 2 );
    InfoPropertyTestAssert< BaseArrayIndexProperty >( -1 );
    InfoPropertyTestDefault< BaseArrayIndexProperty >( 0 );
}

TEST( TextureArrayRange, ArrayLength )
{
    typedef InfoProperty< nn::gfx::TextureArrayRange, int,
        &nn::gfx::TextureArrayRange::SetArrayLength,
        &nn::gfx::TextureArrayRange::GetArrayLength > ArrayLengthProperty;

    InfoPropertyTest< ArrayLengthProperty >( 0, 1, 2 );
    InfoPropertyTestAssert< ArrayLengthProperty >( -1 );
    InfoPropertyTestDefault< ArrayLengthProperty >( 1 );
}

TEST( TextureSubresource, MipLevel )
{
    typedef InfoProperty< nn::gfx::TextureSubresource, int,
        &nn::gfx::TextureSubresource::SetMipLevel,
        &nn::gfx::TextureSubresource::GetMipLevel > MipLevelProperty;

    InfoPropertyTest< MipLevelProperty >( 0, 1, 2 );
    InfoPropertyTestAssert< MipLevelProperty >( -1, 73913 );
    InfoPropertyTestDefault< MipLevelProperty >( 0 );
}

TEST( TextureSubresource, ArrayIndex )
{
    typedef InfoProperty< nn::gfx::TextureSubresource, int,
        &nn::gfx::TextureSubresource::SetArrayIndex,
        &nn::gfx::TextureSubresource::GetArrayIndex > ArrayIndexProperty;

    InfoPropertyTest< ArrayIndexProperty >( 0, 2 );
    InfoPropertyTestAssert< ArrayIndexProperty >( -1 );
    InfoPropertyTestDefault< ArrayIndexProperty >( 0 );
}

TEST( TextureCopyRegion, OffsetU )
{
    typedef InfoProperty< nn::gfx::TextureCopyRegion, int,
        &nn::gfx::TextureCopyRegion::SetOffsetU,
        &nn::gfx::TextureCopyRegion::GetOffsetU > OffsetUProperty;

    InfoPropertyTest< OffsetUProperty >( 0, 100 );
    InfoPropertyTestAssert< OffsetUProperty >( -2 );
    InfoPropertyTestDefault< OffsetUProperty >( 0 );
}

TEST( TextureCopyRegion, OffsetV )
{
    typedef InfoProperty< nn::gfx::TextureCopyRegion, int,
        &nn::gfx::TextureCopyRegion::SetOffsetV,
        &nn::gfx::TextureCopyRegion::GetOffsetV > OffsetVProperty;

    InfoPropertyTest< OffsetVProperty >( 0, 100 );
    InfoPropertyTestAssert< OffsetVProperty >( -2 );
    InfoPropertyTestDefault< OffsetVProperty >( 0 );
}

TEST( TextureCopyRegion, OffsetW )
{
    typedef InfoProperty< nn::gfx::TextureCopyRegion, int,
        &nn::gfx::TextureCopyRegion::SetOffsetW,
        &nn::gfx::TextureCopyRegion::GetOffsetW > OffsetWProperty;

    InfoPropertyTest< OffsetWProperty >( 0, 100 );
    InfoPropertyTestAssert< OffsetWProperty >( -2 );
    InfoPropertyTestDefault< OffsetWProperty >( 0 );
}

TEST( TextureCopyRegion, Width )
{
    typedef InfoProperty< nn::gfx::TextureCopyRegion, int,
        &nn::gfx::TextureCopyRegion::SetWidth,
        &nn::gfx::TextureCopyRegion::GetWidth > WidthProperty;

    InfoPropertyTest< WidthProperty >( 0, 100 );
    InfoPropertyTestAssert< WidthProperty >( -4 );
    InfoPropertyTestDefault< WidthProperty >( 1 );
}

TEST( TextureCopyRegion, Height )
{
    typedef InfoProperty< nn::gfx::TextureCopyRegion, int,
        &nn::gfx::TextureCopyRegion::SetHeight,
        &nn::gfx::TextureCopyRegion::GetHeight > HeightProperty;

    InfoPropertyTest< HeightProperty >( 0, 100 );
    InfoPropertyTestAssert< HeightProperty >( -4 );
    InfoPropertyTestDefault< HeightProperty >( 1 );
}

TEST( TextureCopyRegion, Depth )
{
    typedef InfoProperty< nn::gfx::TextureCopyRegion, int,
        &nn::gfx::TextureCopyRegion::SetDepth,
        &nn::gfx::TextureCopyRegion::GetDepth > DepthProperty;

    InfoPropertyTest< DepthProperty >( 0, 100 );
    InfoPropertyTestAssert< DepthProperty >( -4 );
    InfoPropertyTestDefault< DepthProperty >( 1 );
}

TEST( TextureCopyRegion, ArrayLength )
{
    typedef InfoProperty< nn::gfx::TextureCopyRegion, int,
        &nn::gfx::TextureCopyRegion::SetArrayLength,
        &nn::gfx::TextureCopyRegion::GetArrayLength > ArrayLengthProperty;

    InfoPropertyTest< ArrayLengthProperty >( 0, 2 );
    InfoPropertyTestAssert< ArrayLengthProperty >( -12 );
    InfoPropertyTestDefault< ArrayLengthProperty >( 1 );
}

TEST( BufferTextureCopyRegionTest, BufferOffset )
{
    typedef InfoProperty< nn::gfx::BufferTextureCopyRegion, int,
        &nn::gfx::BufferTextureCopyRegion::SetBufferOffset,
        &nn::gfx::BufferTextureCopyRegion::GetBufferOffset > BufferOffsetProperty;

    InfoPropertyTest< BufferOffsetProperty >( 0, 32 );
    InfoPropertyTestAssert< BufferOffsetProperty >( -2 );
    InfoPropertyTestDefault< BufferOffsetProperty >( 0 );
}

TEST( BufferTextureCopyRegionTest, BufferImageWidth )
{
    // TODO: SIGLO-28356
    /*typedef InfoProperty< nn::gfx::BufferTextureCopyRegion, int,
        &nn::gfx::BufferTextureCopyRegion::SetBufferImageWidth,
        &nn::gfx::BufferTextureCopyRegion::GetBufferImageHeight > BufferImageWidthProperty;

    InfoPropertyTest< BufferImageWidthProperty >( 0, 32 );
    InfoPropertyTestDefault< BufferImageWidthProperty >( 0 );*/
}

TEST( BufferTextureCopyRegionTest, BufferImageHeight )
{
    typedef InfoProperty< nn::gfx::BufferTextureCopyRegion, int,
        &nn::gfx::BufferTextureCopyRegion::SetBufferImageHeight,
        &nn::gfx::BufferTextureCopyRegion::GetBufferImageHeight > BufferImageHeightProperty;

    InfoPropertyTest< BufferImageHeightProperty >( 0, 32 );
    InfoPropertyTestDefault< BufferImageHeightProperty >( 0 );
}
