﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cmath>

#include <gfx/demo.h>

#include "textureHandle.h"
#include "gsParticle.h"

// GS particle shader (VS + GS + PS).
extern int g_ShaderFileIdx;
static const char* const GS_PARTICLE[] =
{
    "shaders/gpuParticle/gsParticle",
    "shaders/gpuParticle/gsParticleHlslcc",
};

// Shader objects.
static DEMOGfxPipeline s_GsParticlePipeline;

#if NN_GFX_IS_TARGET_GX
// Ring buffer.
static void* s_pInputRingBuffer;
static void* s_pOutputRingBuffer;
static u32 s_InputRingBufferSize;
static u32 s_OutputRingBufferSize;
#endif

static int s_BufferPointer = 0;

// Simple particle attribute.
enum GS_PARTICLE_ATTRIB
{
    GS_PARTICLE_VPOS = 0,
    GS_PARTICLE_VCOLOR,
    GS_PARTICLE_VLIFE,
    GS_PARTICLE_VATTR_0,
    GS_PARTICLE_VATTR_1,
    GS_PARTICLE_MAX_ATTR,
};


// View-projection matrix.
typedef struct VIEWPROJ
{
    Mtx44 view;
    Mtx44 viewInv;
    Mtx44 proj;
} TViewProj;
static DEMOGfxBuffer s_ViewProj[2];
static int s_ViewProjLoc = -1;

// System.
typedef struct SYSTEM
{
    Mtx44 model;
    f32 loopAttr[4];
} TSystem;
static DEMOGfxBuffer s_System[2];
static int s_SystemLoc = -1;

// Node attribute.
typedef struct NODEATTRIB
{
    Mtx44 bbPos;
    Mtx44 bbSize;
    Mtx44 nodeColor;
    s32 pattern[4];
} TNodeAttr;
static DEMOGfxBuffer s_NodeAttr[2];
static u32 s_NodeAttribCnt = 0;
static int s_NodeAttrLoc = -1;
static bool s_InvalidateNodeAttr = false;

// Texture
typedef struct TEXTURE
{
    f32 texture[4];
} TTexture;
static DEMOGfxBuffer s_Texture[2];
static u32 s_TextureCnt = 0;
static int s_TextureLoc = -1;
static bool s_InvalidateTexture = false;

// Pixel Shader
typedef struct PIXEL
{
    f32 alphaTest[4];
} TPixel;
static DEMOGfxBuffer s_Pixel[2];
static int s_PixelLoc = -1;

static int s_GsParticleBaseMapLoc = -1;

// Blend state.
enum BLEND_STATE_GS
{
    BLEND_NONE_GS = 0,
    BLEND_ALPHA_GS,
    BLEND_ADD_GS,
};
static BLEND_STATE_GS s_BlendStateGS = BLEND_NONE_GS;

static nn::gfx::DepthStencilState s_GsParticleDepthStencilStates[ BLEND_ADD_GS + 1];
static nn::gfx::BlendState s_GsParticleBlendStates[ BLEND_ADD_GS + 1];

// Alpha test.
static bool s_AlphaTestGS = false;

// Write depth buffer.

// Model matrix.
static Mtx44 s_ModelMatrix44GS;

// Particle scale.
static f32 s_ParticleScaleGS = 1.0f;



// Sampler.
static DEMOGfxTexture* s_pBaseMapGS = NULL;
static nn::gfx::Sampler s_BaseMapSamplerGS;
static nn::gfx::DescriptorSlot s_BaseMapSamplerGSSlot;

// Vertices.
typedef struct GS_PARTICLE_VTX
{
    f32 pos[3];
    u32 color;
    u16 life[2];
    u8 attr0[4];
    u16 attr1[2];
} TGsParticle;
static DEMOGfxBuffer s_GsParticleVtx;

static void InitGsParticlePipelines()
{
    s_GsParticlePipeline.SetDefaults();

    // Setup a pipeline with the initial common state
    s_GsParticlePipeline.depthStencilStateInfo.SetDepthTestEnabled( true );
    s_GsParticlePipeline.depthStencilStateInfo.SetDepthComparisonFunction( nn::gfx::ComparisonFunction_LessEqual );

    s_GsParticlePipeline.colorTargetStateCount = 1;
    s_GsParticlePipeline.blendTargetStateCount = 1;
    s_GsParticlePipeline.colorTargetStateInfoArray[ 0 ].SetDefault();
    s_GsParticlePipeline.colorTargetStateInfoArray[ 0 ].SetFormat( DEMOColorBufferInfo.GetImageFormat() );
    s_GsParticlePipeline.blendTargetStateInfoArray[ 0 ].SetDefault();

    // Turn on z clipping
    s_GsParticlePipeline.rasterizerStateInfo.SetDepthClipEnabled( true );

    s_GsParticlePipeline.Initialize( &DEMODevice );

    // Setup BLEND_NONE_GS state
    {
        s_GsParticlePipeline.depthStencilStateInfo.SetDepthWriteEnabled( true );
        s_GsParticlePipeline.blendTargetStateInfoArray[ 0 ].SetBlendEnabled( false );
        size_t size = nn::gfx::BlendState::GetRequiredMemorySize( s_GsParticlePipeline.blendStateInfo );
        void* data = DEMOGfxAllocMEM2( size, nn::gfx::BlendState::RequiredMemoryInfo_Alignment );
        s_GsParticleBlendStates[ BLEND_NONE_GS ].SetMemory( data, size );
        s_GsParticleBlendStates[ BLEND_NONE_GS ].Initialize( &DEMODevice, s_GsParticlePipeline.blendStateInfo );
        s_GsParticleDepthStencilStates[ BLEND_NONE_GS ].Initialize( &DEMODevice, s_GsParticlePipeline.depthStencilStateInfo );
    }

    // Setup BLEND_ALPHA_GS state
    {
        s_GsParticlePipeline.depthStencilStateInfo.SetDepthWriteEnabled( false );
        s_GsParticlePipeline.blendTargetStateInfoArray[ 0 ].SetBlendEnabled( true );
        s_GsParticlePipeline.blendTargetStateInfoArray[ 0 ].SetColorBlendFunction( nn::gfx::BlendFunction_Add );
        s_GsParticlePipeline.blendTargetStateInfoArray[ 0 ].SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
        s_GsParticlePipeline.blendTargetStateInfoArray[ 0 ].SetDestinationColorBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
        s_GsParticlePipeline.blendTargetStateInfoArray[ 0 ].SetAlphaBlendFunction( nn::gfx::BlendFunction_Add );
        s_GsParticlePipeline.blendTargetStateInfoArray[ 0 ].SetSourceAlphaBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
        s_GsParticlePipeline.blendTargetStateInfoArray[ 0 ].SetDestinationAlphaBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
        size_t size = nn::gfx::BlendState::GetRequiredMemorySize( s_GsParticlePipeline.blendStateInfo );
        void* data = DEMOGfxAllocMEM2( size, nn::gfx::BlendState::RequiredMemoryInfo_Alignment );
        s_GsParticleBlendStates[ BLEND_ALPHA_GS ].SetMemory( data, size );
        s_GsParticleBlendStates[ BLEND_ALPHA_GS ].Initialize( &DEMODevice, s_GsParticlePipeline.blendStateInfo );
        s_GsParticleDepthStencilStates[ BLEND_ALPHA_GS ].Initialize( &DEMODevice, s_GsParticlePipeline.depthStencilStateInfo );
    }

    // Setup BLEND_ADD_GS state
    {
        s_GsParticlePipeline.depthStencilStateInfo.SetDepthWriteEnabled( false );
        s_GsParticlePipeline.blendTargetStateInfoArray[ 0 ].SetBlendEnabled( true );
        s_GsParticlePipeline.blendTargetStateInfoArray[ 0 ].SetColorBlendFunction( nn::gfx::BlendFunction_Add );
        s_GsParticlePipeline.blendTargetStateInfoArray[ 0 ].SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
        s_GsParticlePipeline.blendTargetStateInfoArray[ 0 ].SetDestinationColorBlendFactor( nn::gfx::BlendFactor_One );
        s_GsParticlePipeline.blendTargetStateInfoArray[ 0 ].SetAlphaBlendFunction( nn::gfx::BlendFunction_Add );
        s_GsParticlePipeline.blendTargetStateInfoArray[ 0 ].SetSourceAlphaBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
        s_GsParticlePipeline.blendTargetStateInfoArray[ 0 ].SetDestinationAlphaBlendFactor( nn::gfx::BlendFactor_One );
        size_t size = nn::gfx::BlendState::GetRequiredMemorySize( s_GsParticlePipeline.blendStateInfo );
        void* data = DEMOGfxAllocMEM2( size, nn::gfx::BlendState::RequiredMemoryInfo_Alignment );
        s_GsParticleBlendStates[ BLEND_ADD_GS ].SetMemory( data, size );
        s_GsParticleBlendStates[ BLEND_ADD_GS ].Initialize( &DEMODevice, s_GsParticlePipeline.blendStateInfo );
        s_GsParticleDepthStencilStates[ BLEND_ADD_GS ].Initialize( &DEMODevice, s_GsParticlePipeline.depthStencilStateInfo );
    }
}

// Initialize GS particle.
void InitGsParticle()
{
    DEMOGfxLoadShadersFromFile( &s_GsParticlePipeline.shaders, 0, GS_PARTICLE[g_ShaderFileIdx] );

    // Texture location.
    s_GsParticleBaseMapLoc = s_GsParticlePipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_basemap" );

    // Initialize attibute stream.
    {
        u32 offset = 0;
        DEMOGfxInitShaderAttribute( &s_GsParticlePipeline.shaders, "a_position", 0, offset, nn::gfx::AttributeFormat_32_32_32_Float );
        offset += 3 * sizeof(f32);
        DEMOGfxInitShaderAttribute( &s_GsParticlePipeline.shaders, "a_color", 0, offset, nn::gfx::AttributeFormat_8_8_8_8_Unorm );
        offset += 4 * sizeof(u8);
        DEMOGfxInitShaderAttribute( &s_GsParticlePipeline.shaders, "a_life", 0, offset, nn::gfx::AttributeFormat_16_16_Uint );
        offset += 2 * sizeof(u16);
        DEMOGfxInitShaderAttribute( &s_GsParticlePipeline.shaders, "a_attr0", 0, offset, nn::gfx::AttributeFormat_8_8_8_8_Uint );
        offset += 4 * sizeof(u8);
        DEMOGfxInitShaderAttribute( &s_GsParticlePipeline.shaders, "a_attr1", 0, offset, nn::gfx::AttributeFormat_16_16_Uint );

        // Setup vertex buffer
        DEMOGfxInitShaderVertexBuffer( &s_GsParticlePipeline.shaders, 0, sizeof( TGsParticle ), 0 );
    }

    // Set-up uniform blocks.
    for ( int idx = 0; idx < 2; idx++ )
    {
        s_ViewProj[ idx ].Initialize( sizeof( TViewProj ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );

        s_System[ idx ].Initialize( sizeof( TSystem ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );

        s_NodeAttr[ idx ].Initialize( sizeof( TNodeAttr ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );

        s_Texture[ idx ].Initialize( sizeof( TTexture ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
    }
    s_ViewProjLoc = s_GsParticlePipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Geometry,
        nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_viewProj" );
    s_SystemLoc = s_GsParticlePipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex,
        nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_systemConst" );
    s_NodeAttrLoc = s_GsParticlePipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex,
        nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_nodeAttr" );
    s_TextureLoc = s_GsParticlePipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Geometry,
        nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_textureConst" );

#if NN_GFX_IS_TARGET_GX
    // Set-up geometry shader.
    s_InputRingBufferSize = GX2CalcGeometryShaderInputRingBufferSize( reinterpret_cast< GX2VertexShader* >( s_GsParticlePipeline.shaders.vertexShaderData )->ringItemsize );
    s_pInputRingBuffer = DEMOGfxAllocMEM1( s_InputRingBufferSize, 256 );

    s_OutputRingBufferSize = GX2CalcGeometryShaderOutputRingBufferSize( reinterpret_cast< GX2GeometryShader* >( s_GsParticlePipeline.shaders.geomShaderData )->ringItemsize );
    s_pOutputRingBuffer = DEMOGfxAllocMEM1( s_OutputRingBufferSize, 256 );
#endif

    // Initialize sampler.
    DEMOGfxInitSampler( &s_BaseMapSamplerGS, &s_BaseMapSamplerGSSlot,
        nn::gfx::TextureAddressMode_ClampToEdge,
        nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint,
        nn::gfx::ComparisonFunction_Always );

    // Create & set-up vertex buffer.
    // GS particle only requires 1 vertex per particle.
    s_GsParticleVtx.Initialize( sizeof( TGsParticle ) * MAX_PARTICLE, NULL, nn::gfx::GpuAccess_VertexBuffer, 0 );

    TGsParticle* pvtx = s_GsParticleVtx.Map< TGsParticle >();

    // Reset random number.
    DEMOSRand(1);
    for (u32 ui = 0; ui < MAX_PARTICLE; ui++)
    {
        // Vertex position [-1.0..1.0].
        // This position is relative to bounding box position and size. Assigning 1.0 or -1.0
        // will make vertex positioned at the farthest point of particular axis in the bounding
        // box.
        f32 posx = ( DEMOFRand() * 2 ) - 1.0f;
        f32 posy = ( DEMOFRand() * 2 ) - 1.0f;
        f32 posz = ( DEMOFRand() * 2 ) - 1.0f;

        // Particle birth time and life.
        // Birth time assumes the first frame a particle being displayed.
        u16 birth = DEMORand() % 100;
        // Life time assumes the number of frames required to travel from the first node to
        // the last node. Smaller value will make a particle moves faster.
        u16 life = (DEMORand() % 500) + 60;

        // Particle size.
        u8 particle_size = DEMORand() % 3 + 2;
        // Time shift, used to decrease regularity for particles with similar property.
        u8 time_shift = DEMORand() % 256;
        // Flip animation start index and end index.
        u8 animation_start = 0;
        u8 animation_end = 255;

        // Rotation (in degrees).
        u16 rotation_start = DEMORand() % 360;
        u16 rotation_end = DEMORand() % 360;

        pvtx->pos[0] = posx;
        pvtx->pos[1] = posy;
        pvtx->pos[2] = posz;
        // Vertex color (vertex color will be multiplied with fragment color).
        pvtx->color = 0xffffffff;
        pvtx->life[0] = birth;
        pvtx->life[1] = life;
        pvtx->attr0[0] = particle_size;
        pvtx->attr0[1] = time_shift;
        pvtx->attr0[2] = animation_start;
        pvtx->attr0[3] = animation_end;
        pvtx->attr1[0] = rotation_start;
        pvtx->attr1[1] = rotation_end;

        // Write 1 vertex only.
        pvtx++;
    }

    // Send vertex data to GPU.
    s_GsParticleVtx.Unmap();

    // Initialize model matrix.
    MTX44Identity( s_ModelMatrix44GS );

    // Initialize Alpha Test
    for ( int idx = 0; idx < 2; idx++ )
    {
        s_Pixel[ idx ].Initialize( sizeof( TPixel ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
    }
    s_PixelLoc = s_GsParticlePipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_alpha" );

    InitGsParticlePipelines();

    // Set the default mode
    GsParticleMode(SNOW);
}

// Update GS particle.
void DrawGsParticle(Mtx44 viewMatrix, Mtx44 projMatrix)
{
    Mtx44 invView;
    MTX44Inverse(viewMatrix, invView);

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_GEOMETRY_SHADER );
#endif
    // Set the default state
    DEMOCommandBuffer.SetPipeline( &s_GsParticlePipeline.pipeline );

    // Set the blend/depth state specific to this state
    DEMOCommandBuffer.SetBlendState( &s_GsParticleBlendStates[ s_BlendStateGS ] );
    DEMOCommandBuffer.SetDepthStencilState( &s_GsParticleDepthStencilStates[ s_BlendStateGS ] );

#if NN_GFX_IS_TARGET_GX
    GX2SetGeometryShaderInputRingBuffer( s_pInputRingBuffer, s_InputRingBufferSize );
    GX2SetGeometryShaderOutputRingBuffer(  s_pOutputRingBuffer, s_OutputRingBufferSize );
#endif

    // Invalidate & set-up texture information if necessary.
    if (s_InvalidateTexture)
    {
        s_TextureCnt ^= 1;
        s_InvalidateTexture = false;
    }
    DEMOCommandBuffer.SetConstantBuffer( s_TextureLoc, nn::gfx::ShaderStage_Geometry, s_Texture[ s_TextureCnt ].gpuAddress, s_Texture[ s_TextureCnt ].size );

    // Invalidate & set-up node attributes if necessary.
    if (s_InvalidateNodeAttr)
    {
        s_NodeAttribCnt ^= 1;
        s_InvalidateNodeAttr = false;
    }
    DEMOCommandBuffer.SetConstantBuffer( s_NodeAttrLoc, nn::gfx::ShaderStage_Vertex, s_NodeAttr[ s_NodeAttribCnt ].gpuAddress, s_NodeAttr[ s_NodeAttribCnt ].size );

    // Set-up alpha test state.
    TPixel* pPixel = s_Pixel[ s_BufferPointer ].Map< TPixel >();
    pPixel->alphaTest[ 0 ] = ( s_AlphaTestGS ? 0.5f : -1.0f );
    pPixel->alphaTest[ 1 ] = 0.0f;
    pPixel->alphaTest[ 2 ] = 0.0f;
    pPixel->alphaTest[ 3 ] = 0.0f;

#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pPixel, sizeof( TPixel ) );
#endif

    s_Pixel[ s_BufferPointer ].Unmap();

    DEMOCommandBuffer.SetConstantBuffer( s_PixelLoc,
        nn::gfx::ShaderStage_Pixel, s_Pixel[ s_BufferPointer ].gpuAddress, s_Pixel[ s_BufferPointer ].size );

    // Set-up dynamic uniform blocks.
    TViewProj* pViewProj = s_ViewProj[ s_BufferPointer ].Map< TViewProj >();
    MTX44Copy(viewMatrix, pViewProj->view);
    MTX44Copy(invView, pViewProj->viewInv);
    MTX44Copy(projMatrix, pViewProj->proj);

#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pViewProj, sizeof( TViewProj ) );
#endif
    s_ViewProj[ s_BufferPointer ].Unmap();
    DEMOCommandBuffer.SetConstantBuffer( s_ViewProjLoc, nn::gfx::ShaderStage_Geometry, s_ViewProj[ s_BufferPointer ].gpuAddress, s_ViewProj[ s_BufferPointer ].size );

    TSystem* pSystem = s_System[s_BufferPointer].Map< TSystem >();
    MTX44Copy(s_ModelMatrix44GS, pSystem->model);
    pSystem->loopAttr[0] = loopLength;
    pSystem->loopAttr[1] = currentLoop;
    pSystem->loopAttr[2] = 0.0f;
    pSystem->loopAttr[3] = s_ParticleScaleGS;

#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pSystem, sizeof( *pSystem ) );
#endif
    s_System[ s_BufferPointer ].Unmap();
    DEMOCommandBuffer.SetConstantBuffer( s_SystemLoc, nn::gfx::ShaderStage_Vertex, s_System[ s_BufferPointer ].gpuAddress,  s_System[ s_BufferPointer ].size );

    DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_ConstantBuffer );

    // Swap uniform block buffer.
    s_BufferPointer ^= 1;

    // Set-up vertex attributes & vertex buffer.
    DEMOCommandBuffer.SetVertexBuffer( 0, s_GsParticleVtx.gpuAddress, sizeof( TGsParticle ), s_GsParticleVtx.size );

    // Set-up texture.
    DEMOCommandBuffer.SetTextureAndSampler( s_GsParticleBaseMapLoc, nn::gfx::ShaderStage_Pixel,
        s_pBaseMapGS->GetDescriptorSlot( 0 ), s_BaseMapSamplerGSSlot );

    // Draw points (only one vertex needed to replicate vertices in GS).
    DEMOCommandBuffer.Draw( nn::gfx::PrimitiveTopology_PointList, currentParticleNum, 0 );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
}

// Terminate GS particle.
void TermGsParticle()
{
    // Free pipeline and other states
    s_GsParticlePipeline.Finalize( &DEMODevice );

    for ( int idx = 0; idx < BLEND_ADD_GS + 1; idx++ )
    {
        void* data = s_GsParticleBlendStates[ idx ].GetMemory();
        s_GsParticleBlendStates[ idx ].Finalize( &DEMODevice );
        DEMOGfxFreeMEM2( data );
        s_GsParticleDepthStencilStates[ idx ].Finalize( &DEMODevice );
    }

    // Free buffers.

#if NN_GFX_IS_TARGET_GX
    // MEM1 buffers need special treatment during shutdown.
    DEMOGfxFreeMEM1( s_pInputRingBuffer );
    DEMOGfxFreeMEM1( s_pOutputRingBuffer );
#endif

    s_ViewProj[ 0 ].Finalize();
    s_ViewProj[ 1 ].Finalize();
    s_System[ 0 ].Finalize();
    s_System[ 1 ].Finalize();
    s_NodeAttr[ 0 ].Finalize();
    s_NodeAttr[ 1 ].Finalize();
    s_Texture[ 0 ].Finalize();
    s_Texture[ 1 ].Finalize();

    s_Pixel[ 0 ].Finalize();
    s_Pixel[ 1 ].Finalize();

    s_GsParticleVtx.Finalize();

    s_BaseMapSamplerGS.Finalize( &DEMODevice );
}

// Set GS particle mode.
void GsParticleMode(PARTICLE_MODE mode)
{
    switch (mode)
    {
    case SNOW:
        // Snow/blizzard showcase.
        s_pBaseMapGS = &g_SnowTex;
        s_BlendStateGS = BLEND_ALPHA_GS;
        s_AlphaTestGS = false;

        // Particle scale, scales each particle equally.
        s_ParticleScaleGS = 0.2f;

        {
            u32 ntex = s_TextureCnt ^ 1;
            TTexture* pTexture = s_Texture[ ntex ].Map< TTexture >();

            // Texture size: 32*32 pixels.
            pTexture->texture[0] = 32.0f;
            pTexture->texture[1] = 32.0f;
            pTexture->texture[2] = 1.0f / 32.0f;
            pTexture->texture[3] = 1.0f / 32.0f;

#if NN_GFX_IS_TARGET_GX
            GX2EndianSwap( pTexture, sizeof( *pTexture ) );
#endif
            s_Texture[ ntex ].Unmap();
            s_InvalidateTexture = true;
        }

        // Node information.
        {
            u32 nattr = s_NodeAttribCnt ^ 1;

            TNodeAttr* pNodeAttr = s_NodeAttr[nattr].Map< TNodeAttr >();

            // Position for bounding box 1-4.
            Mtx44 bb_pos = {
                { -40.0f, 100.0f, 0.0f, 1.0f },
                { 20.0f, 60.0f, 0.0f, 1.0f },
                { -30.0f, 50.0f, 0.0f, 1.0f },
                { 0.0f, 0.0f, 0.0f, 1.0f },

            };
            // Bounding box size (x: width, y: height, z: depth, w: unused).
            Mtx44 bb_size = {
                { 250.0f, 20.0f, 250.0f, 0.0f },
                { 250.0f, 1.0f, 250.0f, 0.0f },
                { 250.0f, 1.0f, 250.0f, 0.0f },
                { 400.0f, 1.0f, 400.0f, 0.0f },
            };
            // Bounding box color (RGBA)
            Mtx44 n_color = {
                { 0.25f, 0.25f, 0.25f, 1.0f },
                { 0.8f, 0.8f, 0.8f, 1.0f },
                { 1.0f, 1.0f, 1.0f, 1.0f },
                { 0.3f, 0.3f, 0.3f, 1.0f },
            };
            // Assign to node informations.
            MTX44Copy(bb_pos, pNodeAttr->bbPos);
            MTX44Copy(bb_size, pNodeAttr->bbSize);
            MTX44Copy(n_color, pNodeAttr->nodeColor);

            // 1 row, 1 column, 1 pattern.
            pNodeAttr->pattern[0] = 1;
            pNodeAttr->pattern[1] = 1;
            pNodeAttr->pattern[2] = 1;
            // Unused.
            pNodeAttr->pattern[3] = 0;

#if NN_GFX_IS_TARGET_GX
            GX2EndianSwap( pNodeAttr, sizeof( *pNodeAttr ) );
#endif
            s_NodeAttr[ nattr ].Unmap();

            s_InvalidateNodeAttr = true;
        }

        break;
    case SAKURA:
        // Sakura showcase.
        s_pBaseMapGS = &g_SakuraTex;
        s_BlendStateGS = BLEND_NONE_GS;
        s_AlphaTestGS = true;

        // Particle scale.
        s_ParticleScaleGS = 0.5f;

        {
            u32 ntex = s_TextureCnt ^ 1;
            TTexture* pTexture = s_Texture[ ntex ].Map< TTexture >();

            // Texture size: 1024*1024 pixels.
            pTexture->texture[0] = 1024.0f;
            pTexture->texture[1] = 1024.0f;
            pTexture->texture[2] = 1.0f / 1024.0f;
            pTexture->texture[3] = 1.0f / 1024.0f;

#if NN_GFX_IS_TARGET_GX
            GX2EndianSwap( pTexture, sizeof( *pTexture ) );
#endif
            s_Texture[ ntex ].Unmap();
            s_InvalidateTexture = true;
        }

        // Node information.
        {
            u32 nattr = s_NodeAttribCnt ^ 1;

            TNodeAttr* pNodeAttr = s_NodeAttr[nattr].Map< TNodeAttr >();

            Mtx44 bb_pos = {
                { -40.0f, 100.0f, 0.0f, 1.0f },
                { 20.0f, 80.0f, 0.0f, 1.0f },
                { 0.0f, 40.0f, 0.0f, 1.0f },
                { 40.0f, 0.0f, 0.0f, 1.0f },
            };
            Mtx44 bb_size = {
                { 250.0f, 40.0f, 250.0f, 0.0f },
                { 250.0f, 20.0f, 250.0f, 0.0f },
                { 250.0f, 50.0f, 250.0f, 0.0f },
                { 250.0f, 10.0f, 250.0f, 0.0f },
            };
            Mtx44 n_color = {
                { 0.50f, 0.25f, 0.25f, 1.0f },
                { 0.9f, 0.75f, 0.75f, 1.0f },
                { 1.0f, 0.75f, 0.75f, 1.0f },
                { 0.5f, 0.3f, 0.3f, 1.0f },
            };
            MTX44Copy(bb_pos, pNodeAttr->bbPos);
            MTX44Copy(bb_size, pNodeAttr->bbSize);
            MTX44Copy(n_color, pNodeAttr->nodeColor);

            // 16 rows, 16 columns, 256 patterns.
            pNodeAttr->pattern[0] = 16;
            pNodeAttr->pattern[1] = 16;
            pNodeAttr->pattern[2] = 256;
            // Unused.
            pNodeAttr->pattern[3] = 0;

#if NN_GFX_IS_TARGET_GX
            GX2EndianSwap( pNodeAttr, sizeof( *pNodeAttr ) );
#endif
            s_NodeAttr[nattr].Unmap();
            s_InvalidateNodeAttr = true;
        }

        break;
    case MIST:
        // Mist showcase.
        s_pBaseMapGS = &g_MistTex;
        s_BlendStateGS = BLEND_ADD_GS;
        s_AlphaTestGS = false;

        // Particle scale.
        s_ParticleScaleGS = 5.0f;

        {
            u32 ntex = s_TextureCnt ^ 1;

            TTexture* pTexture = s_Texture[ ntex ].Map< TTexture >();
            // Texture size.
            pTexture->texture[0] = 1024.0f;
            pTexture->texture[1] = 1024.0f;
            pTexture->texture[2] = 1.0f / 1024.0f;
            pTexture->texture[3] = 1.0f / 1024.0f;

#if NN_GFX_IS_TARGET_GX
            GX2EndianSwap( pTexture, sizeof( *pTexture ) );
#endif
            s_Texture[ ntex ].Unmap();
            s_InvalidateTexture = true;
        }

        // Node information.
        {
            u32 nattr = s_NodeAttribCnt ^ 1;

            TNodeAttr* pNodeAttr = s_NodeAttr[nattr].Map< TNodeAttr >();

            Mtx44 bb_pos = {
                { 0.0f, 50.0f, -200.0f, 1.0f },
                { 0.0f, 50.0f, -100.0f, 1.0f },
                { 0.0f, 50.0f, 100.0f, 1.0f },
                { 0.0f, 50.0f, 200.0f, 1.0f },
            };
            Mtx44 bb_size = {
                { 250.0f, 250.0f, 100.0f, 0.0f },
                { 250.0f, 250.0f, 100.0f, 0.0f },
                { 250.0f, 250.0f, 100.0f, 0.0f },
                { 250.0f, 250.0f, 200.0f, 0.0f },
            };
            Mtx44 n_color = {
                { 0.8f, 0.8f, 0.8f, 0.0f },
                { 0.4f, 0.4f, 0.4f, 1.0f },
                { 0.4f, 0.4f, 0.4f, 1.0f },
                { 0.8f, 0.8f, 0.8f, 0.0f },
            };
            MTX44Copy(bb_pos, pNodeAttr->bbPos);
            MTX44Copy(bb_size, pNodeAttr->bbSize);
            MTX44Copy(n_color, pNodeAttr->nodeColor);

            // 8 rows, 8 columns, 32 patterns.
            pNodeAttr->pattern[0] = 8;
            pNodeAttr->pattern[1] = 8;
            pNodeAttr->pattern[2] = 32;
            // Unused.
            pNodeAttr->pattern[3] = 0;

#if NN_GFX_IS_TARGET_GX
            GX2EndianSwap( pNodeAttr, sizeof( *pNodeAttr ) );
#endif
            s_NodeAttr[ nattr ].Unmap();
            s_InvalidateNodeAttr = true;
        }

        break;
    default:
        break;
    }
} // NOLINT(impl/function_size)

// Vertex buffer size.
u32 GsParticleVtxBufSize()
{
    // 1 particle is a point.
    return currentParticleNum * sizeof(TGsParticle);
}
