﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>
#include <cmath>

#include <gfx/demo.h>

#if NN_GFX_IS_TARGET_GX
#include <cafe/gx2ut.h>
#endif

#include <nnt.h>
#include <nnt/nnt_Argument.h>

// There are 3 techniques being demoed
enum Technique
{
    RESOLVE_COLOR,
    RENDER2TEXTURE,
    EXPAND_COLOR
};

// ----- Surface Information

static const int RENDERTARGET_WIDTH = 120;
static const int RENDERTARGET_HEIGHT = 80;

#define SURFACE_WIDTH  (DEMOColorBufferInfo.GetWidth())
#define SURFACE_HEIGHT  (DEMOColorBufferInfo.GetHeight())

// ----- Shader information

static int g_ShaderFileIdx = 0;
static const char * const GSH_SHADER_FILE_1ST[] =
{
    "shaders/expandColor/transform",
    "shaders/expandColor/transformHlslcc",
};
static const char * const GSH_SHADER_FILE_2ND[] =
{
    "shaders/expandColor/texture2D",
    "shaders/expandColor/texture2DHlslcc",
};
static const char * const GSH_SHADER_FILE_2ND_MS[] =
{
    "shaders/expandColor/texture2DMS",
    "shaders/expandColor/texture2DMSHlslcc",
};

struct vertexUniforms {
    Mat44 viewMtx;
    Mat44 projMtx;
    Mat44 modelMtx;
};

typedef struct _ImageData
{
    nn::gfx::Texture texture;
    nn::gfx::TextureView textureView;
    nn::gfx::ColorTargetView colorView;
    nn::gfx::DepthStencilView depthView;
    nn::gfx::DescriptorSlot descriptorSlot;
    DEMOGfxMemPool* pPool;
    bool freed;
} ImageData;

static struct _FirstPass
{
    DEMOGfxPipeline pipeline;
    nn::gfx::ViewportScissorState viewportScissorState;
    void* pViewportScissorData;

    // -- Shader Location -- //
    u32 posLoc;
    u32 clrLoc;

    // Constant Buffers
    int vertexUniformBlockLoc;
    DEMOGfxBuffer vertexUniformBuffer1;
    DEMOGfxBuffer vertexUniformBuffer2;

    // -- Initial Data -- //
    int numAttrib;
    int numIndices;
    f32 posInitData[9];
    f32 clrInitData[9];
    u32 idxInitData[3];
    DEMOGfxBuffer positionBuffer;
    DEMOGfxBuffer colorBuffer;
    DEMOGfxBuffer indexBuffer;

    // -- Texture Setting -- //
    ImageData* pMyColorBuffer;
    ImageData  depthBuffer;

} FirstPassResolveColor, FirstPassRender2Texture, FirstPassExpandColor;

static struct _SecondPass
{
    DEMOGfxPipeline pipeline;

    // -- Shader Location -- //
    u32 posLoc;
    u32 texcoordLoc;
    u32 textureLoc;

    u32 offsetLoc;
    DEMOGfxBuffer offsetBuffer;

    // -- Attribute Buffers -- //
    int numAttrib;
    int numIndices;
    f32 posInitData[12];
    f32 texInitData[8];
    u32 idxInitData[4];
    DEMOGfxBuffer positionBuffer;
    DEMOGfxBuffer texcoordBuffer;
    DEMOGfxBuffer indexBuffer;

    // -- Texture Setting -- //
    ImageData* pMyTextureBuffer;
    nn::gfx::Sampler mySampler;
    nn::gfx::DescriptorSlot mySamplerDescriptorSlot;
} SecondPassResolveColor, SecondPassRender2Texture, SecondPassExpandColor;

static ImageData ResolveColorData, Render2TextureData, ExpandColorData;
static ImageData ResolveSecondData;

static nn::gfx::ViewportScissorState wholeViewport;
static void* wholeViewportData;

static float rotateY;

// Prototype
static void SceneDraw(Technique technique, _FirstPass *FirstPass, _SecondPass *SecondPass, float xOffset, float yOffset);
static void SceneFree(Technique technique, _FirstPass *FirstPass, _SecondPass *SecondPass);
static void AnimTick(void);
static void PrintInfo(void);

static void SetupTextureBuffer(nn::gfx::Texture* pTexture,
    nn::gfx::TextureView* pTextureView,
    nn::gfx::DescriptorSlot* pSlot,
    nn::gfx::ColorTargetView* pColorView,
    nn::gfx::DepthStencilView* pDepthView,
    DEMOGfxMemPool** pMemPool,
    int width,
    int height,
    int depth,
    int mipCount,
    nn::gfx::ImageDimension dimensions,
    nn::gfx::ImageFormat format,
    nn::gfx::DepthStencilFetchMode fetchMode,
    int multisample)
{
    DEMOGfxMemPool* pool = *pMemPool;

    if (pTexture)
    {
        nn::gfx::Texture::InfoType info;

        info.SetDefault();
        info.SetWidth(width);
        info.SetHeight(height);
        info.SetDepth(depth);
        info.SetImageStorageDimension(DEMOConvertToStorageDimension(dimensions));
        info.SetImageFormat(format);
        info.SetMultiSampleCount(multisample);
        info.SetSwizzle(0);
        info.SetMipCount(mipCount);
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture |
            (((format & (nn::gfx::TypeFormat_Bits - 1)) == nn::gfx::TypeFormat_DepthStencil) ?
            nn::gfx::GpuAccess_DepthStencil : nn::gfx::GpuAccess_ColorBuffer));

        if (!pool)
        {
            size_t memPoolSize = nn::gfx::Texture::CalculateMipDataSize(&DEMODevice, info);
            pool = DEMOGfxMemPool::AllocNewPool(memPoolSize, nn::gfx::Texture::CalculateMipDataAlignment(&DEMODevice, info), nn::gfx::MemoryPoolProperty_Compressible | nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached);
            *pMemPool = pool;
        }
        pTexture->Initialize(&DEMODevice, info, pool->GetPool(), pool->GetBaseOffset(), pool->GetSize());
    }

    if (pColorView)
    {
        DEMOGfxSetupColorView(pTexture, pColorView, dimensions, format);
    }

    if (pTextureView)
    {
        DEMOGfxSetupTextureView(pTexture, pTextureView, pSlot, dimensions, format, fetchMode);
    }

    if (pDepthView)
    {
        DEMOGfxSetupDepthView(pTexture, pDepthView, dimensions);
    }
}

// The init function for the rendering portions of this app
static void SceneInit(Technique technique, _FirstPass *FirstPass, _SecondPass *SecondPass, ImageData* pImage)
{
    rotateY = 0.0f;
    int aaSamples = 1;

    switch(technique)
    {
    case RESOLVE_COLOR:
    case EXPAND_COLOR:
        aaSamples = 4;
        break;

    case RENDER2TEXTURE:
    default:
        aaSamples = 1;
        break;
    }

    // ----------------- //
    // First pass setup
    // ----------------- //

    // geometry setup
    FirstPass->pipeline.SetDefaults();
    SecondPass->pipeline.SetDefaults();

    FirstPass->posInitData[0] = -0.4f; FirstPass->posInitData[1] = -0.4f;
    FirstPass->posInitData[2] =  0.0f; FirstPass->posInitData[3] =  0.4f;
    FirstPass->posInitData[4] = -0.4f; FirstPass->posInitData[5] =  0.0f;
    FirstPass->posInitData[6] =  0.0f; FirstPass->posInitData[7] =  0.4f;
    FirstPass->posInitData[8] =  0.0f;

    FirstPass->clrInitData[0] =  1.0f; FirstPass->clrInitData[1] =  0.0f;
    FirstPass->clrInitData[2] =  0.0f; FirstPass->clrInitData[3] =  0.0f;
    FirstPass->clrInitData[4] =  1.0f; FirstPass->clrInitData[5] =  0.0f;
    FirstPass->clrInitData[6] =  0.0f; FirstPass->clrInitData[7] =  0.0f;
    FirstPass->clrInitData[8] =  1.0f;

    FirstPass->idxInitData[0] = 0;
    FirstPass->idxInitData[1] = 1;
    FirstPass->idxInitData[2] = 2;

    FirstPass->numAttrib = 2;
    FirstPass->numIndices = 3;

    // Load shader binary to memory
    DEMOGfxShader* pShader = &FirstPass->pipeline.shaders;
    DEMOGfxLoadShadersFromFile(pShader, 0, GSH_SHADER_FILE_1ST[g_ShaderFileIdx]);

    // Attribute Location Lookup
    FirstPass->posLoc = FirstPass->pipeline.shaders.GetInterfaceSlot(nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_position");
    FirstPass->clrLoc = FirstPass->pipeline.shaders.GetInterfaceSlot(nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_color");

    // Uniform Location Lookup
    FirstPass->vertexUniformBuffer1.Initialize(sizeof(vertexUniforms), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0);
    FirstPass->vertexUniformBuffer2.Initialize(sizeof(vertexUniforms), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0);
    FirstPass->vertexUniformBlockLoc = FirstPass->pipeline.shaders.GetInterfaceSlot(nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_vertex");

    // Attribute Format Setup
    DEMOGfxInitShaderAttribute(&FirstPass->pipeline.shaders, "a_position", FirstPass->posLoc, 0, nn::gfx::AttributeFormat_32_32_32_Float);
    DEMOGfxInitShaderAttribute(&FirstPass->pipeline.shaders, "a_color", FirstPass->clrLoc, 0, nn::gfx::AttributeFormat_32_32_32_Float);
    DEMOGfxInitShaderVertexBuffer(&FirstPass->pipeline.shaders, FirstPass->posLoc, sizeof(float) * 3, 0);
    DEMOGfxInitShaderVertexBuffer(&FirstPass->pipeline.shaders, FirstPass->clrLoc, sizeof(float) * 3, 0);

    // Vertex position buffer
    FirstPass->positionBuffer.Initialize(sizeof(FirstPass->posInitData), FirstPass->posInitData,
        nn::gfx::GpuAccess_VertexBuffer, 0);

    // Vertex color buffer
    FirstPass->colorBuffer.Initialize(sizeof(FirstPass->clrInitData), FirstPass->clrInitData,
        nn::gfx::GpuAccess_VertexBuffer, 0);

    // Index buffer
    FirstPass->indexBuffer.Initialize(sizeof(FirstPass->idxInitData), FirstPass->idxInitData,
        nn::gfx::GpuAccess_IndexBuffer, 0);


    // Initialize render color buffer
    nn::gfx::ImageDimension dim;

    // Initialize texture buffer
    {

        switch (technique)
        {
        case EXPAND_COLOR:
        case RESOLVE_COLOR:
            dim = nn::gfx::ImageDimension_2dMultisample;
            break;
        case RENDER2TEXTURE:
        default:
            dim = nn::gfx::ImageDimension_2d;
            break;
        }
        SetupTextureBuffer(&pImage->texture, &pImage->textureView, &pImage->descriptorSlot, &pImage->colorView, NULL, &pImage->pPool,
            RENDERTARGET_WIDTH, RENDERTARGET_HEIGHT, 1, 1,
            dim, nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm, nn::gfx::DepthStencilFetchMode_DepthComponent, aaSamples);
    }

    FirstPass->pMyColorBuffer = pImage;
    FirstPass->pipeline.blendTargetStateCount = 1;
    FirstPass->pipeline.colorTargetStateCount = 1;
    FirstPass->pipeline.blendTargetStateInfoArray[0].SetDefault();
    FirstPass->pipeline.colorTargetStateInfoArray[0].SetDefault();
    FirstPass->pipeline.colorTargetStateInfoArray[0].SetFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);

    if (technique == RESOLVE_COLOR || technique == EXPAND_COLOR)
    {
        FirstPass->pipeline.rasterizerStateInfo.SetMultisampleEnabled(true);
        FirstPass->pipeline.rasterizerStateInfo.EditMultisampleStateInfo().SetDefault();
        FirstPass->pipeline.rasterizerStateInfo.EditMultisampleStateInfo().SetSampleCount(aaSamples);
    }
    FirstPass->pipeline.Initialize(&DEMODevice);

    // Initialize render depth buffer
    SetupTextureBuffer(&FirstPass->depthBuffer.texture, NULL, NULL, NULL, &FirstPass->depthBuffer.depthView, &FirstPass->depthBuffer.pPool,
        RENDERTARGET_WIDTH, RENDERTARGET_HEIGHT, 1, 1, dim,
        nn::gfx::ImageFormat_D32_Float, nn::gfx::DepthStencilFetchMode_DepthComponent, aaSamples);

    // Set up the sampler object
#if NN_GFX_IS_TARGET_D3D
    // Comparison sampling cannot be set to usual texture in d3d11.
    DEMOGfxInitSampler( &SecondPass->mySampler, &SecondPass->mySamplerDescriptorSlot, nn::gfx::TextureAddressMode_ClampToEdge,
        nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint, nn::gfx::ComparisonFunction_Always );
#else
    DEMOGfxInitSampler( &SecondPass->mySampler, &SecondPass->mySamplerDescriptorSlot, nn::gfx::TextureAddressMode_ClampToEdge,
        nn::gfx::FilterMode_Comparison_MinPoint_MagPoint_MipPoint, nn::gfx::ComparisonFunction_Always );
#endif

    DEMOGfxSetViewportScissorState( &FirstPass->viewportScissorState, &FirstPass->pViewportScissorData,
        0.0f, 0.0f, (float)RENDERTARGET_WIDTH, (float)RENDERTARGET_HEIGHT, 0.0f, 1.0f,
        (float)RENDERTARGET_HEIGHT, true );

    // ------------------ //
    // Second pass setup
    // ------------------ //

    SecondPass->posInitData[0] = -0.3f; SecondPass->posInitData[1] = -0.4f;
    SecondPass->posInitData[2] =  0.0f; SecondPass->posInitData[3] =  0.3f;
    SecondPass->posInitData[4] = -0.4f; SecondPass->posInitData[5] =  0.0f;
    SecondPass->posInitData[6] =  0.3f; SecondPass->posInitData[7] =  0.4f;
    SecondPass->posInitData[8] =  0.0f; SecondPass->posInitData[9] = -0.3f;
    SecondPass->posInitData[10] = 0.4f; SecondPass->posInitData[11] = 0.0f;

    SecondPass->texInitData[0] =  0.0f; SecondPass->texInitData[1] =  1.0f;
    SecondPass->texInitData[2] =  1.0f; SecondPass->texInitData[3] =  1.0f;
    SecondPass->texInitData[4] =  1.0f; SecondPass->texInitData[5] =  0.0f;
    SecondPass->texInitData[6] =  0.0f; SecondPass->texInitData[7] =  0.0f;

    SecondPass->idxInitData[0] = 0; SecondPass->idxInitData[1] = 1;
    SecondPass->idxInitData[2] = 3; SecondPass->idxInitData[3] = 2;
    SecondPass->numAttrib = 2;
    SecondPass->numIndices = 4;

    if (technique == RESOLVE_COLOR)
    {
        SecondPass->pMyTextureBuffer = &ResolveSecondData;
        SetupTextureBuffer(&ResolveSecondData.texture, &ResolveSecondData.textureView, &ResolveSecondData.descriptorSlot, &ResolveSecondData.colorView, NULL, &ResolveSecondData.pPool,
            RENDERTARGET_WIDTH, RENDERTARGET_HEIGHT, 1, 1,
            nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm, nn::gfx::DepthStencilFetchMode_DepthComponent, 0);
    }
    else
    {
        SecondPass->pMyTextureBuffer = pImage;
    }
    // Load shader binary to memory
    switch(technique)
    {
    case RESOLVE_COLOR:
    case RENDER2TEXTURE:
    default:
        DEMOGfxLoadShadersFromFile(&SecondPass->pipeline.shaders, 0, GSH_SHADER_FILE_2ND[g_ShaderFileIdx]);
        break;

    case EXPAND_COLOR:
        DEMOGfxLoadShadersFromFile(&SecondPass->pipeline.shaders, 0, GSH_SHADER_FILE_2ND_MS[g_ShaderFileIdx]);
        break;
    }

    // Attribute Location Lookup
    SecondPass->posLoc = SecondPass->pipeline.shaders.GetInterfaceSlot(nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_position");
    SecondPass->texcoordLoc = SecondPass->pipeline.shaders.GetInterfaceSlot(nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_texCoord");
    SecondPass->textureLoc = SecondPass->pipeline.shaders.GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_texture");

    // Uniform Location lookup
    SecondPass->offsetLoc = SecondPass->pipeline.shaders.GetInterfaceSlot(nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_offsetbuf");
    SecondPass->offsetBuffer.Initialize(4 * sizeof(f32), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0);

    // Attribute Format Setup
    DEMOGfxInitShaderAttribute(&SecondPass->pipeline.shaders, "a_position", SecondPass->posLoc, 0, nn::gfx::AttributeFormat_32_32_32_Float);
    DEMOGfxInitShaderAttribute(&SecondPass->pipeline.shaders, "a_texCoord", SecondPass->texcoordLoc, 0, nn::gfx::AttributeFormat_32_32_Float);
    DEMOGfxInitShaderVertexBuffer(&SecondPass->pipeline.shaders, SecondPass->posLoc, sizeof(float) * 3, 0);
    DEMOGfxInitShaderVertexBuffer(&SecondPass->pipeline.shaders, SecondPass->texcoordLoc, sizeof(float) * 2, 0);

    SecondPass->pipeline.blendTargetStateCount = 1;
    SecondPass->pipeline.colorTargetStateCount = 1;
    SecondPass->pipeline.blendTargetStateInfoArray[0].SetDefault();
    SecondPass->pipeline.colorTargetStateInfoArray[0].SetDefault();
    SecondPass->pipeline.colorTargetStateInfoArray[0].SetFormat(DEMOColorBufferInfo.GetImageFormat());

    SecondPass->pipeline.Initialize(&DEMODevice);

    // Vertex position buffer
    SecondPass->positionBuffer.Initialize(sizeof(SecondPass->posInitData), SecondPass->posInitData,
        nn::gfx::GpuAccess_VertexBuffer, 0);

    // Vertex texcoord buffer
    SecondPass->texcoordBuffer.Initialize(sizeof(SecondPass->texInitData), SecondPass->texInitData,
        nn::gfx::GpuAccess_VertexBuffer, 0);

    // Index buffer
    SecondPass->indexBuffer.Initialize(sizeof(SecondPass->idxInitData), SecondPass->idxInitData,
        nn::gfx::GpuAccess_IndexBuffer, 0);
} // NOLINT(impl/function_size)

// The draw function for the rendering portions of this app
static void SceneDraw(Technique technique, _FirstPass *FirstPass, _SecondPass *SecondPass, float xOffset, float yOffset)
{
    // ------------------------------- //
    //   1ST PASS (RENDERING TEXTURE)
    // ------------------------------- //
    DEMOCommandBuffer.ClearColor( &FirstPass->pMyColorBuffer->colorView, 0.4f, 0.4f, 0.4f, 0.0f, NULL );
    DEMOCommandBuffer.ClearDepthStencil( &FirstPass->depthBuffer.depthView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif

    nn::gfx::ColorTargetView* colorTargets[] = { &FirstPass->pMyColorBuffer->colorView };


    DEMOCommandBuffer.SetRenderTargets( 1, colorTargets, &FirstPass->depthBuffer.depthView );
    DEMOCommandBuffer.SetViewportScissorState( &FirstPass->viewportScissorState );

    DEMOCommandBuffer.SetPipeline( &FirstPass->pipeline.pipeline );
    DEMOCommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_IndexBuffer);

    // Bind position buffer
    DEMOCommandBuffer.SetVertexBuffer( FirstPass->posLoc, FirstPass->positionBuffer.gpuAddress, sizeof( float ) * 3, sizeof( float ) * 3 * FirstPass->positionBuffer.size );
    DEMOCommandBuffer.SetVertexBuffer( FirstPass->clrLoc, FirstPass->colorBuffer.gpuAddress, sizeof( float ) * 3, sizeof( float ) * 3 * FirstPass->colorBuffer.size );

    float modelmat[4][4] = { { 1.5f, 0.0f, 0.0f, -0.5f },
                             { 0.0f, 1.5f, 0.0f, 0.0f },
                             { 0.0f, 0.0f, 1.0f, 0.0f },
                             { 0.0f,  0.0f,  0.0f,  1.0f } };
    float projmat[4][4] = {  { 1.0f,  0.0f,  0.0f,  0.0f },
                             { 0.0f, 1.0f, 0.0f, 0.0f },
                             { 0.0f, 0.0f, 1.0f, 0.0f },
                             { 0.0f, 0.0f, 0.0f, 1.0f } };
    float viewmat[4][4] = { { 1.0f, 0.0f, 0.0f, 0.0f },
                            { 0.0f, 1.0f, 0.0f, 0.0f },
                            { 0.0f, 0.0f, 1.0f, 0.0f },
                            { 0.0f, 0.0f, 0.0f, 1.0f } };
    modelmat[0][0] = cos(rotateY);  modelmat[2][0] = -sin(rotateY);
    modelmat[0][2] = sin(rotateY);  modelmat[2][2] = cos(rotateY);


    // Uniform Location Lookup
    struct vertexUniforms *vs = FirstPass->vertexUniformBuffer1.Map< struct vertexUniforms >();
    memcpy(&vs->viewMtx, &viewmat[0], sizeof(Mat44));
    memcpy(&vs->projMtx, &projmat[0], sizeof(Mat44));
    memcpy(&vs->modelMtx, &modelmat[0], sizeof(Mat44));
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap(vs, sizeof(*vs));
#endif
    FirstPass->vertexUniformBuffer1.Unmap();
    DEMOCommandBuffer.SetConstantBuffer(FirstPass->vertexUniformBlockLoc, nn::gfx::ShaderStage_Vertex, FirstPass->vertexUniformBuffer1.gpuAddress, FirstPass->vertexUniformBuffer1.size);

    // -----------------------
    // draw first triangle
    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList,
        nn::gfx::IndexFormat_Uint32, FirstPass->indexBuffer.gpuAddress, FirstPass->numIndices, 0 );

    // -----------------------
    // draw second triangle
    modelmat[0][0] = cos(-rotateY / 2.0f);  modelmat[2][0] = -sin(-rotateY / 2.0f);
    modelmat[0][2] = sin(-rotateY / 2.0f);  modelmat[2][2] =  cos(-rotateY / 2.0f);
    modelmat[0][3] += 1.0f;

    // Uniform Location Lookup
    vs = FirstPass->vertexUniformBuffer2.Map< struct vertexUniforms >();
    memcpy(&vs->viewMtx, &viewmat[0], sizeof(Mat44));
    memcpy(&vs->projMtx, &projmat[0], sizeof(Mat44));
    memcpy(&vs->modelMtx, &modelmat[0], sizeof(Mat44));
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap(vs, sizeof(*vs));
#endif
    FirstPass->vertexUniformBuffer2.Unmap();
    DEMOCommandBuffer.SetConstantBuffer(FirstPass->vertexUniformBlockLoc, nn::gfx::ShaderStage_Vertex, FirstPass->vertexUniformBuffer2.gpuAddress, FirstPass->vertexUniformBuffer2.size);

    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList,
        nn::gfx::IndexFormat_Uint32, FirstPass->indexBuffer.gpuAddress, FirstPass->numIndices, 0 );

    if(technique == EXPAND_COLOR)
    {
#if NN_GFX_IS_TARGET_GX
        GX2UTSetExpandAAColorState(GX2_ENABLE);
        GX2UTExpandAAColorBufferOp( FirstPass->pMyColorBuffer->colorView.ToData()->pGx2ColorBuffer );
        GX2UTSetExpandAAColorState(GX2_DISABLE);
#endif
    }
    DEMOCommandBuffer.FlushMemory(nn::gfx::GpuAccess_ColorBuffer);
    DEMOCommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_Texture);

    if(technique == RESOLVE_COLOR)
    {
        DEMOCommandBuffer.Resolve(&SecondPass->pMyTextureBuffer->texture, 0, 0, &FirstPass->pMyColorBuffer->colorView, NULL);
        DEMOCommandBuffer.FlushMemory(nn::gfx::GpuAccess_ColorBuffer);
        DEMOCommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_Texture);
#if NN_GFX_IS_TARGET_GX
        GX2UTSetResolveAAColorState(GX2_ENABLE);
        GX2UTResolveAAColorBufferOp(FirstPass->pMyColorBuffer->colorView.ToData()->pGx2ColorBuffer,
            SecondPass->pMyTextureBuffer->texture.ToData()->pGx2Surface, 0, 0);
        GX2UTSetResolveAAColorState(GX2_DISABLE);
#endif
    }

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif

    // ------------------------------- //
    //   2ND PASS (VISALIZE TEXTURE)
    // ------------------------------- //

    f32 mOffset[4] = { xOffset, yOffset, 0.0f, 0.0f };

    // Set to use regular demo color/depth buffers
    DEMOGfxSetDefaultRenderTarget();

    DEMOCommandBuffer.SetViewportScissorState(&wholeViewport);
    DEMOCommandBuffer.SetPipeline(&SecondPass->pipeline.pipeline);
    DEMOCommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_ColorBuffer | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_Texture);
    float* uf = SecondPass->offsetBuffer.Map<float>();
    memcpy(uf, mOffset, sizeof(mOffset));
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap(uf, sizeof(mOffset));
#endif
    SecondPass->offsetBuffer.Unmap();
    DEMOCommandBuffer.SetConstantBuffer(SecondPass->offsetLoc, nn::gfx::ShaderStage_Vertex, SecondPass->offsetBuffer.gpuAddress, SecondPass->offsetBuffer.size);

    // Bind position buffer
    DEMOCommandBuffer.SetVertexBuffer(SecondPass->posLoc, SecondPass->positionBuffer.gpuAddress, sizeof(float) * 3, SecondPass->positionBuffer.size);

    // Bind texcoord buffer
    DEMOCommandBuffer.SetVertexBuffer(SecondPass->texcoordLoc, SecondPass->texcoordBuffer.gpuAddress,
        sizeof(float) * 2, SecondPass->texcoordBuffer.size);

    DEMOCommandBuffer.SetTextureAndSampler(SecondPass->textureLoc, nn::gfx::ShaderStage_Pixel, SecondPass->pMyTextureBuffer->descriptorSlot, SecondPass->mySamplerDescriptorSlot);

    // -----------------------
    // draw first quad
    DEMOCommandBuffer.DrawIndexed(nn::gfx::PrimitiveTopology_TriangleStrip, nn::gfx::IndexFormat_Uint32,
        SecondPass->indexBuffer.gpuAddress, SecondPass->numIndices, 0);
}

// deallocated resources used by techniques
static void SceneFree(Technique technique, _FirstPass *FirstPass, _SecondPass *SecondPass)
{
    NN_UNUSED( technique );

    FirstPass->pipeline.Finalize( &DEMODevice );
    SecondPass->pipeline.Finalize( &DEMODevice );

    FirstPass->depthBuffer.depthView.Finalize(&DEMODevice);
    FirstPass->depthBuffer.texture.Finalize(&DEMODevice);
    FirstPass->depthBuffer.pPool->Finalize();

    FirstPass->colorBuffer.Finalize(&DEMODevice);
    FirstPass->indexBuffer.Finalize(&DEMODevice);
    FirstPass->positionBuffer.Finalize(&DEMODevice);
    FirstPass->vertexUniformBuffer1.Finalize(&DEMODevice);
    FirstPass->vertexUniformBuffer2.Finalize(&DEMODevice);
    FirstPass->viewportScissorState.Finalize(&DEMODevice);
    if ( FirstPass->pViewportScissorData )
    {
        DEMOGfxFreeMEM2( FirstPass->pViewportScissorData );
    }

    SecondPass->indexBuffer.Finalize(&DEMODevice);
    SecondPass->mySampler.Finalize(&DEMODevice);
    SecondPass->offsetBuffer.Finalize(&DEMODevice);
    SecondPass->positionBuffer.Finalize(&DEMODevice);
    SecondPass->texcoordBuffer.Finalize(&DEMODevice);
}

static void AnimTick()
{
    rotateY += 0.06f;
}

static void PrintInfo()
{
    DEMOFontPrintf(11, 11, "Resolve Color");
    DEMOFontPrintf(39, 11, "Render to Texture");
    DEMOFontPrintf(26, 12, "Expand Color");
}

static void FreeImageData(ImageData* pData)
{
    pData->colorView.Finalize(&DEMODevice);
    pData->textureView.Finalize(&DEMODevice);
    pData->texture.Finalize(&DEMODevice);
    pData->pPool->Finalize();
    delete pData->pPool;
    pData->pPool = NULL;
    pData->freed = true;
}

//extern "C" void nnMain()
TEST(GfxExpandColor, Run)
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    DEMOInit();
    DEMOTestInit(argc, argv);
    DEMOGfxInit(argc, argv);
    DEMOFontInit();
    DEMOTestIsUseHlslccGlsl() ? g_ShaderFileIdx = 1 : g_ShaderFileIdx = 0;

    DEMOGfxSetViewportScissorState(&wholeViewport, &wholeViewportData, 0.0, 0.0, (float)SURFACE_WIDTH, (float)SURFACE_HEIGHT, 0.0f, 1.0f, (float)SURFACE_HEIGHT, false);

    SceneInit(RESOLVE_COLOR, &FirstPassResolveColor, &SecondPassResolveColor, &ResolveColorData);
    SceneInit(RENDER2TEXTURE, &FirstPassRender2Texture, &SecondPassRender2Texture, &Render2TextureData);
    SceneInit(EXPAND_COLOR, &FirstPassExpandColor, &SecondPassExpandColor, &ExpandColorData);

    while (DEMOIsRunning())
    {
        DEMOGfxBeforeRender();

        // clear buffers
        DEMOCommandBuffer.ClearColor( DEMOGetColorBufferView(), 0.5f, 0.7f, 0.8f, 0.0f, NULL );
        DEMOCommandBuffer.ClearDepthStencil( &DEMODepthBufferView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

#if NN_GFX_IS_TARGET_GX
        GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif

        SceneDraw(RESOLVE_COLOR, &FirstPassResolveColor, &SecondPassResolveColor, -0.5f, 0.5f);
        SceneDraw(RENDER2TEXTURE, &FirstPassRender2Texture, &SecondPassRender2Texture, 0.5f, 0.5f);
        SceneDraw(EXPAND_COLOR, &FirstPassExpandColor, &SecondPassExpandColor, 0.0f, -0.5f);
        PrintInfo();

        DEMOGfxDoneRender();

        AnimTick();
    }

    SceneFree(RESOLVE_COLOR, &FirstPassResolveColor, &SecondPassResolveColor);
    SceneFree(RENDER2TEXTURE, &FirstPassRender2Texture, &SecondPassRender2Texture);
    SceneFree(EXPAND_COLOR, &FirstPassExpandColor, &SecondPassExpandColor);

    FreeImageData(&ResolveColorData);
    FreeImageData(&Render2TextureData);
    FreeImageData(&ExpandColorData);
    FreeImageData(&ResolveSecondData);

    wholeViewport.Finalize(&DEMODevice);
    if ( wholeViewportData )
    {
        DEMOGfxFreeMEM2( wholeViewportData );
    }

    DEMOFontShutdown();
    DEMOGfxShutdown();
    DEMOTestShutdown();
    DEMOShutdown();

    SUCCEED();
}
