﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>

#include <gfx/demo.h>
#include <nnt.h>
#include <nnt/nnt_Argument.h>

#ifdef CAFE
#include <cafe/gx2.h>
#endif

// ----- Surface Information

#define SURFACE_WIDTH  (DEMOColorBufferInfo.GetWidth())
#define SURFACE_HEIGHT  (DEMOColorBufferInfo.GetHeight())

static int g_ShaderFileIdx = 0;
static const char* SHADER_FILE_NAME[] =
{
    "shaders/textureCubeLOD",
    "shaders/textureCubeLODHlslcc",
};

static const char* TEXTURE_FILE = "textures/cubemapmip/cubemapmip";

static DEMOGfxShader theShader;

// Uniforms
struct Uniforms
{
    Mtx44   modelMtx44;
    Mtx44   viewMtx44;
    Mtx44   projMtx44;
    Qtrn    scale;
};

static int posLoc;
static int mtxLoc;
static int samplerLoc;

//---------------------------------------------------------------------------*
//  Model Data
//---------------------------------------------------------------------------*/

static const char* MODEL_FILES[2]
    = {"geometries/spherePosition.dat",
       "geometries/sphereIdx.dat"};

// Pointers to GPU buffers
static f32 * posBuf;
static u32 * idxBuf;

static u32    modelLen[2];

static u32 MODEL_VTX_COUNT = 2880;

static const int MODEL_VTX_STRIDE = sizeof(f32) * 3;

static const int NUM_ATTRIB =  1;

// ----- GX2 Triangle Data

static const int POSITION_STRIDE = (sizeof(f32) * 3);
static const int POSITION_OFFSET = 0;

static const int POSITION_BUFFER_IDX = 0;

static const int NUM_LEVELS = 9;

// ---- nn::gfx objects
static nn::gfx::Pipeline myPipeline;
static nn::gfx::Sampler mySampler;
static nn::gfx::DescriptorSlot mySamplerSlot;
static DEMOGfxTexture myTexture;
static nn::gfx::ViewportScissorState viewportScissorState;

static DEMOGfxBuffer uniforms[NUM_LEVELS];
static DEMOGfxBuffer vertexBuffer;
static DEMOGfxBuffer indexBuffer;

static Mtx44   viewMtx44;
static Mtx44   projMtx44;

static f32 s_yrad;
static f32 s_xrad;
static bool s_PrintForTest = false;

// Prototype
static void TexInit(void);
static void SceneInit(void);
static void SceneDraw(void);
static void ModelTick(DEMOGfxBuffer& buf, const Vec& pos, const Qtrn& pScale);
static void PrintInfo(nn::gfx::TextureInfo& info);
static void CameraInit(Mtx44* resultProjMtx44, Mtx44* resultViewMtx44);
static void SetupUniformBuffer(DEMOGfxBuffer&, const Mtx44&, const Mtx44&, const Qtrn& );

// Init function for setting projection matrix
static void CameraInit(Mtx44* resultProjMtx44, Mtx44* resultViewMtx44)
{
    // row major matricies
    Mtx   lookAtMtx34;

    Vec     up = { 0.0f,  1.0f, 0.0f };
    Vec  objPt = { 0.0f, 0.0f, 0.0f };
    Vec camLoc = { 0.0f, 0.0f, 520.0f };

    f32   pers = 50.0f; // degrees
    f32 aspect = (f32)SURFACE_WIDTH / (f32)SURFACE_HEIGHT;
    f32  znear = 50.0f;
    f32   zfar = 2000.0f;

    // Compute perspective matrix
    MTXPerspective( *resultProjMtx44, pers, aspect, znear, zfar );

    // Compute lookAt matrix
    MTXLookAt( lookAtMtx34, &camLoc, &up, &objPt );
    MTX34To44( lookAtMtx34, *resultViewMtx44 );
}

static void SetupUniformBuffer(DEMOGfxBuffer& buffer, const Mtx44& resultProjMtx44, const Mtx44& resultViewMtx44, const Qtrn& pScale)
{

    buffer.Initialize(sizeof(Uniforms), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0);

    struct Uniforms* pUniforms = buffer.Map< struct Uniforms >();
    MTX44Identity( pUniforms->modelMtx44 );
    memcpy(&pUniforms->viewMtx44, &resultViewMtx44, sizeof(resultViewMtx44));
    memcpy(&pUniforms->projMtx44, &resultProjMtx44, sizeof(resultProjMtx44));
    memcpy(&pUniforms->scale, &pScale, sizeof(pScale));
#ifdef CAFE
    GX2EndianSwap( pUniforms, sizeof(*pUniforms) );
#endif
    buffer.Unmap();
}

// Update Model function for setting model matrix
static void ModelTick(DEMOGfxBuffer& buf, const Vec& pos, const Qtrn& pScale)
{
    // row major matrix
    Mtx rotXMtx34;
    Mtx rotYMtx34;
    Mtx rotMtx34;
    struct Uniforms* pMtx = buf.Map< struct Uniforms >( );
#ifdef CAFE
    GX2EndianSwap(pMtx, sizeof(*pMtx));
#endif

    // Copy
    pMtx->scale = pScale;

    // Compute rotation matrix
    MTXRotRad( rotYMtx34, 'y', s_yrad * 2 );
    MTXRotRad( rotXMtx34, 'x', s_xrad * 2 );

    // Compute model matrix
    MTXConcat( rotXMtx34, rotYMtx34, rotMtx34 );
    MTX34To44( rotMtx34, pMtx->modelMtx44 );

    // Translate
    pMtx->modelMtx44[ 0 ][ 3 ] = pos.x;
    pMtx->modelMtx44[ 1 ][ 3 ] = pos.y;
    pMtx->modelMtx44[ 2 ][ 3 ] = pos.z;

#ifdef CAFE
    GX2EndianSwap(pMtx, sizeof(*pMtx));
#endif
    buf.Unmap();
}


static void InitBuffers()
{
#ifdef CAFE
    // PPC_IO_BUFFER_ALIGN is set by DEMOGfxLoadAssetFile.
    // This number must fit to alignment of vertex and index buffer.
    ASSERT(PPC_IO_BUFFER_ALIGN % GX2_VERTEX_BUFFER_ALIGNMENT == 0);
    ASSERT(PPC_IO_BUFFER_ALIGN % GX2_INDEX_BUFFER_ALIGNMENT == 0);
#endif
    // Vertex buffer
    posBuf = (f32*)DEMOGfxLoadModelFile(MODEL_FILES[0], &modelLen[0]);

    vertexBuffer.Initialize(modelLen[0], posBuf, nn::gfx::GpuAccess_VertexBuffer, 0);

    // Index buffer
    idxBuf = (u32*)DEMOGfxLoadModelFile(MODEL_FILES[1], &modelLen[1]);

    indexBuffer.Initialize(modelLen[1], idxBuf, nn::gfx::GpuAccess_IndexBuffer, 0);
}

static void TexInit()
{
    bool ok = myTexture.Initialize( TEXTURE_FILE );
    ASSERT( ok );
    NN_UNUSED( ok );

    DEMOGfxInitSampler( &mySampler, &mySamplerSlot, nn::gfx::TextureAddressMode_ClampToEdge,
        nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint, nn::gfx::ComparisonFunction_Always );
}

static void SetupViewportScissor()
{
    nn::gfx::ViewportScissorState::InfoType info;
    nn::gfx::ViewportStateInfo viewportStateInfo;
    nn::gfx::ScissorStateInfo scissorStateInfo;

    viewportStateInfo.SetDefault();
    viewportStateInfo.SetWidth( ( float )DEMOColorBufferInfo.GetWidth() );
    viewportStateInfo.SetHeight( ( float )DEMOColorBufferInfo.GetHeight() );
    viewportStateInfo.SetOriginX( 0.0f );
    viewportStateInfo.SetOriginY( 0.0f );
    viewportStateInfo.SetMinDepth( 0.0f );
    viewportStateInfo.SetMaxDepth( 1.0f );

    scissorStateInfo.SetDefault();
    scissorStateInfo.SetOriginX( 0 );
    scissorStateInfo.SetOriginY( 0 );
    scissorStateInfo.SetWidth( DEMOColorBufferInfo.GetWidth() );
    scissorStateInfo.SetHeight( DEMOColorBufferInfo.GetHeight() );

    info.SetDefault();
    info.SetScissorEnabled( true );
    info.SetScissorStateInfoArray( &scissorStateInfo, 1 );
    info.SetViewportStateInfoArray( &viewportStateInfo, 1 );

    size_t size = nn::gfx::ViewportScissorState::GetRequiredMemorySize( info );
    if ( size != 0 )
    {
        void* pViewportScissorData = DEMOGfxAllocMEM2( size, nn::gfx::ViewportScissorState::RequiredMemoryInfo_Alignment );
        viewportScissorState.SetMemory( pViewportScissorData, size );
    }
    viewportScissorState.Initialize( &DEMODevice, info );
}

static void InitShaders()
{
    DEMOGfxLoadShadersFromFile(&theShader, 0, SHADER_FILE_NAME[g_ShaderFileIdx]);
}

// The init function for the rendering portions of this app
static void SceneInit()
{
    nn::gfx::Pipeline::InfoType pipelineInfo;
    nn::gfx::VertexStateInfo vertexStateInfo;
    nn::gfx::BlendStateInfo blendStateInfo;
    nn::gfx::RasterizerStateInfo rasterizerStateInfo;
    nn::gfx::RenderTargetStateInfo renderTargetStateInfo;
    nn::gfx::DepthStencilStateInfo depthStencilStateInfo;

    // Start setting up the pipeline
    pipelineInfo.SetDefault();

    s_yrad = 0.0f;
    s_xrad = 0.0f;

    InitShaders();

    pipelineInfo.SetShaderPtr(theShader.GetShader());

    // Uniform Location lookup
    mtxLoc = theShader.GetInterfaceSlot(nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_uniforms");

    // Texture Sampler lookup
    samplerLoc = theShader.GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_texture");
    // Attribute Location lookup
    posLoc = theShader.GetInterfaceSlot(nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_position");

    // Setup vertex buffers and index buffers
    vertexStateInfo.SetDefault();
    nn::gfx::VertexAttributeStateInfo attribs[ NUM_ATTRIB ];
    nn::gfx::VertexBufferStateInfo buffer[ 1 ];
    attribs[ 0 ].SetDefault();
    attribs[ 0 ].SetBufferIndex( POSITION_BUFFER_IDX );
    attribs[ 0 ].SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
    attribs[ 0 ].SetNamePtr( "a_position" );
    attribs[ 0 ].SetOffset( POSITION_OFFSET * sizeof(float) );
    attribs[ 0 ].SetShaderSlot( posLoc );
    buffer[ 0 ].SetDefault();
    buffer[ 0 ].SetStride( POSITION_STRIDE );
    vertexStateInfo.SetVertexBufferStateInfoArray( buffer, 1 );
    vertexStateInfo.SetVertexAttributeStateInfoArray( attribs, NUM_ATTRIB );
    pipelineInfo.SetVertexStateInfo(&vertexStateInfo);

    // Set Color Mask
    nn::gfx::BlendTargetStateInfo blendTargets[ 1 ];
    blendStateInfo.SetDefault();
    blendTargets[ 0 ].SetDefault();
    blendTargets[ 0 ].SetBlendEnabled( false );
    blendTargets[ 0 ].SetChannelMask( nn::gfx::ChannelMask_All );
    blendStateInfo.SetBlendTargetStateInfoArray( blendTargets, 1 );
    pipelineInfo.SetBlendStateInfo(&blendStateInfo);

    // Color targets
    nn::gfx::ColorTargetStateInfo colorTargets[ 1 ];
    colorTargets[ 0 ].SetDefault();
    colorTargets[ 0 ].SetFormat( DEMOColorBufferInfo.GetImageFormat() );
    renderTargetStateInfo.SetDefault();
    renderTargetStateInfo.SetColorTargetStateInfoArray( colorTargets, 1 );
    pipelineInfo.SetRenderTargetStateInfo(&renderTargetStateInfo);

    depthStencilStateInfo.SetDefault();
    depthStencilStateInfo.SetDepthWriteEnabled( true );
    depthStencilStateInfo.SetDepthTestEnabled( true );
    depthStencilStateInfo.SetDepthComparisonFunction( nn::gfx::ComparisonFunction_Less );
    depthStencilStateInfo.SetStencilTestEnabled( false );
    pipelineInfo.SetDepthStencilStateInfo(&depthStencilStateInfo);

    rasterizerStateInfo.SetDefault();
    rasterizerStateInfo.SetFillMode( nn::gfx::FillMode_Solid );
    rasterizerStateInfo.SetCullMode( nn::gfx::CullMode_None );
    rasterizerStateInfo.SetFrontFace( nn::gfx::FrontFace_Ccw );
    rasterizerStateInfo.SetScissorEnabled( true );
    pipelineInfo.SetRasterizerStateInfo(&rasterizerStateInfo);

    size_t size = nn::gfx::Pipeline::GetRequiredMemorySize( pipelineInfo );
    void *data = DEMOGfxAllocMEM2( size, nn::gfx::Pipeline::RequiredMemoryInfo_Alignment );
    myPipeline.SetMemory( data, size );
    myPipeline.Initialize( &DEMODevice, pipelineInfo );

    InitBuffers();

    // Texture setup
    TexInit();

    CameraInit(&projMtx44, &viewMtx44);
    Qtrn initScale = { 0.0f, 0.0f, 0.0f, 0.0f };
    for(int idx = 0; idx < NUM_LEVELS; idx++)
    {
        SetupUniformBuffer(uniforms[idx], projMtx44, viewMtx44, initScale);
    }

    // Setup the viewport/scissor
    SetupViewportScissor();
}

// Draw Cubemap with mip
static void DrawCubemapMip(f32 x, f32 y, f32 z, u32 miplevel)
{
    Vec pos = { x, y, z };
    Qtrn scale = { static_cast< f32 >( miplevel ), 0.0f, 0.0f, 0.0f };
    ModelTick(uniforms[miplevel], pos, scale);

    DEMOCommandBuffer.SetPipeline(&myPipeline);
    DEMOCommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_ConstantBuffer);

    // Set Texture
    DEMOCommandBuffer.SetTextureAndSampler(samplerLoc, nn::gfx::ShaderStage_Pixel, myTexture.GetDescriptorSlot( 0 ), mySamplerSlot);

    // Bind vertex buffer
    DEMOCommandBuffer.SetVertexBuffer( posLoc, vertexBuffer.gpuAddress, MODEL_VTX_STRIDE, vertexBuffer.size );
    DEMOCommandBuffer.SetConstantBuffer( mtxLoc, nn::gfx::ShaderStage_Vertex, uniforms[miplevel].gpuAddress, uniforms[miplevel].size );

    DEMOCommandBuffer.DrawIndexed(nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32, indexBuffer.gpuAddress, MODEL_VTX_COUNT, 0);
}

// Print Informations
static void PrintInfo(nn::gfx::TextureInfo& info)
{
    static OSTime lastTime = 0;
    OSTime now = OSGetTime();
    f32 dtime = OSTicksToMilliseconds(now - lastTime) / 1000.0f;
    lastTime = now;

    DEMOFontSetSpacing(0);
    DEMOFontSetGridSize(100,33);
    if ( s_PrintForTest )
    {
        DEMOFontPrintf(5, 2, "<Cubemap mipmap>");
    }
    else
    {
        DEMOFontPrintf(5, 2, "<Cubemap mipmap: %.1f fps>", 1.0f / dtime);
    }
    DEMOFontPrintf(6, 3, "TB: [ width: %d height: %d format: %x first mip: %d num mip: %d]",
             info.GetWidth(),
             info.GetHeight(),
             info.GetImageFormat(),
             0,
             info.GetMipCount());

    DEMOCommandBuffer.SetPipeline(&myPipeline);
}

// The draw function for the rendering portions of this app
static void SceneDraw()
{
    u32 level=0;

    DEMOGfxBeforeRender();

    nn::gfx::ColorTargetView* pCurrentScanBuffer = DEMOGetColorBufferView();
    DEMOCommandBuffer.ClearColor(pCurrentScanBuffer, 0.15f, 0.17f, 0.3f, 1.0f, NULL);
    DEMOCommandBuffer.ClearDepthStencil(&DEMODepthBufferView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL);

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif

    // Setup the color/depth buffers
    const nn::gfx::ColorTargetView* renderTargets[1];
    renderTargets[ 0 ] = pCurrentScanBuffer;
    DEMOCommandBuffer.SetRenderTargets( 1, renderTargets, &DEMODepthBufferView );

    // Setup viewport scissor
    DEMOCommandBuffer.SetViewportScissorState(&viewportScissorState);

    // set font spacing
    DEMOFontSetSpacing(0);
    DEMOFontSetGridSize(100,33);

    // Testing Mip Level 0-8
    DrawCubemapMip(-150.0f, 90.0f, -3.0f, level);
    DEMOFontPrintf(20,5, "LEVEL: %d", level);
    level++;

    DrawCubemapMip(0.0f, 90.0f, -3.0f, level);
    DEMOFontPrintf(38,5, "LEVEL: %d", level);
    level++;

    DrawCubemapMip(150.0f, 90.0f, -3.0f, level);
    DEMOFontPrintf(56,5, "LEVEL: %d", level);
    level++;

    DrawCubemapMip(-150.0f, -30.0f, -3.0f, level);
    DEMOFontPrintf(20,14, "LEVEL: %d", level);
    level++;

    DrawCubemapMip(0.0f, -30.0f, -3.0f, level);
    DEMOFontPrintf(38,14, "LEVEL: %d", level);
    level++;

    DrawCubemapMip(150.0f, -30.0f, -3.0f, level);
    DEMOFontPrintf(56,14, "LEVEL: %d", level);
    level++;

    DrawCubemapMip(-150.0f, -150.0f, -3.0f, level);
    DEMOFontPrintf(20,22, "LEVEL: %d", level);
    level++;

    DrawCubemapMip(0.0f, -150.0f, -3.0f, level);
    DEMOFontPrintf(38,22, "LEVEL: %d", level);
    level++;

    DrawCubemapMip(150.0f, -150.0f, -3.0f, level);
    DEMOFontPrintf(56,22, "LEVEL: %d", level);

    // Print texture info
    PrintInfo( *myTexture.GetTextureInfo( 0 ) );

    DEMOGfxDoneRender();
}

//extern "C" void nnMain()
TEST(GfxCubemapmip, Run)
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    // Init DEMO lib
    DEMOInit();
    DEMOTestInit(argc, argv);
    DEMOGfxInit(argc, argv);
    DEMOFontInit();
    DEMOTestIsUseHlslccGlsl() ? g_ShaderFileIdx = 1 : g_ShaderFileIdx = 0;
    //DEMOFontDrawEnable(false); //FIXME

    // Disable printing FPS during test
    for ( int i = 0; i < argc; i++ )
    {
        if ( !strstr(argv[i], "TEST_SELECT" ) )
        {
            s_PrintForTest = true;
        }
    }

    SceneInit();
    while (DEMOIsRunning())
    {
        SceneDraw();
        s_yrad += 0.0005f;
        s_xrad += 0.0004f;
    }

    // Free textures
    myTexture.Finalize();
    mySampler.Finalize( &DEMODevice );

    void* freeData = NULL;

    // Free shaders
    DEMOGfxFreeShaders(&theShader);

    // Free the viewport
    freeData = viewportScissorState.GetMemory();
    viewportScissorState.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2( freeData );

    freeData = myPipeline.GetMemory();
    myPipeline.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2( freeData );

    vertexBuffer.Finalize( );
    DEMOGfxFreeMEM2( posBuf );

    indexBuffer.Finalize( );
    DEMOGfxFreeMEM2( idxBuf );

    for ( int idx = 0; idx < NUM_LEVELS; idx++ )
    {
        uniforms[ idx ].Finalize( );
    }

    DEMOFontShutdown();
    DEMOTestShutdown();
    DEMOGfxShutdown();
    DEMOShutdown();

    SUCCEED();
}
