﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>
#include <cmath>

#include <gfx/demo.h>

#include <nnt.h>
#include <nnt/nnt_Argument.h>

#if NN_GFX_IS_TARGET_GX || NN_GFX_IS_TARGET_D3D
inline float FLIP_Y(float y)
{
    return 1.0f - y;
}
#else
inline float FLIP_Y(float y)
{
    return y;
}
#endif

// ----- GX2 Texture

static int g_ShaderFileIdx = 0;
static const char* const SHADER_FILE_NAME[] =
{
    "shaders/copySurfaceRect",
    "shaders/copySurfaceRectHlslcc",
};
static const char* const TEXTURE_FILE = "textures/copySurfaceRect/seattle_A32B32G32R32_1280x720";

static DEMOGfxTexture g_ExampleTexture;

// Temporary Color Buffer
static nn::gfx::Texture g_ExampleColorBuffer;
static nn::gfx::TextureView g_ExampleColorBufferTextureView;
static nn::gfx::ColorTargetView g_ExampleColorBufferView;
static nn::gfx::DescriptorSlot g_ExampleColorBufferDescriptorSlot;
static DEMOGfxMemPool* g_ExampleColorBufferData;

// Stretch Blit
static nn::gfx::ViewportScissorState g_viewportScissorState;
static void* g_pViewportScissorData;
static DEMOGfxPipeline g_stretchBlitPipeline;

static const int VERTEX_COUNT = 4;
static const int POSITION_BUFFER_SIZE = sizeof( float ) * 3 * VERTEX_COUNT;
static const int TEXCOORD_BUFFER_SIZE = sizeof( float ) * 2 * VERTEX_COUNT;
static DEMOGfxBuffer g_stretchPositionBuffer;
static DEMOGfxBuffer g_stretchTexCoordBuffer;
static DEMOGfxBuffer g_noflipTexCoordBuffer;
static DEMOGfxBuffer g_copyTexCoordBuffer;
static DEMOGfxBuffer g_copyPositionBuffer;
static nn::gfx::Sampler g_stretchSampler;
static nn::gfx::DescriptorSlot g_stretchSamplerDescriptorSlot;
static int g_texId = -1;
static int g_PositionLocation = -1;
static int g_TexCoordLocation = -1;

// Prototype
static void SceneInit(void);
static void SceneDraw(void);

static void SetupStretchBlit()
{
    DEMOGfxSetViewportScissorState( &g_viewportScissorState, &g_pViewportScissorData,
        0.0f, 0.0f,
        static_cast< float >( DEMOColorBufferInfo.GetWidth() ),
        static_cast< float >( DEMOColorBufferInfo.GetHeight() ),
        0.0f, 1.0f,
        static_cast< float >( DEMOColorBufferInfo.GetHeight() ),
        false );

    DEMOGfxLoadShadersFromFile(&g_stretchBlitPipeline.shaders, 0, SHADER_FILE_NAME[g_ShaderFileIdx]);
    DEMOGfxInitShaderAttribute( &g_stretchBlitPipeline.shaders, "i_positions", 0, 0, nn::gfx::AttributeFormat_32_32_32_Float );
    DEMOGfxInitShaderAttribute( &g_stretchBlitPipeline.shaders, "i_texCoords", 1, 0, nn::gfx::AttributeFormat_32_32_Float );

    DEMOGfxInitShaderVertexBuffer( &g_stretchBlitPipeline.shaders, 0, POSITION_BUFFER_SIZE / VERTEX_COUNT, 0 );
    DEMOGfxInitShaderVertexBuffer( &g_stretchBlitPipeline.shaders, 1, TEXCOORD_BUFFER_SIZE / VERTEX_COUNT, 0 );

    g_stretchBlitPipeline.blendTargetStateInfoArray[ 0 ].SetDefault();
    g_stretchBlitPipeline.colorTargetStateInfoArray[ 0 ].SetDefault();
    g_stretchBlitPipeline.colorTargetStateInfoArray[ 0 ].SetFormat( DEMOColorBufferInfo.GetImageFormat() );
    g_stretchBlitPipeline.blendTargetStateCount = 1;
    g_stretchBlitPipeline.colorTargetStateCount = 1;

    // Override a few defaults
    g_stretchBlitPipeline.depthStencilStateInfo.SetDepthTestEnabled( false );
    g_stretchBlitPipeline.depthStencilStateInfo.SetDepthWriteEnabled( false );

    g_stretchBlitPipeline.Initialize( &DEMODevice );

    // Grab the vertex/texture coordinate location
    g_PositionLocation = g_stretchBlitPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "i_positions" );
    g_TexCoordLocation = g_stretchBlitPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "i_texCoords" );

    // Grab the sampler ID
    g_texId = g_stretchBlitPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "u_Sampler" );

    u32 width = DEMOColorBufferInfo.GetWidth();
    u32 height = DEMOColorBufferInfo.GetHeight();

    // Setup the vertex/texture coordinate buffers
    float positions[ 12 ];
    positions[ 0 ] = -1.0f; // v0
    positions[ 1 ] = -1.0f;
    positions[ 2 ] = 0.0f;

    positions[ 3 ] = -1.0f; // v1
    positions[ 4 ] = 1.0f;
    positions[ 5 ] = 0.0f;

    positions[ 6 ] = 1.0f; // v2
    positions[ 7 ] = -1.0f;
    positions[ 8 ] = 0.0f;
    positions[ 9 ] = 1.0f; // v3
    positions[ 10 ] = 1.0f;
    positions[ 11 ] = 0.0f;

    float texCoords[ 8 ];
    texCoords[ 0 ] = 1.0f / static_cast< float >( width ); // v0
    texCoords[ 1 ] = FLIP_Y( 1.0f / static_cast< float >( height ) );
    texCoords[ 2 ] = 1.0f / static_cast< float >( width ); // v1
    texCoords[ 3 ] = FLIP_Y( 1.0f - 1.0f / static_cast< float >( height ) );
    texCoords[ 4 ] = 1.0f - 1.0f / static_cast< float >( width ); // v2
    texCoords[ 5 ] = FLIP_Y( 1.0f / static_cast< float >( height ) );
    texCoords[ 6 ] = 1.0f - 1.0f / static_cast< float >( width ); // v3
    texCoords[ 7 ] = FLIP_Y( 1.0f - 1.0f / static_cast< float >( height ) );

    g_stretchPositionBuffer.Initialize( POSITION_BUFFER_SIZE, positions, nn::gfx::GpuAccess_VertexBuffer, 0 );
    g_stretchTexCoordBuffer.Initialize( TEXCOORD_BUFFER_SIZE, texCoords, nn::gfx::GpuAccess_VertexBuffer, 0 );

    DEMOGfxInitSampler( &g_stretchSampler, &g_stretchSamplerDescriptorSlot,
        nn::gfx::TextureAddressMode_ClampToEdge, nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint, nn::gfx::ComparisonFunction_Always );

    // Create dynamic buffers for copying
    g_copyTexCoordBuffer.Initialize( TEXCOORD_BUFFER_SIZE, NULL, nn::gfx::GpuAccess_VertexBuffer, 0 );
    g_copyPositionBuffer.Initialize( POSITION_BUFFER_SIZE, NULL, nn::gfx::GpuAccess_VertexBuffer, 0 );
}

// The init function for the rendering portions of this app
static void SceneInit()
{
    bool ok = g_ExampleTexture.Initialize( TEXTURE_FILE );
    ASSERT( ok );
    NN_UNUSED( ok );

    // Initialize (temp) color buffer
    DEMOGfxSetupTextureBuffer( &g_ExampleColorBuffer, &g_ExampleColorBufferTextureView, &g_ExampleColorBufferDescriptorSlot, &g_ExampleColorBufferView,
        NULL, &g_ExampleColorBufferData, DEMOColorBufferInfo.GetWidth(), DEMOColorBufferInfo.GetHeight(), 1, 1,
        nn::gfx::ImageDimension_2d, DEMOColorBufferInfo.GetImageFormat(),
        nn::gfx::DepthStencilFetchMode_DepthComponent, 0 );

    DEMOCommandBuffer.Begin();
    {
        const nn::gfx::ColorTargetView* colorTargets[ 1 ] = { &g_ExampleColorBufferView };
        DEMOCommandBuffer.SetRenderTargets( 1, colorTargets, NULL );
        DEMOCommandBuffer.ClearColor( &g_ExampleColorBufferView, 0.0f, 0.0f, 0.0f, 0.0f, NULL );
    }
    DEMOCommandBuffer.End();
    DEMOQueue.ExecuteCommand( &DEMOCommandBuffer, NULL );
    DEMOQueue.Sync();

    SetupStretchBlit();
}

// The draw function for the rendering portions of this app
static void SceneDraw()
{
    static nn::gfx::Texture* s_pLastColorBuffer = NULL;
    // When using the DEMO library, it is necessary to call
    // DEMOGfxBeforeRender and DEMOGfxDoneRender before and after drawing.
    DEMOGfxBeforeRender();

    // Copy entire color buffer
    u32 width = DEMOColorBufferInfo.GetWidth();
    u32 height = DEMOColorBufferInfo.GetHeight();

    // Copy the color buffer's surface to the temp color buffer's surface
    if ( s_pLastColorBuffer != NULL )
    {
        DEMOCommandBuffer.SetTextureStateTransition( s_pLastColorBuffer, NULL,
            nn::gfx::TextureState_ColorTarget, nn::gfx::ShaderStageBit_Pixel,
            nn::gfx::TextureState_CopySource, nn::gfx::ShaderStageBit_Pixel );

        DEMOCommandBuffer.SetTextureStateTransition( g_ExampleTexture.GetTexture( 0 ), NULL,
            nn::gfx::TextureState_ShaderRead, nn::gfx::ShaderStageBit_Pixel,
            nn::gfx::TextureState_CopyDestination, nn::gfx::ShaderStageBit_Pixel );

        nn::gfx::TextureSubresource copyDestination;
        nn::gfx::TextureCopyRegion copySourceRegion;
        copyDestination.SetDefault();
        copySourceRegion.SetDefault();
        copySourceRegion.SetWidth( width );
        copySourceRegion.SetHeight( height );

        DEMOCommandBuffer.CopyImage( &g_ExampleColorBuffer, copyDestination, 0, 0, 0,
            s_pLastColorBuffer, copySourceRegion );

        DEMOCommandBuffer.SetTextureStateTransition( g_ExampleTexture.GetTexture( 0 ), NULL,
            nn::gfx::TextureState_CopyDestination, nn::gfx::ShaderStageBit_Pixel,
            nn::gfx::TextureState_ColorTarget, nn::gfx::ShaderStageBit_Pixel );
    }

    // Copy a region of the texture's surface to a region of the temp color buffer's surface
    {
        static f32 angle = 0.0f;
        angle += 0.025f;
        s32 xOff = s32(100.0f * cos(angle));
        s32 yOff = s32(100.0f * sin(angle));

        Vec dstTL = { 2.0f * (200 + xOff) / static_cast< float >( width ) - 1.0f,
            -1.0f * (2.0f * (200 + yOff) / static_cast< float >( height ) - 1.0f),
            0.0f };
        Vec dstBR = { 2.0f * (width - 200 + xOff) / static_cast< float >( width ) - 1.0f,
            -1.0f * (2.0f * (height - 200 + yOff) / static_cast< float >( height ) - 1.0f),
            0.0f };
        Vec* pPos = g_copyPositionBuffer.Map< Vec >();
        // Setup positions as follows
        // 0 1
        // 2 3
        pPos[ 0 ] = dstTL;
        pPos[ 1 ].x = dstBR.x;
        pPos[ 1 ].y = dstTL.y;
        pPos[ 1 ].z = 0.0f;
        pPos[ 2 ].x = dstTL.x;
        pPos[ 2 ].y = dstBR.y;
        pPos[ 2 ].z = 0.0f;
        pPos[ 3 ] = dstBR;
        g_copyPositionBuffer.Unmap();

        Vec2 srcTL = { (100 + xOff) / static_cast< float >( width ),
            ( (100 + yOff) / static_cast< float >( height ) ) };
        Vec2 srcBR = { (1280 - 100 + xOff) / static_cast< float >( width ),
            ( (720 - 100 + yOff) / static_cast< float >( height ) ) };
        Vec2* pTex = g_copyTexCoordBuffer.Map< Vec2 >();
        pTex[ 0 ] = srcTL;
        pTex[ 1 ].x = srcBR.x;
        pTex[ 1 ].y = srcTL.y;
        pTex[ 2 ].x = srcTL.x;
        pTex[ 2 ].y = srcBR.y;
        pTex[ 3 ] = srcBR;
        g_copyTexCoordBuffer.Unmap();

        const nn::gfx::ColorTargetView* colorTargets[ 1 ] = { &g_ExampleColorBufferView };
#if NN_GFX_IS_TARGET_GX
        GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
        DEMOCommandBuffer.SetRenderTargets( 1, colorTargets, &DEMODepthBufferView );
        DEMOCommandBuffer.SetViewportScissorState( &g_viewportScissorState );
        DEMOCommandBuffer.SetPipeline( &g_stretchBlitPipeline.pipeline );
        DEMOCommandBuffer.SetVertexBuffer( g_PositionLocation, g_copyPositionBuffer.gpuAddress, POSITION_BUFFER_SIZE / VERTEX_COUNT, POSITION_BUFFER_SIZE );
        DEMOCommandBuffer.SetVertexBuffer( g_TexCoordLocation, g_copyTexCoordBuffer.gpuAddress, TEXCOORD_BUFFER_SIZE / VERTEX_COUNT, TEXCOORD_BUFFER_SIZE  );
        DEMOCommandBuffer.SetTextureAndSampler( g_texId, nn::gfx::ShaderStage_Pixel, g_ExampleTexture.GetDescriptorSlot(0), g_stretchSamplerDescriptorSlot );
        DEMOCommandBuffer.Draw( nn::gfx::PrimitiveTopology_TriangleStrip, VERTEX_COUNT, 0 );
    }

    DEMOCommandBuffer.SetTextureStateTransition( g_ExampleTexture.GetTexture( 0 ), NULL,
        nn::gfx::TextureState_ColorTarget, nn::gfx::ShaderStageBit_Pixel,
        nn::gfx::TextureState_ShaderRead, nn::gfx::ShaderStageBit_Pixel );

    // Use a special shader to do a stretch blit.
    {
        nn::gfx::ColorTargetView* pScanBufferView = DEMOGetColorBufferView();
        const nn::gfx::ColorTargetView* colorTargets[ 1 ] = { pScanBufferView };
#if NN_GFX_IS_TARGET_GX
        GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
        DEMOCommandBuffer.SetRenderTargets( 1, colorTargets, &DEMODepthBufferView );
        DEMOCommandBuffer.SetViewportScissorState( &g_viewportScissorState );
        DEMOCommandBuffer.SetPipeline( &g_stretchBlitPipeline.pipeline );
        DEMOCommandBuffer.SetVertexBuffer( g_PositionLocation, g_stretchPositionBuffer.gpuAddress, POSITION_BUFFER_SIZE / VERTEX_COUNT, POSITION_BUFFER_SIZE );
        DEMOCommandBuffer.SetVertexBuffer( g_TexCoordLocation, g_stretchTexCoordBuffer.gpuAddress, TEXCOORD_BUFFER_SIZE / VERTEX_COUNT, TEXCOORD_BUFFER_SIZE  );
        DEMOCommandBuffer.SetTextureAndSampler( g_texId, nn::gfx::ShaderStage_Pixel, g_ExampleColorBufferDescriptorSlot, g_stretchSamplerDescriptorSlot );
        DEMOCommandBuffer.Draw( nn::gfx::PrimitiveTopology_TriangleStrip, VERTEX_COUNT, 0 );
    }

    s_pLastColorBuffer = DEMOGetColorBuffer();
    DEMOGfxDoneRender();
}

//extern "C" void nnMain()
TEST(GfxCopySurfaceRect, Run)
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    DEMOInit();
    DEMOTestInit(argc, argv);
    DEMOGfxInit(argc, argv);
    DEMOFontInit();
    DEMOTestIsUseHlslccGlsl() ? g_ShaderFileIdx = 1 : g_ShaderFileIdx = 0;

    SceneInit();

    while (DEMOIsRunning())
    {
        SceneDraw();
    }

    // Free texture
    g_ExampleTexture.Finalize();

    g_copyPositionBuffer.Finalize();
    g_copyTexCoordBuffer.Finalize();
    g_stretchPositionBuffer.Finalize();
    g_stretchTexCoordBuffer.Finalize();

    g_stretchBlitPipeline.Finalize( &DEMODevice );
    g_ExampleColorBufferView.Finalize( &DEMODevice );
    g_ExampleColorBufferTextureView.Finalize( &DEMODevice );
    g_ExampleColorBuffer.Finalize( &DEMODevice );
    g_ExampleColorBufferData->Finalize();

    g_viewportScissorState.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2( g_pViewportScissorData );

    g_stretchSampler.Finalize( &DEMODevice );

    DEMOFontShutdown();
    DEMOTestShutdown();
    DEMOGfxShutdown();
    DEMOShutdown();

    SUCCEED();
}
