﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/ae.h>
#include <nn/fs.h>
#include <nn/hid.h>
#include <nn/applet/applet.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_StringUtil.h>

#include <nn/vi.private.h>

#include "../Common/Graphics.h"

#include <nnt.h>

namespace {

const size_t ThreadStackSize = 32 * 1024;
NN_OS_ALIGNAS_THREAD_STACK char g_GraphicsThreadStack[ThreadStackSize];
nn::os::ThreadType  g_GraphicsThread;

void* Allocate( size_t size )
{
    return malloc( size );
}

void Deallocate( void* ptr, size_t )
{
    free( ptr );
}

} // namespace


//-----------------------------------------------------------------------------
//  メモリ関連の初期化です。
//
NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[38 * 1024 * 1024];
extern "C" void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ae::SetMemoryHeapSizeWithRetry(0x400000));
    nn::init::InitializeAllocator( g_MallocBuffer, sizeof(g_MallocBuffer) );
}

//-----------------------------------------------------------------------------
//  メイン関数です。
//
void OverlayAppletMain(nn::ae::OverlayAppletParameters* param)
{
    int64_t begin = nn::os::GetSystemTick().ToTimeSpan().GetSeconds();
    NN_UNUSED(param);

    // DummyAPP が起動し終わるまで待機する
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));

    int appPresentInterval = 1;
    char** argv = nn::os::GetHostArgv();
    if (!nn::util::Strncmp(argv[1], "1", 2))
    {
        appPresentInterval = 1;
    }
    else if (!nn::util::Strncmp(argv[1], "2", 2))
    {
        appPresentInterval = 2;
    }

    NN_LOG("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@\n");
    NN_LOG("        OA                   \n");
    NN_LOG("   PresentInterval = %d      \n", appPresentInterval);
    //APP 側のフレームワーク
    const char* appFrameworkMode = NULL;
    if (!nn::util::Strncmp(argv[2], "1", 2))
    {
        NN_LOG("    DeferredSubmission   \n");
        appFrameworkMode = "DeferredSubmission";
    }
    else if (!nn::util::Strncmp(argv[2], "2", 2))
    {
        NN_LOG("    DeferredExecution    \n");
        appFrameworkMode = "DeferredExecution";
    }
    NN_LOG("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@\n");

    const char* appStatus = NULL;
    if (!nn::util::Strncmp(argv[3], "1", 2))
    {
        appStatus = "HighLoad";
    }
    else
    {
        appStatus = "LowLoad";
    }

    // 誤差でテストが失敗することがあるため、計測に失敗しても3回まで再試行できる。
    for (int trialCount = 0; ; trialCount++)
    {
        Graphics::SetMeasureParameters(appPresentInterval);
        nn::fs::SetAllocator(Allocate, Deallocate);
        Graphics::Rgba clearColor = { 0.0f, 0.0f, 0.0f, 0.0f };
        Graphics::InitializeGraphics(clearColor, "OverlayApplet", Graphics::FrameworkMode_DeferredExecution, Graphics::TestMode_OverlayTest, 1);

        // スレッドを生成して実行
        nn::Result result = nn::os::CreateThread(&g_GraphicsThread, Graphics::GraphicsThreadFunction, NULL, g_GraphicsThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority);
        NN_ASSERT(result.IsSuccess(), "OA: Cannot create GraphicsThread.");
        nn::os::StartThread(&g_GraphicsThread);
        nn::os::WaitThread(&g_GraphicsThread);
        nn::os::DestroyThread(&g_GraphicsThread);

        if (appPresentInterval == 1)
        {
            // 1,2 回目の試行で微妙な誤差で失敗したら再試行する
            if (trialCount < 2)
            {
                if (Graphics::GetGpuLoadAverage() > 325 && Graphics::GetGpuLoadAverage() < 335)
                {
                    continue;
                }
            }
            // OA が使用できる最大 GPU 時間が 335～410us の範囲にあることをテスト
            // APP の GPU が高い場合は 350us, 低い場合は 395us が目安
            EXPECT_LT(Graphics::GetGpuLoadAverage(), 420);
            EXPECT_GT(Graphics::GetGpuLoadAverage(), 335);
            NN_LOG("##teamcity[buildStatisticValue key='TrialCount_OA_APP_%s_%s_60' value='%d']\n", appFrameworkMode, appStatus, trialCount);
            NN_LOG("##teamcity[buildStatisticValue key='OA_APP_%s_%s_60_Max' value='%d']\n", appFrameworkMode, appStatus, Graphics::GetGpuLoadAverage());
            break;
        }
        else if (appPresentInterval == 2)
        {
            if (trialCount < 2)
            {
                if (Graphics::GetGpuLoadAverage() > 820 && Graphics::GetGpuLoadAverage() < 840)
                {
                    continue;
                }
            }
            // OA が使用できる最大 GPU 時間が 840～925us の範囲にあることをテスト
            // APP の GPU が高い場合は 853us, 低い場合は 395us が目安
            EXPECT_LT(Graphics::GetGpuLoadAverage(), 935);
            EXPECT_GT(Graphics::GetGpuLoadAverage(), 840);
            NN_LOG("##teamcity[buildStatisticValue key='TrialCount_OA_APP_%s_%s_30' value='%d']\n", appFrameworkMode, appStatus, trialCount);
            NN_LOG("##teamcity[buildStatisticValue key='OA_APP_%s_%s_30_Max' value='%d']\n", appFrameworkMode, appStatus, Graphics::GetGpuLoadAverage());
            break;
        }
    }

    // 終了処理
    Graphics::FinalizeGraphics();
    int64_t end = nn::os::GetSystemTick().ToTimeSpan().GetSeconds();
    NN_LOG("TOTAL_TIME = %lld\n", end - begin);
}

TEST(CheckGpuResourceOa, AllData)
{
    nn::ae::InvokeOverlayAppletMain(nn::ae::AppletId_OverlayApplet, OverlayAppletMain);
    SUCCEED();
    return;
}
