﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nn/init.h>
#include <nn/os.h>
#include <nn/hid.h>
#include <nn/fs.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/util/util_StringUtil.h>
#include <nn/oe.h>

#include "../Common/Graphics.h"

#include <nnt.h>

namespace {

const size_t ThreadStackSize = 32 * 1024;
NN_OS_ALIGNAS_THREAD_STACK char g_GraphicsThreadStack[ThreadStackSize];
nn::os::ThreadType  g_GraphicsThread;

void* Allocate( size_t size )
{
    return malloc( size );
}

void Deallocate( void* ptr, size_t )
{
    free( ptr );
}

} // namespace

NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[256 * 1024 * 1024];
extern "C" void nninitStartup()
{
    nn::os::SetMemoryHeapSize( 0x400000 );
    nn::init::InitializeAllocator( g_MallocBuffer, sizeof(g_MallocBuffer) );
}

//extern "C" void nnMain()
TEST(CheckGpuResourceApp, AllData)
{
    nn::oe::FinishStartupLogo();
    int64_t begin = nn::os::GetSystemTick().ToTimeSpan().GetSeconds();

    int presentInterval = 1;
    Graphics::FrameworkMode frameworkMode = Graphics::FrameworkMode_DeferredSubmission;
    char** argv = nn::os::GetHostArgv();
    if (!nn::util::Strncmp(argv[1], "1", 2))
    {
        presentInterval = 1;
    }
    else if (!nn::util::Strncmp(argv[1], "2", 2))
    {
        presentInterval = 2;
    }

    NN_LOG("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@\n");
    NN_LOG("        APP                  \n");
    NN_LOG("   PresentInterval = %d      \n", presentInterval);
    if (!nn::util::Strncmp(argv[2], "1", 2))
    {
        frameworkMode = Graphics::FrameworkMode_DeferredSubmission;
        NN_LOG("   DeferredSubmission    \n");
    }
    else if (!nn::util::Strncmp(argv[2], "2", 2))
    {
        frameworkMode = Graphics::FrameworkMode_DeferredExecution;
        NN_LOG("   DeferredExecution     \n");
    }
    NN_LOG("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@\n");

    const char* oaStatus = NULL;
    if (!nn::util::Strncmp(argv[3], "1", 2))
    {
        oaStatus = "Visible";
    }
    else
    {
        oaStatus = "Invisible";
    }

    nn::fs::SetAllocator(Allocate, Deallocate);
    Graphics::Rgba clearColor = { 0.1f, 0.1f, 0.3f, 1.0f };
    Graphics::InitializeGraphics(clearColor, "Application", frameworkMode, Graphics::TestMode_ApplicationTest, presentInterval);

    // 誤差でテストが失敗することがあるため、計測に失敗しても3回まで再試行できる。
    for(int trialCount = 0; ; trialCount++)
    {
        // スレッドを生成して実行
        Graphics::SetMeasureParameters(presentInterval);
        nn::Result result = nn::os::CreateThread(&g_GraphicsThread, Graphics::GraphicsThreadFunction, NULL, g_GraphicsThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority);
        NN_ASSERT(result.IsSuccess(), "APP: Cannot create GraphicsThread.");
        nn::os::StartThread(&g_GraphicsThread);
        nn::os::WaitThread(&g_GraphicsThread);
        nn::os::DestroyThread(&g_GraphicsThread);

        if (presentInterval == 1)
        {
            // 1,2 回目の試行で微妙な誤差で失敗したら再試行する
            if (trialCount < 2)
            {
                if (Graphics::GetGpuLoadAverage() > 15850 && Graphics::GetGpuLoadAverage() < 15870)
                {
                    continue;
                }
            }

            // 3回目の試行が完了、またはテスト結果が確定した場合はログを出して終了。
            EXPECT_LT(Graphics::GetGpuLoadAverage(), 15945);
            EXPECT_GT(Graphics::GetGpuLoadAverage(), 15870);
            EXPECT_LT(Graphics::GetGpuDelayAverage(), 660);
            EXPECT_GT(Graphics::GetGpuDelayAverage(), 640);
            if (frameworkMode == Graphics::FrameworkMode_DeferredSubmission)
            {
                NN_LOG("##teamcity[buildStatisticValue key='TrialCount_APP_DeferredSubmission_OA_%s_60' value='%d']\n", oaStatus, trialCount);
                NN_LOG("##teamcity[buildStatisticValue key='APP_DeferredSubmission_OA_%s_60_Max' value='%d']\n", oaStatus, Graphics::GetGpuLoadAverage());
                NN_LOG("##teamcity[buildStatisticValue key='APP_DeferredSubmission_OA_%s_60_Delay' value='%d']\n", oaStatus, Graphics::GetGpuDelayAverage());
            }
            else
            {
                NN_LOG("##teamcity[buildStatisticValue key='TrialCount_APP_DeferredExecution_OA_%s_60' value='%d']\n", oaStatus, trialCount);
                NN_LOG("##teamcity[buildStatisticValue key='APP_DeferredExecution_OA_%s_60_Max' value='%d']\n", oaStatus, Graphics::GetGpuLoadAverage());
                NN_LOG("##teamcity[buildStatisticValue key='APP_DeferredExecution_OA_%s_60_Delay' value='%d']\n", oaStatus, Graphics::GetGpuDelayAverage());
            }
            break;
        }
        else if (presentInterval == 2)
        {
            if (trialCount < 2)
            {
                if (Graphics::GetGpuLoadAverage() > 32010 && Graphics::GetGpuLoadAverage() < 32040)
                {
                    continue;
                }
            }

            // APP が使用できる最大 GPU 時間が 32045～32085us(96.13%～96.3%) の範囲にあることをテスト
            // 2フレ目は GPU コンテキスト切り替えが無いので presentinterval=1 と比べて使用できる時間割合は高い
            EXPECT_LT(Graphics::GetGpuLoadAverage(), 32100);
            EXPECT_GT(Graphics::GetGpuLoadAverage(), 32040);
            EXPECT_LT(Graphics::GetGpuDelayAverage(), 1160);
            EXPECT_GT(Graphics::GetGpuDelayAverage(), 1140);
            if (frameworkMode == Graphics::FrameworkMode_DeferredSubmission)
            {
                NN_LOG("##teamcity[buildStatisticValue key='TrialCount_APP_DeferredSubmission_OA_%s_30' value='%d']\n", oaStatus, trialCount);
                NN_LOG("##teamcity[buildStatisticValue key='APP_DeferredSubmission_OA_%s_30_Max' value='%d']\n", oaStatus, Graphics::GetGpuLoadAverage());
                NN_LOG("##teamcity[buildStatisticValue key='APP_DeferredSubmission_OA_%s_30_Delay' value='%d']\n", oaStatus, Graphics::GetGpuDelayAverage());
            }
            else
            {
                NN_LOG("##teamcity[buildStatisticValue key='TrialCount_APP_DeferredExecution_OA_%s_30' value='%d']\n", oaStatus, trialCount);
                NN_LOG("##teamcity[buildStatisticValue key='APP_DeferredExecution_OA_%s_30_Max' value='%d']\n", oaStatus, Graphics::GetGpuLoadAverage());
                NN_LOG("##teamcity[buildStatisticValue key='APP_DeferredExecution_OA_%s_30_Delay' value='%d']\n", oaStatus, Graphics::GetGpuDelayAverage());
            }
            break;
        }
    }
    // 終了処理
    Graphics::FinalizeGraphics();

    int64_t end = nn::os::GetSystemTick().ToTimeSpan().GetSeconds();
    NN_LOG("TOTAL_TIME = %lld\n", end - begin);

    SUCCEED();
    return;
} // NOLINT
