﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/ae.h>
#include <nn/fs.h>
#include <nn/hid.h>
#include <nn/applet/applet.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_StringUtil.h>

#include <nn/vi.private.h>

#include "../Common/Graphics.h"


namespace {

const size_t ThreadStackSize = 32 * 1024;
NN_OS_ALIGNAS_THREAD_STACK char g_GraphicsThreadStack[ThreadStackSize];
nn::os::ThreadType  g_GraphicsThread;
Graphics::TestMode g_TestMode = Graphics::TestMode_OverlayTest;

void* Allocate( size_t size )
{
    return malloc( size );
}

void Deallocate( void* ptr, size_t )
{
    free( ptr );
}

} // namespace


//-----------------------------------------------------------------------------
//  メモリ関連の初期化です。
//
NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[38 * 1024 * 1024];
extern "C" void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ae::SetMemoryHeapSizeWithRetry(0x400000));
    nn::init::InitializeAllocator( g_MallocBuffer, sizeof(g_MallocBuffer) );
}

//-----------------------------------------------------------------------------
//  メイン関数です。
//
void OverlayAppletMain(nn::ae::OverlayAppletParameters* param)
{
    NN_UNUSED(param);

    char** argv = nn::os::GetHostArgv();
    NN_LOG("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@\n");
    if (!nn::util::Strncmp(argv[3], "1", 2))
    {
        NN_LOG("      DymmyOA - Visible     \n");
        g_TestMode = Graphics::TestMode_OverlayVisible;
    }
    else if (!nn::util::Strncmp(argv[3], "2", 2))
    {
        NN_LOG("    DymmyOA - Invisible     \n");
        g_TestMode = Graphics::TestMode_OverlayInvisible;
    }
    NN_LOG("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@\n");

    // 初期化処理
    nn::fs::SetAllocator(Allocate, Deallocate);
    Graphics::Rgba clearColor = { 0.0f, 0.0f, 0.0f, 0.0f };
    Graphics::InitializeGraphics(clearColor, "OverlayApplet", Graphics::FrameworkMode_DeferredExecution, g_TestMode, 1);

    // スレッドを生成して実行
    nn::Result result = nn::os::CreateThread( &g_GraphicsThread, Graphics::GraphicsThreadFunction, NULL, g_GraphicsThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority);
    NN_ASSERT( result.IsSuccess(), "OA: Cannot create GraphicsThread." );
    nn::os::StartThread( &g_GraphicsThread );
    nn::os::WaitThread( &g_GraphicsThread );
    nn::os::DestroyThread( &g_GraphicsThread );

    // 終了処理
    Graphics::FinalizeGraphics();
}

extern "C" void nnMain()
{
    nn::ae::InvokeOverlayAppletMain(nn::ae::AppletId_OverlayApplet, OverlayAppletMain);
    return;
}
