﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nn/init.h>
#include <nn/os.h>
#include <nn/hid.h>
#include <nn/fs.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/util/util_StringUtil.h>
#include <nn/oe.h>

#include "../Common/Graphics.h"

namespace {

const size_t ThreadStackSize = 32 * 1024;
NN_OS_ALIGNAS_THREAD_STACK char g_GraphicsThreadStack[ThreadStackSize];
nn::os::ThreadType  g_GraphicsThread;
Graphics::TestMode g_TestMode = Graphics::TestMode_ApplicationTest;

void* Allocate( size_t size )
{
    return malloc( size );
}

void Deallocate( void* ptr, size_t )
{
    free( ptr );
}

} // namespace

NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[256 * 1024 * 1024];
extern "C" void nninitStartup()
{
    nn::os::SetMemoryHeapSize( 0x400000 );
    nn::init::InitializeAllocator( g_MallocBuffer, sizeof(g_MallocBuffer) );
}

extern "C" void nnMain()
{
    nn::oe::FinishStartupLogo();
    int presentInterval = 1;
    Graphics::FrameworkMode frameworkMode = Graphics::FrameworkMode_DeferredSubmission;
    char** argv = nn::os::GetHostArgv();
    if (!nn::util::Strncmp(argv[1], "1", 2))
    {
        presentInterval = 1;
    }
    else if (!nn::util::Strncmp(argv[1], "2", 2))
    {
        presentInterval = 2;
    }
    Graphics::SetMeasureParameters(presentInterval);

    NN_LOG("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@\n");
    NN_LOG("        DummyAPP             \n");
    NN_LOG("   PresentInterval = %d      \n", presentInterval);
    if (!nn::util::Strncmp(argv[2], "1", 2))
    {
        NN_LOG("    DeferredSubmission   \n");
        frameworkMode = Graphics::FrameworkMode_DeferredSubmission;
    }
    else if (!nn::util::Strncmp(argv[2], "2", 2))
    {
        NN_LOG("    DeferredExecution    \n");
        frameworkMode = Graphics::FrameworkMode_DeferredExecution;
    }

    if (!nn::util::Strncmp(argv[3], "1", 2))
    {
        NN_LOG("        HighLoad         \n");
        g_TestMode = Graphics::TestMode_ApplicationHighLoad;
    }
    else if (!nn::util::Strncmp(argv[3], "2", 2))
    {
        NN_LOG("         LowLoad         \n");
        g_TestMode = Graphics::TestMode_ApplicationLowLoad;
    }
    NN_LOG("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@\n");


    // 初期化処理
    nn::fs::SetAllocator(Allocate, Deallocate);
    Graphics::Rgba clearColor = { 0.1f, 0.1f, 0.3f, 1.0f };
    Graphics::InitializeGraphics(clearColor, "Application", frameworkMode, g_TestMode, presentInterval);

    // スレッドを生成して実行
    nn::Result result = nn::os::CreateThread( &g_GraphicsThread, Graphics::GraphicsThreadFunction, NULL, g_GraphicsThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority);
    NN_ASSERT( result.IsSuccess(), "APP: Cannot create GraphicsThread." );
    nn::os::StartThread( &g_GraphicsThread );
    nn::os::WaitThread( &g_GraphicsThread );
    nn::os::DestroyThread( &g_GraphicsThread );

    // 終了処理
    Graphics::FinalizeGraphics();

    return;
}
