﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cmath>

#include <gfx/demo.h>
#include <nn/gfx.h>

#include <nnt.h>
#include <nnt/nnt_Argument.h>

//#define USE_ATTRIB_PER_DRAW

// Each object will have two instances
static const int NUM_INSTANCE = 6;
static const int NUM_X = 1;

static const int MAX_BUFFER_ALIGNMENT = 4096;

// We use instance for color material
// Color changes every instance
static const int INSTANCE_INTERVAL_COLOR = 3;

// We use another instance for object shift in image screen
// Each instance has different shift value
static const int INSTANCE_INTERVAL_SHIFT = 1;

static int g_ShaderFileIdx = 0;
static const char * const SHADER_FILE_NAME[] =
{
    "shaders/baseInstanceShader",
    "shaders/baseInstanceShaderHlslcc",
};

static const int MAX_MODELS = 4;

//---------------------------------------------------------------------------*
//  Model Data
//---------------------------------------------------------------------------*/
static const char* MODEL_FILES[MAX_MODELS][3]
    = {
       // Test Draw
       {"geometries/cubePosition.dat",
       NULL,
       NULL},
       {"geometries/pyramidPosition.dat",
       NULL,
       NULL},

       // Test DrawIndexed
       {"geometries/elephantPosition.dat",
       NULL,
       "geometries/elephantIdx.dat"},
       {"geometries/teapotPosition.dat",
       NULL,
       "geometries/teapotIdx.dat"},
};


static const int MODEL_VTX_STRIDE = sizeof(f32) * 3;
static const int MODEL_NORM_STRIDE = sizeof(f32) * 3;
static const int MODEL_CLR_STRIDE = sizeof(f32) * 3;
static const int MODEL_MOVE_STRIDE = sizeof(f32) * 3;

static const float MOVE_STEP_X = 0.5f;
static const float MOVE_STEP_Y = 0.3f;
static const float MOVE_STEP_Z = 0.001f;

static const int MODEL_OFFSET = 0;

static const int NUM_ATTRIB = 4;

// -----

typedef struct _InstancedObject InstancedObject;
typedef void (*PDRAWFUNC)(InstancedObject *pObj);
enum AttribType
{
    AttribTypeNone = 0,
    AttribTypePos,
    AttribTypeTexCoord,
    AttribTypeNormal,
    AttribTypeColor,
    AttribTypeMove,
};

struct Attribute
{
    AttribType type;
    u32 bufferId;
    u32 stride;
    u32 size;
    void *data;
#ifdef USE_ATTRIB_PER_DRAW
    nn::gfx::Buffer buffer;
    nn::gfx::GpuAddress addr;
#endif
};

struct _InstancedObject
{
    PDRAWFUNC draw;
    void *indices;
    struct Attribute attribs[16];
    u32 numAttrs;
    u32 count;
    u32 numInstances;
    u32 baseVertex;
    u32 baseInstance;

    void* pPipelineData;
    nn::gfx::Pipeline pipeline;
    nn::gfx::Buffer indexBuffer;
    nn::gfx::GpuAddress indexBufferAddress;
    nn::gfx::IndexFormat indexFormat;
};

typedef struct _Matrices {
    Mtx44   modelMtx44;
    Mtx44   viewMtx44;
    Mtx44   projMtx44;
} Matrices;

static struct _AppState {
    DEMOGfxShader theShader;

    int posLoc;
    int clrLoc;
    int normLoc;
    int moveLoc;

    // Constant Buffers
    int uniformLoc;

#ifndef USE_ATTRIB_PER_DRAW
    // Global Vertex Data
    void *vertexData;
    void *normData;
    void *colorData;
    void *moveData;
    u32 vertexSize;
    u32 normSize;
    u32 colorSize;
    u32 moveSize;
    nn::gfx::Buffer vertexBuffer;
    nn::gfx::GpuAddress vertexBufferAddress;
    nn::gfx::Buffer normBuffer;
    nn::gfx::GpuAddress normBufferAddress;
    nn::gfx::Buffer colorBuffer;
    nn::gfx::GpuAddress colorBufferAddress;
    nn::gfx::Buffer moveBuffer;
    nn::gfx::GpuAddress moveBufferAddress;
#endif

    Matrices* pMatricesData;
    nn::gfx::Buffer matricesBuffer;
    nn::gfx::GpuAddress matricesBufferAddress;

    u32 NUM_MODEL;
    InstancedObject* instancedObjects[MAX_MODELS];
    u32 s_ticks;
    f32 s_yrad;
    f32 s_xrad;

    // Default viewport
    nn::gfx::ViewportScissorState viewportScissorState;
    void* pViewportScissorData;

} applicationState;

#if !NN_GFX_IS_TARGET_GX
static void swapBuffer(void* bufp, u32 len)
{
    uint8_t* buf = static_cast< uint8_t* >( bufp );
    uint8_t a;

    while (len > 0)
    {
        a = buf[0];
        buf[0] = buf[3];
        buf[3] = a;
        a = buf[1];
        buf[1] = buf[2];
        buf[2] = a;
        len -= 4;
        buf += 4;
    }
}
#endif

static void InitAppState()
{
    applicationState.posLoc = 0;
    applicationState.clrLoc = 0;
    applicationState.normLoc = 0;
    applicationState.moveLoc = 0;

    // Constant Buffers
    applicationState.uniformLoc = 0;

#ifndef USE_ATTRIB_PER_DRAW
    // Global Vertex Data
    applicationState.vertexData = NULL;
    applicationState.normData = NULL;
    applicationState.colorData = NULL;
    applicationState.moveData = NULL;
    applicationState.vertexSize = 0;
    applicationState.normSize = 0;
    applicationState.colorSize = 0;
    applicationState.moveSize = 0;
#endif

    for (int idx = 0; idx < MAX_MODELS; idx++)
    {
        // We must use the new operator because InstancedObject contants nn::gfx classes
        applicationState.instancedObjects[idx] = new InstancedObject;
    }

    applicationState.NUM_MODEL = MAX_MODELS;
    applicationState.s_ticks = 0;
    applicationState.s_yrad = 0.0f;
    applicationState.s_xrad = 0.0f;
}

#ifdef USE_ATTRIB_PER_DRAW
void SetupBuffers(InstancedObject *pObj)
{
    for (u32 i = 0; i < pObj->numAttrs; i++)
    {
        DEMOCommandBuffer.SetVertexBuffer(pObj->attribs[i].bufferId, pObj->attribs[i].addr, pObj->attribs[i].stride, pObj->attribs[i].size);
    }
}
#endif // USE_ATTRIB_PER_DRAW

void DrawArray(InstancedObject *pObj)
{
#ifdef USE_ATTRIB_PER_DRAW
    SetupBuffers(pObj);
#endif // USE_ATTRIB_PER_DRAW
    DEMOCommandBuffer.SetPipeline(& pObj->pipeline );
    DEMOCommandBuffer.Draw(nn::gfx::PrimitiveTopology_TriangleList, pObj->count, pObj->baseVertex, pObj->numInstances, 0);
}

void DrawArray2(InstancedObject *pObj)
{
#ifdef USE_ATTRIB_PER_DRAW
    SetupBuffers(pObj);
#endif // USE_ATTRIB_PER_DRAW
    DEMOCommandBuffer.SetPipeline( &pObj->pipeline );
    DEMOCommandBuffer.Draw(nn::gfx::PrimitiveTopology_TriangleList, pObj->count, pObj->baseVertex, pObj->numInstances, pObj->baseInstance);

}

void DrawIndexed(InstancedObject *pObj)
{
#ifdef USE_ATTRIB_PER_DRAW
    SetupBuffers(pObj);
#endif // USE_ATTRIB_PER_DRAW
    DEMOCommandBuffer.SetPipeline( &pObj->pipeline);
    DEMOCommandBuffer.DrawIndexed(nn::gfx::PrimitiveTopology_TriangleList, pObj->indexFormat, pObj->indexBufferAddress, pObj->count, pObj->baseVertex, pObj->numInstances, pObj->baseInstance);
}

// Prototypes
static void CameraInit(nn::gfx::Buffer&);
static int SceneInit(void);
static int SceneDraw(void);
static void ModelTick(nn::gfx::Buffer&);
static void PrintInfo(void);

// Init function for setting projection matrix
static void CameraInit(nn::gfx::Buffer& uniformBlock)
{
    // row major matrices
    Mtx   lookAtMtx34;

    Vec     up = {0.0f,  1.0f, 0.0f};
    Vec  objPt = {0.0f, 0.0f, 0.0f};
    Vec camLoc = {400.0f, 0.0f, 0.0f};

    f32   pers = 60.0f;
    f32 aspect = (f32)DEMOColorBufferInfo.GetWidth() / (f32)DEMOColorBufferInfo.GetHeight();
    f32  znear = 10.0f;
    f32   zfar = 2000.0f;

    Matrices* pMatrix = uniformBlock.Map< Matrices >( );

    // Compute perspective matrix
    MTXPerspective(pMatrix->projMtx44, pers, aspect, znear, zfar);

    // Compute lookAt matrix
    MTXLookAt(lookAtMtx34, &camLoc, &up, &objPt);
    MTX34To44(lookAtMtx34, pMatrix->viewMtx44);
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pMatrix, sizeof( *pMatrix ) );
#endif
    uniformBlock.FlushMappedRange( 0, sizeof( Matrices ) );
    uniformBlock.Unmap();
}

// Update Model function for setting model matrix
static void ModelTick(nn::gfx::Buffer& uniformBlock)
{
    // row major matrix
    Mtx  rotXMtx34;
    Mtx  rotYMtx34;
    Mtx  modelMtx34;

    Matrices* pMatrix = uniformBlock.Map< Matrices >();
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pMatrix, sizeof( *pMatrix ) );
#endif

    // Compute rotation matrix
    MTXRotRad(rotYMtx34, 'y', applicationState.s_yrad * 2);
    MTXRotRad(rotXMtx34, 'x', applicationState.s_xrad * 2);

    // Compute model matrix
    MTXConcat(rotXMtx34, rotYMtx34, modelMtx34);
    MTX34To44(modelMtx34, pMatrix->modelMtx44);

#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pMatrix, sizeof( *pMatrix ) );
#endif
    uniformBlock.FlushMappedRange( 0, sizeof( Matrices ) );
    uniformBlock.Unmap();

    applicationState.s_yrad += 0.003f;
    applicationState.s_xrad += 0.006f;
}

static void SetDefaultBlendTargetStateInfo(nn::gfx::BlendTargetStateInfo& info)
{
    // Match GX2 default state
    info.SetDefault();
    info.SetBlendEnabled( false );
    info.SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    info.SetDestinationColorBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    info.SetColorBlendFunction( nn::gfx::BlendFunction_Add );
    info.SetChannelMask( nn::gfx::ChannelMask_All );
    info.SetSourceAlphaBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    info.SetDestinationAlphaBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    info.SetAlphaBlendFunction( nn::gfx::BlendFunction_Add );
}

static void SetDefaultColorTargetStateInfo(nn::gfx::ColorTargetStateInfo& info)
{
    // Match GX2 default state
    info.SetDefault();
    info.SetFormat( DEMOColorBufferInfo.GetImageFormat() );
}

static void SetDefaultViewportScissorInfo(nn::gfx::ViewportScissorStateInfo& info,
                                          nn::gfx::ViewportStateInfo& viewportStateInfo,
                                          nn::gfx::ScissorStateInfo& scissorStateInfo)
{
    viewportStateInfo.SetDefault();
    viewportStateInfo.SetWidth( static_cast< float >( DEMOColorBufferInfo.GetWidth() ) );
    viewportStateInfo.SetHeight( static_cast< float >( DEMOColorBufferInfo.GetHeight() ) );
    viewportStateInfo.SetOriginX( 0.0f );
    viewportStateInfo.SetOriginY( 0.0f );
    viewportStateInfo.SetMinDepth( 0.0f );
    viewportStateInfo.SetMaxDepth( 1.0f );

    scissorStateInfo.SetDefault();
    scissorStateInfo.SetOriginX( 0 );
    scissorStateInfo.SetOriginY( 0 );
    scissorStateInfo.SetWidth( DEMOColorBufferInfo.GetWidth() );
    scissorStateInfo.SetHeight( DEMOColorBufferInfo.GetHeight() );

    info.SetDefault();
    info.SetScissorEnabled( true );
    info.SetScissorStateInfoArray( &scissorStateInfo, 1 );
    info.SetViewportStateInfoArray( &viewportStateInfo, 1 );
}

static void SetDefaultPipelineStateInfo(nn::gfx::Pipeline::InfoType& info,
                                        nn::gfx::BlendTargetStateInfo* blendTargetStateInfoArray,
                                        int blendTargetStateCount,
                                        nn::gfx::ColorTargetStateInfo* colorTargetStateInfoArray,
                                        int colorTargetStateCount,
                                        nn::gfx::VertexAttributeStateInfo* vertexAttributeStateInfoArray,
                                        int vertexAttributeStateCount,
                                        nn::gfx::VertexBufferStateInfo* vertexBufferStateInfoArray,
                                        int vertexBufferStateCount)
{
    // Setup the default state like in Cafe
    // these have to be static so they survive the pipeline init
    static nn::gfx::DepthStencilStateInfo depthStencilStateInfo;
    static nn::gfx::RasterizerStateInfo rasterizerStateInfo;
    static nn::gfx::BlendStateInfo blendStateInfo;
    static nn::gfx::RenderTargetStateInfo renderTargetStateInfo;
    static nn::gfx::VertexStateInfo vertexStateInfo;

    info.SetDefault();
    depthStencilStateInfo.SetDefault();
    rasterizerStateInfo.SetDefault();
    blendStateInfo.SetDefault();
    renderTargetStateInfo.SetDefault();
    vertexStateInfo.SetDefault();

    depthStencilStateInfo.SetDepthComparisonFunction( nn::gfx::ComparisonFunction_Less );
    depthStencilStateInfo.SetDepthTestEnabled( true );
    depthStencilStateInfo.SetDepthWriteEnabled( true );
    depthStencilStateInfo.SetStencilTestEnabled( false );
    depthStencilStateInfo.SetStencilReadMask( 0xFF );
    depthStencilStateInfo.SetStencilWriteMask( 0xFF );
    depthStencilStateInfo.EditBackStencilStateInfo().SetComparisonFunction( nn::gfx::ComparisonFunction_Always );
    depthStencilStateInfo.EditBackStencilStateInfo().SetDepthPassOperation( nn::gfx::StencilOperation_Replace );
    depthStencilStateInfo.EditBackStencilStateInfo().SetDepthFailOperation( nn::gfx::StencilOperation_Replace );
    depthStencilStateInfo.EditBackStencilStateInfo().SetStencilFailOperation( nn::gfx::StencilOperation_Replace );
    depthStencilStateInfo.EditBackStencilStateInfo().SetStencilRef( 0x1 );
    depthStencilStateInfo.EditFrontStencilStateInfo().SetComparisonFunction( nn::gfx::ComparisonFunction_Always );
    depthStencilStateInfo.EditFrontStencilStateInfo().SetDepthPassOperation( nn::gfx::StencilOperation_Replace );
    depthStencilStateInfo.EditFrontStencilStateInfo().SetDepthFailOperation( nn::gfx::StencilOperation_Replace );
    depthStencilStateInfo.EditFrontStencilStateInfo().SetStencilFailOperation( nn::gfx::StencilOperation_Replace );
    depthStencilStateInfo.EditFrontStencilStateInfo().SetStencilRef( 0x1 );

    rasterizerStateInfo.SetFrontFace( nn::gfx::FrontFace_Ccw );
    rasterizerStateInfo.SetCullMode( nn::gfx::CullMode_None );
    rasterizerStateInfo.SetFillMode( nn::gfx::FillMode_Solid );
    rasterizerStateInfo.SetDepthBias( 0 );
    rasterizerStateInfo.SetMultisampleEnabled( false );
    rasterizerStateInfo.SetRasterEnabled( true );
    rasterizerStateInfo.SetScissorEnabled( true );
    rasterizerStateInfo.SetSlopeScaledDepthBias( 0.0f );
    rasterizerStateInfo.SetDepthClipEnabled( false );

    blendStateInfo.SetBlendConstant( 0.0f, 0.0f, 0.0f, 0.0f );
    blendStateInfo.SetBlendTargetStateInfoArray( blendTargetStateInfoArray, blendTargetStateCount );

    renderTargetStateInfo.SetDepthStencilFormat( DEMODepthBufferInfo.GetImageFormat() );
    renderTargetStateInfo.SetColorTargetStateInfoArray( colorTargetStateInfoArray, colorTargetStateCount );

    vertexStateInfo.SetVertexAttributeStateInfoArray( vertexAttributeStateInfoArray, vertexAttributeStateCount );
    vertexStateInfo.SetVertexBufferStateInfoArray( vertexBufferStateInfoArray, vertexBufferStateCount );

    info.SetDepthStencilStateInfo(&depthStencilStateInfo);
    info.SetRasterizerStateInfo(&rasterizerStateInfo);
    info.SetBlendStateInfo(&blendStateInfo);
    info.SetRenderTargetStateInfo(&renderTargetStateInfo);
    info.SetVertexStateInfo(&vertexStateInfo);
}

static void SetupVertexAttributeBuffer( nn::gfx::VertexAttributeStateInfo& attribute,
                                       nn::gfx::VertexBufferStateInfo& buffer,
                                       int location,
                                       nn::gfx::AttributeFormat attributeFormat,
                                       int stride )
{
    // Setup vertex attribute info
    attribute.SetDefault();
    attribute.SetBufferIndex( location );
    attribute.SetFormat( attributeFormat );
    attribute.SetShaderSlot( location );

    // Setup vertex buffer info
    buffer.SetDefault();
    buffer.SetStride( stride );
}

static void SetupBufferView( nn::gfx::Buffer& buffer,
            nn::gfx::GpuAddress& addr,
            int gpuAccessFlags,
            void* data,
            int size,
            DEMOGfxMemPool** pPool)
{
    if (size == 0)
    {
        return;
    }
    DEMOGfxMemPool* pool;
    pool = DEMOGfxCreateBuffer(&buffer, gpuAccessFlags, data, size);

    // Setup View
    buffer.GetGpuAddress(&addr);
    if (pPool)
    {
        *pPool = pool;
    }
}

static void SetupAttribute( struct Attribute& attribute,
    enum AttribType attributeType,
    int location,
    int stride,
    void* data,
    int size)
{
    attribute.type = attributeType;
    attribute.bufferId = location;
    attribute.stride = stride;
    attribute.data = data;
    attribute.size = size;
}

static void SetupObjectData( nn::gfx::VertexAttributeStateInfo& attribute,
                             nn::gfx::VertexBufferStateInfo& buffer,
                             Attribute& applicationAttribute,
                             int location,
                             int gpuAccessFlags,
                             nn::gfx::AttributeFormat attributeFormat,
                             enum AttribType attributeType,
                             int stride,
                             void* data,
                             int size,
                             int offset )
{
    ASSERT(offset == 0);
    NN_UNUSED( offset );

    SetupVertexAttributeBuffer( attribute,
        buffer,
        location,
        attributeFormat,
        stride  );

    SetupAttribute( applicationAttribute,
        attributeType,
        location,
        stride,
        data,
        size );

#ifdef USE_ATTRIB_PER_DRAW
    SetupBufferView( applicationAttribute.buffer,
       applicationAttribute.addr,
        gpuAccessFlags,
        data,
        size,
        0 );
#else
    NN_UNUSED( gpuAccessFlags );
#endif // USE_ATTRIB_PER_DRAW

}
static void CreateColorData( float* colorBuf )
{
    // Only fill in the first NUM_INSTANCE / INSTANCE_INTERVAL_COLOR
    // entries. The other color values are just there for padding.
    for (u32 j = 0; j < NUM_INSTANCE / INSTANCE_INTERVAL_COLOR; j++)
    {
        colorBuf[3 * j]   = DEMOFRand();
        colorBuf[3 * j + 1] = DEMOFRand();
        colorBuf[3 * j + 2] = DEMOFRand();
    }
    for (u32 j = NUM_INSTANCE / INSTANCE_INTERVAL_COLOR; j < NUM_INSTANCE; j++)
    {
        // Pad with 0s till next baseInstance step
        colorBuf[3 * j]   = 0.0;
        colorBuf[3 * j + 1] = 0.0;
        colorBuf[3 * j + 2] = 0.0;
    }
}

static void CreateMoveBuffer( float* moveBuf, int index )
{
    for (u32 i = 0; i < NUM_INSTANCE / INSTANCE_INTERVAL_SHIFT; i++)
    {
        moveBuf[3 * i]   = index * NUM_X * MOVE_STEP_X;
        moveBuf[3 * i + 1] = i * MOVE_STEP_Y;
        moveBuf[3 * i + 2] = i * MOVE_STEP_Z;
    }
    for (u32 i = NUM_INSTANCE / INSTANCE_INTERVAL_SHIFT;
         i < NUM_INSTANCE; i++)
    {
        // Pad with 0s till next baseInstance step
        moveBuf[3 * i]   = 0.0;
        moveBuf[3 * i + 1] = 0.0;
        moveBuf[3 * i + 2] = 0.0;
    }
}
static void SetupApplicationPipeline( nn::gfx::Pipeline::InfoType& pipelineInfo,
        nn::gfx::BlendTargetStateInfo* blendTargetStateInfoArray,
        nn::gfx::ColorTargetStateInfo* colorTargetStateInfoArray,
        nn::gfx::VertexAttributeStateInfo* vertexAttributeStateInfoArray,
        nn::gfx::VertexBufferStateInfo* vertexBufferStateInfoArray,
        int index,
        int attribId,
        int count )
{
    // Setup the pipeline for this object.
    SetDefaultBlendTargetStateInfo(blendTargetStateInfoArray[ 0 ]);
    SetDefaultColorTargetStateInfo(colorTargetStateInfoArray[ 0 ]);

    SetDefaultPipelineStateInfo( pipelineInfo, blendTargetStateInfoArray, 1, colorTargetStateInfoArray, 1,
        vertexAttributeStateInfoArray, attribId, vertexBufferStateInfoArray, attribId );


    pipelineInfo.SetShaderPtr( applicationState.theShader.GetShader() );

    size_t pipelineSize = nn::gfx::Pipeline::GetRequiredMemorySize( pipelineInfo );
    applicationState.instancedObjects[index]->pPipelineData = DEMOGfxAllocMEM2( pipelineSize, nn::gfx::Pipeline::RequiredMemoryInfo_Alignment );
    applicationState.instancedObjects[index]->pipeline.SetMemory( applicationState.instancedObjects[index]->pPipelineData, pipelineSize );
    applicationState.instancedObjects[index]->pipeline.Initialize( &DEMODevice, pipelineInfo );

    applicationState.instancedObjects[index]->numAttrs = attribId;
    applicationState.instancedObjects[index]->count = count;
    applicationState.instancedObjects[index]->numInstances = NUM_INSTANCE;
    applicationState.instancedObjects[index]->baseVertex = 0;
    applicationState.instancedObjects[index]->baseInstance = 0;
}

static int LoadObject(u32 index, const char *positionFile, const char *normalFile, const char *indexFile)
{
    nn::gfx::Pipeline::InfoType pipelineInfo;
    nn::gfx::BlendTargetStateInfo blendTargetStateInfoArray[ 1 ];
    nn::gfx::ColorTargetStateInfo colorTargetStateInfoArray[ 1 ];
    nn::gfx::VertexAttributeStateInfo vertexAttributeStateInfoArray[ NUM_ATTRIB ];
    nn::gfx::VertexBufferStateInfo vertexBufferStateInfoArray[ NUM_ATTRIB ];
    u32 attribId = 0;
    u32 count = 0;

    // Select the drawing function based on the parameters
    if (indexFile)
    {
        applicationState.instancedObjects[index]->draw = DrawIndexed;
    }
    else
    {
        applicationState.instancedObjects[index]->draw = DrawArray2;
    }

    // Load the Position Data
    if (positionFile)
    {
        void* data = DEMOGfxLoadAssetFile(positionFile,
            &applicationState.instancedObjects[index]->attribs[attribId].size);
#if !NN_GFX_IS_TARGET_GX
        swapBuffer(data, applicationState.instancedObjects[index]->attribs[attribId].size);
#endif
        SetupObjectData( vertexAttributeStateInfoArray[ attribId ],
            vertexBufferStateInfoArray[ applicationState.posLoc ],
            applicationState.instancedObjects[index]->attribs[attribId],
            applicationState.posLoc,
            nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Read,
            nn::gfx::AttributeFormat_32_32_32_Float,
            AttribTypePos, MODEL_VTX_STRIDE,
            data,
            applicationState.instancedObjects[index]->attribs[attribId].size,
            MODEL_OFFSET * sizeof(f32) );

        // Save the number of vertices
        count = applicationState.instancedObjects[index]->attribs[attribId].size /
                applicationState.instancedObjects[index]->attribs[attribId].stride;

        attribId++;
    }

    // Load the Normal Data
    if (normalFile)
    {
        void* data = DEMOGfxLoadAssetFile(normalFile,
            &applicationState.instancedObjects[index]->attribs[attribId].size);
#if !NN_GFX_IS_TARGET_GX
        swapBuffer(data, applicationState.instancedObjects[index]->attribs[attribId].size);
#endif
        SetupObjectData( vertexAttributeStateInfoArray[ attribId ],
            vertexBufferStateInfoArray[ applicationState.normLoc ],
            applicationState.instancedObjects[index]->attribs[attribId],
            applicationState.normLoc,
            nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Read,
            nn::gfx::AttributeFormat_32_32_32_Float, AttribTypeNormal, MODEL_NORM_STRIDE,
            data, applicationState.instancedObjects[index]->attribs[attribId].size,
            MODEL_OFFSET * sizeof(f32) );

        attribId++;
    }

    // Load the Index Data
    if (indexFile)
    {
        applicationState.instancedObjects[index]->indices = DEMOGfxLoadAssetFile(indexFile, &count);
#if !NN_GFX_IS_TARGET_GX
        swapBuffer(applicationState.instancedObjects[index]->indices, count);
#endif
        SetupBufferView( applicationState.instancedObjects[index]->indexBuffer,
            applicationState.instancedObjects[index]->indexBufferAddress,
            nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_Read,
            applicationState.instancedObjects[index]->indices, count, 0 );

        applicationState.instancedObjects[index]->indexFormat = nn::gfx::IndexFormat_Uint32;

        // Convert num bytes read to num indices
        count /= sizeof(u32);
    }
    else
    {
        applicationState.instancedObjects[index]->indices = NULL;
    }

    // Generate instancing data for each object
    {
        // Allocate extra space so that baseInstance works as expected
        f32 *colorBuf = (f32 *)DEMOGfxAllocMEM2( MODEL_CLR_STRIDE * NUM_INSTANCE, MAX_BUFFER_ALIGNMENT);

        CreateColorData( colorBuf );

        SetupObjectData( vertexAttributeStateInfoArray[ attribId ], vertexBufferStateInfoArray[ applicationState.clrLoc ],
            applicationState.instancedObjects[index]->attribs[attribId], applicationState.clrLoc,
            nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Read, nn::gfx::AttributeFormat_32_32_32_Float,
            AttribTypeColor, MODEL_CLR_STRIDE, colorBuf, MODEL_CLR_STRIDE * NUM_INSTANCE, MODEL_OFFSET * sizeof(f32) );

        vertexBufferStateInfoArray[ applicationState.clrLoc ].SetDivisor( INSTANCE_INTERVAL_COLOR );

        attribId++;

        // Setup the instanced position buffer, each row is a different object.
        // Normally we would only allocate
        // NUM_INSTANCE / INSTANCE_INTERVAL_SHIFT entries however if
        // baseInstance is used this will not work correctly when the interval
        // is > 1.
        f32 *moveBuf = (f32 *)DEMOGfxAllocMEM2(
                MODEL_MOVE_STRIDE * NUM_INSTANCE,
                MAX_BUFFER_ALIGNMENT);

        CreateMoveBuffer( moveBuf, index );

        SetupObjectData( vertexAttributeStateInfoArray[ attribId ],
            vertexBufferStateInfoArray[ applicationState.moveLoc ],
            applicationState.instancedObjects[index]->attribs[attribId],
            applicationState.moveLoc,
            nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Read,
            nn::gfx::AttributeFormat_32_32_32_Float,
            AttribTypeMove,
            MODEL_MOVE_STRIDE,
            moveBuf,
            MODEL_MOVE_STRIDE * (NUM_INSTANCE / INSTANCE_INTERVAL_SHIFT ),
            MODEL_OFFSET * sizeof(f32) );

        vertexBufferStateInfoArray[ applicationState.moveLoc ].SetDivisor( INSTANCE_INTERVAL_SHIFT );

        attribId++;
    }

    ASSERT(attribId <= NUM_ATTRIB);

    SetupApplicationPipeline( pipelineInfo,
        blendTargetStateInfoArray,
        colorTargetStateInfoArray,
        vertexAttributeStateInfoArray,
        vertexBufferStateInfoArray,
        index, attribId, count );

    return 1;
}

static int FreeObject(u32 index)
{
    u32 attribId = 0;

    for (attribId = 0; attribId < applicationState.instancedObjects[index]->numAttrs; attribId++)
    {
#ifdef USE_ATTRIB_PER_DRAW
        applicationState.instancedObjects[index]->attribs[attribId].buffer.Finalize( &DEMODevice );
#endif
        DEMOGfxFreeMEM2(applicationState.instancedObjects[index]->attribs[attribId].data);
    }

    if (applicationState.instancedObjects[index]->indices)
    {
        applicationState.instancedObjects[index]->indexBuffer.Finalize( &DEMODevice );
        DEMOGfxFreeMEM2(applicationState.instancedObjects[index]->indices);
    }

    applicationState.instancedObjects[index]->pipeline.Finalize(&DEMODevice);
    DEMOGfxFreeMEM2(applicationState.instancedObjects[index]->pPipelineData);

    delete applicationState.instancedObjects[index];
    applicationState.instancedObjects[index] = NULL;

    return 1;
}

static int ShaderInit()
{
    DEMOGfxLoadShadersFromFile(&applicationState.theShader, 0, SHADER_FILE_NAME[g_ShaderFileIdx]);

    // Uniform Location Lookup
    applicationState.uniformLoc = applicationState.theShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "Matrices");

    // Attribute Location Lookup
    applicationState.posLoc = applicationState.theShader.GetInterfaceSlot(nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_position");
    applicationState.clrLoc = applicationState.theShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_color");
    applicationState.normLoc = applicationState.theShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_normal");
    applicationState.moveLoc = applicationState.theShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_movement");

    return 1;
}

static void SetupViewportScissor()
{
    nn::gfx::ViewportScissorStateInfo info;
    nn::gfx::ViewportStateInfo viewportStateInfo;
    nn::gfx::ScissorStateInfo scissorStateInfo;
    size_t size;
    SetDefaultViewportScissorInfo(info, viewportStateInfo, scissorStateInfo);

    size = nn::gfx::ViewportScissorState::GetRequiredMemorySize( info );
    if ( size )
    {
        applicationState.pViewportScissorData = DEMOGfxAllocMEM2( size, nn::gfx::ViewportScissorState::RequiredMemoryInfo_Alignment );
        applicationState.viewportScissorState.SetMemory( applicationState.pViewportScissorData, size );
    }
    applicationState.viewportScissorState.Initialize( &DEMODevice, info );
}

#ifndef USE_ATTRIB_PER_DRAW
static void ConsolidateAttributeBuffers()
{
    // Counters for data compaction
    u32 vertexOffset = 0;
    u32 normOffset = 0;
    u32 colorOffset = 0;
    u32 moveOffset = 0;

    // Compact the vertex, normal, color and movement buffers
    // into one buffer
    if (applicationState.vertexSize)
    {
        applicationState.vertexData = DEMOGfxAllocMEM2(applicationState.vertexSize, MAX_BUFFER_ALIGNMENT);
    }
    if (applicationState.normSize)
    {
        applicationState.normData = DEMOGfxAllocMEM2(applicationState.normSize, MAX_BUFFER_ALIGNMENT);
    }
    if (applicationState.colorSize)
    {
        applicationState.colorData = DEMOGfxAllocMEM2(applicationState.colorSize, MAX_BUFFER_ALIGNMENT);
    }
    if (applicationState.moveSize)
    {
        applicationState.moveData = DEMOGfxAllocMEM2(applicationState.moveSize, MAX_BUFFER_ALIGNMENT);
    }

    for (u32 i = 0; i < applicationState.NUM_MODEL; i++)
    {
        // baseVertex is based on the vertex buffer. Since
        // vertexOffset and normalOffset should have the same baseVertex,
        // set it to the vertex offset in vertices
        applicationState.instancedObjects[i]->baseVertex = (vertexOffset / sizeof(f32) / 3);

        for (u32 j = 0; j < applicationState.instancedObjects[i]->numAttrs; j++)
        {
            // Copy the object's data to the appropriate buffer
            switch (applicationState.instancedObjects[i]->attribs[j].type)
            {
                case AttribTypePos:
                     memcpy((u8*)applicationState.vertexData + vertexOffset,
                            applicationState.instancedObjects[i]->attribs[j].data,
                            applicationState.instancedObjects[i]->attribs[j].size);
                     vertexOffset += applicationState.instancedObjects[i]->attribs[j].size;
                     break;
                case AttribTypeNormal:
                     memcpy((u8*)applicationState.normData + normOffset,
                            applicationState.instancedObjects[i]->attribs[j].data,
                            applicationState.instancedObjects[i]->attribs[j].size);
                     normOffset += applicationState.instancedObjects[i]->attribs[j].size;
                     break;
                case AttribTypeColor:
                     memcpy((u8*)applicationState.colorData + colorOffset,
                            applicationState.instancedObjects[i]->attribs[j].data,
                            applicationState.instancedObjects[i]->attribs[j].size);
                     colorOffset += applicationState.instancedObjects[i]->attribs[j].size;
                     break;
                case AttribTypeMove:
                     memcpy((u8*)applicationState.moveData + moveOffset,
                            applicationState.instancedObjects[i]->attribs[j].data,
                            applicationState.instancedObjects[i]->attribs[j].size);
                     moveOffset += applicationState.instancedObjects[i]->attribs[j].size;
                     break;
                default:
                     break;
            }
        }

        // The baseInstance should be set as the instance ID offset
        // to the start of the instancing data for that primitive.
        //
        // This must always be used otherwise the primitves will be drawn
        // using the same instance data for object 0.
        applicationState.instancedObjects[i]->baseInstance = NUM_INSTANCE * i;
    }

    // Vertex Buffer/View
    SetupBufferView( applicationState.vertexBuffer,
            applicationState.vertexBufferAddress,
            nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Read,
            applicationState.vertexData, applicationState.vertexSize, 0 );

    // Color Buffer/View
    SetupBufferView( applicationState.colorBuffer,
            applicationState.colorBufferAddress,
            nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Read,
            applicationState.colorData, applicationState.colorSize, 0 );

    // Move Buffer/View
    SetupBufferView( applicationState.moveBuffer,
            applicationState.moveBufferAddress,
            nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Read,
            applicationState.moveData, applicationState.moveSize, 0 );

    // Normal Buffer/View
    SetupBufferView( applicationState.normBuffer,
            applicationState.normBufferAddress,
            nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Read,
            applicationState.normData, applicationState.normSize, 0 );
}
#endif // !USE_ATTRIB_PER_DRAW

// The init function for the rendering portions of this app
static int SceneInit()
{
    applicationState.s_ticks = 0;
    applicationState.s_yrad = 0.0f;
    applicationState.s_xrad = 0.0f;

    // Initialize random number seed to get consistent values back
    DEMOSRand(0);

    // Load the VS/PS/FS
    if (!ShaderInit())
    {
        return 0;
    }

    // Load each of the models
    for (u32 i = 0; i < applicationState.NUM_MODEL; i++)
    {
        // Load the objects
        if (!LoadObject(i,
                        MODEL_FILES[i][0],
                        MODEL_FILES[i][1],
                        MODEL_FILES[i][2]))
        {
            return 0;
        }

#ifndef USE_ATTRIB_PER_DRAW
        // Keep track of total space needed for compacting the primitive buffers
        for (u32 j = 0; j < applicationState.instancedObjects[i]->numAttrs; j++)
        {
            switch (applicationState.instancedObjects[i]->attribs[j].type)
            {
                case AttribTypePos:
                     applicationState.vertexSize += applicationState.instancedObjects[i]->attribs[j].size;
                     break;
                case AttribTypeNormal:
                     applicationState.normSize += applicationState.instancedObjects[i]->attribs[j].size;
                     break;
                case AttribTypeColor:
                     applicationState.colorSize += applicationState.instancedObjects[i]->attribs[j].size;
                     break;
                case AttribTypeMove:
                     applicationState.moveSize += applicationState.instancedObjects[i]->attribs[j].size;
                     break;
                default:
                     break;
            }
        }
#endif // !USE_ATTRIB_PER_DRAW
    }

#ifndef USE_ATTRIB_PER_DRAW
    ConsolidateAttributeBuffers();
#endif // !USE_ATTRIB_PER_DRAW

    // Swap the first and last to test the resetting of baseInstance
    InstancedObject* tmp = applicationState.instancedObjects[0];
    applicationState.instancedObjects[0] = applicationState.instancedObjects[applicationState.NUM_MODEL - 1];
    tmp->draw = DrawArray;
    applicationState.instancedObjects[applicationState.NUM_MODEL - 1] = tmp;

    // Setup the uniform buffer/view
    applicationState.pMatricesData = (Matrices*) DEMOGfxAllocMEM2( sizeof(Matrices), MAX_BUFFER_ALIGNMENT );
    SetupBufferView( applicationState.matricesBuffer,
            applicationState.matricesBufferAddress,
            nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_Read,
            applicationState.pMatricesData, sizeof(Matrices), 0 );

    CameraInit(applicationState.matricesBuffer);

    //  Setup the viewport/scissor
    SetupViewportScissor();

    return 1;
}

static void PrintInfo()
{
    // Set Demo Font String
    DEMOFontPrintf(2, 1, "<Instance Demo>");
}

// The draw function for the rendering portions of this app
static int SceneDraw()
{
    DEMOGfxBeforeRender();
    nn::gfx::ColorTargetView* pScanBufferView = DEMOGetColorBufferView();

    // Clear buffers
    DEMOCommandBuffer.ClearColor(pScanBufferView, 0.1f, 0.1f, 0.1f, 1.0f, NULL);
    DEMOCommandBuffer.ClearDepthStencil(&DEMODepthBufferView, 1.0, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL);

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif

    DEMOGfxSetDefaultRenderTarget();
    DEMOCommandBuffer.SetViewportScissorState( &applicationState.viewportScissorState );
    // Update Model Matrix Uniform
    ModelTick(applicationState.matricesBuffer);
    DEMOCommandBuffer.SetConstantBuffer(applicationState.uniformLoc, nn::gfx::ShaderStage_Vertex, applicationState.matricesBufferAddress, sizeof(Matrices));
#ifndef USE_ATTRIB_PER_DRAW
    // Set the attribute buffers once
    DEMOCommandBuffer.SetVertexBuffer(applicationState.posLoc, applicationState.vertexBufferAddress, MODEL_VTX_STRIDE, applicationState.vertexSize);
    if (applicationState.normLoc > 0)
    {
        DEMOCommandBuffer.SetVertexBuffer(applicationState.normLoc, applicationState.normBufferAddress, MODEL_NORM_STRIDE, applicationState.normSize);
    }
    DEMOCommandBuffer.SetVertexBuffer(applicationState.clrLoc, applicationState.colorBufferAddress, MODEL_CLR_STRIDE, applicationState.colorSize);
    DEMOCommandBuffer.SetVertexBuffer(applicationState.moveLoc, applicationState.moveBufferAddress, MODEL_MOVE_STRIDE, applicationState.moveSize);
#endif // !USE_ATTRIB_PER_DRAW
    // Draw all of the instanced objects
    for (u32 i = 0; i < applicationState.NUM_MODEL; i++)
    {
        applicationState.instancedObjects[i]->draw(applicationState.instancedObjects[i]);
    }
    // Draw Infomation
    PrintInfo();

    DEMOGfxDoneRender();

    applicationState.s_ticks++;

    return 1;
}

// extern "C" void nnMain()
TEST(GfxBaseInstance, Run)
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    DEMOInit();
    DEMOTestInit(argc, argv);
    DEMOGfxInit(argc, argv);
    DEMOFontInit();
    DEMOTestIsUseHlslccGlsl() ? g_ShaderFileIdx = 1 : g_ShaderFileIdx = 0;

    InitAppState();

    // Process parameters
    for (int i = 0; i < argc; i++)
    {
        char *pStr;

        if ( (pStr = strstr(argv[i], "NUM_MODELS=")) != NULL)
        {
            applicationState.NUM_MODEL = (u32) atoi(pStr + strlen("NUM_MODELS="));
            ASSERT(applicationState.NUM_MODEL <= MAX_MODELS);
        }
    }

    SceneInit();

    while (DEMOIsRunning())
    {
        SceneDraw();
    }

    // Free shaders
    DEMOGfxFreeShaders( &applicationState.theShader );

    for (u32 i = 0; i < applicationState.NUM_MODEL; i++)
    {
        FreeObject(i);
    }

#ifndef USE_ATTRIB_PER_DRAW
    // Free consoldiated buffers
    if (applicationState.vertexData)
    {
        applicationState.vertexBuffer.Finalize(&DEMODevice);
        DEMOGfxFreeMEM2(applicationState.vertexData);
    }
    if (applicationState.colorData)
    {
        applicationState.colorBuffer.Finalize(&DEMODevice);
        DEMOGfxFreeMEM2(applicationState.colorData);
    }
    if (applicationState.normData)
    {
        applicationState.normBuffer.Finalize(&DEMODevice);
        DEMOGfxFreeMEM2(applicationState.normData);
    }
    if (applicationState.moveData)
    {
        applicationState.moveBuffer.Finalize(&DEMODevice);
        DEMOGfxFreeMEM2(applicationState.moveData);
    }
#endif // !USE_ATTRIB_PER_DRAW

    applicationState.matricesBuffer.Finalize(&DEMODevice);
    if (applicationState.pMatricesData)
    {
        DEMOGfxFreeMEM2(applicationState.pMatricesData);
    }

    applicationState.viewportScissorState.Finalize( &DEMODevice );
    if ( applicationState.pViewportScissorData )
    {
        DEMOGfxFreeMEM2( applicationState.pViewportScissorData );
    }

    DEMOFontShutdown();
    DEMOTestShutdown();
    DEMOGfxShutdown();
    DEMOShutdown();

    SUCCEED();
}
