﻿// NOLINT(style/copyright)
/*
 * Copyright 2014-2016 NVIDIA Corporation.  All rights reserved.
 *
 * NOTICE TO USER:
 *
 * This source code is subject to NVIDIA ownership rights under U.S. and
 * international Copyright laws.
 *
 * This software and the information contained herein is PROPRIETARY and
 * CONFIDENTIAL to NVIDIA and is being provided under the terms and conditions
 * of a form of NVIDIA software license agreement.
 *
 * NVIDIA MAKES NO REPRESENTATION ABOUT THE SUITABILITY OF THIS SOURCE
 * CODE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS OR
 * IMPLIED WARRANTY OF ANY KIND.  NVIDIA DISCLAIMS ALL WARRANTIES WITH
 * REGARD TO THIS SOURCE CODE, INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY, NONINFRINGEMENT, AND FITNESS FOR A PARTICULAR PURPOSE.
 * IN NO EVENT SHALL NVIDIA BE LIABLE FOR ANY SPECIAL, INDIRECT, INCIDENTAL,
 * OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS,  WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION,  ARISING OUT OF OR IN CONNECTION WITH THE USE
 * OR PERFORMANCE OF THIS SOURCE CODE.
 *
 * U.S. Government End Users.   This source code is a "commercial item" as
 * that term is defined at  48 C.F.R. 2.101 (OCT 1995), consisting  of
 * "commercial computer  software"  and "commercial computer software
 * documentation" as such terms are  used in 48 C.F.R. 12.212 (SEPT 1995)
 * and is provided to the U.S. Government only as a commercial end item.
 * Consistent with 48 C.F.R.12.212 and 48 C.F.R. 227.7202-1 through
 * 227.7202-4 (JUNE 1995), all U.S. Government End Users acquire the
 * source code with only those rights set forth herein.
 *
 * Any use of this source code in individual and commercial software must
 * include, in the user documentation and internal comments to the code,
 * the above Disclaimer and U.S. Government End Users Notice.
 */

#pragma once

#include <nvperfapi.h>
#include <iostream>
#include <string>
#include <cstdlib>

#ifndef ENABLE_PERFWORKS_PROFILING
#define ENABLE_PERFWORKS_PROFILING 1 // NOLINT(preprocessor/const)
#endif

#define DUMP_NVPA_CALLS 0 // NOLINT(preprocessor/const)
#if DUMP_NVPA_CALLS
#define DUMP(X) std::cout << #X << std::endl
#else
#define DUMP(X)
#endif

#define NVPA_STATUS_VAR status_L##__LINE__

#define INSTRUMENT_NVPA_IMPL(X, file, line)                             \
    DUMP(X);                                                            \
    if (NVPA_Status NVPA_STATUS_VAR = (X))                              \
        std::cout                                                       \
            << file << ":" << line << ": \""                            \
            << #X "\" failed with return code " << NVPA_STATUS_VAR      \
            << " (" << ::NvPerfApiSamples::detail::ToStr(NVPA_STATUS_VAR) << ")\n"  \
            << std::endl,                                               \
    exit(1)

#if ENABLE_PERFWORKS_PROFILING
#define INSTRUMENT_NVPA(X)                                              \
    INSTRUMENT_NVPA_IMPL(X, __FILE__, __LINE__)
#else
#define INSTRUMENT_NVPA(X)
#endif

#define CHECK_NVPA_IMPL(X, expectedStatus, permutation, file, line)     \
    [&](){                                                              \
        DUMP(X);                                                        \
        do {                                                            \
            NVPA_Status NVPA_STATUS_VAR = (X);                          \
            if (NVPA_STATUS_VAR != expectedStatus)                      \
            {                                                           \
                std::cout                                               \
                    << file << ":" << line << ": permutation "          \
                    << permutation << ": \""                            \
                    << #X "\" failed with return code "                 \
                    << ::NvPerfApiSamples::detail::ToStr(NVPA_STATUS_VAR)  \
                    << " (expected "                                    \
                    << ::NvPerfApiSamples::detail::ToStr(expectedStatus)      \
                    << ")\n"                                            \
                    << std::endl;                                       \
                    exit(1);                                            \
            }                                                           \
        } while (false);                                                \
                                                                        \
        return true;                                                    \
    }()

#define CHECK_NVPA_STATUS(X, expectedStatus, permutation)               \
    CHECK_NVPA_IMPL(X, expectedStatus, permutation, __FILE__, __LINE__)

namespace NvPerfApiSamples {

    namespace detail {

        inline std::string ToStr(NVPA_Status s)
        {
            switch (s)
            {
#define CASE(x) case x: return #x
                CASE(NVPA_STATUS_SUCCESS);
                CASE(NVPA_STATUS_ERROR);
                CASE(NVPA_STATUS_INTERNAL_ERROR);
                CASE(NVPA_STATUS_NOT_INITIALIZED);
                CASE(NVPA_STATUS_NOT_LOADED);
                CASE(NVPA_STATUS_FUNCTION_NOT_FOUND);
                CASE(NVPA_STATUS_NOT_SUPPORTED);
                CASE(NVPA_STATUS_NOT_IMPLEMENTED);
                CASE(NVPA_STATUS_INVALID_ARGUMENT);
                CASE(NVPA_STATUS_INVALID_METRIC_ID);
                CASE(NVPA_STATUS_DRIVER_NOT_LOADED);
                CASE(NVPA_STATUS_OUT_OF_MEMORY);
                CASE(NVPA_STATUS_INVALID_THREAD_STATE);
                CASE(NVPA_STATUS_FAILED_CONTEXT_ALLOC);
                CASE(NVPA_STATUS_UNSUPPORTED_GPU);
                CASE(NVPA_STATUS_INSUFFICIENT_DRIVER_VERSION);
                CASE(NVPA_STATUS_OBJECT_NOT_REGISTERED);
                CASE(NVPA_STATUS_INSUFFICIENT_PRIVILEGE);
                CASE(NVPA_STATUS_INVALID_CONTEXT_STATE);
                CASE(NVPA_STATUS_INVALID_OBJECT_STATE);
                CASE(NVPA_STATUS_RESOURCE_UNAVAILABLE);
                CASE(NVPA_STATUS_DRIVER_LOADED_TOO_LATE);
                CASE(NVPA_STATUS_INSUFFICIENT_SPACE);
                CASE(NVPA_STATUS_OBJECT_MISMATCH);
                CASE(NVPA_STATUS__COUNT);
#undef CASE
                default: return "[Unknown NVPA_Status value]";
            }
        }

        inline std::string ToStr(NVPA_MetricEnableError reason)
        {
            switch(reason)
            {
#define CASE(x) case x: return #x
                CASE(NVPA_METRIC_ENABLE_ERROR_NONE);
                CASE(NVPA_METRIC_ENABLE_ERROR_UNKNOWN);
                CASE(NVPA_METRIC_ENABLE_ERROR_TOO_MANY_PASSES);
                CASE(NVPA_METRIC_ENABLE_ERROR_NOT_IMPLEMENTED);
                CASE(NVPA_METRIC_ENABLE_ERROR__COUNT);
                default: return "[Unknown NVPA_MetricEnableError value]";
#undef CASE
            }
        }

        inline std::string ToStr(NVPA_MetricValueError reason)
        {
            switch(reason)
            {
#define CASE(x) case x: return #x
                CASE(NVPA_METRIC_VALUE_ERROR_NONE);
                CASE(NVPA_METRIC_VALUE_ERROR_UNINITIALIZED);
                CASE(NVPA_METRIC_VALUE_ERROR_INVALID);
                CASE(NVPA_METRIC_VALUE_ERROR_OUT_OF_MEMORY);
                CASE(NVPA_METRIC_VALUE_ERROR_OVERFLOWED);
                CASE(NVPA_METRIC_VALUE_ERROR_DIVIDE_BY_ZERO);
                CASE(NVPA_METRIC_VALUE_ERROR_NOT_AVAILABLE);
                CASE(NVPA_METRIC_VALUE_ERROR_NOT_READY);
                CASE(NVPA_METRIC_VALUE_ERROR_NOT_PERMITTED);
                CASE(NVPA_METRIC_VALUE_ERROR_MIN_MAX_NOT_COLLECTIBLE);
                default: return "[Unknown NVPA_MetricValueError value]";
#undef CASE
            }
        }

        template <typename T>
        inline void Check(T value, std::string const& msg)
        {
            if (!value)
            {
                std::cout << "Check(" << value << ") failed; " << msg << "\n";
                exit(1);
            }
        }

        template <typename T>
        inline void CheckNot(T value, std::string const& msg)
        {
            if (value)
            {
                std::cout << "CheckNot(" << value << ") failed; " << msg << "\n";
                exit(1);
            }
        }

        template <typename T>
        inline void CheckEq(T value1, T value2, std::string const& msg)
        {
            if (value1 != value2)
            {
                std::cout << "CheckEq(" << value1 << ", " << value2 << ") failed; " << msg << "\n";
                exit(1);
            }
        }

        template <typename T>
        inline void CheckNeq(T value1, T value2, std::string const& msg)
        {
            if (value1 == value2)
            {
                std::cout << "CheckNeq(" << value1 << ", " << value2 << ") failed; " << msg << "\n";
                exit(1);
            }
        }

        template <typename T>
        inline void CheckLE(T value1, T value2, std::string const& msg)
        {
            if (value2 < value1)
            {
                std::cout << "CheckLE(" << value1 << ", " << value2 << ") failed; " << msg << "\n";
                exit(1);
            }
        }
    }
}
