﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

////===========================================================================
///  demoSystem.c
///
///     This is system code for the demo library.
///
////===========================================================================
#include <gfx/demo.h>
#if NN_GFX_IS_TARGET_GX
#include <cafe/procui.h>
#else
#include <nn/time/time_Api.h>
#include <nn/os.h>
#endif

#include <nn/nn_Macro.h>
#include <nn/mem.h>

#include <cstdio>
#include <cstdlib>
#include <cstdarg>

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_WIN32 )
NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_FROM_WINDOWS_SDK_HEADERS
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nn/nn_Windows.h>
NN_PRAGMA_POP_WARNINGS
#endif

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
#include <nn/init.h>
#include <nn/fs.h>
#endif

void* s_pDemoSystemMemoryMEM2 = NULL;
#if NN_BUILD_CONFIG_OS_SUPPORTS_WIN32
size_t s_DemoSystemMemoryMEM2Size = 512 * 1024 * 1024;
#else
size_t s_DemoSystemMemoryMEM2Size = 256 * 1024 * 1024;
#endif
static nn::mem::StandardAllocator s_DemoSystemMemoryMEM2;

void (*DEMOReleaseCallback)(void);

#if NN_GFX_IS_TARGET_GX
extern unsigned int TCoreMEM2Limit();
extern unsigned int TCoreMEM2Limit()
{
    // Limit TCore MEM2 Usage to: 12MB
    // TOD0: Pick a less randmon TCore MEM limit
    return 0x00C00000;
}
#endif

u32 DEMOSaveCallback(void* )
{
#if NN_GFX_IS_TARGET_GX
    // Finish any saving before calling OSSavesDone_ReadyToRelease

    OSSavesDone_ReadyToRelease();
#endif
    return 0;
}

void DEMOEmptyReleaseCallback()
{
#if NN_GFX_IS_TARGET_GX
    // If fading or something else is desired, this call can be postponed
    ProcUIDrawDoneRelease();
#endif
}

void DEMOSetReleaseCallback(DEMOReleaseCallbackFunc func)
{
    DEMOReleaseCallback = func;
}

static BOOL gDemoRunningFlag;
static int sMainCore;

static DEMODefaultAllocateFunc gpAllocFunc = NULL;
static DEMODefaultFreeFunc     gpFreeFunc  = NULL;

void* DefaultAllocator( size_t byteCount, size_t alignment )
{
    return s_DemoSystemMemoryMEM2.Allocate(byteCount, alignment);
}

void DefaultFreeFunc(void* pMem)
{
    s_DemoSystemMemoryMEM2.Free( pMem );
}

#if !NN_GFX_IS_TARGET_GX
static void* Allocate(size_t size)
{
    return std::malloc( size );// DEMOAlloc( size );
}

static void Deallocate(void* p, size_t size)
{
    NN_UNUSED( size );
    std::free( p );// DEMOFree( p );
}

#endif


void DEMOInit()
{
    OSReport("DEMO: Build date - %s %s\n", __DATE__, __TIME__);
    gDemoRunningFlag = TRUE;
#if NN_GFX_IS_TARGET_GX
    sMainCore = OSGetCoreId();

    ProcUIInitEx(&DEMOSaveCallback, NULL);
#else
    nn::os::SetThreadCoreMask( nn::os::GetCurrentThread(), 0, 1 );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::Initialize() );
#endif
    DEMOPadInit();
    DEMOReleaseCallback = &DEMOEmptyReleaseCallback;

    if ( NULL == gpAllocFunc && NULL == gpFreeFunc )
    {
#if NN_GFX_IS_TARGET_GX
        s_pDemoSystemMemoryMEM2 = MEMAllocFromDefaultHeapEx(s_DemoSystemMemoryMEM2Size, PPC_IO_BUFFER_ALIGN);
#else
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
        s_pDemoSystemMemoryMEM2 = aligned_alloc( sizeof(ptrdiff_t), s_DemoSystemMemoryMEM2Size );
        //DEMOAssert( !posix_memalign( &s_pDemoSystemMemoryMEM2, s_DemoSystemMemoryMEM2Size, PPC_IO_BUFFER_ALIGN ) );
#else
        s_pDemoSystemMemoryMEM2 = _aligned_malloc( s_DemoSystemMemoryMEM2Size, sizeof(ptrdiff_t) );
#endif
#endif
        DEMOAssert( NULL != s_pDemoSystemMemoryMEM2 );

        s_DemoSystemMemoryMEM2.Initialize( s_pDemoSystemMemoryMEM2, s_DemoSystemMemoryMEM2Size );
        DEMOSetDefaultAllocator( DefaultAllocator, DefaultFreeFunc );
    }

#if !NN_GFX_IS_TARGET_GX
    nn::fs::SetAllocator( Allocate, Deallocate );
#endif
}

void DEMOSetMainCore(int core)
{
    sMainCore = core;
}

void DEMOStopRunning()
{
    gDemoRunningFlag = FALSE;
}

void DEMOShutdown()
{
    DEMOStopRunning();
#if NN_GFX_IS_TARGET_GX
    DEMOPadShutdown();
#endif
    gpAllocFunc = NULL;
    gpFreeFunc  = NULL;

    if ( NULL != s_pDemoSystemMemoryMEM2 )
    {
        s_DemoSystemMemoryMEM2.Finalize();
#if NN_GFX_IS_TARGET_GX
        MEMFreeToDefaultHeap( s_pDemoSystemMemoryMEM2 );
#else
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
        free( s_pDemoSystemMemoryMEM2 );
#else
        _aligned_free( s_pDemoSystemMemoryMEM2 );
#endif
#endif
    }

    OSReport("\nEnd of demo\n");
}

BOOL DEMOIsRunning()
{
#if NN_GFX_IS_TARGET_GX
    // Only do extra on the proper core
    if (OSGetCoreId() != sMainCore)
    {
        ProcUISubProcessMessages(TRUE);
        return gDemoRunningFlag;
    }
#endif

    // See if test action needs to be taken
    DEMOTestCheck();

#if NN_GFX_IS_TARGET_GX
    // See if any process switching or exiting is going on
    ProcUIStatus status;

    // Block if we're in the background
    status = ProcUIProcessMessages(TRUE);

        // Handle exiting or release
    if (status == PROCUI_STATUS_EXIT)
    {
        DEMOStopRunning();
    } else if (status == PROCUI_STATUS_RELEASING)
    {
            DEMOReleaseCallback();
    }
#else
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::Finalize() );
#endif

    if (!gDemoRunningFlag)
    {
            // This function can only be called when in the foreground
            // This must be called prior to freeing any shaders or other memory the GPU might be using
        DEMOGfxDrawDone();

#if NN_GFX_IS_TARGET_GX
        ProcUIShutdown();
#endif
    }

    return gDemoRunningFlag;
}

void DEMOSetDefaultAllocator(DEMODefaultAllocateFunc pfnAlloc, DEMODefaultFreeFunc pfnFree)
{
    gpAllocFunc = pfnAlloc;
    gpFreeFunc  = pfnFree;
}

void DEMOGetDefaultAllocator(DEMODefaultAllocateFunc *ppfnAlloc, DEMODefaultFreeFunc *ppfnFree)
{
    if (ppfnAlloc)
    {
        *ppfnAlloc = gpAllocFunc;
    }
    if (ppfnFree)
    {
        *ppfnFree  = gpFreeFunc;
    }
}

void DEMODumper()
{
    s_DemoSystemMemoryMEM2.Dump();
}

void* DEMOAlloc(size_t size)
{
    return gpAllocFunc(size, PPC_IO_BUFFER_ALIGN);
}

void* DEMOAllocEx( size_t size, size_t align )
{
    return gpAllocFunc(size, align);
}

void DEMOFree(void* ptr)
{
    gpFreeFunc(ptr);
}

size_t DEMOGetUsedMemory()
{
    if ( gpAllocFunc == DefaultAllocator )
    {
        return s_DemoSystemMemoryMEM2.GetTotalFreeSize();
    }

    return 0;
}

// ----------------------------------------------------------------------------
//  Random
// ----------------------------------------------------------------------------

static u32 holdrand = 0;

void DEMOSRand( u32 seed )
{
    holdrand = seed;
}

f32 DEMOFRand()
{
    return ( f32 ) DEMORand() / DEMO_RAND_MAX;
}

u32 DEMORand()
{
    return ( ( ( holdrand = holdrand * 214013L + 2531011L ) >> 16 ) & 0xffff );
}

// ----------------------------------------------------------------------------
//  Print
// ----------------------------------------------------------------------------

#if !defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
void DEMOPrintf( const char* msg, ... )
{
    va_list args;
    va_start(args, msg);
    vprintf( msg, args );
    va_end(args);
}
#endif

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_WIN32 )
u64 OSGetTime()
{
    LARGE_INTEGER li;
    QueryPerformanceCounter( &li );

    return li.QuadPart;
}

u64 OSTicksToMilliseconds( u64 x )
{
    LARGE_INTEGER frequency;

    QueryPerformanceFrequency( &frequency );
    return x * 1000 / frequency.QuadPart;
}

u64 OSTicksToMicroseconds( u64 x )
{
    LARGE_INTEGER frequency;

    QueryPerformanceFrequency( &frequency );
    return x * 1000000 / frequency.QuadPart;
}
#elif defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
u64 OSGetTime()
{
    nn::os::Tick t = nn::os::GetSystemTick();
    return static_cast< u64 >( t.GetInt64Value() );
}

u64 OSTicksToMilliseconds( u64 x )
{
    nn::os::Tick t( static_cast< int64_t >( x ) );
    nn::TimeSpan ts = ConvertToTimeSpan(t);
    return static_cast< u64 >( ts.GetMilliSeconds() );
}
u64 OSTicksToMicroseconds( u64 x )
{
    nn::os::Tick t( static_cast< int64_t >( x ) );
    nn::TimeSpan ts = ConvertToTimeSpan(t);
    return static_cast< u64 >( ts.GetMicroSeconds() );
}
#endif
