﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

////===========================================================================
///  demoPad.c
///
///     This is a pad code for the demo library.
///
////===========================================================================

#include <gfx/demo.h>

#include <cafe/pad.h>

#ifdef _DEBUG
#include <cstdio>
#endif

// -------------------------------------------------------------------------
//  Global Values
// -------------------------------------------------------------------------
// Demo Pad Status
DemoPadStatus  g_DemoPad[ DEMO_PAD_MAX_CONTROLLERS ];

// -------------------------------------------------------------------------
// Pad Status
static PADStatus g_Pad[PAD_MAX_CONTROLLERS]; // internal use only

// Pad chanel mask
static u32 g_PadChanMask[PAD_MAX_CONTROLLERS] =
{
    PAD_CHAN0_BIT, PAD_CHAN1_BIT, PAD_CHAN2_BIT, PAD_CHAN3_BIT
};

// ------------------------------------------------------------------------
//  Macro definitions
// ------------------------------------------------------------------------
#define padButtonDown(buttonLast, button)   \
    ((u16) (((buttonLast) ^ (button)) & (button)))

#define padButtonUp(buttonLast, button)     \
    ((u16) (((buttonLast) ^ (button)) & (buttonLast)))

//--------------------------------------------------------------------------
//   Forward references
//--------------------------------------------------------------------------
static void SetPadInfo ( PADStatus* pad, DemoPadStatus* dmpad );

// -------------------------------------------------------------------------
//  Internal Fuctions
// -------------------------------------------------------------------------
// Attaches informations such as down/up, stick direction.
static void SetPadInfo ( PADStatus* pad, DemoPadStatus* dmpad )
{
    u16  dirs;

    if ( pad->err != PAD_ERR_TRANSFER )
    {
        // Detects which direction is the stick(s) pointing.
        // This can be used when we want to use a stick as direction pad.
        dirs = 0;
        if ( pad->stickX    < - DEMO_STICK_THRESHOLD )
        {
            dirs |= DEMO_STICK_LEFT;
        }
        if ( pad->stickX    >   DEMO_STICK_THRESHOLD )
        {
            dirs |= DEMO_STICK_RIGHT;
        }
        if ( pad->stickY    < - DEMO_STICK_THRESHOLD )
        {
            dirs |= DEMO_STICK_DOWN;
        }
        if ( pad->stickY    >   DEMO_STICK_THRESHOLD )
        {
            dirs |= DEMO_STICK_UP;
        }
        if ( pad->substickX < - DEMO_STICK_THRESHOLD )
        {
            dirs |= DEMO_SUBSTICK_LEFT;
        }
        if ( pad->substickX >   DEMO_STICK_THRESHOLD )
        {
            dirs |= DEMO_SUBSTICK_RIGHT;
        }
        if ( pad->substickY < - DEMO_STICK_THRESHOLD )
        {
            dirs |= DEMO_SUBSTICK_DOWN;
        }
        if ( pad->substickY >   DEMO_STICK_THRESHOLD )
        {
            dirs |= DEMO_SUBSTICK_UP;
        }

        // Get the direction newly detected / released
        dmpad->dirsNew      = padButtonDown(dmpad->dirs, dirs);
        dmpad->dirsReleased = padButtonUp(dmpad->dirs, dirs);
        dmpad->dirs         = dirs;

        // Get DOWN/UP status of all buttons
        dmpad->buttonDown = padButtonDown(dmpad->button, pad->button);
        dmpad->buttonUp   = padButtonUp(dmpad->button, pad->button);

        // Get delta of analogs
        dmpad->stickDeltaX = (s16)(pad->stickX - dmpad->stickX);
        dmpad->stickDeltaY = (s16)(pad->stickY - dmpad->stickY);
        dmpad->substickDeltaX = (s16)(pad->substickX - dmpad->substickX);
        dmpad->substickDeltaY = (s16)(pad->substickY - dmpad->substickY);

        dmpad->lastButton = dmpad->button;

        // Copy current status into DEMOPadStatus field
        dmpad->button = pad->button;
        dmpad->stickX = pad->stickX;
        dmpad->stickY = pad->stickY;
        dmpad->substickX = pad->substickX;
        dmpad->substickY = pad->substickY;
        dmpad->triggerLeft  = pad->triggerLeft;
        dmpad->triggerRight = pad->triggerRight;
        dmpad->analogA = pad->analogA;
        dmpad->analogB = pad->analogB;
        dmpad->err = pad->err;
    }
    else
    {
        // Get the direction newly detected / released
        dmpad->dirsNew = dmpad->dirsReleased = 0;

        // Get DOWN/UP status of all buttons
        dmpad->buttonDown = dmpad->buttonUp = 0;

        // Get delta of analogs
        dmpad->stickDeltaX =    dmpad->stickDeltaY    = 0;
        dmpad->substickDeltaX = dmpad->substickDeltaY = 0;
    }
}

// -------------------------------------------------------------------------
//  Initialize
// -------------------------------------------------------------------------
void DEMOPadInit()
{
    u32 i;

    // Pad init
    PADInit();

    // Reset exported pad status
    for (i = 0 ; i < DEMO_PAD_MAX_CONTROLLERS ; i++)
    {
        g_DemoPad[i].button = 0;
        g_DemoPad[i].stickX = 0;
        g_DemoPad[i].stickY = 0;
        g_DemoPad[i].substickX = 0;
        g_DemoPad[i].substickY = 0;
        g_DemoPad[i].triggerLeft = 0;
        g_DemoPad[i].triggerRight = 0;
        g_DemoPad[i].analogA = 0;
        g_DemoPad[i].analogB = 0;
        g_DemoPad[i].err = 0;
        g_DemoPad[i].lastButton = 0;
        g_DemoPad[i].buttonDown = 0;
        g_DemoPad[i].buttonUp = 0;
        g_DemoPad[i].dirs = 0;
        g_DemoPad[i].dirsNew = 0;
        g_DemoPad[i].dirsReleased = 0;
        g_DemoPad[i].stickDeltaX = 0;
        g_DemoPad[i].stickDeltaY = 0;
        g_DemoPad[i].substickDeltaX = 0;
        g_DemoPad[i].substickDeltaY = 0;
    }
}

// -------------------------------------------------------------------------
//  ShutDown
// -------------------------------------------------------------------------
void DEMOPadShutdown()
{
    u32 i;
    // Reset exported pad status
    for (i = 0 ; i < DEMO_PAD_MAX_CONTROLLERS ; i++)
    {
        g_DemoPad[i].button = 0;
        g_DemoPad[i].stickX = 0;
        g_DemoPad[i].stickY = 0;
        g_DemoPad[i].substickX = 0;
        g_DemoPad[i].substickY = 0;
        g_DemoPad[i].triggerLeft = 0;
        g_DemoPad[i].triggerRight = 0;
        g_DemoPad[i].analogA = 0;
        g_DemoPad[i].analogB = 0;
        g_DemoPad[i].err = 0;
        g_DemoPad[i].lastButton = 0;
        g_DemoPad[i].buttonDown = 0;
        g_DemoPad[i].buttonUp = 0;
        g_DemoPad[i].dirs = 0;
        g_DemoPad[i].dirsNew = 0;
        g_DemoPad[i].dirsReleased = 0;
        g_DemoPad[i].stickDeltaX = 0;
        g_DemoPad[i].stickDeltaY = 0;
        g_DemoPad[i].substickDeltaX = 0;
        g_DemoPad[i].substickDeltaY = 0;
    }
}

// -------------------------------------------------------------------------
//  Read
// -------------------------------------------------------------------------

u32 DEMOPadRead ()
{
    u32 i;

    u32 ResetReq = 0; // for error handling

    // Read current PAD status
    PADRead( g_Pad );
    PADClampCircle( g_Pad );

    for ( i = 0 ; i < DEMO_PAD_MAX_CONTROLLERS ; i++ )
    {
        // Connection check
        if ( g_Pad[i].err == PAD_ERR_NO_CONTROLLER )
        {
            ResetReq |= g_PadChanMask[i];
        }

        SetPadInfo( &g_Pad[i], &g_DemoPad[i] );
    }

    return TRUE;
}

