﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>
#include <cstdio>
#include <gfx/demo_Mtx.h>
#include "demo_MtxAssert.h"
#include "demo_Mtx44Assert.h"

/*---------------------------------------------------------------------*


                             PROJECTION SECTION


 *---------------------------------------------------------------------*/

/*---------------------------------------------------------------------*

Name:           MTXFrustum

Description:    compute a 4x4 perspective projection matrix from a
                specified view volume.


Arguments:      m        4x4 matrix to be set

                t        top coord. of view volume at the near clipping plane

                b        bottom coord of view volume at the near clipping plane

                lf       left coord. of view volume at near clipping plane

                r        right coord. of view volume at near clipping plane

                n        positive distance from camera to near clipping plane

                f        positive distance from camera to far clipping plane


Return:         none

 *---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*
    C version
 *---------------------------------------------------------------------*/
void C_MTXFrustum ( Mtx44 m, f32 t, f32 b, f32 lf, f32 r, f32 n, f32 f )
{
    f32 tmp;

    ASSERTMSG( (m != 0),  MTX_FRUSTUM_1     );
    ASSERTMSG( (t != b),  MTX_FRUSTUM_2     );
    ASSERTMSG( (lf != r), MTX_FRUSTUM_3     );
    ASSERTMSG( (n != f),  MTX_FRUSTUM_4     );

    tmp     =  1.0f / (r - lf);
    m[0][0] =  (2 * n) * tmp;
    m[0][1] =  0.0f;
    m[0][2] =  (r + lf) * tmp;
    m[0][3] =  0.0f;

    tmp     =  1.0f / (t - b);
    m[1][0] =  0.0f;
    m[1][1] =  (2 * n) * tmp;
    m[1][2] =  (t + b) * tmp;
    m[1][3] =  0.0f;

    m[2][0] =  0.0f;
    m[2][1] =  0.0f;

    tmp     =  1.0f / (f - n);

    // scale z to (-w, w) range (different than Wii's -w...0 range)
    m[2][2] = -(f + n) * tmp;
    m[2][3] = -(2 * f * n) * tmp;

    m[3][0] =  0.0f;
    m[3][1] =  0.0f;
    m[3][2] = -1.0f;
    m[3][3] =  0.0f;
}

/*---------------------------------------------------------------------*

Name:           MTXPerspective

Description:    compute a 4x4 perspective projection matrix from
                field of view and aspect ratio.


Arguments:      m       4x4 matrix to be set

                fovy    total field of view in in degrees in the YZ plane

                aspect  ratio of view window width:height (X / Y)

                n       positive distance from camera to near clipping plane

                f       positive distance from camera to far clipping plane


Return:         none

 *---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*
    C version
 *---------------------------------------------------------------------*/
void C_MTXPerspective ( Mtx44 m, f32 fovY, f32 aspect, f32 n, f32 f )
{
    f32 angle;
    f32 cot;
    f32 tmp;

    ASSERTMSG( (m != 0),                             MTX_PERSPECTIVE_1    );
    ASSERTMSG( ( (fovY > 0.0) && ( fovY < 180.0) ),  MTX_PERSPECTIVE_2    );
    ASSERTMSG( (aspect != 0),                        MTX_PERSPECTIVE_3    );

    // find the cotangent of half the (YZ) field of view
    angle = fovY * 0.5f;
    angle = MTXDegToRad( angle );

    cot = 1.0f / tanf(angle);

    m[0][0] =  cot / aspect;
    m[0][1] =  0.0f;
    m[0][2] =  0.0f;
    m[0][3] =  0.0f;

    m[1][0] =  0.0f;
    m[1][1] =   cot;
    m[1][2] =  0.0f;
    m[1][3] =  0.0f;

    m[2][0] =  0.0f;
    m[2][1] =  0.0f;

    tmp     = 1.0f / (f - n);

    // scale z to (-w, +w) range (different than Wii's -w...0 range)
    m[2][2] = -(f + n) * tmp;
    m[2][3] = -(2 * f * n) * tmp;

    m[3][0] =  0.0f;
    m[3][1] =  0.0f;
    m[3][2] = -1.0f;
    m[3][3] =  0.0f;
}

/*---------------------------------------------------------------------*

Name:           MTXOrtho

Description:    compute a 4x4 orthographic projection matrix.


Arguments:      m        4x4 matrix to be set

                t        top coord. of parallel view volume

                b        bottom coord of parallel view volume

                lf       left coord. of parallel view volume

                r        right coord. of parallel view volume

                n        positive distance from camera to near clipping plane

                f        positive distance from camera to far clipping plane


Return:         none

 *---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*
    C version
 *---------------------------------------------------------------------*/
void C_MTXOrtho ( Mtx44 m, f32 t, f32 b, f32 lf, f32 r, f32 n, f32 f )
{
    f32 tmp;

    ASSERTMSG( (m != 0),  MTX_ORTHO_1  );
    ASSERTMSG( (t != b),  MTX_ORTHO_2  );
    ASSERTMSG( (lf != r), MTX_ORTHO_3  );
    ASSERTMSG( (n != f),  MTX_ORTHO_4  );

    tmp     =  1.0f / (r - lf);
    m[0][0] =  2.0f * tmp;
    m[0][1] =  0.0f;
    m[0][2] =  0.0f;
    m[0][3] = -(r + lf) * tmp;

    tmp     =  1.0f / (t - b);
    m[1][0] =  0.0f;
    m[1][1] =  2.0f * tmp;
    m[1][2] =  0.0f;
    m[1][3] = -(t + b) * tmp;

    m[2][0] =  0.0f;
    m[2][1] =  0.0f;

    tmp     =  1.0f / (f - n);

    // scale z to (-1, 1) range (different than Wii's -1...0 range)
    m[2][2] = -2.0f * tmp;
    m[2][3] = -(f + n) * tmp;

    m[3][0] =  0.0f;
    m[3][1] =  0.0f;
    m[3][2] =  0.0f;
    m[3][3] =  1.0f;
}

/*---------------------------------------------------------------------*


                             GENERAL SECTION


 *---------------------------------------------------------------------*/

/* NOTE: Prototypes for these functions are defined in "mtx44ext.h".   */

/*---------------------------------------------------------------------*
Name:           MTX44Identity

Description:    sets a matrix to identity

Arguments:      m :  matrix to be set

Return:         none

 *---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*
    C version
 *---------------------------------------------------------------------*/
void C_MTX44Identity( Mtx44 m )
{
    ASSERTMSG( (m != 0), MTX44_IDENTITY_1 );

    m[0][0] = 1.0f; m[0][1] = 0.0f; m[0][2] = 0.0f; m[0][3] = 0.0f;
    m[1][0] = 0.0f; m[1][1] = 1.0f; m[1][2] = 0.0f; m[1][3] = 0.0f;
    m[2][0] = 0.0f; m[2][1] = 0.0f; m[2][2] = 1.0f; m[2][3] = 0.0f;
    m[3][0] = 0.0f; m[3][1] = 0.0f; m[3][2] = 0.0f; m[3][3] = 1.0f;
}

/*---------------------------------------------------------------------*
Name:           MTX44Copy

Description:    copies the contents of one matrix into another

Arguments:      src        source matrix for copy
                dst        destination matrix for copy


Return:         none
 *---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*
    C version
 *---------------------------------------------------------------------*/
void C_MTX44Copy( MTX_CONST Mtx44 src, Mtx44 dst )
{
    ASSERTMSG( (src != 0) , MTX44_COPY_1 );
    ASSERTMSG( (dst != 0) , MTX44_COPY_2 );

    if( src == dst )
    {
        return;
    }

    dst[0][0] = src[0][0]; dst[0][1] = src[0][1]; dst[0][2] = src[0][2]; dst[0][3] = src[0][3];
    dst[1][0] = src[1][0]; dst[1][1] = src[1][1]; dst[1][2] = src[1][2]; dst[1][3] = src[1][3];
    dst[2][0] = src[2][0]; dst[2][1] = src[2][1]; dst[2][2] = src[2][2]; dst[2][3] = src[2][3];
    dst[3][0] = src[3][0]; dst[3][1] = src[3][1]; dst[3][2] = src[3][2]; dst[3][3] = src[3][3];
}

/*---------------------------------------------------------------------*
Name:           MTX44Concat

Description:    concatenates two matrices.
                order of operation is A x B = AB.
                ok for any of ab == a == b.

                saves a MTXCopy operation if ab != to a or b.

Arguments:      a        first matrix for concat.
                b        second matrix for concat.
                ab       resultant matrix from concat.

Return:         none
 *---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*
    C version
 *---------------------------------------------------------------------*/
void C_MTX44Concat( MTX_CONST Mtx44 a, MTX_CONST Mtx44 b, Mtx44 ab )
{
    Mtx44       mTmp;
    Mtx44Ptr    m;

    ASSERTMSG( (a  != 0), MTX44_CONCAT_1 );
    ASSERTMSG( (b  != 0), MTX44_CONCAT_2 );
    ASSERTMSG( (ab != 0), MTX44_CONCAT_3 );

    if( (ab == a) || (ab == b) )
    {
        m = mTmp;
    }
    else
    {
        m = ab;
    }

    // compute (a x b) -> m

    m[0][0] = a[0][0]*b[0][0] + a[0][1]*b[1][0] + a[0][2]*b[2][0] + a[0][3]*b[3][0];
    m[0][1] = a[0][0]*b[0][1] + a[0][1]*b[1][1] + a[0][2]*b[2][1] + a[0][3]*b[3][1];
    m[0][2] = a[0][0]*b[0][2] + a[0][1]*b[1][2] + a[0][2]*b[2][2] + a[0][3]*b[3][2];
    m[0][3] = a[0][0]*b[0][3] + a[0][1]*b[1][3] + a[0][2]*b[2][3] + a[0][3]*b[3][3];

    m[1][0] = a[1][0]*b[0][0] + a[1][1]*b[1][0] + a[1][2]*b[2][0] + a[1][3]*b[3][0];
    m[1][1] = a[1][0]*b[0][1] + a[1][1]*b[1][1] + a[1][2]*b[2][1] + a[1][3]*b[3][1];
    m[1][2] = a[1][0]*b[0][2] + a[1][1]*b[1][2] + a[1][2]*b[2][2] + a[1][3]*b[3][2];
    m[1][3] = a[1][0]*b[0][3] + a[1][1]*b[1][3] + a[1][2]*b[2][3] + a[1][3]*b[3][3];

    m[2][0] = a[2][0]*b[0][0] + a[2][1]*b[1][0] + a[2][2]*b[2][0] + a[2][3]*b[3][0];
    m[2][1] = a[2][0]*b[0][1] + a[2][1]*b[1][1] + a[2][2]*b[2][1] + a[2][3]*b[3][1];
    m[2][2] = a[2][0]*b[0][2] + a[2][1]*b[1][2] + a[2][2]*b[2][2] + a[2][3]*b[3][2];
    m[2][3] = a[2][0]*b[0][3] + a[2][1]*b[1][3] + a[2][2]*b[2][3] + a[2][3]*b[3][3];

    m[3][0] = a[3][0]*b[0][0] + a[3][1]*b[1][0] + a[3][2]*b[2][0] + a[3][3]*b[3][0];
    m[3][1] = a[3][0]*b[0][1] + a[3][1]*b[1][1] + a[3][2]*b[2][1] + a[3][3]*b[3][1];
    m[3][2] = a[3][0]*b[0][2] + a[3][1]*b[1][2] + a[3][2]*b[2][2] + a[3][3]*b[3][2];
    m[3][3] = a[3][0]*b[0][3] + a[3][1]*b[1][3] + a[3][2]*b[2][3] + a[3][3]*b[3][3];

    // overwrite a or b if needed
    if(m == mTmp)
    {
        C_MTX44Copy( *((MTX_CONST Mtx44 *)&mTmp), ab );
    }
}

/*---------------------------------------------------------------------*
Name:           MTX44Transpose

Description:    computes the transpose of a matrix.

Arguments:      src       source matrix.
                xPose     destination (transposed) matrix.
                          ok if src == xPose.

Return:         none
 *---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*
    C version
 *---------------------------------------------------------------------*/
void C_MTX44Transpose ( MTX_CONST Mtx44 src, Mtx44 xPose )
{
    Mtx44       mTmp;
    Mtx44Ptr    m;

    ASSERTMSG( (src   != 0), MTX44_TRANSPOSE_1  );
    ASSERTMSG( (xPose != 0), MTX44_TRANSPOSE_2  );

    if(src == xPose)
    {
        m = mTmp;
    }
    else
    {
        m = xPose;
    }

    m[0][0] = src[0][0];    m[0][1] = src[1][0];    m[0][2] = src[2][0];    m[0][3] = src[3][0];
    m[1][0] = src[0][1];    m[1][1] = src[1][1];    m[1][2] = src[2][1];    m[1][3] = src[3][1];
    m[2][0] = src[0][2];    m[2][1] = src[1][2];    m[2][2] = src[2][2];    m[2][3] = src[3][2];
    m[3][0] = src[0][3];    m[3][1] = src[1][3];    m[3][2] = src[2][3];    m[3][3] = src[3][3];

    // copy back if needed
    if( m == mTmp )
    {
        C_MTX44Copy( *((MTX_CONST Mtx44 *)&mTmp), xPose );
    }
}

/*---------------------------------------------------------------------*
Name:           MTX44Inverse

Description:    computes a fast inverse of a matrix.
                uses Gauss-Jordan(with partial pivoting)

Arguments:      src       source matrix.
                inv       destination (inverse) matrix.
                          ok if src == inv.

Return:         0 if src is not invertible.
                1 on success.
 *---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*
    C version only
 *---------------------------------------------------------------------*/
static const int NUM = 4;
#define SWAPF(a,b)  { f32 tmp; tmp = (a); (a) = (b); (b)=tmp; }

u32 C_MTX44Inverse( MTX_CONST Mtx44 src, Mtx44 inv )
{
    Mtx44       gjm;
    s32         i, j, k;
    f32         w;

    ASSERTMSG( (src != 0), MTX44_INVERSE_1 );
    ASSERTMSG( (inv != 0), MTX44_INVERSE_2 );

    C_MTX44Copy(src, gjm);
    C_MTX44Identity(inv);

    for ( i = 0 ; i < NUM ; ++i )
    {
        f32 max = 0.0f;
        s32 swp = i;

        // ---- partial pivoting -----
        for( k = i ; k < NUM ; k++ )
        {
            f32 ftmp;
            ftmp = fabsf(gjm[k][i]);
            if ( ftmp > max )
            {
                max = ftmp;
                swp = k;
            }
        }

        // check singular matrix
        //(or can't solve inverse matrix with this algorithm)
        if ( max == 0.0f )
        {
            return 0;
        }

        // swap row
        if( swp != i )
        {
            for ( k = 0 ; k < NUM ; k++ )
            {
                SWAPF(gjm[i][k], gjm[swp][k]);
                SWAPF(inv[i][k], inv[swp][k]);
            }
        }

        // ---- pivoting end ----

        w = 1.0F / gjm[i][i];
        for ( j = 0 ; j < NUM ; ++j )
        {
            gjm[i][j] *= w;
            inv[i][j] *= w;
        }

        for ( k = 0 ; k < NUM ; ++k )
        {
            if ( k == i )
                continue;

            w = gjm[k][i];
            for ( j = 0 ; j < NUM ; ++j )
            {
                gjm[k][j] -= gjm[i][j] * w;
                inv[k][j] -= inv[i][j] * w;
            }
        }
    }

    return 1;
}

#undef SWAPF
#undef NUM

/*---------------------------------------------------------------------*


                             MODEL SECTION


 *---------------------------------------------------------------------*/

/* NOTE: Prototypes for these functions are defined in "mtx44ext.h".   */

/*---------------------------------------------------------------------*
Name:           MTX44Trans

Description:    sets a translation matrix.

Arguments:       m        matrix to be set
                xT        x component of translation.
                yT        y component of translation.
                zT        z component of translation.

Return:         none
 *---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*
    C version
 *---------------------------------------------------------------------*/
void C_MTX44Trans ( Mtx44 m, f32 xT, f32 yT, f32 zT )
{
    ASSERTMSG( (m != 0), MTX44_TRANS_1 );

    m[0][0] = 1.0f;     m[0][1] = 0.0f;  m[0][2] = 0.0f;  m[0][3] =  xT;
    m[1][0] = 0.0f;     m[1][1] = 1.0f;  m[1][2] = 0.0f;  m[1][3] =  yT;
    m[2][0] = 0.0f;     m[2][1] = 0.0f;  m[2][2] = 1.0f;  m[2][3] =  zT;
    m[3][0] = 0.0f;     m[3][1] = 0.0f;  m[3][2] = 0.0f;  m[3][3] =  1.0f;
}

/*---------------------------------------------------------------------*
Name:           MTX44TransApply

Description:    This function performs the operation equivalent to
                MTXTrans + MTXConcat.

Arguments:      src       matrix to be operated.
                dst       resultant matrix from concat.
                xT        x component of translation.
                yT        y component of translation.
                zT        z component of translation.

Return:         none
 *---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*
    C version
 *---------------------------------------------------------------------*/
void C_MTX44TransApply ( MTX_CONST Mtx44 src, Mtx44 dst, f32 xT, f32 yT, f32 zT )
{
    ASSERTMSG( (src != 0), MTX44_TRANSAPPLY_1 );
    ASSERTMSG( (dst != 0), MTX44_TRANSAPPLY_1 );

    if ( src != dst )
    {
        dst[0][0] = src[0][0];    dst[0][1] = src[0][1];    dst[0][2] = src[0][2];
        dst[1][0] = src[1][0];    dst[1][1] = src[1][1];    dst[1][2] = src[1][2];
        dst[2][0] = src[2][0];    dst[2][1] = src[2][1];    dst[2][2] = src[2][2];
        dst[3][0] = src[3][0];    dst[3][1] = src[3][1];    dst[3][2] = src[3][2];
        dst[3][3] = src[3][3];
    }

    dst[0][3] = src[0][3] + xT;
    dst[1][3] = src[1][3] + yT;
    dst[2][3] = src[2][3] + zT;
}

/*---------------------------------------------------------------------*
Name:            MTX44Scale

Description:     sets a scaling matrix.

Arguments:       m        matrix to be set
                xS        x scale factor.
                yS        y scale factor.
                zS        z scale factor.

Return:         none
 *---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*
    C version
 *---------------------------------------------------------------------*/
void C_MTX44Scale ( Mtx44 m, f32 xS, f32 yS, f32 zS )
{
    ASSERTMSG( (m != 0), MTX44_SCALE_1 );

    m[0][0] = xS;      m[0][1] = 0.0f;  m[0][2] = 0.0f;  m[0][3] = 0.0f;
    m[1][0] = 0.0f;    m[1][1] = yS;    m[1][2] = 0.0f;  m[1][3] = 0.0f;
    m[2][0] = 0.0f;    m[2][1] = 0.0f;  m[2][2] = zS;    m[2][3] = 0.0f;
    m[3][0] = 0.0f;    m[3][1] = 0.0f;  m[3][2] = 0.0f;  m[3][3] = 1.0f;
}


/*---------------------------------------------------------------------*
Name:           MTX44ScaleApply

Description:    This function performs the operation equivalent to
                MTXScale + MTXConcat

Arguments:      src       matrix to be operated.
                dst       resultant matrix from concat.
                xS        x scale factor.
                yS        y scale factor.
                zS        z scale factor.

Return:         none
*---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*
    C version
 *---------------------------------------------------------------------*/
void C_MTX44ScaleApply ( MTX_CONST Mtx44 src, Mtx44 dst, f32 xS, f32 yS, f32 zS )
{
    ASSERTMSG( (src != 0), MTX44_SCALEAPPLY_1 );
    ASSERTMSG( (dst != 0), MTX44_SCALEAPPLY_2 );

    dst[0][0] = src[0][0] * xS;     dst[0][1] = src[0][1] * xS;
    dst[0][2] = src[0][2] * xS;     dst[0][3] = src[0][3] * xS;

    dst[1][0] = src[1][0] * yS;     dst[1][1] = src[1][1] * yS;
    dst[1][2] = src[1][2] * yS;     dst[1][3] = src[1][3] * yS;

    dst[2][0] = src[2][0] * zS;     dst[2][1] = src[2][1] * zS;
    dst[2][2] = src[2][2] * zS;     dst[2][3] = src[2][3] * zS;

    dst[3][0] = src[3][0] ; dst[3][1] = src[3][1];
    dst[3][2] = src[3][2] ; dst[3][3] = src[3][3];
}

/*---------------------------------------------------------------------*
Name:           MTX44RotRad

Description:    sets a rotation matrix about one of the X, Y or Z axes

Arguments:      m       matrix to be set
                axis    major axis about which to rotate.
                        axis is passed in as a character.
                        it must be one of 'X', 'x', 'Y', 'y', 'Z', 'z'
                deg     rotation angle in radians.
                        note:  counter-clockwise rotation is positive.

Return:         none
 *---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*
    C version
 *---------------------------------------------------------------------*/
void C_MTX44RotRad ( Mtx44 m, char axis, f32 rad )
{
    f32 sinA, cosA;

    ASSERTMSG( (m != 0), MTX44_ROTRAD_1 );

    // verification of "axis" will occur in MTXRotTrig

    sinA = sinf(rad);
    cosA = cosf(rad);

    C_MTX44RotTrig( m, axis, sinA, cosA );
}

/*---------------------------------------------------------------------*
Name:           MTX44RotTrig

Arguments:      m       matrix to be set
                axis    major axis about which to rotate.
                        axis is passed in as a character.
                        It must be one of 'X', 'x', 'Y', 'y', 'Z', 'z'
                sinA    sine of rotation angle.
                cosA    cosine of rotation angle.
                        note:  counter-clockwise rotation is positive.

Return:         none
 *---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*
    C version
 *---------------------------------------------------------------------*/
void C_MTX44RotTrig ( Mtx44 m, char axis, f32 sinA, f32 cosA )
{
    ASSERTMSG( (m != 0), MTX44_ROTTRIG_1 );

    axis |= 0x20;
    switch(axis)
    {

    case 'x':
        m[0][0] =  1.0f;  m[0][1] =  0.0f;    m[0][2] =  0.0f;  m[0][3] = 0.0f;
        m[1][0] =  0.0f;  m[1][1] =  cosA;    m[1][2] = -sinA;  m[1][3] = 0.0f;
        m[2][0] =  0.0f;  m[2][1] =  sinA;    m[2][2] =  cosA;  m[2][3] = 0.0f;
        m[3][0] =  0.0f;  m[3][1] =  0.0f;    m[3][2] =  0.0f;  m[3][3] = 1.0f;
        break;

    case 'y':
        m[0][0] =  cosA;  m[0][1] =  0.0f;    m[0][2] =  sinA;  m[0][3] = 0.0f;
        m[1][0] =  0.0f;  m[1][1] =  1.0f;    m[1][2] =  0.0f;  m[1][3] = 0.0f;
        m[2][0] = -sinA;  m[2][1] =  0.0f;    m[2][2] =  cosA;  m[2][3] = 0.0f;
        m[3][0] =  0.0f;  m[3][1] =  0.0f;    m[3][2] =  0.0f;  m[3][3] = 1.0f;
        break;

    case 'z':
        m[0][0] =  cosA;  m[0][1] = -sinA;    m[0][2] =  0.0f;  m[0][3] = 0.0f;
        m[1][0] =  sinA;  m[1][1] =  cosA;    m[1][2] =  0.0f;  m[1][3] = 0.0f;
        m[2][0] =  0.0f;  m[2][1] =  0.0f;    m[2][2] =  1.0f;  m[2][3] = 0.0f;
        m[3][0] =  0.0f;  m[3][1] =  0.0f;    m[3][2] =  0.0f;  m[3][3] = 1.0f;
        break;

    default:
        ASSERTMSG( 0, MTX44_ROTTRIG_2 );
        break;
    }
}

/*---------------------------------------------------------------------*
Name:           C_MTX44RotAxisRad
 *---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*
    C version
 *---------------------------------------------------------------------*/
void C_MTX44RotAxisRad( Mtx44 m, const Vec *axis, f32 rad )
{
    Vec vN;
    f32 s, c;             // sinTheta, cosTheta
    f32 t;                // ( 1 - cosTheta )
    f32 x, y, z;          // x, y, z components of normalized axis
    f32 xSq, ySq, zSq;    // x, y, z squared

    ASSERTMSG( (m    != 0), MTX44_ROTAXIS_1  );
    ASSERTMSG( (axis != 0), MTX44_ROTAXIS_2  );

    s = sinf(rad);
    c = cosf(rad);
    t = 1.0f - c;

    C_VECNormalize( axis, &vN );

    x = vN.x;
    y = vN.y;
    z = vN.z;

    xSq = x * x;
    ySq = y * y;
    zSq = z * z;

    m[0][0] = ( t * xSq )   + ( c );
    m[0][1] = ( t * x * y ) - ( s * z );
    m[0][2] = ( t * x * z ) + ( s * y );
    m[0][3] =    0.0f;

    m[1][0] = ( t * x * y ) + ( s * z );
    m[1][1] = ( t * ySq )   + ( c );
    m[1][2] = ( t * y * z ) - ( s * x );
    m[1][3] =    0.0f;

    m[2][0] = ( t * x * z ) - ( s * y );
    m[2][1] = ( t * y * z ) + ( s * x );
    m[2][2] = ( t * zSq )   + ( c );
    m[2][3] =    0.0f;

    m[3][0] = 0.0f;
    m[3][1] = 0.0f;
    m[3][2] = 0.0f;
    m[3][3] = 1.0f;
}

/*---------------------------------------------------------------------------*
    MATRIX CONVERSION
 *---------------------------------------------------------------------------*/
void C_MTX34To44 ( MTX_CONST Mtx src, Mtx44 dst)
{
    dst[0][0] = src[0][0];    dst[0][1] = src[0][1];    dst[0][2] = src[0][2];    dst[0][3] = src[0][3];
    dst[1][0] = src[1][0];    dst[1][1] = src[1][1];    dst[1][2] = src[1][2];    dst[1][3] = src[1][3];
    dst[2][0] = src[2][0];    dst[2][1] = src[2][1];    dst[2][2] = src[2][2];    dst[2][3] = src[2][3];
    dst[3][0] = 0.0f;         dst[3][1] = 0.0f;         dst[3][2] = 0.0f;         dst[3][3] = 1.0f;
}

#if defined(CAFE)
/*
* The Green Hills versions of 32 bit trig functions are problematic (differ
* in low bits from the more accurate GNU ones). So we use the 64 bit versions
* and round down. Unfortunately, this can still produce different results
* from the accurate 32 bit versions (due to double rounding) and so
* we have to use the 64->32 bit versions on all platforms to get the
* same results
*/
float cosf( float x )
{
    return ( float ) cos( ( double ) x );
}

float sinf( float x )
{
    return ( float ) sin( ( double ) x );
}

float tanf( float x )
{
    return ( float ) tan( ( double ) x );
}
#endif

