﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

////===========================================================================
///  demoGfxShader.c
///
///     This is graphics shader system code for the DEMO library.
///
////===========================================================================

#include <cstdio>
#include <gfx/demo.h>

static const int MAX_FILE_NAME_LEN = 256;

#if NN_GFX_IS_TARGET_GX
#include <cafe/procui.h>
#include <cafe/gfd.h>

BOOL DEMOGfxLoadShaders(DEMOGfxShader *pShader, u32 index, const void *pData)
{
    BOOL fOK = FALSE;

    // Load shaders from memory.
    if ( DEMOGFDReadVertexShader( &pShader->vertexShaderData, index, pData ) )
    {
        fOK = TRUE;
    }

    if ( DEMOGFDReadPixelShader( &pShader->pixelShaderData, index, pData ) )
    {
        fOK = TRUE;
    }

    if ( DEMOGFDReadGeometryShader( &pShader->geomShaderData, index, pData ) )
    {
        fOK = TRUE;
    }

    if ( DEMOGFDReadComputeShader( &pShader->computeShaderData, index, pData ) )
    {
        fOK = TRUE;
    }

    if ( !fOK )
    {
        OSReport("Warning : The file did not contain any shaders!\n");
        return fOK;
    }
    nn::gfx::ShaderInitializeResult result = nn::gfx::ShaderInitializeResult_SetupFailed;
    {
        nn::gfx::Shader::InfoType info;
        info.SetDefault();

        // Separate shaders are not allowed when there is a geometry shader
        info.SetSeparationEnabled( !pShader->geomShaderData );
        info.SetCodeType( nn::gfx::ShaderCodeType_Binary );
        info.SetShaderCodePtr( nn::gfx::ShaderStage_Vertex,
            pShader->vertexShaderData );
        info.SetShaderCodePtr( nn::gfx::ShaderStage_Geometry,
            pShader->geomShaderData );
        info.SetShaderCodePtr( nn::gfx::ShaderStage_Pixel,
            pShader->pixelShaderData );
        info.SetShaderCodePtr( nn::gfx::ShaderStage_Compute,
            pShader->computeShaderData );
        pShader->SetShader( new nn::gfx::Shader );
        result = pShader->GetShader()->Initialize( &DEMODevice, info );
        DEMOAssert( result == nn::gfx::ShaderInitializeResult_Success );
    }

    return result == nn::gfx::ShaderInitializeResult_Success;
}

void DEMOGfxLoadShadersFromFile(DEMOGfxShader* pShader, u32 index, const char* fileName)
{
    void * gshBuf;
    u32 gshLen;
    char fullFileName[MAX_FILE_NAME_LEN];
    size_t nameLen;

    nameLen = 1 + strlen(".gsh") + strlen(fileName);
    DEMOAssert(nameLen < MAX_FILE_NAME_LEN);
    NN_UNUSED( nameLen );
    strcpy(fullFileName, fileName);
    strcat(fullFileName, ".gsh");

    // Load shader binary to memory
    gshBuf = DEMOGfxLoadAssetFile(fullFileName, &gshLen);
    DEMOAssert(gshBuf != NULL);

    // Load shaders from memory
    BOOL fOK = DEMOGfxLoadShaders( pShader, 0, gshBuf );
    DEMOAssert(fOK && "Shader file malformed");

    // Free memory used to load the shader file
    DEMOFree(gshBuf);
}
#else

static const char SHADER_EXTENSION[] = ".grsf";

void DEMOGfxLoadShadersFromFile(DEMOGfxShader* pShader, u32 index, const char* fileName)
{
    void* buffer;
    size_t align = nn::gfx::ResShaderFile::GetMaxFileAlignment();
    u32 fileLen;

    char fullFileName[MAX_FILE_NAME_LEN];
    size_t nameLen;

    nameLen = 1 + strlen(SHADER_EXTENSION) + strlen(fileName);
    DEMOAssert(nameLen < MAX_FILE_NAME_LEN);
    strcpy(fullFileName, fileName);
    strcat(fullFileName, SHADER_EXTENSION);

    buffer = DEMOGfxLoadAssetFileAlign(fullFileName, &fileLen, static_cast< u32 >(align));

    ASSERT(buffer != NULL && "Unable to read shader file");
    pShader->pResShaderFile = nn::gfx::ResShaderFile::ResCast(buffer);
    pShader->pResShaderContainer = pShader->pResShaderFile->GetShaderContainer();
    NN_ASSERT_NOT_NULL(pShader->pResShaderContainer);
    pShader->pResShaderContainer->Initialize(&DEMODevice);

    nn::gfx::ResShaderVariation* pVariation = pShader->pResShaderContainer->GetResShaderVariation(index);
    NN_ASSERT_NOT_NULL(pVariation);
    nn::gfx::ShaderInitializeResult shaderResult = nn::gfx::ShaderInitializeResult_SetupFailed;
    nn::gfx::ShaderCodeType codeType = nn::gfx::ShaderCodeType_Binary;

    nn::gfx::ResShaderProgram* program;
    program = pVariation->GetResShaderProgram(codeType);
    if (program != NULL)
    {
        shaderResult = program->Initialize(&DEMODevice);
    }
    if (shaderResult != nn::gfx::ShaderInitializeResult_Success)
    {
        codeType = nn::gfx::ShaderCodeType_Source;
        program = pVariation->GetResShaderProgram(codeType);
        NN_ASSERT_NOT_NULL(program);
        shaderResult = program->Initialize(&DEMODevice);
    }

#if NN_GFX_IS_TARGET_NVN
    size_t scratchSize = program->NvnGetRecommendedScrachMemorySize( &DEMODevice );
    if ( scratchSize > DEMOGfxShaderScratchMemory->GetSize() )
    {
        DEMOPrintf( "Warning: shader '%s' desired scratch memory is: %d. DEMOGfxShaderScratchMemory size is: %d. Performance may suffer.\n",
            fullFileName, scratchSize, DEMOGfxShaderScratchMemory->GetSize() );
    }
#endif

    NN_ASSERT_EQUAL(shaderResult, nn::gfx::ShaderInitializeResult_Success);
    pShader->SetShader(pVariation->GetResShaderProgram(codeType)->GetShader());
}

#endif

BOOL DEMOGfxInitShaderAttribute(DEMOGfxShader *pShader, const char *name,
                                u32 bufferIndex, u32 offset, nn::gfx::AttributeFormat format)
{
    if(!pShader)
    {
        OSReport("Error : Null pointer.\n");
        return FALSE;
    }
    if(bufferIndex > DEMO_MAX_ATTRIB_BUFFERS )
    {
        OSReport("Warning : The attribute buffer index (%d) is over %d\n", bufferIndex, DEMO_MAX_ATTRIB_BUFFERS);
        return FALSE;
    }

    // Get location
    u32 location = (u32)pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, name);
    ASSERT(DEMOGfxCheckShaderLocation(location));

    // set up the shader attribute
    pShader->vertexAttributes[pShader->attribCount].SetDefault();
    pShader->vertexAttributes[pShader->attribCount].SetBufferIndex( bufferIndex );
    pShader->vertexAttributes[pShader->attribCount].SetFormat( format );
    pShader->vertexAttributes[pShader->attribCount].SetNamePtr( name );
    pShader->vertexAttributes[pShader->attribCount].SetOffset( offset );
    pShader->vertexAttributes[pShader->attribCount].SetShaderSlot( location );

    // Increase count
    pShader->attribCount++;

    return TRUE;
}

void DEMOGfxInitShaderVertexBuffer( DEMOGfxShader *pShader, u32 bufferIndex,
    u32 stride, u32 divisor )
{
    // Keep track of the number of buffers that are used
    pShader->bufferCount = ( pShader->bufferCount < bufferIndex + 1 ) ? ( bufferIndex + 1 ) : pShader->bufferCount;

    pShader->vertexBuffers[ bufferIndex ].SetDefault();
    pShader->vertexBuffers[ bufferIndex ].SetStride( stride );
    pShader->vertexBuffers[ bufferIndex ].SetDivisor( divisor );
}

BOOL DEMOGfxGetVertexShaderSamplerLocation(DEMOGfxShader *pShader, const char *name)
{
    ASSERT(pShader != NULL);

    if(pShader->samplersVS.count > DEMO_MAX_SAMPLERS )
    {
        OSReport("Warning : The sampler count (%d) in VS is over %d\n", pShader->samplersVS.count, DEMO_MAX_SAMPLERS);
        return FALSE;
    }

    // Set sampler data
    pShader->samplersVS.location[pShader->samplersVS.count] =
    (u32)pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Sampler, name);
    ASSERT(DEMOGfxCheckShaderLocation(pShader->samplersVS.location[pShader->samplersVS.count]));

    // Increase count
    pShader->samplersVS.count++;
    return TRUE;
}

BOOL DEMOGfxGetPixelShaderSamplerLocation(DEMOGfxShader *pShader, const char *name)
{
    ASSERT(pShader != NULL);

    if(pShader->samplersPS.count > DEMO_MAX_SAMPLERS )
    {
        OSReport("Warning : The sampler count (%d) in PS is over %d\n", pShader->samplersPS.count, DEMO_MAX_SAMPLERS);
        return FALSE;
    }

    // Get sampler location
    pShader->samplersPS.location[pShader->samplersPS.count] =
    (u32)pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, name );
    ASSERT(DEMOGfxCheckShaderLocation(pShader->samplersPS.location[pShader->samplersPS.count]));

    // Increase count
    pShader->samplersPS.count++;
    return TRUE;
}

BOOL DEMOGfxGetGeometryShaderSamplerLocation(DEMOGfxShader *pShader, const char *name)
{
    ASSERT(pShader != NULL);

    if(pShader->samplersGS.count > DEMO_MAX_SAMPLERS )
    {
        OSReport("Warning : The sampler count (%d) in GS is over %d\n", pShader->samplersGS.count, DEMO_MAX_SAMPLERS);
        return FALSE;
    }

    // Get sampler location
    pShader->samplersGS.location[pShader->samplersGS.count] =
    (u32)pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Geometry, nn::gfx::ShaderInterfaceType_Sampler, name);
    ASSERT(DEMOGfxCheckShaderLocation(pShader->samplersGS.location[pShader->samplersGS.count]));

    // Increase count
    pShader->samplersGS.count++;
    return TRUE;
}

#if 0
// not supported in nn::gfx
BOOL DEMOGfxGetVertexShaderUniformLocation(DEMOGfxShader *pShader, const char *name)
{
    if(!pShader)
    {
        OSReport("Error : Null pointer.\n");
        return FALSE;
    }
    if(pShader->uniformsVS.count > GX2_MAX_VS_UNIFORM_VARS )
    {
        OSReport("Warning : The uniform count (%d) in VS is over %d\n", pShader->uniformsVS.count, GX2_MAX_VS_UNIFORM_VARS);
        return FALSE;
    }

    // Get uniform location
    pShader->uniformsVS.location[pShader->uniformsVS.count] =
    (u32)GX2GetVertexUniformVarOffset(pShader->pVertexShader, name);
    ASSERT(DEMOGfxCheckShaderLocation(pShader->uniformsVS.location[pShader->uniformsVS.count]));

    // Increase count
    pShader->uniformsVS.count++;
    return TRUE;
}

BOOL DEMOGfxGetPixelShaderUniformLocation(DEMOGfxShader *pShader, const char *name)
{
    if(!pShader)
    {
        OSReport("Error : Null pointer.\n");
        return FALSE;
    }
    if(pShader->uniformsPS.count > GX2_MAX_PS_UNIFORM_VARS )
    {
        OSReport("Warning : The uniform count (%d) in PS is over %d\n", pShader->uniformsPS.count, GX2_MAX_PS_UNIFORM_VARS);
        return FALSE;
    }
    // Get uniform location
    pShader->uniformsPS.location[pShader->uniformsPS.count] =
    (u32)GX2GetPixelUniformVarOffset(pShader->pPixelShader, name);
    ASSERT(DEMOGfxCheckShaderLocation(pShader->uniformsPS.location[pShader->uniformsPS.count]));

    // Increase count
    pShader->uniformsPS.count++;
    return TRUE;
}
#endif

BOOL DEMOGfxGetVertexShaderUniformBlockLocation(DEMOGfxShader *pShader, const char *name)
{
    ASSERT(pShader != NULL);

    if(pShader->uniformBlocksVS.count > DEMO_MAX_UNIFORM_BLOCKS )
    {
        OSReport("Warning : The uniform block count (%d) in VS is over %d\n", pShader->uniformBlocksVS.count, DEMO_MAX_UNIFORM_BLOCKS);
        return FALSE;
    }

    // Get uniform block location and size
    pShader->uniformBlocksVS.location[pShader->uniformBlocksVS.count] = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, name );
    ASSERT(DEMOGfxCheckShaderLocation(pShader->uniformBlocksVS.location[pShader->uniformBlocksVS.count]));

    // Increase count
    pShader->uniformBlocksVS.count++;
    return TRUE;
}

BOOL DEMOGfxGetPixelShaderUniformBlockLocation(DEMOGfxShader *pShader, const char *name)
{
    ASSERT(pShader != NULL);

    if(pShader->uniformBlocksPS.count > DEMO_MAX_UNIFORM_BLOCKS )
    {
        OSReport("Warning : The uniform block count (%d) in PS is over %d\n", pShader->uniformBlocksPS.count, DEMO_MAX_UNIFORM_BLOCKS);
        return FALSE;
    }

    // Get uniform block location and size
    pShader->uniformBlocksPS.location[pShader->uniformBlocksPS.count] = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, name );
    ASSERT(DEMOGfxCheckShaderLocation(pShader->uniformBlocksPS.location[pShader->uniformBlocksPS.count]));

    // Increase count
    pShader->uniformBlocksPS.count++;
    return TRUE;
}

BOOL DEMOGfxGetGeometryShaderUniformBlockLocation(DEMOGfxShader *pShader, const char *name)
{
    ASSERT(pShader != NULL);

    if(pShader->uniformBlocksGS.count > DEMO_MAX_UNIFORM_BLOCKS )
    {
        OSReport("Warning : The uniform block count (%d) in GS is over %d\n", pShader->uniformBlocksGS.count, DEMO_MAX_UNIFORM_BLOCKS);
        return FALSE;
    }

    // Get uniform block location and size
    pShader->uniformBlocksGS.location[pShader->uniformBlocksGS.count] = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Geometry, nn::gfx::ShaderInterfaceType_ConstantBuffer, name );
    ASSERT(DEMOGfxCheckShaderLocation(pShader->uniformBlocksGS.location[pShader->uniformBlocksGS.count]));

    // Increase count
    pShader->uniformBlocksGS.count++;
    return TRUE;
}

BOOL DEMOGfxFreeShaders(DEMOGfxShader *pShader)
{
    if(!pShader)
    {
        OSReport("Error : Null pointer.\n");
        return FALSE;
    }

    pShader->GetShader()->Finalize(&DEMODevice);

#ifdef CAFE
    if(pShader->vertexShaderData)
    {
        DEMOGFDFreeVertexShader(pShader->vertexShaderData);
    }
    if(pShader->pixelShaderData)
    {
        DEMOGFDFreePixelShader(pShader->pixelShaderData);
    }
    if(pShader->geomShaderData)
    {
        DEMOGFDFreeGeometryShader(pShader->geomShaderData);
    }
#else
    pShader->pResShaderContainer->Finalize( &DEMODevice );
    if (pShader->pixelShaderData)
    {
        DEMOFree( pShader->pixelShaderData );
    }
    if (pShader->vertexShaderData)
    {
        DEMOFree( pShader->vertexShaderData );
    }
    if (pShader->geomShaderData)
    {
        DEMOFree( pShader->geomShaderData );
    }
    if (pShader->hullShaderData)
    {
        DEMOFree( pShader->hullShaderData );
    }
    if (pShader->domainShaderData)
    {
        DEMOFree( pShader->domainShaderData );
    }
#endif
    pShader->SetShader(NULL);
    return TRUE;
}
