﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

////===========================================================================
///  DEMO_Fs.c
///
///     This is file system code for the DEMO library.
///
////===========================================================================

#include <gfx/demo.h>

#include <cstring>

#include <nnt/nnt_Argument.h>

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
#include <nn/nn_Abort.h>
#endif

namespace {

// -------------------------------------------------------

// "Safe" string concat function

#if !defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
// Like strncat, but len = size of dst buffer
inline char *strlcat(char *dst, const char *src, size_t len)
{
    size_t n = strlen(dst);
    if (n > len - 1)
    {
        n = len - 1;
    }
    return strncat(dst, src, len - 1 - n);
}
#endif

u32 g_FileSystemInitializeRefCount = 0;
void* g_pRomCache = NULL;

void InitRomFs()
{
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
    // Setup Rom
    size_t romCacheSize = 0;
    nn::Result result = nn::fs::QueryMountRomCacheSize( &romCacheSize );
    NN_ABORT_UNLESS(result.IsSuccess());

    g_pRomCache = DEMOAlloc( romCacheSize );
    NN_ABORT_UNLESS_NOT_NULL( g_pRomCache );

    result = nn::fs::MountRom( "rom", g_pRomCache, romCacheSize );
    NN_ABORT_UNLESS(result.IsSuccess());
#endif
}

DEMOFSResultCode _DemoInitFS()
{
    if (!g_FileSystemInitializeRefCount)
    {
        const size_t ASSET_ROOT_LENGTH = 512;
        char assetRoot[ ASSET_ROOT_LENGTH ];
        char** argv = nnt::GetHostArgv();
        char* path = NULL;

        NN_ASSERT_NOT_NULL( argv );
        NN_ASSERT_NOT_NULL( argv[ 0 ] );

        // Copy the root from the application path
        std::strncpy( assetRoot, argv[ 0 ], ASSET_ROOT_LENGTH );

        // Remove Tests\Outputs.*
        path = std::strstr( assetRoot, "\\Tests\\Outputs" );
        NN_ASSERT_NOT_NULL( path );
        *path = '\0';

        nn::Result result = nn::fs::MountHostRoot();
        NN_ABORT_UNLESS(result.IsSuccess());

        result = nn::fs::MountHost( "assets", assetRoot );
        NN_ABORT_UNLESS(result.IsSuccess());

        InitRomFs();
    }
    g_FileSystemInitializeRefCount++;

    return DEMO_FS_RESULT_OK;
}
}

s32 DEMOFSOpenFile(const char* path, DEMOFSFileInfo* info)
{
    return DEMOFSOpenFileMode(path, info, "rb");
}


s32 DEMOFSOpenFileMode(const char* path, DEMOFSFileInfo* info, const char* mode)
{
    DEMOFSResultCode fsresult = _DemoInitFS();
    if (DEMO_FS_RESULT_OK != fsresult)
    {
        DEMOPrintf("DEMOFS Error: _DemoInitFS failed!\n");
        return DEMO_FS_RESULT_FATAL_ERROR;
    }

    {
        int openMode = 0;
        openMode |= ( NULL != std::strchr( mode, 'r' ) ) ? nn::fs::OpenMode_Read : 0;
        openMode |= ( NULL != std::strchr( mode, 'w' ) ) ?
            ( nn::fs::OpenMode_AllowAppend | nn::fs::OpenMode_Write ) : 0;
        openMode |= ( NULL != std::strchr( mode, 'a' ) ) ?
            ( nn::fs::OpenMode_AllowAppend | nn::fs::OpenMode_Write ) : 0;
        nn::Result result = nn::fs::OpenFile( info, path, openMode );

        if ( ( openMode & nn::fs::OpenMode_Write ) && nn::fs::ResultPathNotFound::Includes( result ) )
        {
            // Create the file and retry
            result = nn::fs::CreateFile( path, 0 );
            NN_ABORT_UNLESS( result.IsSuccess() );

            result = nn::fs::OpenFile( info, path, openMode );
        }

        NN_ABORT_UNLESS( result.IsSuccess() );
        if ( !result.IsSuccess() )
        {
            return DEMO_FS_RESULT_FATAL_ERROR;
        }
    }

    return DEMO_FS_RESULT_OK;
}


s32 DEMOFSGetLength( const DEMOFSFileInfo* fileInfo, u32* length )
{
    int64_t bigLength;
    nn::Result result = nn::fs::GetFileSize( &bigLength, *fileInfo );
    *length = static_cast< u32 >( bigLength );
    NN_ABORT_UNLESS( result.IsSuccess() );
    if ( !result.IsSuccess() )
    {
        return DEMO_FS_RESULT_FATAL_ERROR;
    }

    return DEMO_FS_RESULT_OK;
}

s32 DEMOFSRead( DEMOFSFileInfo* fileInfo, void* addr, size_t length, size_t offset )
{
    DEMOFSResultCode fsresult = _DemoInitFS();
    if (DEMO_FS_RESULT_OK != fsresult)
    {
        DEMOPrintf("DEMOFS Error: _DemoInitFS failed!\n");
        return DEMO_FS_RESULT_FATAL_ERROR;
    }

    {
        nn::Result result = nn::fs::ReadFile( *fileInfo, offset, addr, length );
        NN_ABORT_UNLESS( result.IsSuccess() );
        if ( !result.IsSuccess() )
        {
            return DEMO_FS_RESULT_FATAL_ERROR;
        }
    }

    return DEMO_FS_RESULT_OK;
}

s32 DEMOFSWrite(DEMOFSFileInfo* fileInfo, const void* addr, size_t length)
{
    DEMOFSResultCode fsresult = _DemoInitFS();
    if (DEMO_FS_RESULT_OK != fsresult)
    {
        DEMOPrintf("DEMOFS Error: _DemoInitFS failed!\n");
        return DEMO_FS_RESULT_FATAL_ERROR;
    }

    {
        nn::fs::WriteOption options = nn::fs::WriteOption::MakeValue( nn::fs::WriteOptionFlag_Flush );
        nn::Result result = nn::fs::WriteFile( *fileInfo, 0, addr, length, options );
        NN_ABORT_UNLESS( result.IsSuccess() );
        if ( !result.IsSuccess() )
        {
            return DEMO_FS_RESULT_FATAL_ERROR;
        }
    }

    return DEMO_FS_RESULT_OK;
}

s32 DEMOFSCloseFile( DEMOFSFileInfo* fileInfo )
{
    nn::fs::CloseFile(*fileInfo);

    return DEMO_FS_RESULT_OK;
}


void * DEMOFSSimpleRead(const char * filename, u32 * len)
{
    DEMOFSFileInfo file;
    u32 fileLen;
    u32 fileAllocLen;
    s32 fileRet;
    void *fileBuf = NULL;

    fileRet = DEMOFSOpenFile(filename, &file);
    DEMOAssert(fileRet == DEMO_FS_RESULT_OK && "couldn't open file");

    fileRet = DEMOFSGetLength(&file, &fileLen);
    DEMOAssert(fileRet == DEMO_FS_RESULT_OK && "couldn't get file length");
    fileAllocLen = DEMORoundUpForIO(fileLen + 1); // Add 1 to make it null terminated

    fileBuf = DEMOAllocEx(fileAllocLen, PPC_IO_BUFFER_ALIGN);
    DEMOAssert(fileBuf && "couldn't alloc memory");

    fileRet = DEMOFSRead( &file, fileBuf, ( s32 )fileLen, 0 );
    DEMOAssert(fileRet == DEMO_FS_RESULT_OK && "couldn't read file");

    fileRet = DEMOFSCloseFile(&file);
    if (fileRet != DEMO_FS_RESULT_OK)
    {
        DEMOAssert(!"couldn't close file");
    }

    // Make it null terminated in case something tries to use strlen on it.
    reinterpret_cast< char* >( fileBuf )[ fileLen ] = '\0';

    *len = fileLen;
    return fileBuf;
}

void * DEMOFSSimpleReadAlign(const char * filename, u32 * len, u32 alignSize)
{
    DEMOFSFileInfo file;
    u32 fileLen;
    u32 fileAllocLen;
    s32 fileRet;
    void *fileBuf = NULL;

    fileRet = DEMOFSOpenFile(filename, &file);
    DEMOAssert(fileRet == DEMO_FS_RESULT_OK && "couldn't open file");

    fileRet = DEMOFSGetLength(&file, &fileLen);
    DEMOAssert(fileRet == DEMO_FS_RESULT_OK && "couldn't get file length");
    fileAllocLen = DEMORoundUpForIO(fileLen);

    fileBuf = DEMOAllocEx(fileAllocLen, alignSize);
    DEMOAssert(fileBuf && "couldn't alloc memory");

    fileRet = DEMOFSRead( &file, fileBuf, ( s32 )fileLen, 0 );
    DEMOAssert(fileRet == DEMO_FS_RESULT_OK && "couldn't read file");

    fileRet = DEMOFSCloseFile(&file);
    if (fileRet != DEMO_FS_RESULT_OK)
    {
        DEMOAssert(!"couldn't close file");
    }

    *len = fileLen;
    return fileBuf;
}

s32 DEMOFSScanDir(const char *pSearchPath, const char *pPrefixPath, u32 *pFileCount, u32 MaxFiles, char** ppFileNames )
{
    NN_UNUSED( ppFileNames );
    NN_UNUSED( MaxFiles );
    NN_UNUSED( pFileCount );
    u32             fileCount = 0;
    u32             dirCount = 0;

    DEMOFSResultCode demoFsResult = _DemoInitFS();
    if ( DEMO_FS_RESULT_OK != demoFsResult )
    {
        DEMOPrintf("DEMOFS Error: _DemoInitFS failed!\n");
        return DEMO_FS_RESULT_FATAL_ERROR;
    }

    nn::Result result;
    nn::fs::DirectoryHandle directoryHandle;

    result = nn::fs::OpenDirectory( &directoryHandle, pSearchPath, nn::fs::OpenDirectoryMode_All );

    if ( !result.IsSuccess() )
    {
        DEMOPrintf( "nn::fs::OpenDirectory Error: %d\n", result );
        return DEMO_FS_RESULT_FATAL_ERROR;
    }

    int64_t totalDirectoryEntries = 0;
    result = nn::fs::GetDirectoryEntryCount( &totalDirectoryEntries, directoryHandle );
    if ( !result.IsSuccess() )
    {
        DEMOPrintf( "nn::fs::GetDirectoryEntryCount Error!\n" );
        return DEMO_FS_RESULT_FATAL_ERROR;
    }

    for ( int64_t index = 0; index < totalDirectoryEntries; index++ )
    {
        nn::fs::DirectoryEntry entry;
        int64_t entriesReturned = 0;

        result = nn::fs::ReadDirectory( &entriesReturned, &entry, directoryHandle, 1 );
        if ( result.IsFailure() )
        {
            break;
        }

        if ( entry.directoryEntryType == nn::fs::DirectoryEntryType_Directory )
        {
            dirCount++;
        }
        else /* this is file */
        {
            if(fileCount < MaxFiles)
            {
              // get file name (not including the volume, DemoSimpleRead puts it back on)
                const u32 MaxNameLen = 256;
                u32 nameLen = static_cast< u32 >( (strlen( pSearchPath ) + 1 + strlen( entry.name )));

                    // be safe, if file name too long, don't return it. (Assert)!
                if(nameLen < MaxNameLen)
                {
                    char *pBuff = (char*) DEMOAlloc(nameLen + 1);   // allocate new space for this string
                    DEMOAssert(pBuff && "couldn't alloc memory");
                    strncpy(pBuff, pPrefixPath, nameLen + 1);
                    if(pBuff[0] && pBuff[strlen(pBuff) - 1] != '/')
                    {
                        strlcat(pBuff, "/", nameLen + 1);
                    }
                    strlcat(pBuff, entry.name, nameLen + 1);

                    ppFileNames[fileCount] = pBuff;
                    fileCount++;
                } else {
                    DEMOPrintf("DEMO Error: filename found in dir that is too long\n");
                    return DEMO_FS_RESULT_FATAL_ERROR;
                }
            }
        }
    }
    *pFileCount = fileCount;
    return DEMO_FS_RESULT_OK;
} //NOLINT(readability/fn_size)

s32 DEMOFSSimpleScanDir(const char *pPath, u32 *pFileCount, u32 MaxFiles, char** ppFileNames )
{
    return DEMOFSScanDir(pPath, pPath, pFileCount, MaxFiles, ppFileNames);
}

s32 DEMOFSRename(const char *oldpath, const char *newpath)
{
    DEMOFSResultCode result = _DemoInitFS();
    if (DEMO_FS_RESULT_OK != result)
    {
        DEMOPrintf("DEMOFS Error: _DemoInitFS failed!\n");
        return DEMO_FS_RESULT_FATAL_ERROR;
    }

    nn::Result renameResult = nn::fs::RenameFile( oldpath, newpath );
    if ( renameResult.IsFailure() )
    {
        DEMOPrintf("DEMOFS Error: Failed to rename file!\n");
        return DEMO_FS_RESULT_ACCESS;
    }

    return DEMO_FS_RESULT_OK;
}

s32 DEMOFSRemove(const char *path, BOOL ignoreError)
{
    nn::Result result = nn::fs::DeleteFile( path );

    if ( result.IsFailure() && !ignoreError )
    {
        DEMOPrintf("DEMOFS Error: FSRemove(%s) returned %d!\n", path, result);
        return DEMO_FS_RESULT_FATAL_ERROR;
    }

    return DEMO_FS_RESULT_OK;
}

s32 DEMOFSCreateDirectory( const char* path )
{
    DEMOFSResultCode demoFsResult = _DemoInitFS();
    if ( DEMO_FS_RESULT_OK != demoFsResult )
    {
        DEMOPrintf("DEMOFS Error: _DemoInitFS failed!\n");
        return DEMO_FS_RESULT_FATAL_ERROR;
    }

    nn::Result result;
    nn::fs::DirectoryHandle directoryHandle;

    result = nn::fs::OpenDirectory( &directoryHandle, path, nn::fs::OpenDirectoryMode_All );

    if ( result.IsSuccess() )
    {
        // Directory already exists
        nn::fs::CloseDirectory( directoryHandle );
        return DEMO_FS_RESULT_OK;
    }

    // Recursively create the directory if necessary
    result = nn::fs::CreateDirectory( path );
    if ( nn::fs::ResultPathNotFound().Includes( result ) )
    {
        size_t length = strlen( path );
        char* pStr = new char[ length + 1 ];
        const char* p = strchr( path, '/' );
        if( p == NULL )
        {
            p = strchr( path, '\\' );
        }
        const char* prevDir = nullptr;
        do
        {
            size_t copyLength = static_cast< size_t >( reinterpret_cast< ptrdiff_t >( p ) - reinterpret_cast< ptrdiff_t >( path ) + 1 );
            strncpy( pStr, path, copyLength );
            pStr[ copyLength ] = '\0';
            result = nn::fs::CreateDirectory( pStr );
            prevDir = p;
        } while ( ( p = strchr( prevDir + 1, '/' ) ) == NULL ? ( p = strchr( prevDir + 1, '\\' ) ) != NULL : true &&
                  ( nn::fs::ResultPathAlreadyExists().Includes( result ) || result.IsSuccess() ) );

        if ( result.IsSuccess() )
        {
            // Create the final directory
            result = nn::fs::CreateDirectory( path );
        }

        DEMOAssert( result.IsSuccess()|| nn::fs::ResultPathAlreadyExists().Includes( result ) );

        delete[] pStr;
    }

    if ( !result.IsSuccess() && !nn::fs::ResultPathAlreadyExists().Includes( result ) )
    {
        return DEMO_FS_RESULT_FATAL_ERROR;
    }

    return DEMO_FS_RESULT_OK;
}
// --------------------------------------------------------
