﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// -----------------------------------------------------------------------------
//  demoGfx.h
//
// -----------------------------------------------------------------------------

#pragma once

#include <gfx/demo.h>
#include <gfx/demo_GfxTypes.h>
#include <gfx/demo_GfxShader.h>
#include <gfx/demo_Mtx.h>

/// @addtogroup demoRender
/// @{

/// \brief The maximum length allowed for the asset directory passed with DEMO_ASSETS_DIR
static const int MAX_ASSET_DIR_LEN = 1000;

/// \brief The maximum total length allowed for the asset's path in DEMOGfxLoadAssetFile
static const int MAX_ASSET_DIR_FULL_LEN = 2000;

/// \brief Default device for DEMOGfx (global)
extern nn::gfx::Device DEMODevice;

/// \brief Default queue for DEMOGfx (global)
extern nn::gfx::Queue DEMOQueue;

/// \brief Default command buffer for DEMOGfx (global)
extern nn::gfx::CommandBuffer DEMOCommandBuffer;

/// \brief Default swap chain for DEMOGfx (global)
extern nn::gfx::SwapChain DEMOSwapChain;

/// \brief Default color buffer texture info for DEMOGfx (global)
extern nn::gfx::Texture::InfoType DEMOColorBufferInfo;

/// \brief Default depth buffer texture info for DEMOGfx (global)
extern nn::gfx::Texture::InfoType DEMODepthBufferInfo;

/// \brief Default depth buffer texture for DEMOGfx (global)
extern nn::gfx::Texture DEMODepthBuffer;

/// \brief Default depth buffer view for DEMOGfx (global)
extern nn::gfx::DepthStencilView DEMODepthBufferView;

/// \brief Global texture descriptor pool for the demos
extern nn::gfx::DescriptorPool DEMOTextureDescriptorPool;

/// \brief Global sampler descriptor pool for the demos
extern nn::gfx::DescriptorPool DEMOSamplerDescriptorPool;

/// \brief Global buffer view descriptor pool for the demos
extern nn::gfx::DescriptorPool DEMOBufferViewDescriptorPool;

#if NN_GFX_IS_TARGET_NVN
class DEMOGfxMemPool;

/// \brief Global shader scratch memory
extern DEMOGfxMemPool* DEMOGfxShaderScratchMemory;
#endif

int DEMOGfxRegisterTextureView( nn::gfx::TextureView* pTextureView );
int DEMOGfxRegisterSampler( nn::gfx::Sampler* pSampler );
int DEMOGfxRegisterBufferView( const nn::gfx::GpuAddress& gpuAddress, size_t size );

class DEMOGfxPipeline
{
public:
    DEMOGfxShader shaders;

    int vertexBufferStateCount;
    int blendTargetStateCount;
    int colorTargetStateCount;
    nn::gfx::BlendTargetStateInfo blendTargetStateInfoArray[ 6 ];
    nn::gfx::ColorTargetStateInfo colorTargetStateInfoArray[ 6 ];

    nn::gfx::BlendStateInfo blendStateInfo;
    nn::gfx::VertexStateInfo vertexStateInfo;
    nn::gfx::RenderTargetStateInfo renderTargetStateInfo;
    nn::gfx::RasterizerStateInfo rasterizerStateInfo;
    nn::gfx::DepthStencilStateInfo depthStencilStateInfo;
    nn::gfx::TessellationStateInfo tessellationStateInfo;

    nn::gfx::Pipeline::InfoType pipelineInfo;
    nn::gfx::Pipeline pipeline;
    void* pPipelineData;

    DEMOGfxPipeline();
    void SetDefaults();
    void Initialize( nn::gfx::Device* pDevice );
    void Finalize( nn::gfx::Device* pDevice );
    void Finalize()
    {
        Finalize( &DEMODevice );
    }
};

class DEMOGfxMemPool
{
private:
    nn::gfx::MemoryPool* pMemPool;  // underlying memory pool
    u8* dataPtr;
    size_t curPos;  // current offset
    size_t origPos; // original offset
    size_t size;
    size_t end;
    bool dataNeedsFree;
    bool poolNeedsFree;

public:
    void* GetDataPtr()
    {
        return static_cast<void*>(dataPtr + origPos);
    }
    nn::gfx::MemoryPool* GetPool()
    {
        return pMemPool;
    }
    size_t GetBaseOffset()
    {
        return origPos;
    }
    size_t GetSize()
    {
        return size;
    }

    /// \brief Allocate space within a pool, and return an offset to that space
    size_t AllocSpace(size_t size, size_t align);
    /// \brief Reset all allocations within the pool (effectively freeing all memory allocated with AllocSpace)
    void   Reset();
    /// \brief create a sub-pool from a memory pool; like AllocSpace, but actually creates a new pool
    DEMOGfxMemPool* AllocSubPool(size_t size, size_t align);

    /// \brief destroy the memory pool
    void Finalize();

    /// \brief Create a memory pool from already allocated memory (must be properly aligned)
    static DEMOGfxMemPool* CreateFromMemory(void* memPtr, size_t size, int memoryPoolProperty);
    /// \brief Allocate memory and create a new pool from it
    static DEMOGfxMemPool* AllocNewPool(size_t size, size_t align, int memoryPoolProperty);
};

class DEMOGfxBuffer
{
public:
    void* data;
    size_t size;
    size_t offset;
    nn::gfx::Buffer buffer;
    nn::gfx::GpuAddress gpuAddress;
    DEMOGfxMemPool* memPool;

    void Initialize( size_t size, void* data, int gpuAccessFlags, int offset );
    void AllocFromPool(DEMOGfxMemPool* pool, size_t size, void* data, int gpuAccessFlags, int offset);
    void CreateFromPool(DEMOGfxMemPool* pool, size_t size, int gpuAccessFlags);
    void Finalize(nn::gfx::Device* dev);
    void Finalize() { Finalize(&DEMODevice); }

    template <typename TType>
    TType* Map()
    {
        TType* pData = buffer.Map< TType >( );
        buffer.InvalidateMappedRange( 0, size );
        return pData;
    }

    void Unmap();
};

class DEMOGfxTexture
{
private:
    void* pFileData;
    int totalTextures;
    nn::gfx::DescriptorSlot* pSlots;
    nn::gfx::ResTextureFile* pResTextureFile;


public:
    bool Initialize( const char* pFileName );
    void Finalize();

    nn::gfx::Texture* GetTexture( int index );
    nn::gfx::TextureInfo* GetTextureInfo( int index );
    nn::gfx::TextureView* GetTextureView( int index );
    nn::gfx::DescriptorSlot& GetDescriptorSlot( int index );
};

extern DEMOGfxMemPool* DEMOGfxSharedPool; /// pool of memory that may be accessed by both the CPU and GPU
extern DEMOGfxMemPool* DEMOGfxGpuPool;    /// pool of memory that may be accessed only by the GPU

/// \brief GPU Timestamp Abstraction
class DEMOGfxGpuTimestamp
{
private:
    DEMOGfxMemPool* pMemPool;
    nn::gfx::Buffer buffer;
    nn::gfx::GpuAddress address;

public:
    void Initialize();
    void Finalize();
    void QueryTimestamp( nn::gfx::CommandBuffer* pCmd = &DEMOCommandBuffer );
    uint64_t GetTimestampResult( nn::gfx::Queue* pQueue = &DEMOQueue, nn::gfx::CommandBuffer* pCmd = &DEMOCommandBuffer );
    static float TicksToMicroseconds( uint64_t tick );

#if NN_GFX_IS_TARGET_GL
    friend void DEMOGfxTimestampQueryCallback( const void* pParam );
    friend void DEMOGfxTimestampGetResultCallback( const void* pParam );
#endif
};

/// \brief Struct for models
///

////////////////////////////////////////////////////
//
//Model file data structure
//
//-----------------------------------
//ModelHeader   header
//JointData     joints         [header.jointCount]
//AttributeData attributes     [header.AttributeCount]
//TextureData   textures       [header.TextureCount]
//MaterialData  materials      [header.MaterialCount]
//MeshData      meshes         [header.MeshCount]
//u8            attributeBuffer[header.AttributeBufferSize]
//u8            indexBuffer    [header.IndexBufferSize]
//-----------------------------------
//
////////////////////////////////////////////////////

// Model data magic number
static const int DEMO_MODEL_DATA_MAGIC = 0x004d4f44;

// Model data version
static const int DEMO_MODEL_DATA_VERSION = 0x0001;

// Maximum length of texture name
static const int DEMO_MAX_TEXTURE_NAME = 64;

// Texture data
typedef struct DemoTextureData_
{
    char name[DEMO_MAX_TEXTURE_NAME];

} DemoTextureData;

// Attribute types
typedef enum
{
    DEMO_ATTRIBUTE_POSITION,      // Local coordinates
    DEMO_ATTRIBUTE_NORMAL,        // Normal vectors
    DEMO_ATTRIBUTE_TEXCOORD,      // Texture coordinates
    DEMO_ATTRIBUTE_SKIN_WEIGHTS,  // Skinning matrix weights
    DEMO_ATTRIBUTE_SKIN_INDICES,  // Skinning matrix indices

} DemoAttributeType;

// Joint data
typedef struct DemoJointData_
{
    s32   parentIndex;       // Parent joint indices
    Mtx44 basePoseInverse;   // Base pose inverse matrix
    Mtx44 basePoseLocal;     // Base pose local matrix

} DemoJointData;

// Attribute data
typedef struct DemoAttributeData_
{
    DemoAttributeType   type;    // Attribute types
    u32                offset;  // Byte offset to interleaved attributes
    nn::gfx::AttributeFormat format;  // Attribute format

} DemoAttributeData;

// Material type
typedef enum
{
    DEMO_MATERIAL_OPAQUE,            // Opaque
    DEMO_MATERIAL_ALPHA_TO_COVERAGE, // Alpha to coverage
    DEMO_MATERIAL_OVERLAP,           // Translucent without Z-write
    DEMO_MATERIAL_TRANSPARENT,       // Translucent

} DemoMaterialType;

// Material data
typedef struct DemoMaterialData_
{
    DemoMaterialType type;       // Material type
    s32 textureIndex;        // Texture index

} DemoMaterialData;

// Mesh data
typedef struct DemoMeshData_
{
    s32 materialIndex;        // Material index
    u32 indexOffset;          // Byte offset of index data
    u32 indexCount;           // Number of indices
    u32 attributeOffset;      // Byte offset of attribute buffer
    u32 attributeStride;      // Bytes per attribute-buffer vertex

} DemoMeshData;

// Model header
typedef struct DemoModelHeader_
{
    u32  magic;               // Model data magic number (=MODEL_DATA_MAGIC)
    u32  version;             // Model data version (=MODE_DATA_VERSION)
    u32  jointCount;          // Number of joints
    s32  attributeCount;      // Number of attributes
    s32  textureCount;        // Texture count
    s32  materialCount;       // Number of materials
    s32  meshCount;           // Number of meshes
    u32  attributeBufferSize; // Byte size of attribute buffer
    u32  indexBufferSize;     // Byte size of index buffer
} DemoModelHeader;

typedef struct DemoBufferData_
{
    void* data;
    size_t size;
    size_t stride;
    size_t elementCount;
    DEMOGfxBuffer buffer;
} DemoBufferData;

// Model data
typedef struct DemoModelData_
{
    DemoModelHeader    header;     // Model data header

    DemoJointData*     joints;     // Joint data
    DemoAttributeData* attributes; // Attribute data
    DEMOGfxTexture*    pTextures;  // Textures
    DemoMaterialData*  materials;  // Material data
    DemoMeshData*      meshes;     // Mesh data

    DemoBufferData      attributeBuffer;      // Attribute buffer
    DemoBufferData      indexBuffer;          // Index buffer
} DemoModelData;
/// \brief Memory allocator that allocates MEM1 for graphics
///
/// This function calls \ref GX2NotifyMemAlloc which enables
/// various (future) debugging features.
void *DEMOGfxAllocMEM1(size_t size, size_t align);

/// \brief Memory allocator that allocates from the foreground bucket for graphics
///
/// This function calls \ref GX2NotifyMemAlloc which enables
/// various (future) debugging features.
void *DEMOGfxAllocBucket(size_t size, size_t align);

/// \brief Memory allocator that allocates MEM2 for graphics
///
/// This function calls \ref GX2NotifyMemAlloc which enables
/// various (future) debugging features.
void *DEMOGfxAllocMEM2(size_t size, size_t align);

/// \brief Free MEM1
///
void DEMOGfxFreeMEM1(void * ptr);

/// \brief Free memory from the foreground bucket.
///
void DEMOGfxFreeBucket(void * ptr);

/// \brief Free MEM2
///
void DEMOGfxFreeMEM2(void * ptr);

/// \brief Initializes GX2 & AVM; sets up default render/depth/scan buffers.
///
/// It must be called before calling any other GX2 functions.\n
/// The following arguments can be specified:\n
///
///  -DEMO_WIDTH=n       (set TV RT width to n) \n
///  -DEMO_HEIGHT=n      (set TV RT height to n) \n
///  -DEMO_CB_FORMAT=str (set TV color buffer format to 8_8_8_8, 10_10_10_2, 2_10_10_10, 16_16_16_16F, or 32_32_32_32F) \n
///  -DEMO_SCAN_FORMAT=str (set TV scan-out buffer format to 8_8_8_8, 10_10_10_2, or 2_10_10_10) \n
///  -DEMO_DB_FORMAT=str (set TV depth buffer format to 16, 32F, 8_24, X24_8_32F) \n
///  -DEMO_AA_MODE=n     (set TV AA mode to 0/1/2/3) \n
///  -DEMO_FONT_DISABLE  (disables drawing of DEMOFont API)\n
///
/// Set up defaults for all options are (1280/720/8_8_8_8/32F/no AA).\n
/// Unknown args are ignored, and the arg list is left unchanged.\n
///
/// To configure the DRC, please see \ref DEMODRCInit.
///
/// This function will call DEMOGfxMem1HeapInit(), which will allocate all available memory from the MEM1
/// frame heap and initialize the expanded heap used by DEMOGfxAllocMEM1().
///
/// \param argc number of arguments
/// \param argv argument values
///
/// \retval A pointer to the instance created by this function
bool DEMOGfxInit(int argc, char *argv[]);

/// \brief Releases allocated buffers and shuts down GX2.
///
void DEMOGfxShutdown(void);

/// \brief Releases foreground only resources and GX2.
///
void DEMOGfxReleaseForeground(void);

/// \brief Re-acquires GX2 resources and prepares for rendering.
///
void DEMOGfxAcquiredForeground(void);

/// \brief Get demo graphics running state
///
/// \retval TRUE if \ref DEMOGfxInit() has been called; false otherwise.
BOOL DEMOGfxIsRunning(void);

/// \brief Currently, this function waits for the previously rendered frame to appear.
/// Functionality may change in the future.
///
void DEMOGfxBeforeRender(void);

/// \brief Swaps the video (scan) buffers, requests new frame to be displayed.
/// Note that it does not wait for the new frame to appear.
///
void DEMOGfxDoneRender(void);

/// \brief Wrapper for GX2DrawDone that works in the background and if
/// DEMOGfx has not been initialized.
///
void DEMOGfxDrawDone(void);

/// \brief Synchronize rendering with video display/swap in a specific way
///
/// This is a very important function that every app should include in some fashion.
/// It prevents the CPU and GPU from running too far ahead of the video display.
/// This function shows an example of how to specify a precise amount of
/// latency with respect to when video swap occurs.
///
/// \param depth Number of frames rendering may run ahead of display
/// \param percent Additional fraction of a frame to wait after specified swap
///                (expressed as an integer 0...100 percent of the swap interval time)
void DEMOGfxWaitForSwap(u32 depth, u32 percent);

/// \brief Returns the color target view for the current scan buffer
nn::gfx::ColorTargetView* DEMOGetColorBufferView();

/// \brief Returns the texture for the current scan buffer
nn::gfx::Texture* DEMOGetColorBuffer();

/// \brief Returns a string with the name of a given GX2 attrib format.
///
const char *DEMOGfxGetAttribFormatName(nn::gfx::AttributeFormat format);

/// \brief Returns a string with the name of a given GX2 surface format.
///
const char *DEMOGfxGetSurfaceFormatName(nn::gfx::ImageFormat format);

/// \brief A helper function for adding debugging tags into command stream.
///
void DEMOGfxDebugTagIndent( const char* formatString, ... );

void DEMOGfxDebugTagUndent();

void DEMOGfxDebugTagComment( const char* formatString, ... );

/// \brief A helper function to reset render targets to DEMO buffers.
///
void DEMOGfxSetDefaultRenderTarget();

/// \brief A helper function to set the default viewport/scissor state
///
void DEMOGfxSetDefaultViewportScissor();

/// \brief Initializes the expanded heap for MEM1
///
/// This function will allocate all available memory from the MEM1 frame heap and
/// initialize the expanded heap used by DEMOGfxAllocMEM1().
/// This is normally called by DEMOGfxInit().
/// A real game shouldn't use the expanded heap to manage MEM1, but for demos it should be fine.
void DEMOGfxMem1HeapInit(void);

/// \brief Destroys the expanded heap for MEM1
///
/// This function will destroy the expanded heap used by DEMOGfxAllocMEM1().
/// This is normally called by DEMOGfxShutdown().
void DEMOGfxMem1HeapDestroy(void);

/// \brief Initializes the expanded heap for the foreground bucket
///
/// This function will initialize the expanded heap used by DEMOGfxAllocBucket().
/// This is normally called by DEMOGfxInit().
void DEMOGfxBucketHeapInit(void);

/// \brief Destroys the expanded heap for the foreground bucket
///
/// This function will destroy the expanded heap used by DEMOGfxAllocBucket().
/// This is normally called by DEMOGfxShutdown().
void DEMOGfxBucketHeapDestroy(void);

/// \brief Appends the file name "name" to the asset directory to return a full path
/// The path name will have to be freed later by "free".
///
/// \param name File string name to use
///
char* DEMOGfxAssetFileName(const char *name);

/// \brief Reads an asset from the asset directory into an automatically created and allocated buffer.
///
/// File is automatically closed after it's read from.
/// Call \ref DEMOFree() to free this buffer.
///
/// \param path File string name to read (full path is \<asset_dir>\<path>)
/// \param len Length of file returned here
///
void* DEMOGfxLoadAssetFile(const char* path, u32* len);

/// \brief Open an asset file for reading.
///
/// Opens a file for reading.
///
/// The DEMOGfxOpenAssetFile function returns one of 4 error codes.
///
/// \param path Pointer to file name to open
/// \param info Pointer to file information to use
///
/// \retval DEMO_FS_RESULT_OK if it is ok.
/// \retval DEMO_FS_RESULT_PERMISSION if it is not permitted to open current file.
/// \retval DEMO_FS_RESULT_CORRUPTION if a file is corrupted.
/// \retval DEMO_FS_FATAL_ERROR if a fatal error occurs during a read, the function returns.
s32 DEMOGfxOpenAssetFile(const char* path, DEMOFSFileInfo* info);

/// \brief Gets the size of the opened asset file.
///
/// Gets in advance the size of a file opened with the demoFSGetLength function.
///
/// This function is actually a wrapper for DEMOFSGetLength.
///
/// \param fileInfo File information of the file
/// \param length The file size. As a return value, this size may not always be a multiple of 32. For this reason, when preparing buffers for reading files, this value must be rounded up to a multiple of 32
///
/// \retval DEMO_FS_RESULT_OK if it is successed.
/// \retval DEMO_FS_RESULT_FATAL_ERROR if a fatal error occurs during a read, the function returns.
s32 DEMOGfxGetAssetFileLength(const DEMOFSFileInfo* fileInfo, u32* length);

/// \brief Synchronously reads from the opened asset file.
///
/// Synchronously reads data from a file. First opens the file with the DEMOGfxOpenAssetFile function.
///
/// This function is actually a wrapper for DEMOFSRead.
///
/// \param fileInfo File information of the file.
/// \param addr Buffer address. Must be \c PPC_IO_BUFFER_ALIGN byte aligned.
/// \param length Number of bytes to read. Must be a multiple of 32.
/// \param offset File position to start reading from. Must be a multiple of 4.
///
/// \retval DEMO_FS_RESULT_OK if it is ok.
/// \retval DEMO_FS_RESULT_ACCESS if it is no right to access file.
/// \retval DEMO_FS_RESULT_CORRUPTION if a file is corrupted.
/// \retval DEMO_FS_FATAL_ERROR if a fatal error occurs during a read, the function returns.
s32 DEMOGfxReadAssetFile(DEMOFSFileInfo* fileInfo, void* addr, s32 length, s32 offset);

/// \brief Closes the asset file.
///
/// Closes the specified file.
/// if the FS library is not expecting a call (for example, during transfer) to the demoFSCloseFile function, the call to the function takes time.
/// Ensure that this situation doesn't occur.
/// (Namely, when calling demoFSCloseFile, make sure the transfer is complete.)
///
/// This function is actually a wrapper for DEMOFSClose.
///
/// \param fileInfo File information for the file to close.
///
/// \retval DEMO_FS_RESULT_OK if it is ok.
/// \retval DEMO_FS_RESULT_CORRUPTION if a file is corrupted.
/// \retval DEMO_FS_FATAL_ERROR if a fatal error occurs during a read, the function returns.
s32 DEMOGfxCloseAssetFile(DEMOFSFileInfo* fileInfo);

/// \brief Reads an asset from the asset directory into an automatically created and allocated buffer.
///
/// File is automatically closed after it's read from.
/// Call \ref DEMOFree() to free this buffer.
///
/// \param path File string name to read (full path is \<asset_dir\>\<path\>)
/// \param len Length of file returned here
///
void* DEMOGfxLoadAssetFile(const char* path, u32* len);

/// \brief Reads an asset from the asset directory into an automatically created and allocated buffer.
///
/// This buffer is aligned with alignSize.
/// File is automatically closed after it's read from.
/// Call \ref DEMOFree() to free this buffer.
///
/// \param path File string name to read (full path is \<asset_dir\>\<path\>)
/// \param len Length of file returned here
/// \param alignSize The alignment to use
///
void* DEMOGfxLoadAssetFileAlign(const char* path, u32* len, u32 alignSize);

/// \brief Reads a model data file from the asset directory into an automatically created and allocated buffer.
///
/// File is automatically closed after it's read from.
/// Little endian systems byte swap the data after reading it.
/// Call \ref DEMOFree() to free this buffer.
///
/// \param path File string name to read (full path is \<asset_dir\>\<path\>)
/// \param len Length of file returned here
///
void* DEMOGfxLoadModelFile( const char* path, u32* len );

/// \brief Scans directory for all the assets in it.
///
/// This returns a list of the names of all the files in a directory.  Subdirectories
/// are automatically excluded.  Pathnames are returned, suitable for passing directly
/// to DEMOGfxOpenAssetFile() or DEMOGfxLoadAssetFile().
///
/// \param path        Directory name
/// \param pFileCount  Number of files in the directory
/// \param maxFiles    Maximum number of file names to return from this function
/// \param ppFileNames Returned array of names of files.  The base array must be specified
///                    by the caller. (e.g char *ppFileNames[maxFiles]; )
///                    Each non null string is allocated using DEMOAlloc() in this routine.
///                    Call DEMOFree() in caller to free them.
/// \return DEMO_FS_RESULT_OK on success, or error value.
s32 DEMOGfxScanAssetDir(const char *path, u32 *pFileCount, u32 maxFiles, char** ppFileNames);

/// \brief Helper to initialize an nn::gfx::Buffer from a DEMOGfxMemPool
///
/// \param pBuffer Buffer to initialize
/// \param pView BufferView to initialize
/// \param gpuAccessFlags nn::gfx::GpuAccess flags for the buffer
/// \param data Initialization data. This may be used as the backing GPU memory in some instances.
/// \param size Size of the buffer;
/// \return returns true if initialization went correctly
///
bool DEMOGfxInitBuffer(DEMOGfxMemPool* pool, nn::gfx::Buffer* pBuffer, int gpuAccessFlags, void* data, size_t size);

/// \brief Helper to create a fresh memory pool and initialize nn::gfx::Buffer
///
/// \param pBuffer Buffer to initialize
/// \param pView BufferView to initialize
/// \param gpuAccessFlags nn::gfx::GpuAccess flags for the buffer
/// \param data Initialization data. This may be used as the backing GPU memory in some instances.
/// \param size Size of the buffer;
/// \return returns the memory pool associated with the buffer if one is necessary, otherwise NULL
///
DEMOGfxMemPool* DEMOGfxCreateBuffer( nn::gfx::Buffer* pBuffer, int gpuAccessFlags, void* data, size_t size );

/// \brief Helper to initialize nn::gfx::ViewportScissorState for single viewport/scissor use cases.
///
/// \param pState ViewportScissorState to initialize
/// \param ppData Pointer to data address passed into ViewportScissorState::SetMemory
/// \param xOrig X Origin
/// \param yOrig Y Origin
/// \param width Width of the viewport/scissor
/// \param height Height of the viewport/scissor
/// \param minZ Z near value
/// \param maxZ Z far value
/// \param renderTargetHeight Height of the destination render target.
/// \param isTargetTexure set if the destinatation is a render from texture target
///
void DEMOGfxSetViewportScissorState( nn::gfx::ViewportScissorState* pState, void** ppData,
    float xOrig, float yOrig, float width, float height, float minZ, float maxZ,
    float renderTargetHeight, bool isTargetTexture );

/// \brief Setup nn::gfx::TextureView from a given nn::gfx::Texture
///
/// \param pTexture Pointer to nn::gfx::Texture used to create the view.
/// \param pTextureView  Pointer to the nn::gfx::TextureView that will be initialized.
/// \param dimensions ImageDimension of the texture.
/// \param format ImageFormat of the texture
/// \param fetchMode nn::gfx::DepthStencilFetchMode of the texture
///
void DEMOGfxSetupTextureView( nn::gfx::Texture* pTexture,
    nn::gfx::TextureView* pTextureView,
    nn::gfx::DescriptorSlot* pDescriptorSlot,
    nn::gfx::ImageDimension dimensions,
    nn::gfx::ImageFormat format,
    nn::gfx::DepthStencilFetchMode fetchMode );

/// \brief Setup nn::gfx::ColorTargetView from a given nn::gfx::Texture
///
/// \param pTexture Pointer to nn::gfx::Texture used to create the view.
/// \param pColorView  Pointer to the nn::gfx::ColorTargetView that will be initialized.
/// \param dimensions ImageDimension of the texture.
/// \param format ImageFormat of the texture
///
void DEMOGfxSetupColorView( nn::gfx::Texture* pTexture,
    nn::gfx::ColorTargetView* pColorView,
    nn::gfx::ImageDimension dimensions,
    nn::gfx::ImageFormat format );

/// \brief Setup nn::gfx::DepthStencilView from a given nn::gfx::Texture
///
/// \param pTexture Pointer to nn::gfx::Texture used to create the view.
/// \param pDepthView  Pointer to the nn::gfx::DepthStencilView that will be initialized.
/// \param dimensions ImageDimension of the texture.
/// \param fetchMode DepthStencilFetchMode.
///
void DEMOGfxSetupDepthView( nn::gfx::Texture* pTexture,
    nn::gfx::DepthStencilView* pDepthView,
    nn::gfx::ImageDimension dimensions );

/// \brief Setup nn::gfx::Texture and the various views.
///
/// \param pTexture Pointer to nn::gfx::Texture used to create the view.
/// \param pTextureView  Pointer to the nn::gfx::TextureView that will be initialized.
/// \param pColorView  Pointer to the nn::gfx::ColorTargetView that will be initialized.
/// \param pDepthView  Pointer to the nn::gfx::DepthStencilView that will be initialized.
/// \param ppMemPool Pointer to the memory pool created for the texture. If this is passed in as NULL a new one will be allocated.
/// \param width Width of the texture
/// \param height Height of the texture
/// \param depth Depth of the texture
/// \param mipCount Mipmap count of the texture
/// \param dimensions ImageDimension of the texture.
/// \param format ImageFormat of the texture
/// \param swizzle nn::gfx::Texture swizzle value.
///
void DEMOGfxSetupTextureBuffer( nn::gfx::Texture* pTexture,
    nn::gfx::TextureView* pTextureView,
    nn::gfx::DescriptorSlot* pDescriptorSlot,
    nn::gfx::ColorTargetView* pColorView,
    nn::gfx::DepthStencilView* pDepthView,
    DEMOGfxMemPool **ppData,
    int width,
    int height,
    int depth,
    int mipCount,
    nn::gfx::ImageDimension dimensions,
    nn::gfx::ImageFormat format,
    nn::gfx::DepthStencilFetchMode fetchMode,
    int swizzle );

/// \brief Wrapper for initialize an nn::gfx::Sampler
///
/// \param pSampler nn::gfx::Sampler to initialize
/// \param addressMode nn::gfx::TextureAddressMode to use for the sampler
/// \param filterMode nn::gfx::FilterMode to use for the sampler
/// \param compareFunction nn::gfx::ComparisonFunction to use for the sampler
///
void DEMOGfxInitSampler( nn::gfx::Sampler* pSampler,
    nn::gfx::DescriptorSlot* pDescriptorSlot,
    nn::gfx::TextureAddressMode addressMode,
    nn::gfx::FilterMode filterMode,
    nn::gfx::ComparisonFunction compareFunction );

/// \brief swap 32 bit data in a buffer
void DEMOSwapBuffer32(void *bufp, size_t len);

/// \brief swap 16 bit data in a buffer
void DEMOSwapBuffer16(void *bufp, size_t len);

/// \brief Load a model from a file
///
/// \param pModelData the model we will load
/// \param pModelFilePath string pointing to the model file
/// \param pTextureDirectoryPath string giving the texture which contains all textures for the model
///
void DEMOLoadModelData(DemoModelData* pModelData,
    const char* pModelFilePath,
    const char* pTextureDirectoryPath);

/// \brief Free memory for a model loaded by DEMOLoadModelData
void DEMOFreeModelData(DemoModelData* pModelData);

nn::gfx::ImageStorageDimension DEMOConvertToStorageDimension(nn::gfx::ImageDimension imDim);

/// @}  // demoGfx
