﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// -----------------------------------------------------------------------------
//  demoFont.h
//
// -----------------------------------------------------------------------------

#pragma once

#include <gfx/demo.h>

/// @addtogroup demoFont
/// @{

/// \brief Structure used to define the font data for an instance of DEMOFont
typedef struct
{
    /// \brief Mag Scale X,Y of characters
    f32 *charMagScale;

    /// \brief Mag Scale X,Y of characters (proportional)
    f32 charMagScaleF[2]; // xxx should it be [4]?

    /// \brief Mag Scale X,Y of characters (fixed)
    f32 charMagScaleP[2]; // xxx should it be [4]?

    /// \brief Depth value
    f32 depth;

    /// \brief Color
    f32 color[4];

} DEMOFontFontData;

/// \brief Structure used to define an instance of DEMOFont
typedef struct
{
    /// \brief Graphics pipeline for DEMOFont (instance)
    nn::gfx::Pipeline pipeline;

    /// \brief The font data
    DEMOFontFontData font;

    /// \brief TRUE if the font is currently proportional
    BOOL proportional;

    /// \brief TRUE if fonts are currently enabled
    BOOL enabled;

} DEMOFontInstance;

/// \brief Initializes the font data and font shader program
///
/// Sets up demo font context state and font texture object.
/// Creates the font shader program, Sets up font initial values.
//  Sets up initial grid = 60x24(proportional), 80x24(mono-spacing), color = (1.0f,1.0f,1.0f,1.0f), zValue=-1.0f, proportional=TRUE.
/// At default, DEMOFont uses DEMOColorBuffer and viewport which is created in DEMOGfxInit.
/// If you want to change the buffer, need to call DEMOFontSetTarget after init.
///
/// \retval A pointer to the instance created by this function
DEMOFontInstance* DEMOFontInit(void);

/// \brief Deletes all the font data
void DEMOFontShutdown(void);

/// \brief Lets the DEMOFont system know a new frame has started
/// This is mostly for memory management
void DEMOFontNewFrame(void);

/// \brief Enable/disable fonts from drawing
///
/// \param enable If true, fonts will draw. If false, they will not draw.
void DEMOFontDrawEnable(BOOL enable);

/// \brief Output the string into Screen by character units step
///
/// Need to call this API after DEMOFontSetContextState.
///
/// \param column Column position of the string by character units step (0 is left side of Screen)
/// \param line Line position of the string by character units step (0 is the top of Screen)
/// \param pFmt Pointer to a null-terminated string including format specification (equivalent to C's standard output function)
void DEMOFontPrintf(f32 column, f32 line, const char* pFmt, ... );

/// \brief Output the pre-formatted string into Screen by character units step
///
/// Need to call this API after DEMOFontSetContextState.
///
/// \param column Column position of the string by character units step (0 is left side of Screen)
/// \param line Line position of the string by character units step (0 is the top of Screen)
/// \param pStr Pointer to a null-terminated string
void DEMOFontPuts(f32 column, f32 line, const char* pStr);

/// \brief Set the font color
///
/// \param r The red color for the font
/// \param g The green color for the font
/// \param b The blue color for the font
/// \param a The alpha color for the font
void DEMOFontSetColor(f32 r, f32 g, f32 b, f32 a);

/// \brief Set the grid size
///
/// \param xGrid The number of fixed-width columns
///  \param yGrid The number of fixed-width lines
void DEMOFontSetGridSize(f32 xGrid, f32 yGrid);

/// \brief Set the font spacing type
///
/// \param proportional True means to use proportional spacing, false means mono spacing
void DEMOFontSetSpacing(BOOL proportional);

/// \brief Set the Z value
///
///    \param zValue  -1.0 = near plane, 1.0 = far plane
void DEMOFontSetZValue(f32 zValue);

/// \brief Get the maximum texel size of the character.
///
/// \param pCharWidth Pointer to the width texel size of character cell
/// \param pCharHeight Pointer to the height texel size of character cell
void DEMOFontGetCharSize(f32 *pCharWidth, f32 *pCharHeight);

/// \brief Creates a new instance of DEMOFont
///
/// This will create a new instance, in addition to setting the instance.
/// This is called by DEMOFontInit().
DEMOFontInstance* DEMOFontAddInstance(void);

/// \brief Deletes an instance of DEMOFont
///
/// This is called by DEMOFontShutdown().
void DEMOFontDeleteInstance(DEMOFontInstance *instance);

/// @}
