﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// -----------------------------------------------------------------------------
//  demoCapture.h
//
// -----------------------------------------------------------------------------
#pragma once

#include <gfx/demo.h>

/// @addtogroup demoCapture
/// @{

/// \brief supported demo capture format
typedef enum _DEMOCaptureFormat
{
    I8,
    I16,
    RGBA8,
    SRGB8,
    DEPTH16,
    DEPTH32,
} DEMOCaptureFormat;

/// \brief Data for demo capture
typedef struct _DEMOCaptureDataStore {
    DEMOCaptureFormat captureFormat;
    struct {
        bool initialized;
        u32 width;
        u32 height;
        u32 pitch;
        nn::gfx::ImageFormat format;
        size_t size;
        DEMOGfxBuffer buffer;
    } colorBuffer;
    u8 *TGAData;
    u32 TGALength;
} DEMOCaptureDataStore;

extern DEMOCaptureDataStore DEMOCaptureData;

/// \brief Sets up surface for demo capture.
///
/// The capture API allows you to copy a surface to the capture surface.
/// This function must be called first to set up that surface.
/// Various capture formats are supported (see \ref DEMOCaptureFormat).
/// Can be called as necessary to reconfigure.
///
/// Format can be I8, I16, RGBA8, or SRGB8.
/// For I8 & I16, capture always grabs the first (R) component of source.
/// Number format is always UNORM.
///
/// \param width   width of capture surface
/// \param height  height of capture surface
/// \param format  format for capture surface & file
void DEMOCaptureInit(u32 width, u32 height, DEMOCaptureFormat format);

/// \brief Frees resources used for demo capture.
///
void DEMOCaptureShutdown(void);

/// \brief Copies the specified src surface src to the dst file.
///
/// If dstFilePath == NULL, capture to buffer only.
/// If dstFilePath == "", write captured image to serial output.
/// If srcBuffer == NULL, use a previously captured image.
///
/// If the pathname begins with '/', it is used as an absolute path name.
/// If it does not, then "/vol/content/" is prepended to the path name
/// (this is done by DEMOFS).
///
/// \note If /vol/content has been remapped to use DVDFS, then you may not
///       open files there in "w" (write) mode.  Use /vol/save instead.
///
/// \note For serial output, data is written using UUENCODE format.
///       You must uudecode the captured output to convert to binary.
///       (Install cygwin uudecode.exe from sharutils package.)
///       This is done to overcome serial capture limitations.
///       See http://en.wikipedia.org/wiki/Uuencode for details.
///
/// \note You must call \ref DEMOCaptureInit before calling this function.
///
/// The output file is in TGA format with RLE compression.
/// - I8 format = 8-bit grayscale TGA
/// - I16 format = 16-bit grayscale TGA
/// - RGBA8 = 32-bit full color with alpha TGA
/// - SRGB8 = 32-bit non-linear color with alpha TGA<p>
/// See http://en.wikipedia.org/wiki/Truevision_TGA for more details.
///
/// GX2 tests/demos in the SDK can capture an image using the following make command,
/// where # is the frame number you wish to capture:
/// \verbatim make run RUN_ARGS="CAPTURE_SELECT=#" \endverbatim
/// The image is saved as: $CAFE_SAVE_DIR/common/GX2Capture.tga
///
/// \param srcBuffer  Pointer to buffer to capture (typically DEMOColorBuffer)
/// \param dstFilePath  Path name to capture file to write, or NULL for serial output
///
void DEMOCaptureCopy(nn::gfx::Texture *srcBuffer, const char *dstFilePath);

/// @} // demoCapture
