﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testG3d_ScopedMemoryLeakDetector.h"
#include "testG3d_ViewerTestAllocator.h"
#include "testG3d_FileIoUtility.h"
#include "testG3d_ViewerTestUtility.h"

#include "nnt/g3d/testG3d_TestUtility.h"

#include "util\g3d_Map.h"
#include "util\g3d_SynchronizedDynamicPtrArray.h"
#include "util\g3d_DynamicArray.h"
#include "util\g3d_DynamicRingBuffer.h"
#include "g3d_ViewerKeyManager.h"

#include <nnt/nntest.h>

using namespace nn::g3d;
using namespace nn::g3d::viewer;
using namespace nn::g3d::viewer::detail;
using namespace G3dTest;

TEST(G3dViewerUtilityTest, ViewerTestAllocatorTest)
{
    TestFuncPrePostProcessHandler prePostProcessHandler(__FUNCTION__);
    G3dTest::ScopedMemoryLeakDetector memoryLeakDetector;
    G3dTest::ViewerTestAllocator allocator;

    void* pBuffer = allocator.Allocate(50000, 8192);
    allocator.Free(pBuffer);
}

TEST(G3dViewerUtilityTest, DynamicPtrArrayIterator)
{
    TestFuncPrePostProcessHandler prePostProcessHandler(__FUNCTION__);
    G3dTest::ViewerTestAllocator testAlocator;
    nn::g3d::viewer::detail::Allocator allocator(
        ViewerTestAllocator::AllocateWithUserData, ViewerTestAllocator::FreeWithUserData, &testAlocator, &testAlocator);
    nn::g3d::viewer::detail::SynchronizedDynamicPtrArray<int> ptrArray(&allocator, nn::DefaultAlignment);

    // 要素 0 個でイテレートをしても問題がないかテスト
    for (nn::g3d::viewer::detail::Iter<int*> iter = ptrArray.Begin(), end = ptrArray.End(); iter != end; ++iter);

    int testData[] = {1, 2, 3, 4, 5};
    int testDataCount = static_cast<int>(sizeof(testData) / sizeof(int));
    for (int i = 0; i < testDataCount; ++i)
    {
        ptrArray.PushBack(&testData[i]);
    }

    EXPECT_EQ(3, ptrArray.IndexOf(&testData[3]));

    // ++iter でイテレートできるかテスト
    {
        int i = 0;
        for (nn::g3d::viewer::detail::Iter<int*> iter = ptrArray.Begin(), end = ptrArray.End(); iter != end; ++iter, ++i)
        {
            int* pElem = *iter;
            EXPECT_EQ(testData[i], *pElem);
        }
    }

    // iter++ でイテレートできるかテスト
    {
        int i = 0;
        for (nn::g3d::viewer::detail::Iter<int*> iter = ptrArray.Begin(), end = ptrArray.End(); iter != end; iter++, ++i)
        {
            int* pElem = *iter;
            EXPECT_EQ(testData[i], *pElem);
        }
    }

    // 要素の削除テスト
    {
        int currentCount = ptrArray.GetCount();
        for (nn::g3d::viewer::detail::Iter<int*> iter = ptrArray.Begin(), end = ptrArray.End(); iter != end; ++iter)
        {
            iter = ptrArray.Erase(iter);
            --currentCount;
            EXPECT_EQ(currentCount, ptrArray.GetCount());
        }

        EXPECT_EQ(0, ptrArray.GetCount());
    }
}


TEST(G3dViewerUtilityTest, DynamicArrayIterator)
{
    TestFuncPrePostProcessHandler prePostProcessHandler(__FUNCTION__);
    G3dTest::ViewerTestAllocator testAlocator;
    nn::g3d::viewer::detail::Allocator allocator(
        ViewerTestAllocator::AllocateWithUserData, ViewerTestAllocator::FreeWithUserData, &testAlocator, &testAlocator);
    nn::g3d::viewer::detail::DynamicArray<int*> ptrArray(&allocator, nn::DefaultAlignment, NULL);

    // 要素 0 個でイテレートをしても問題がないかテスト
    for (nn::g3d::viewer::detail::Iter<int*> iter = ptrArray.Begin(), end = ptrArray.End(); iter != end; ++iter);

    int testData[] = { 1, 2, 3, 4, 5 };
    int testDataCount = static_cast<int>(sizeof(testData) / sizeof(int));
    for (int i = 0; i < testDataCount; ++i)
    {
        ptrArray.PushBack(&testData[i]);
    }
    EXPECT_EQ(testDataCount, ptrArray.GetCount());

    for (int i = 0; i < testDataCount; ++i)
    {
        ptrArray.PopBack();
    }
    EXPECT_EQ(0, ptrArray.GetCount());

    for (int i = 0; i < testDataCount; ++i)
    {
        ptrArray.PushBack(&testData[i]);
    }
    EXPECT_EQ(testDataCount, ptrArray.GetCount());

    // ++iter でイテレートできるかテスト
    {
        int i = 0;
        for (nn::g3d::viewer::detail::Iter<int*> iter = ptrArray.Begin(), end = ptrArray.End(); iter != end; ++iter, ++i)
        {
            int* pElem = *iter;
            EXPECT_EQ(testData[i], *pElem);
        }
    }

    // iter++ でイテレートできるかテスト
    {
        int i = 0;
        for (nn::g3d::viewer::detail::Iter<int*> iter = ptrArray.Begin(), end = ptrArray.End(); iter != end; iter++, ++i)
        {
            int* pElem = *iter;
            EXPECT_EQ(testData[i], *pElem);
        }
    }

    // 要素の削除テスト
    {
        int currentCount = ptrArray.GetCount();
        for (nn::g3d::viewer::detail::Iter<int*> iter = ptrArray.Begin(), end = ptrArray.End(); iter != end; ++iter)
        {
            iter = ptrArray.Erase(iter);
            --currentCount;
            EXPECT_EQ(currentCount, ptrArray.GetCount());
        }

        EXPECT_EQ(0, ptrArray.GetCount());
    }
}

TEST(G3dViewerUtilityTest, Map)
{
    TestFuncPrePostProcessHandler prePostProcessHandler(__FUNCTION__);
    G3dTest::ViewerTestAllocator testAlocator;
    nn::g3d::viewer::detail::Allocator allocator(
        ViewerTestAllocator::AllocateWithUserData, ViewerTestAllocator::FreeWithUserData, &testAlocator, &testAlocator);
    nn::g3d::viewer::detail::Map<int, int*> map(&allocator, -1, NULL);

    int testData[] = { 1, 2, 3, 4, 5 };

    map.Register(0, &testData[0]);
    map.Register(1, &testData[1]);
    EXPECT_TRUE(map.IsKeyRegistered(0));
    EXPECT_FALSE(map.IsKeyRegistered(2));
    EXPECT_TRUE(map.IsValueRegistered(&testData[1]));
    EXPECT_FALSE(map.IsValueRegistered(&testData[3]));
    EXPECT_EQ(2, map.GetCount());

    map.UpdateValue(1, &testData[2]);
    EXPECT_EQ(&testData[2], map.FindValue(1));

    map.Unregister(0);
    EXPECT_EQ(NULL, map.FindValue(0));
    EXPECT_EQ(1, map.GetCount());
}

TEST(G3dViewerUtilityTest, KeyManager)
{
    TestFuncPrePostProcessHandler prePostProcessHandler(__FUNCTION__);
    G3dTest::ViewerTestAllocator testAlocator;
    nn::g3d::viewer::detail::Allocator allocator(
        ViewerTestAllocator::AllocateWithUserData, ViewerTestAllocator::FreeWithUserData, &testAlocator, &testAlocator);

    EXPECT_FALSE(ViewerKeyManager::IsInitialized());
    ViewerKeyManager::Initialize(&allocator);
    EXPECT_TRUE(ViewerKeyManager::IsInitialized());

    int testData[] = { 1, 2, 3, 4, 5 };

    ViewerKeyType key0 = ViewerKeyManager::GetInstance().Register(&testData[0]);
    ViewerKeyType key1 = ViewerKeyManager::GetInstance().Register(&testData[1]);
    EXPECT_TRUE(ViewerKeyManager::GetInstance().IsKeyRegistered(key0));
    EXPECT_FALSE(ViewerKeyManager::GetInstance().IsKeyRegistered(InvalidKey));
    EXPECT_TRUE(ViewerKeyManager::GetInstance().IsRegistered(&testData[1]));
    EXPECT_FALSE(ViewerKeyManager::GetInstance().IsRegistered(&testData[3]));
    EXPECT_EQ(2, ViewerKeyManager::GetInstance().GetRegisteredKeyCount());

    ViewerKeyManager::GetInstance().UpdateData(key1, &testData[2]);
    EXPECT_EQ(&testData[2], ViewerKeyManager::GetInstance().FindData<int>(key1));

    ViewerKeyManager::GetInstance().Unregister(key0);
    EXPECT_EQ(NULL, ViewerKeyManager::GetInstance().FindData<int>(key0));
    EXPECT_EQ(1, ViewerKeyManager::GetInstance().GetRegisteredKeyCount());

    ViewerKeyManager::Finalize();
    EXPECT_FALSE(ViewerKeyManager::IsInitialized());
}

TEST(G3dViewerUtilityTest, DynamicRingBuffer)
{
    TestFuncPrePostProcessHandler prePostProcessHandler(__FUNCTION__);
    G3dTest::ViewerTestAllocator testAllocator;
    nn::g3d::viewer::detail::Allocator allocator(
        ViewerTestAllocator::AllocateWithUserData,
        ViewerTestAllocator::FreeWithUserData,
        &testAllocator,
        &testAllocator
    );

    nn::g3d::viewer::detail::DynamicRingBuffer<int> dynamicRingBuffer(&allocator);
    const int pushCount = 128;

    {

        // push back
        for (int i = 0; i < pushCount; ++i)
        {
            dynamicRingBuffer.PushBack(i);
        }
        EXPECT_EQ(dynamicRingBuffer.GetCount(), pushCount);
        EXPECT_LE(pushCount, dynamicRingBuffer.GetCountMax());

        // pop front / Get / UnsafeGet
        for(int i = 0; i < pushCount; ++i)
        {
            int value = dynamicRingBuffer[0];
            int* pValue = dynamicRingBuffer.UnsafeGet(0);
            EXPECT_EQ(value, *pValue);

            value = dynamicRingBuffer.PopFront();
            EXPECT_EQ(value, *pValue);
            EXPECT_EQ(i, value);
        }
        EXPECT_EQ(dynamicRingBuffer.GetCount(), 0);
    }

    // clear
    {
        dynamicRingBuffer.PushBack(1);
        EXPECT_EQ(dynamicRingBuffer.GetCount(), 1);
        dynamicRingBuffer.Clear();
        EXPECT_EQ(dynamicRingBuffer.GetCount(), 0);
    }


    {
        for (int i = 0; i < pushCount; ++i)
        {
            dynamicRingBuffer.PushBack(i);
            dynamicRingBuffer.PushFront(i);
        }

        EXPECT_EQ(dynamicRingBuffer.GetCount(), pushCount * 2);

        for (int i = pushCount - 1; i >= 0; --i)
        {
            int value = dynamicRingBuffer.PopBack();
            EXPECT_EQ(value, i);
            value = dynamicRingBuffer.PopFront();
            EXPECT_EQ(value, i);
        }

    }
}
